rray or nil
    #   CSV.parse_line(io, **options) -> new_array or nil
    #   CSV.parse_line(string, headers: true, **options) -> csv_row or nil
    #   CSV.parse_line(io, headers: true, **options) -> csv_row or nil
    #
    # Returns the data created by parsing the first line of +string+ or +io+
    # using the specified +options+.
    #
    # - Argument +string+ should be a \String object;
    #   it will be put into a new StringIO object positioned at the beginning.
    # :include: ../doc/csv/arguments/io.rdoc
    # - Argument +options+: see {Options for Parsing}[#class-CSV-label-Options+for+Parsing]
    #
    # ====== Without Option +headers+
    #
    # Without option +headers+, returns the first row as a new \Array.
    #
    # These examples assume prior execution of:
    #   string = "foo,0\nbar,1\nbaz,2\n"
    #   path = 't.csv'
    #   File.write(path, string)
    #
    # Parse the first line from a \String object:
    #   CSV.parse_line(string) # => ["foo", "0"]
    #
    # Parse the first line from a File object:
    #   File.open(path) do |file|
    #     CSV.parse_line(file) # => ["foo", "0"]
    #   end # => ["foo", "0"]
    #
    # Returns +nil+ if the argument is an empty \String:
    #   CSV.parse_line('') # => nil
    #
    # ====== With Option +headers+
    #
    # With {option +headers+}[#class-CSV-label-Option+headers],
    # returns the first row as a CSV::Row object.
    #
    # These examples assume prior execution of:
    #   string = "Name,Count\nfoo,0\nbar,1\nbaz,2\n"
    #   path = 't.csv'
    #   File.write(path, string)
    #
    # Parse the first line from a \String object:
    #   CSV.parse_line(string, headers: true) # => #<CSV::Row "Name":"foo" "Count":"0">
    #
    # Parse the first line from a File object:
    #   File.open(path) do |file|
    #     CSV.parse_line(file, headers: true)
    #   end # => #<CSV::Row "Name":"foo" "Count":"0">
    #
    # ---
    #
    # Raises an exception if the argument is +nil+:
    #   # Raises ArgumentError (Cannot parse nil as CSV):
    #   CSV.parse_line(nil)
    #
    def parse_line(line, **options)
      new(line, **options).each.first
    end

    #
    # :call-seq:
    #   read(source, **options) -> array_of_arrays
    #   read(source, headers: true, **options) -> csv_table
    #
    # Opens the given +source+ with the given +options+ (see CSV.open),
    # reads the source (see CSV#read), and returns the result,
    # which will be either an \Array of Arrays or a CSV::Table.
    #
    # Without headers:
    #   string = "foo,0\nbar,1\nbaz,2\n"
    #   path = 't.csv'
    #   File.write(path, string)
    #   CSV.read(path) # => [["foo", "0"], ["bar", "1"], ["baz", "2"]]
    #
    # With headers:
    #   string = "Name,Value\nfoo,0\nbar,1\nbaz,2\n"
    #   path = 't.csv'
    #   File.write(path, string)
    #   CSV.read(path, headers: true) # => #<CSV::Table mode:col_or_row row_count:4>
    def read(path, **options)
      open(path, **options) { |csv| csv.read }
    end

    # :call-seq:
    #   CSV.readlines(source, **options)
    #
    # Alias for CSV.read.
    def readlines(path, **options)
      read(path, **options)
    end

    # :call-seq:
    #   CSV.table(source, **options)
    #
    # Calls CSV.read with +source+, +options+, and certain default options:
    # - +headers+: +true+
    # - +converters+: +:numeric+
    # - +header_converters+: +:symbol+
    #
    # Returns a CSV::Table object.
    #
    # Example:
    #   string = "Name,Value\nfoo,0\nbar,1\nbaz,2\n"
    #   path = 't.csv'
    #   File.write(path, string)
    #   CSV.table(path) # => #<CSV::Table mode:col_or_row row_count:4>
    def table(path, **options)
      default_options = {
        headers:           true,
        converters:        :numeric,
        header_converters: :symbol,
      }
      options = default_options.merge(options)
      read(path, **options)
    end
  end

  # :call-seq:
  #   CSV.new(string)
  #   CSV.new(io)
  #   CSV.new(string, **options)
  #   CSV.new(io, **options)
  #
  # Returns the new \CSV object created using +string+ or +io+
  # and the specified +options+.
  #
  # - Argument +string+ should be a \String object;
  #   it will be put into a new StringIO object positioned at the beginning.
  # :include: ../doc/csv/arguments/io.rdoc
  # - Argument +options+: See:
  #   * {Options for Parsing}[#class-CSV-label-Options+for+Parsing]
  #   * {Options for Generating}[#class-CSV-label-Options+for+Generating]
  #   For performance reasons, the options cannot be overridden
  #   in a \CSV object, so those specified here will endure.
  #
  # In addition to the \CSV instance methods, several \IO methods are delegated.
  # See {Delegated Methods}[#class-CSV-label-Delegated+Methods].
  #
  # ---
  #
  # Create a \CSV object from a \String object:
  #   csv = CSV.new('foo,0')
  #   csv # => #<CSV io_type:StringIO encoding:UTF-8 lineno:0 col_sep:"," row_sep:"\n" quote_char:"\"">
  #
  # Create a \CSV object from a \File object:
  #   File.write('t.csv', 'foo,0')
  #   csv = CSV.new(File.open('t.csv'))
  #   csv # => #<CSV io_type:File io_path:"t.csv" encoding:UTF-8 lineno:0 col_sep:"," row_sep:"\n" quote_char:"\"">
  #
  # ---
  #
  # Raises an exception if the argument is +nil+:
  #   # Raises ArgumentError (Cannot parse nil as CSV):
  #   CSV.new(nil)
  #
  def initialize(data,
                 col_sep: ",",
                 row_sep: :auto,
                 quote_char: '"',
                 field_size_limit: nil,
                 max_field_size: nil,
                 converters: nil,
                 unconverted_fields: nil,
                 headers: false,
                 return_headers: false,
                 write_headers: nil,
                 header_converters: nil,
                 skip_blanks: false,
                 force_quotes: false,
                 skip_lines: nil,
                 liberal_parsing: false,
                 internal_encoding: nil,
                 external_encoding: nil,
                 encoding: nil,
                 nil_value: nil,
                 empty_value: "",
                 strip: false,
                 quote_empty: true,
                 write_converters: nil,
                 write_nil_value: nil,
                 write_empty_value: "")
    raise ArgumentError.new("Cannot parse nil as CSV") if data.nil?

    if data.is_a?(String)
      if encoding
        if encoding.is_a?(String)
          data_external_encoding, data_internal_encoding = encoding.split(":", 2)
          if data_internal_encoding
            data = data.encode(data_internal_encoding, data_external_encoding)
          else
            data = data.dup.force_encoding(data_external_encoding)
          end
        else
          data = data.dup.force_encoding(encoding)
        end
      end
      @io = StringIO.new(data)
    else
      @io = data
    end                                                                                                                                                                                                                                                                                                                        ruby/pp.rb                                                                                          0000644                 00000044555 15040313426 0006504 0                                                                                                    ustar 00                                                                                                                                                                                                                                                       # frozen_string_literal: true

require 'prettyprint'

##
# A pretty-printer for Ruby objects.
#
##
# == What PP Does
#
# Standard output by #p returns this:
#   #<PP:0x81fedf0 @genspace=#<Proc:0x81feda0>, @group_queue=#<PrettyPrint::GroupQueue:0x81fed3c @queue=[[#<PrettyPrint::Group:0x81fed78 @breakables=[], @depth=0, @break=false>], []]>, @buffer=[], @newline="\n", @group_stack=[#<PrettyPrint::Group:0x81fed78 @breakables=[], @depth=0, @break=false>], @buffer_width=0, @indent=0, @maxwidth=79, @output_width=2, @output=#<IO:0x8114ee4>>
#
# Pretty-printed output returns this:
#   #<PP:0x81fedf0
#    @buffer=[],
#    @buffer_width=0,
#    @genspace=#<Proc:0x81feda0>,
#    @group_queue=
#     #<PrettyPrint::GroupQueue:0x81fed3c
#      @queue=
#       [[#<PrettyPrint::Group:0x81fed78 @break=false, @breakables=[], @depth=0>],
#        []]>,
#    @group_stack=
#     [#<PrettyPrint::Group:0x81fed78 @break=false, @breakables=[], @depth=0>],
#    @indent=0,
#    @maxwidth=79,
#    @newline="\n",
#    @output=#<IO:0x8114ee4>,
#    @output_width=2>
#
##
# == Usage
#
#   pp(obj)             #=> obj
#   pp obj              #=> obj
#   pp(obj1, obj2, ...) #=> [obj1, obj2, ...]
#   pp()                #=> nil
#
# Output <tt>obj(s)</tt> to <tt>$></tt> in pretty printed format.
#
# It returns <tt>obj(s)</tt>.
#
##
# == Output Customization
#
# To define a customized pretty printing function for your classes,
# redefine method <code>#pretty_print(pp)</code> in the class.
# Note that <code>require 'pp'</code> is needed before redefining <code>#pretty_print(pp)</code>.
#
# <code>#pretty_print</code> takes the +pp+ argument, which is an instance of the PP class.
# The method uses #text, #breakable, #nest, #group and #pp to print the
# object.
#
##
# == Pretty-Print JSON
#
# To pretty-print JSON refer to JSON#pretty_generate.
#
##
# == Author
# Tanaka Akira <akr@fsij.org>

class PP < PrettyPrint

  # The version string
  VERSION = "0.6.2"

  # Returns the usable width for +out+.
  # As the width of +out+:
  # 1. If +out+ is assigned to a tty device, its width is used.
  # 2. Otherwise, or it could not get the value, the +COLUMN+
  #    environment variable is assumed to be set to the width.
  # 3. If +COLUMN+ is not set to a non-zero number, 80 is assumed.
  #
  # And finally, returns the above width value - 1.
  # * This -1 is for Windows command prompt, which moves the cursor to
  #   the next line if it reaches the last column.
  def PP.width_for(out)
    begin
      require 'io/console'
      _, width = out.winsize
    rescue LoadError, NoMethodError, SystemCallError
    end
    (width || ENV['COLUMNS']&.to_i&.nonzero? || 80) - 1
  end

  # Outputs +obj+ to +out+ in pretty printed format of
  # +width+ columns in width.
  #
  # If +out+ is omitted, <code>$></code> is assumed.
  # If +width+ is omitted, the width of +out+ is assumed (see
  # width_for).
  #
  # PP.pp returns +out+.
  def PP.pp(obj, out=$>, width=width_for(out))
    q = new(out, width)
    q.guard_inspect_key {q.pp obj}
    q.flush
    #$pp = q
    out << "\n"
  end

  # Outputs +obj+ to +out+ like PP.pp but with no indent and
  # newline.
  #
  # PP.singleline_pp returns +out+.
  def PP.singleline_pp(obj, out=$>)
    q = SingleLine.new(out)
    q.guard_inspect_key {q.pp obj}
    q.flush
    out
  end

  # :stopdoc:
  def PP.mcall(obj, mod, meth, *args, &block)
    mod.instance_method(meth).bind_call(obj, *args, &block)
  end
  # :startdoc:

  if defined? ::Ractor
    class << self
      # Returns the sharing detection flag as a boolean value.
      # It is false (nil) by default.
      def sharing_detection
        Ractor.current[:pp_sharing_detection]
      end
      # Sets the sharing detection flag to b.
      def sharing_detection=(b)
        Ractor.current[:pp_sharing_detection] = b
      end
    end
  else
    @sharing_detection = false
    class << self
      # Returns the sharing detection flag as a boolean value.
      # It is false by default.
      attr_accessor :sharing_detection
    end
  end

  # Module that defines helper methods for pretty_print.
  module PPMethods

    # Yields to a block
    # and preserves the previous set of objects being printed.
    def 