ruby/uri/http.rb                                                                                    0000644                 00000007176 15040313431 0007635 0                                                                                                    ustar 00                                                                                                                                                                                                                                                       # frozen_string_literal: false
# = uri/http.rb
#
# Author:: Akira Yamada <akira@ruby-lang.org>
# License:: You can redistribute it and/or modify it under the same term as Ruby.
#
# See URI for general documentation
#

require_relative 'generic'

module URI

  #
  # The syntax of HTTP URIs is defined in RFC1738 section 3.3.
  #
  # Note that the Ruby URI library allows HTTP URLs containing usernames and
  # passwords. This is not legal as per the RFC, but used to be
  # supported in Internet Explorer 5 and 6, before the MS04-004 security
  # update. See <URL:http://support.microsoft.com/kb/834489>.
  #
  class HTTP < Generic
    # A Default port of 80 for URI::HTTP.
    DEFAULT_PORT = 80

    # An Array of the available components for URI::HTTP.
    COMPONENT = %i[
      scheme
      userinfo host port
      path
      query
      fragment
    ].freeze

    #
    # == Description
    #
    # Creates a new URI::HTTP object from components, with syntax checking.
    #
    # The components accepted are userinfo, host, port, path, query, and
    # fragment.
    #
    # The components should be provided either as an Array, or as a Hash
    # with keys formed by preceding the component names with a colon.
    #
    # If an Array is used, the components must be passed in the
    # order <code>[userinfo, host, port, path, query, fragment]</code>.
    #
    # Example:
    #
    #     uri = URI::HTTP.build(host: 'www.example.com', path: '/foo/bar')
    #
    #     uri = URI::HTTP.build([nil, "www.example.com", nil, "/path",
    #       "query", 'fragment'])
    #
    # Currently, if passed userinfo components this method generates
    # invalid HTTP URIs as per RFC 1738.
    #
    def self.build(args)
      tmp = Util.make_components_hash(self, args)
      super(tmp)
    end

    #
    # == Description
    #
    # Returns the full path for an HTTP request, as required by Net::HTTP::Get.
    #
    # If the URI contains a query, the full path is URI#path + '?' + URI#query.
    # Otherwise, the path is simply URI#path.
    #
    # Example:
    #
    #     uri = URI::HTTP.build(path: '/foo/bar', query: 'test=true')
    #     uri.request_uri #  => "/foo/bar?test=true"
    #
    def request_uri
      return unless @path

      url = @query ? "#@path?#@query" : @path.dup
      url.start_with?(?/.freeze) ? url : ?/ + url
    end

    #
    # == Description
    #
    # Returns the authority for an HTTP uri, as defined in
    # https://www.rfc-editor.org/rfc/rfc3986#section-3.2.
    #
    #
    # Example:
    #
    #     URI::HTTP.build(host: 'www.example.com', path: '/foo/bar').authority #=> "www.example.com"
    #     URI::HTTP.build(host: 'www.example.com', port: 8000, path: '/foo/bar').authority #=> "www.example.com:8000"
    #     URI::HTTP.build(host: 'www.example.com', port: 80, path: '/foo/bar').authority #=> "www.example.com"
    #
    def authority
      if port == default_port
        host
      else
        "#{host}:#{port}"
      end
    end

    #
    # == Description
    #
    # Returns the origin for an HTTP uri, as defined in
    # https://www.rfc-editor.org/rfc/rfc6454.
    #
    #
    # Example:
    #
    #     URI::HTTP.build(host: 'www.example.com', path: '/foo/bar').origin #=> "http://www.example.com"
    #     URI::HTTP.build(host: 'www.example.com', port: 8000, path: '/foo/bar').origin #=> "http://www.example.com:8000"
    #     URI::HTTP.build(host: 'www.example.com', port: 80, path: '/foo/bar').origin #=> "http://www.example.com"
    #     URI::HTTPS.build(host: 'www.example.com', path: '/foo/bar').origin #=> "https://www.example.com"
    #
    def origin
      "#{scheme}://#{authority}"
    end
  end

  register_scheme 'HTTP', HTTP
end
                                                                                                                                                                                                                                                                                                                                                                                                  ruby/uri/ftp.rb                                                                                     0000644                 00000016046 15040313431 0007443 0                                                                                                    ustar 00                                                                                                                                                                                                                                                       # frozen_string_literal: false
# = uri/ftp.rb
#
# Author:: Akira Yamada <akira@ruby-lang.org>
# License:: You can redistribute it and/or modify it under the same term as Ruby.
#
# See URI for general documentation
#

require_relative 'generic'

module URI

  #
  # FTP URI syntax is defined by RFC1738 section 3.2.
  #
  # This class will be redesigned because of difference of implementations;
  # the structure of its path. draft-hoffman-ftp-uri-04 is a draft but it
  # is a good summary about the de facto spec.
  # https://datatracker.ietf.org/doc/html/draft-hoffman-ftp-uri-04
  #
  class FTP < Generic
    # A Default port of 21 for URI::FTP.
    DEFAULT_PORT = 21

    #
    # An Array of the available components for URI::FTP.
    #
    COMPONENT = [
      :scheme,
      :userinfo, :host, :port,
      :path, :typecode
    ].freeze

    #
    # Typecode is "a", "i", or "d".
    #
    # * "a" indicates a text file (the FTP command was ASCII)
    # * "i" indicates a binary file (FTP command IMAGE)
    # * "d" indicates the contents of a directory should be displayed
    #
    TYPECODE = ['a', 'i', 'd'].freeze

    # Typecode prefix ";type=".
    TYPECODE_PREFIX = ';type='.freeze

    def self.new2(user, password, host, port, path,
                  typecode = nil, arg_check = true) # :nodoc:
      # Do not use this method!  Not tested.  [Bug #7301]
      # This methods remains just for compatibility,
      # Keep it undocumented until the active maintainer is assigned.
      typecode = nil if typecode.size == 0
      if typecode && !TYPECODE.include?(typecode)
        raise ArgumentError,
          "bad typecode is specified: #{typecode}"
      end

      # do escape

      self.new('ftp',
               [user, password],
               host, port, nil,
               typecode ? path + TYPECODE_PREFIX + typecode : path,
               nil, nil, nil, arg_check)
    end

    #
    # == Description
    #
    # Creates a new URI::FTP object from components, with syntax checking.
    #
    # The components accepted are +userinfo+, +host+, +port+, +path+, and
    # +typecode+.
    #
    # The components should be provided either as an Array, or as a Hash
    # with keys formed by preceding the component names with a colon.
    #
    # If an Array is used, the components must be passed in the
    # order <code>[userinfo, host, port, path, typecode]</code>.
    #
    # If the path supplied is absolute, it will be escaped in order to
    # make it absolute in the URI.
    #
    # Examples:
    #
    #     require 'uri'
    #
    #     uri1 = URI::FTP.build(['user:password', 'ftp.example.com', nil,
    #       '/path/file.zip', 'i'])
    #     uri1.to_s  # => "ftp://user:password@ftp.example.com/%2Fpath/file.zip;type=i"
    #
    #     uri2 = URI::FTP.build({:host => 'ftp.example.com',
    #       :path => 'ruby/src'})
    #     uri2.to_s  # => "ftp://ftp.example.com/ruby/src"
    #
    def self.build(args)

      # Fix the incoming path to be generic URL syntax
      # FTP path  ->  URL path
      # foo/bar       /foo/bar
      # /foo/bar      /%2Ffoo/bar
      #
      if args.kind_of?(Array)
        args[3] = '/' + args[3].sub(/^\//, '%2F')
      else
        args[:path] = '/' + args[:path].sub(/^\//, '%2F')
      end

      tmp = Util::make_components_hash(self, args)

      if tmp[:typecode]
        if tmp[:typecode].size == 1
          tmp[:typecode] = TYPECODE_PREFIX + tmp[:typecode]
        end
        tmp[:path] << tmp[:typecode]
      end

      return super(tmp)
    end

    #
    # == Description
    #
    # Creates a new URI::FTP object from generic URL components with no
    # syntax checking.
    #
    # Unlike build(), this method does not escape the path component as
    # required by RFC1738; instead it is treated as per RFC2396.
    #
    # Arguments are +scheme+, +userinfo+, +host+, +port+, +registry+, +path+,
    # +opaque+, +query+, and +fragment+, in that order.
    #
    def initialize(scheme,
                   userinfo, host, port, registry,
                   path, opaque,
                   query,
                   fragment,
                   parser = nil,
                   arg_check = false)
      raise InvalidURIError unless path
      path = path.sub(/^\//,'')
      path.sub!(/^%2F/,'/')
      super(scheme, userinfo, host, port, registry, path, opaque,
            query, fragment, parser, arg_check)
      @typecode = nil
      if tmp = @path.index(TYPECODE_PREFIX)
        typecode = @path[tmp + TYPECODE_PREFIX.size..-1]
        @path = @path[0..tmp - 1]

        if arg_check
          self.typecode = typecode
        else
          self.set_typecode(typecode)
        end
      end
    end

    # typecode accessor.
    #
    # See URI::FTP::COMPONENT.
    attr_reader :typecode

    # Validates typecode +v+,
    # returns +true+ or +false+.
    #
    def check_typecode(v)
      if TYPECODE.include?(v)
        return true
      else
        raise InvalidComponentError,
          "bad typecode(expected #{TYPECODE.join(', ')}): #{v}"
      end
    end
    private :check_typecode

    # Private setter for the typecode +v+.
    #
    # See also URI::FTP.typecode=.
    #
    def set_typecode(v)
      @typecode = v
    end
    protected :set_typecode

    #
    # == Args
    #
    # +v+::
    #    String
    #
    # == Description
    #
    # Public setter for the typecode +v+
    # (with validation).
    #
    # See also URI::FTP.check_typecode.
    #
    # == Usage
    #
    #   require 'uri'
    #
    #   uri = URI.parse("ftp://john@ftp.example.com/my_file.img")
    #   #=> #<URI::FTP ftp://john@ftp.example.com/my_file.img>
    #   uri.typecode = "i"
    #   uri
    #   #=> #<URI::FTP ftp://john@ftp.example.com/my_file.img;type=i>
    #
    def typecode=(typecode)
      check_typecode(typecode)
      set_typecode(typecode)
      typecode
    end

    def merge(oth) # :nodoc:
      tmp = super(oth)
      if self != tmp
        tmp.set_typecode(oth.typecode)
      end

      return tmp
    end

    # Returns the path from an FTP URI.
    #
    # RFC 1738 specifically states that the path for an FTP URI does not
    # include the / which separates the URI path from the URI host. Example:
    #
    # <code>ftp://ftp.example.com/pub/ruby</code>
    #
    # The above URI indicates that the client should connect to
    # ftp.example.com then cd to pub/ruby from the initial login directory.
    #
    # If you want to cd to an absolute directory, you must include an
    # escaped / (%2F) in the path. Example:
    #
    # <code>ftp://ftp.example.com/%2Fpub/ruby</code>
    #
    # This method will then return "/pub/ruby".
    #
    def path
      return @path.sub(/^\//,'').sub(/^%2F/,'/')
    end

    # Private setter for the path of the URI::FTP.
    def set_path(v)
      super("/" + v.sub(/^\//, "%2F"))
    end
    protected :set_path

    # Returns a String representation of the URI::FTP.
    def to_s
      save_path = nil
      if @typecode
        save_path = @path
        @path = @path + TYPECODE_PREFIX + @typecode
      end
      str = super
      if @typecode
        @path = save_path
      end

      return str
    end
  end

  register_scheme 'FTP', FTP
end
                                                                                                                                                                                                                                                                                                                                                                                                                                                                                          ruby/uri/ws.rb                                                                                      0000644                 00000004365 15040313431 0007304 0                                                                                                    ustar 00                                                                                                                                                                                                                                                       # frozen_string_literal: false
# = uri/ws.rb
#
# Author:: Matt Muller <mamuller@amazon.com>
# License:: You can redistribute it and/or modify it under the same term as Ruby.
#
# See URI for general documentation
#

require_relative 'generic'

module URI

  #
  # The syntax of WS URIs is defined in RFC6455 section 3.
  #
  # Note that the Ruby URI library allows WS URLs containing usernames and
  # passwords. This is not legal as per the RFC, but used to be
  # supported in Internet Explorer 5 and 6, before the MS04-004 security
  # update. See <URL:http://support.microsoft.com/kb/834489>.
  #
  class WS < Generic
    # A Default port of 80 for URI::WS.
    DEFAULT_PORT = 80

    # An Array of the available components for URI::WS.
    COMPONENT = %i[
      scheme
      userinfo host port
      path
      query
    ].freeze

    #
    # == Description
    #
    # Creates a new URI::WS object from components, with syntax checking.
    #
    # The components accepted are userinfo, host, port, path, and query.
    #
    # The components should be provided either as an Array, or as a Hash
    # with keys formed by preceding the component names with a colon.
    #
    # If an Array is used, the components must be passed in the
    # order <code>[userinfo, host, port, path, query]</code>.
    #
    # Example:
    #
    #     uri = URI::WS.build(host: 'www.example.com', path: '/foo/bar')
    #
    #     uri = URI::WS.build([nil, "www.example.com", nil, "/path", "query"])
    #
    # Currently, if passed userinfo components this method generates
    # invalid WS URIs as per RFC 1738.
    #
    def self.build(args)
      tmp = Util.make_components_hash(self, args)
      super(tmp)
    end

    #
    # == Description
    #
    # Returns the full path for a WS URI, as required by Net::HTTP::Get.
    #
    # If the URI contains a query, the full path is URI#path + '?' + URI#query.
    # Otherwise, the path is simply URI#path.
    #
    # Example:
    #
    #     uri = URI::WS.build(path: '/foo/bar', query: 'test=true')
    #     uri.request_uri #  => "/foo/bar?test=true"
    #
    def request_uri
      return unless @path

      url = @query ? "#@path?#@query" : @path.dup
      url.start_with?(?/.freeze) ? url : ?/ + url
    end
  end

  register_scheme 'WS', WS
end
                                                                                                                                                                                                                                                                           ruby/uri/common.rb                                                                                  0000644                 00000064501 15040313432 0010142 0                                                                                                    ustar 00                                                                                                                                                                                                                                                       # frozen_string_literal: true
#--
# = uri/common.rb
#
# Author:: Akira Yamada <akira@ruby-lang.org>
# License::
#   You can redistribute it and/or modify it under the same term as Ruby.
#
# See URI for general documentation
#

require_relative "rfc2396_parser"
require_relative "rfc3986_parser"

module URI
  # The default parser instance for RFC 2396.
  RFC2396_PARSER = RFC2396_Parser.new
  Ractor.make_shareable(RFC2396_PARSER) if defined?(Ractor)

  # The default parser instance for RFC 3986.
  RFC3986_PARSER = RFC3986_Parser.new
  Ractor.make_shareable(RFC3986_PARSER) if defined?(Ractor)

  # The default parser instance.
  DEFAULT_PARSER = RFC3986_PARSER
  Ractor.make_shareable(DEFAULT_PARSER) if defined?(Ractor)

  # Set the default parser instance.
  def self.parser=(parser = RFC3986_PARSER)
    remove_const(:Parser) if defined?(::URI::Parser)
    const_set("Parser", parser.class)

    remove_const(:REGEXP) if defined?(::URI::REGEXP)
    remove_const(:PATTERN) if defined?(::URI::PATTERN)
    if Parser == RFC2396_Parser
      const_set("REGEXP", URI::RFC2396_REGEXP)
      const_set("PATTERN", URI::RFC2396_REGEXP::PATTERN)
    end

    Parser.new.regexp.each_pair do |sym, str|
      remove_const(sym) if const_defined?(sym, false)
      const_set(sym, str)
    end
  end
  self.parser = RFC3986_PARSER

  def self.const_missing(const) # :nodoc:
    if const == :REGEXP
      warn "URI::REGEXP is obsolete. Use URI::RFC2396_REGEXP explicitly.", uplevel: 1 if $VERBOSE
      URI::RFC2396_REGEXP
    elsif value = RFC2396_PARSER.regexp[const]
      warn "URI::#{const} is obsolete. Use RFC2396_PARSER.regexp[#{const.inspect}] explicitly.", uplevel: 1 if $VERBOSE
      value
    elsif value = RFC2396_Parser.const_get(const)
      warn "URI::#{const} is obsolete. Use RFC2396_Parser::#{const} explicitly.", uplevel: 1 if $VERBOSE
      value
    else
      super
    end
  end

  module Util # :nodoc:
    def make_components_hash(klass, array_hash)
      tmp = {}
      if array_hash.kind_of?(Array) &&
          array_hash.size == klass.component.size - 1
        klass.component[1..-1].each_index do |i|
          begin
            tmp[klass.component[i + 1]] = array_hash[i].clone
          rescue TypeError
            tmp[klass.component[i + 1]] = array_hash[i]
          end
        end

      elsif array_hash.kind_of?(Hash)
        array_hash.each do |key, value|
          begin
            tmp[key] = value.clone
          rescue TypeError
            tmp[key] = value
          end
        end
      else
        raise ArgumentError,
          "expected Array of or Hash of components of #{klass} (#{klass.component[1..-1].join(', ')})"
      end
      tmp[:scheme] = klass.to_s.sub(/\A.*::/, '').downcase

      return tmp
    end
    module_function :make_components_hash
  end

  module Schemes # :nodoc:
  end
  private_constant :Schemes

  # Registers the given +klass+ as the class to be instantiated
  # when parsing a \URI with the given +scheme+:
  #
  #   URI.register_scheme('MS_SEARCH', URI::Generic) # => URI::Generic
  #   URI.scheme_list['MS_SEARCH']                   # => URI::Generic
  #
  # Note that after calling String#upcase on +scheme+, it must be a valid
  # constant name.
  def self.register_scheme(scheme, klass)
    Schemes.const_set(scheme.to_s.upcase, klass)
  end

  # Returns a hash of the defined schemes:
  #
  #   URI.scheme_list
  #   # =>
  #   {"MAILTO"=>URI::MailTo,
  #    "LDAPS"=>URI::LDAPS,
  #    "WS"=>URI::WS,
  #    "HTTP"=>URI::HTTP,
  #    "HTTPS"=>URI::HTTPS,
  #    "LDAP"=>URI::LDAP,
  #    "FILE"=>URI::File,
  #    "FTP"=>URI::FTP}
  #
  # Related: URI.register_scheme.
  def self.scheme_list
    Schemes.constants.map { |name|
      [name.to_s.upcase, Schemes.const_get(name)]
    }.to_h
  end

  INITIAL_SCHEMES = scheme_list
  private_constant :INITIAL_SCHEMES
  Ractor.make_shareable(INITIAL_SCHEMES) if defined?(Ractor)

  # Returns a new object constructed from the given +scheme+, +arguments+,
  # and +default+:
  #
  # - The new object is an instance of <tt>URI.scheme_list[scheme.upcase]</tt>.
  # - The object is initialized by calling the class initializer
  #   using +scheme+ and +arguments+.
  #   See URI::Generic.new.
  #
  # Examples:
  #
  #   values = ['john.doe', 'www.example.com', '123', nil, '/forum/questions/', nil, 'tag=networking&order=newest', 'top']
  #   URI.for('https', *values)
  #   # => #<URI::HTTPS https://john.doe@www.example.com:123/forum/questions/?tag=networking&order=newest#top>
  #   URI.for('foo', *values, default: URI::HTTP)
  #   # => #<URI::HTTP foo://john.doe@www.example.com:123/forum/questions/?tag=networking&order=newest#top>
  #
  def self.for(scheme, *arguments, default: Generic)
    const_name = scheme.to_s.upcase

    uri_class = INITIAL_SCHEMES[const_name]
    uri_class ||= if /\A[A-Z]\w*\z/.match?(const_name) && Schemes.const_defined?(const_name, false)
      Schemes.const_get(const_name, false)
    end
    uri_class ||= default

    return uri_class.new(scheme, *arguments)
  end

  #
  # Base class for all URI exceptions.
  #
  class Error < StandardError; end
  #
  # Not a URI.
  #
  class InvalidURIError < Error; end
  #
  # Not a URI component.
  #
  class InvalidComponentError < Error; end
  #
  # URI is valid, bad usage is not.
  #
  class BadURIError < Error; end

  # Returns a 9-element array representing the parts of the \URI
  # formed from the string +uri+;
  # each array element is a string or +nil+:
  #
  #   names = %w[scheme userinfo host port registry path opaque query fragment]
  #   values = URI.split('https://john.doe@www.example.com:123/forum/questions/?tag=networking&order=newest#top')
  #   names.zip(values)
  #   # =>
  #   [["scheme", "https"],
  #    ["userinfo", "john.doe"],
  #    ["host", "www.example.com"],
  #    ["port", "123"],
  #    ["registry", nil],
  #    ["path", "/forum/questions/"],
  #    ["opaque", nil],
  #    ["query", "tag=networking&order=newest"],
  #    ["fragment", "top"]]
  #
  def self.split(uri)
    DEFAULT_PARSER.split(uri)
  end

  # Returns a new \URI object constructed from the given string +uri+:
  #
  #   URI.parse('https://john.doe@www.example.com:123/forum/questions/?tag=networking&order=newest#top')
  #   # => #<URI::HTTPS https://john.doe@www.example.com:123/forum/questions/?tag=networking&order=newest#top>
  #   URI.parse('http://john.doe@www.example.com:123/forum/questions/?tag=networking&order=newest#top')
  #   # => #<URI::HTTP http://john.doe@www.example.com:123/forum/questions/?tag=networking&order=newest#top>
  #
  # It's recommended to first ::escape string +uri+
  # if it may contain invalid URI characters.
  #
  def self.parse(uri)
    DEFAULT_PARSER.parse(uri)
  end

  # Merges the given URI strings +str+
  # per {RFC 2396}[https://www.rfc-editor.org/rfc/rfc2396.html].
  #
  # Each string in +str+ is converted to an
  # {RFC3986 URI}[https://www.rfc-editor.org/rfc/rfc3986.html] before being merged.
  #
  # Examples:
  #
  #   URI.join("http://example.com/","main.rbx")
  #   # => #<URI::HTTP http://example.com/main.rbx>
  #
  #   URI.join('http://example.com', 'foo')
  #   # => #<URI::HTTP http://example.com/foo>
  #
  #   URI.join('http://example.com', '/foo', '/bar')
  #   # => #<URI::HTTP http://example.com/bar>
  #
  #   URI.join('http://example.com', '/foo', 'bar')
  #   # => #<URI::HTTP http://example.com/bar>
  #
  #   URI.join('http://example.com', '/foo/', 'bar')
  #   # => #<URI::HTTP http://example.com/foo/bar>
  #
  def self.join(*str)
    DEFAULT_PARSER.join(*str)
  end

  #
  # == Synopsis
  #
  #   URI::extract(str[, schemes][,&blk])
  #
  # == Args
  #
  # +str+::
  #   String to extract URIs from.
  # +schemes+::
  #   Limit URI matching to specific schemes.
  #
  # == Description
  #
  # Extracts URIs from a string. If block given, iterates through all matched URIs.
  # Returns nil if block given or array with matches.
  #
  # == Usage
  #
  #   require "uri"
  #
  #   URI.extract("text here http://foo.example.org/bla and here mailto:test@example.com and here also.")
  #   # => ["http://foo.example.com/bla", "mailto:test@example.com"]
  #
  def self.extract(str, schemes = nil, &block) # :nodoc:
    warn "URI.extract is obsolete", uplevel: 1 if $VERBOSE
    DEFAULT_PARSER.extract(str, schemes, &block)
  end

  #
  # == Synopsis
  #
  #   URI::regexp([match_schemes])
  #
  # == Args
  #
  # +match_schemes+::
  #   Array of schemes. If given, resulting regexp matches to URIs
  #   whose scheme is one of the match_schemes.
  #
  # == Description
  #
  # Returns a Regexp object which matches to URI-like strings.
  # The Regexp object returned by this method includes arbitrary
  # number of capture group (parentheses).  Never rely on its number.
  #
  # == Usage
  #
  #   require 'uri'
  #
  #   # extract first URI from html_string
  #   html_string.slice(URI.regexp)
  #
  #   # remove ftp URIs
  #   html_string.sub(URI.regexp(['ftp']), '')
  #
  #   # You should not rely on the number of parentheses
  #   html_string.scan(URI.regexp) do |*matches|
  #     p $&
  #   end
  #
  def self.regexp(schemes = nil)# :nodoc:
    warn "URI.regexp is obsolete", uplevel: 1 if $VERBOSE
    DEFAULT_PARSER.make_regexp(schemes)
  end

  TBLENCWWWCOMP_ = {} # :nodoc:
  256.times do |i|
    TBLENCWWWCOMP_[-i.chr] = -('%%%02X' % i)
  end
  TBLENCURICOMP_ = TBLENCWWWCOMP_.dup.freeze # :nodoc:
  TBLENCWWWCOMP_[' '] = '+'
  TBLENCWWWCOMP_.freeze
  TBLDECWWWCOMP_ = {} # :nodoc:
  256.times do |i|
    h, l = i>>4, i&15
    TBLDECWWWCOMP_[-('%%%X%X' % [h, l])] = -i.chr
    TBLDECWWWCOMP_[-('%%%x%X' % [h, l])] = -i.chr
    TBLDECWWWCOMP_[-('%%%X%x' % [h, l])] = -i.chr
    TBLDECWWWCOMP_[-('%%%x%x' % [h, l])] = -i.chr
  end
  TBLDECWWWCOMP_['+'] = ' '
  TBLDECWWWCOMP_.freeze

  # Returns a URL-encoded string derived from the given string +str+.
  #
  # The returned string:
  #
  # - Preserves:
  #
  #   - Characters <tt>'*'</tt>, <tt>'.'</tt>, <tt>'-'</tt>, and <tt>'_'</tt>.
  #   - Character in ranges <tt>'a'..'z'</tt>, <tt>'A'..'Z'</tt>,
  #     and <tt>'0'..'9'</tt>.
  #
  #   Example:
  #
  #     URI.encode_www_form_component('*.-_azAZ09')
  #     # => "*.-_azAZ09"
  #
  # - Converts:
  #
  #   - Character <tt>' '</tt> to character <tt>'+'</tt>.
  #   - Any other character to "percent notation";
  #     the percent notation for character <i>c</i> is <tt>'%%%X' % c.ord</tt>.
  #
  #   Example:
  #
  #     URI.encode_www_form_component('Here are some punctuation characters: ,;?:')
  #     # => "Here+are+some+punctuation+characters%3A+%2C%3B%3F%3A"
  #
  # Encoding:
  #
  # - If +str+ has encoding Encoding::ASCII_8BIT, argument +enc+ is ignored.
  # - Otherwise +str+ is converted first to Encoding::UTF_8
  #   (with suitable character replacements),
  #   and then to encoding +enc+.
  #
  # In either case, the returned string has forced encoding Encoding::US_ASCII.
  #
  # Related: URI.encode_uri_component (encodes <tt>' '</tt> as <tt>'%20'</tt>).
  def self.encode_www_form_component(str, enc=nil)
    _encode_uri_component(/[^*\-.0-9A-Z_a-z]/, TBLENCWWWCOMP_, str, enc)
  end

  # Returns a string decoded from the given \URL-encoded string +str+.
  #
  # The given string is first encoded as Encoding::ASCII-8BIT (using String#b),
  # then decoded (as below), and finally force-encoded to the given encoding +enc+.
  #
  # The returned string:
  #
  # - Preserves:
  #
  #   - Characters <tt>'*'</tt>, <tt>'.'</tt>, <tt>'-'</tt>, and <tt>'_'</tt>.
  #   - Character in ranges <tt>'a'..'z'</tt>, <tt>'A'..'Z'</tt>,
  #     and <tt>'0'..'9'</tt>.
  #
  #   Example:
  #
  #     URI.decode_www_form_component('*.-_azAZ09')
  #     # => "*.-_azAZ09"
  #
  # - Converts:
  #
  #   - Character <tt>'+'</tt> to character <tt>' '</tt>.
  #   - Each "percent notation" to an ASCII character.
  #
  #   Example:
  #
  #     URI.decode_www_form_component('Here+are+some+punctuation+characters%3A+%2C%3B%3F%3A')
  #     # => "Here are some punctuation characters: ,;?:"
  #
  # Related: URI.decode_uri_component (preserves <tt>'+'</tt>).
  def self.decode_www_form_component(str, enc=Encoding::UTF_8)
    _decode_uri_component(/\+|%\h\h/, str, enc)
  end

  # Like URI.encode_www_form_component, except that <tt>' '</tt> (space)
  # is encoded as <tt>'%20'</tt> (instead of <tt>'+'</tt>).
  def self.encode_uri_component(str, enc=nil)
    _encode_uri_component(/[^*\-.0-9A-Z_a-z]/, TBLENCURICOMP_, str, enc)
  end

  # Like URI.decode_www_form_component, except that <tt>'+'</tt> is preserved.
  def self.decode_uri_component(str, enc=Encoding::UTF_8)
    _decode_uri_component(/%\h\h/, str, enc)
  end

  def self._encode_uri_component(regexp, table, str, enc)
    str = str.to_s.dup
    if str.encoding != Encoding::ASCII_8BIT
      if enc && enc != Encoding::ASCII_8BIT
        str.encode!(Encoding::UTF_8, invalid: :replace, undef: :replace)
        str.encode!(enc, fallback: ->(x){"&##{x.ord};"})
      end
      str.force_encoding(Encoding::ASCII_8BIT)
    end
    str.gsub!(regexp, table)
    str.force_encoding(Encoding::US_ASCII)
  end
  private_class_method :_encode_uri_component

  def self._decode_uri_component(regexp, str, enc)
    raise ArgumentError, "invalid %-encoding (#{str})" if /%(?!\h\h)/.match?(str)
    str.b.gsub(regexp, TBLDECWWWCOMP_).force_encoding(enc)
  end
  private_class_method :_decode_uri_component

  # Returns a URL-encoded string derived from the given
  # {Enumerable}[rdoc-ref:Enumerable@Enumerable+in+Ruby+Classes]
  # +enum+.
  #
  # The result is suitable for use as form data
  # for an \HTTP request whose <tt>Content-Type</tt> is
  # <tt>'application/x-www-form-urlencoded'</tt>.
  #
  # The returned string consists of the elements of +enum+,
  # each converted to one or more URL-encoded strings,
  # and all joined with character <tt>'&'</tt>.
  #
  # Simple examples:
  #
  #   URI.encode_www_form([['foo', 0], ['bar', 1], ['baz', 2]])
  #   # => "foo=0&bar=1&baz=2"
  #   URI.encode_www_form({foo: 0, bar: 1, baz: 2})
  #   # => "foo=0&bar=1&baz=2"
  #
  # The returned string is formed using method URI.encode_www_form_component,
  # which converts certain characters:
  #
  #   URI.encode_www_form('f#o': '/', 'b-r': '$', 'b z': '@')
  #   # => "f%23o=%2F&b-r=%24&b+z=%40"
  #
  # When +enum+ is Array-like, each element +ele+ is converted to a field:
  #
  # - If +ele+ is an array of two or more elements,
  #   the field is formed from its first two elements
  #   (and any additional elements are ignored):
  #
  #     name = URI.encode_www_form_component(ele[0], enc)
  #     value = URI.encode_www_form_component(ele[1], enc)
  #     "#{name}=#{value}"
  #
  #   Examples:
  #
  #     URI.encode_www_form([%w[foo bar], %w[baz bat bah]])
  #     # => "foo=bar&baz=bat"
  #     URI.encode_www_form([['foo', 0], ['bar', :baz, 'bat']])
  #     # => "foo=0&bar=baz"
  #
  # - If +ele+ is an array of one element,
  #   the field is formed from <tt>ele[0]</tt>:
  #
  #     URI.encode_www_form_component(ele[0])
  #
  #   Example:
  #
  #     URI.encode_www_form([['foo'], [:bar], [0]])
  #     # => "foo&bar&0"
  #
  # - Otherwise the field is formed from +ele+:
  #
  #     URI.encode_www_form_component(ele)
  #
  #   Example:
  #
  #     URI.encode_www_form(['foo', :bar, 0])
  #     # => "foo&bar&0"
  #
  # The elements of an Array-like +enum+ may be mixture:
  #
  #   URI.encode_www_form([['foo', 0], ['bar', 1, 2], ['baz'], :bat])
  #   # => "foo=0&bar=1&baz&bat"
  #
  # When +enum+ is Hash-like,
  # each +key+/+value+ pair is converted to one or more fields:
  #
  # - If +value+ is
  #   {Array-convertible}[rdoc-ref:implicit_conversion.rdoc@Array-Convertible+Objects],
  #   each element +ele+ in +value+ is paired with +key+ to form a field:
  #
  #     name = URI.encode_www_form_component(key, enc)
  #     value = URI.encode_www_form_component(ele, enc)
  #     "#{name}=#{value}"
  #
  #   Example:
  #
  #     URI.encode_www_form({foo: [:bar, 1], baz: [:bat, :bam, 2]})
  #     # => "foo=bar&foo=1&baz=bat&baz=bam&baz=2"
  #
  # - Otherwise, +key+ and +value+ are paired to form a field:
  #
  #     name = URI.encode_www_form_component(key, enc)
  #     value = URI.encode_www_form_component(value, enc)
  #     "#{name}=#{value}"
  #
  #   Example:
  #
  #     URI.encode_www_form({foo: 0, bar: 1, baz: 2})
  #     # => "foo=0&bar=1&baz=2"
  #
  # The elements of a Hash-like +enum+ may be mixture:
  #
  #   URI.encode_www_form({foo: [0, 1], bar: 2})
  #   # => "foo=0&foo=1&bar=2"
  #
  def self.encode_www_form(enum, enc=nil)
    enum.map do |k,v|
      if v.nil?
        encode_www_form_component(k, enc)
      elsif v.respond_to?(:to_ary)
        v.to_ary.map do |w|
          str = encode_www_form_component(k, enc)
          unless w.nil?
            str << '='
            str << encode_www_form_component(w, enc)
          end
        end.join('&')
      else
        str = encode_www_form_component(k, enc)
        str << '='
        str << encode_www_form_component(v, enc)
      end
    end.join('&')
  end

  # Returns name/value pairs derived from the given string +str+,
  # which must be an ASCII string.
  #
  # The method may be used to decode the body of Net::HTTPResponse object +res+
  # for which <tt>res['Content-Type']</tt> is <tt>'application/x-www-form-urlencoded'</tt>.
  #
  # The returned data is an array of 2-element subarrays;
  # each subarray is a name/value pair (both are strings).
  # Each returned string has encoding +enc+,
  # and has had invalid characters removed via
  # {String#scrub}[rdoc-ref:String#scrub].
  #
  # A simple example:
  #
  #   URI.decode_www_form('foo=0&bar=1&baz')
  #   # => [["foo", "0"], ["bar", "1"], ["baz", ""]]
  #
  # The returned strings have certain conversions,
  # similar to those performed in URI.decode_www_form_component:
  #
  #   URI.decode_www_form('f%23o=%2F&b-r=%24&b+z=%40')
  #   # => [["f#o", "/"], ["b-r", "$"], ["b z", "@"]]
  #
  # The given string may contain consecutive separators:
  #
  #   URI.decode_www_form('foo=0&&bar=1&&baz=2')
  #   # => [["foo", "0"], ["", ""], ["bar", "1"], ["", ""], ["baz", "2"]]
  #
  # A different separator may be specified:
  #
  #   URI.decode_www_form('foo=0--bar=1--baz', separator: '--')
  #   # => [["foo", "0"], ["bar", "1"], ["baz", ""]]
  #
  def self.decode_www_form(str, enc=Encoding::UTF_8, separator: '&', use__charset_: false, isindex: false)
    raise ArgumentError, "the input of #{self.name}.#{__method__} must be ASCII only string" unless str.ascii_only?
    ary = []
    return ary if str.empty?
    enc = Encoding.find(enc)
    str.b.each_line(separator) do |string|
      string.chomp!(separator)
      key, sep, val = string.partition('=')
      if isindex
        if sep.empty?
          val = key
          key = +''
        end
        isindex = false
      end

      if use__charset_ and key == '_charset_' and e = get_encoding(val)
        enc = e
        use__charset_ = false
      end

      key.gsub!(/\+|%\h\h/, TBLDECWWWCOMP_)
      if val
        val.gsub!(/\+|%\h\h/, TBLDECWWWCOMP_)
      else
        val = +''
      end

      ary << [key, val]
    end
    ary.each do |k, v|
      k.force_encoding(enc)
      k.scrub!
      v.force_encoding(enc)
      v.scrub!
    end
    ary
  end

  private
=begin command for WEB_ENCODINGS_
  curl https://encoding.spec.whatwg.org/encodings.json|
  ruby -rjson -e 'H={}
  h={
    "shift_jis"=>"Windows-31J",
    "euc-jp"=>"cp51932",
    "iso-2022-jp"=>"cp50221",
    "x-mac-cyrillic"=>"macCyrillic",
  }
  JSON($<.read).map{|x|x["encodings"]}.flatten.each{|x|
    Encoding.find(n=h.fetch(n=x["name"].downcase,n))rescue next
    x["labels"].each{|y|H[y]=n}
  }
  puts "{"
  H.each{|k,v|puts %[  #{k.dump}=>#{v.dump},]}
  puts "}"
'
=end
  WEB_ENCODINGS_ = {
    "unicode-1-1-utf-8"=>"utf-8",
    "utf-8"=>"utf-8",
    "utf8"=>"utf-8",
    "866"=>"ibm866",
    "cp866"=>"ibm866",
    "csibm866"=>"ibm866",
    "ibm866"=>"ibm866",
    "csisolatin2"=>"iso-8859-2",
    "iso-8859-2"=>"iso-8859-2",
    "iso-ir-101"=>"iso-8859-2",
    "iso8859-2"=>"iso-8859-2",
    "iso88592"=>"iso-8859-2",
    "iso_8859-2"=>"iso-8859-2",
    "iso_8859-2:1987"=>"iso-8859-2",
    "l2"=>"iso-8859-2",
    "latin2"=>"iso-8859-2",
    "csisolatin3"=>"iso-8859-3",
    "iso-8859-3"=>"iso-8859-3",
    "iso-ir-109"=>"iso-8859-3",
    "iso8859-3"=>"iso-8859-3",
    "iso88593"=>"iso-8859-3",
    "iso_8859-3"=>"iso-8859-3",
    "iso_8859-3:1988"=>"iso-8859-3",
    "l3"=>"iso-8859-3",
    "latin3"=>"iso-8859-3",
    "csisolatin4"=>"iso-8859-4",
    "iso-8859-4"=>"iso-8859-4",
    "iso-ir-110"=>"iso-8859-4",
    "iso8859-4"=>"iso-8859-4",
    "iso88594"=>"iso-8859-4",
    "iso_8859-4"=>"iso-8859-4",
    "iso_8859-4:1988"=>"iso-8859-4",
    "l4"=>"iso-8859-4",
    "latin4"=>"iso-8859-4",
    "csisolatincyrillic"=>"iso-8859-5",
    "cyrillic"=>"iso-8859-5",
    "iso-8859-5"=>"iso-8859-5",
    "iso-ir-144"=>"iso-8859-5",
    "iso8859-5"=>"iso-8859-5",
    "iso88595"=>"iso-8859-5",
    "iso_8859-5"=>"iso-8859-5",
    "iso_8859-5:1988"=>"iso-8859-5",
    "arabic"=>"iso-8859-6",
    "asmo-708"=>"iso-8859-6",
    "csiso88596e"=>"iso-8859-6",
    "csiso88596i"=>"iso-8859-6",
    "csisolatinarabic"=>"iso-8859-6",
    "ecma-114"=>"iso-8859-6",
    "iso-8859-6"=>"iso-8859-6",
    "iso-8859-6-e"=>"iso-8859-6",
    "iso-8859-6-i"=>"iso-8859-6",
    "iso-ir-127"=>"iso-8859-6",
    "iso8859-6"=>"iso-8859-6",
    "iso88596"=>"iso-8859-6",
    "iso_8859-6"=>"iso-8859-6",
    "iso_8859-6:1987"=>"iso-8859-6",
    "csisolatingreek"=>"iso-8859-7",
    "ecma-118"=>"iso-8859-7",
    "elot_928"=>"iso-8859-7",
    "greek"=>"iso-8859-7",
    "greek8"=>"iso-8859-7",
    "iso-8859-7"=>"iso-8859-7",
    "iso-ir-126"=>"iso-8859-7",
    "iso8859-7"=>"iso-8859-7",
    "iso88597"=>"iso-8859-7",
    "iso_8859-7"=>"iso-8859-7",
    "iso_8859-7:1987"=>"iso-8859-7",
    "sun_eu_greek"=>"iso-8859-7",
    "csiso88598e"=>"iso-8859-8",
    "csisolatinhebrew"=>"iso-8859-8",
    "hebrew"=>"iso-8859-8",
    "iso-8859-8"=>"iso-8859-8",
    "iso-8859-8-e"=>"iso-8859-8",
    "iso-ir-138"=>"iso-8859-8",
    "iso8859-8"=>"iso-8859-8",
    "iso88598"=>"iso-8859-8",
    "iso_8859-8"=>"iso-8859-8",
    "iso_8859-8:1988"=>"iso-8859-8",
    "visual"=>"iso-8859-8",
    "csisolatin6"=>"iso-8859-10",
    "iso-8859-10"=>"iso-8859-10",
    "iso-ir-157"=>"iso-8859-10",
    "iso8859-10"=>"iso-8859-10",
    "iso885910"=>"iso-8859-10",
    "l6"=>"iso-8859-10",
    "latin6"=>"iso-8859-10",
    "iso-8859-13"=>"iso-8859-13",
    "iso8859-13"=>"iso-8859-13",
    "iso885913"=>"iso-8859-13",
    "iso-8859-14"=>"iso-8859-14",
    "iso8859-14"=>"iso-8859-14",
    "iso885914"=>"iso-8859-14",
    "csisolatin9"=>"iso-8859-15",
    "iso-8859-15"=>"iso-8859-15",
    "iso8859-15"=>"iso-8859-15",
    "iso885915"=>"iso-8859-15",
    "iso_8859-15"=>"iso-8859-15",
    "l9"=>"iso-8859-15",
    "iso-8859-16"=>"iso-8859-16",
    "cskoi8r"=>"koi8-r",
    "koi"=>"koi8-r",
    "koi8"=>"koi8-r",
    "koi8-r"=>"koi8-r",
    "koi8_r"=>"koi8-r",
    "koi8-ru"=>"koi8-u",
    "koi8-u"=>"koi8-u",
    "dos-874"=>"windows-874",
    "iso-8859-11"=>"windows-874",
    "iso8859-11"=>"windows-874",
    "iso885911"=>"windows-874",
    "tis-620"=>"windows-874",
    "windows-874"=>"windows-874",
    "cp1250"=>"windows-1250",
    "windows-1250"=>"windows-1250",
    "x-cp1250"=>"windows-1250",
    "cp1251"=>"windows-1251",
    "windows-1251"=>"windows-1251",
    "x-cp1251"=>"windows-1251",
    "ansi_x3.4-1968"=>"windows-1252",
    "ascii"=>"windows-1252",
    "cp1252"=>"windows-1252",
    "cp819"=>"windows-1252",
    "csisolatin1"=>"windows-1252",
    "ibm819"=>"windows-1252",
    "iso-8859-1"=>"windows-1252",
    "iso-ir-100"=>"windows-1252",
    "iso8859-1"=>"windows-1252",
    "iso88591"=>"windows-1252",
    "iso_8859-1"=>"windows-1252",
    "iso_8859-1:1987"=>"windows-1252",
    "l1"=>"windows-1252",
    "latin1"=>"windows-1252",
    "us-ascii"=>"windows-1252",
    "windows-1252"=>"windows-1252",
    "x-cp1252"=>"windows-1252",
    "cp1253"=>"windows-1253",
    "windows-1253"=>"windows-1253",
    "x-cp1253"=>"windows-1253",
    "cp1254"=>"windows-1254",
    "csisolatin5"=>"windows-1254",
    "iso-8859-9"=>"windows-1254",
    "iso-ir-148"=>"windows-1254",
    "iso8859-9"=>"windows-1254",
    "iso88599"=>"windows-1254",
    "iso_8859-9"=>"windows-1254",
    "iso_8859-9:1989"=>"windows-1254",
    "l5"=>"windows-1254",
    "latin5"=>"windows-1254",
    "windows-1254"=>"windows-1254",
    "x-cp1254"=>"windows-1254",
    "cp1255"=>"windows-1255",
    "windows-1255"=>"windows-1255",
    "x-cp1255"=>"windows-1255",
    "cp1256"=>"windows-1256",
    "windows-1256"=>"windows-1256",
    "x-cp1256"=>"windows-1256",
    "cp1257"=>"windows-1257",
    "windows-1257"=>"windows-1257",
    "x-cp1257"=>"windows-1257",
    "cp1258"=>"windows-1258",
    "windows-1258"=>"windows-1258",
    "x-cp1258"=>"windows-1258",
    "x-mac-cyrillic"=>"macCyrillic",
    "x-mac-ukrainian"=>"macCyrillic",
    "chinese"=>"gbk",
    "csgb2312"=>"gbk",
    "csiso58gb231280"=>"gbk",
    "gb2312"=>"gbk",
    "gb_2312"=>"gbk",
    "gb_2312-80"=>"gbk",
    "gbk"=>"gbk",
    "iso-ir-58"=>"gbk",
    "x-gbk"=>"gbk",
    "gb18030"=>"gb18030",
    "big5"=>"big5",
    "big5-hkscs"=>"big5",
    "cn-big5"=>"big5",
    "csbig5"=>"big5",
    "x-x-big5"=>"big5",
    "cseucpkdfmtjapanese"=>"cp51932",
    "euc-jp"=>"cp51932",
    "x-euc-jp"=>"cp51932",
    "csiso2022jp"=>"cp50221",
    "iso-2022-jp"=>"cp50221",
    "csshiftjis"=>"Windows-31J",
    "ms932"=>"Windows-31J",
    "ms_kanji"=>"Windows-31J",
    "shift-jis"=>"Windows-31J",
    "shift_jis"=>"Windows-31J",
    "sjis"=>"Windows-31J",
    "windows-31j"=>"Windows-31J",
    "x-sjis"=>"Windows-31J",
    "cseuckr"=>"euc-kr",
    "csksc56011987"=>"euc-kr",
    "euc-kr"=>"euc-kr",
    "iso-ir-149"=>"euc-kr",
    "korean"=>"euc-kr",
    "ks_c_5601-1987"=>"euc-kr",
    "ks_c_5601-1989"=>"euc-kr",
    "ksc5601"=>"euc-kr",
    "ksc_5601"=>"euc-kr",
    "windows-949"=>"euc-kr",
    "utf-16be"=>"utf-16be",
    "utf-16"=>"utf-16le",
    "utf-16le"=>"utf-16le",
  } # :nodoc:
  Ractor.make_shareable(WEB_ENCODINGS_) if defined?(Ractor)

  # :nodoc:
  # return encoding or nil
  # http://encoding.spec.whatwg.org/#concept-encoding-get
  def self.get_encoding(label)
    Encoding.find(WEB_ENCODINGS_[label.to_str.strip.downcase]) rescue nil
  end
end # module URI

module Kernel

  #
  # Returns a \URI object derived from the given +uri+,
  # which may be a \URI string or an existing \URI object:
  #
  #   # Returns a new URI.
  #   uri = URI('http://github.com/ruby/ruby')
  #   # => #<URI::HTTP http://github.com/ruby/ruby>
  #   # Returns the given URI.
  #   URI(uri)
  #   # => #<URI::HTTP http://github.com/ruby/ruby>
  #
  def URI(uri)
    if uri.is_a?(URI::Generic)
      uri
    elsif uri = String.try_convert(uri)
      URI.parse(uri)
    else
      raise ArgumentError,
        "bad argument (expected URI object or URI string)"
    end
  end
  module_function :URI
end
                                                                                                                                                                                               ruby/uri/wss.rb                                                                                     0000644                 00000001023 15040313432 0007454 0                                                                                                    ustar 00                                                                                                                                                                                                                                                       # frozen_string_literal: false
# = uri/wss.rb
#
# Author:: Matt Muller <mamuller@amazon.com>
# License:: You can redistribute it and/or modify it under the same term as Ruby.
#
# See URI for general documentation
#

require_relative 'ws'

module URI

  # The default port for WSS URIs is 443, and the scheme is 'wss:' rather
  # than 'ws:'. Other than that, WSS URIs are identical to WS URIs;
  # see URI::WS.
  class WSS < WS
    # A Default port of 443 for URI::WSS
    DEFAULT_PORT = 443
  end

  register_scheme 'WSS', WSS
end
                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                             ruby/uri/file.rb                                                                                    0000644                 00000004365 15040313432 0007573 0                                                                                                    ustar 00                                                                                                                                                                                                                                                       # frozen_string_literal: true

require_relative 'generic'

module URI

  #
  # The "file" URI is defined by RFC8089.
  #
  class File < Generic
    # A Default port of nil for URI::File.
    DEFAULT_PORT = nil

    #
    # An Array of the available components for URI::File.
    #
    COMPONENT = [
      :scheme,
      :host,
      :path
    ].freeze

    #
    # == Description
    #
    # Creates a new URI::File object from components, with syntax checking.
    #
    # The components accepted are +host+ and +path+.
    #
    # The components should be provided either as an Array, or as a Hash
    # with keys formed by preceding the component names with a colon.
    #
    # If an Array is used, the components must be passed in the
    # order <code>[host, path]</code>.
    #
    # A path from e.g. the File class should be escaped before
    # being passed.
    #
    # Examples:
    #
    #     require 'uri'
    #
    #     uri1 = URI::File.build(['host.example.com', '/path/file.zip'])
    #     uri1.to_s  # => "file://host.example.com/path/file.zip"
    #
    #     uri2 = URI::File.build({:host => 'host.example.com',
    #       :path => '/ruby/src'})
    #     uri2.to_s  # => "file://host.example.com/ruby/src"
    #
    #     uri3 = URI::File.build({:path => URI::escape('/path/my file.txt')})
    #     uri3.to_s  # => "file:///path/my%20file.txt"
    #
    def self.build(args)
      tmp = Util::make_components_hash(self, args)
      super(tmp)
    end

    # Protected setter for the host component +v+.
    #
    # See also URI::Generic.host=.
    #
    def set_host(v)
      v = "" if v.nil? || v == "localhost"
      @host = v
    end

    # do nothing
    def set_port(v)
    end

    # raise InvalidURIError
    def check_userinfo(user)
      raise URI::InvalidURIError, "cannot set userinfo for file URI"
    end

    # raise InvalidURIError
    def check_user(user)
      raise URI::InvalidURIError, "cannot set user for file URI"
    end

    # raise InvalidURIError
    def check_password(user)
      raise URI::InvalidURIError, "cannot set password for file URI"
    end

    # do nothing
    def set_userinfo(v)
    end

    # do nothing
    def set_user(v)
    end

    # do nothing
    def set_password(v)
    end
  end

  register_scheme 'FILE', File
end
                                                                                                                                                                                                                                                                           ruby/uri/ldap.rb                                                                                    0000644                 00000013437 15040313432 0007574 0                                                                                                    ustar 00                                                                                                                                                                                                                                                       # frozen_string_literal: false
# = uri/ldap.rb
#
# Author::
#  Takaaki Tateishi <ttate@jaist.ac.jp>
#  Akira Yamada <akira@ruby-lang.org>
# License::
#   URI::LDAP is copyrighted free software by Takaaki Tateishi and Akira Yamada.
#   You can redistribute it and/or modify it under the same term as Ruby.
#
# See URI for general documentation
#

require_relative 'generic'

module URI

  #
  # LDAP URI SCHEMA (described in RFC2255).
  #--
  # ldap://<host>/<dn>[?<attrs>[?<scope>[?<filter>[?<extensions>]]]]
  #++
  class LDAP < Generic

    # A Default port of 389 for URI::LDAP.
    DEFAULT_PORT = 389

    # An Array of the available components for URI::LDAP.
    COMPONENT = [
      :scheme,
      :host, :port,
      :dn,
      :attributes,
      :scope,
      :filter,
      :extensions,
    ].freeze

    # Scopes available for the starting point.
    #
    # * SCOPE_BASE - the Base DN
    # * SCOPE_ONE  - one level under the Base DN, not including the base DN and
    #   not including any entries under this
    # * SCOPE_SUB  - subtrees, all entries at all levels
    #
    SCOPE = [
      SCOPE_ONE = 'one',
      SCOPE_SUB = 'sub',
      SCOPE_BASE = 'base',
    ].freeze

    #
    # == Description
    #
    # Creates a new URI::LDAP object from components, with syntax checking.
    #
    # The components accepted are host, port, dn, attributes,
    # scope, filter, and extensions.
    #
    # The components should be provided either as an Array, or as a Hash
    # with keys formed by preceding the component names with a colon.
    #
    # If an Array is used, the components must be passed in the
    # order <code>[host, port, dn, attributes, scope, filter, extensions]</code>.
    #
    # Example:
    #
    #     uri = URI::LDAP.build({:host => 'ldap.example.com',
    #       :dn => '/dc=example'})
    #
    #     uri = URI::LDAP.build(["ldap.example.com", nil,
    #       "/dc=example;dc=com", "query", nil, nil, nil])
    #
    def self.build(args)
      tmp = Util::make_components_hash(self, args)

      if tmp[:dn]
        tmp[:path] = tmp[:dn]
      end

      query = []
      [:extensions, :filter, :scope, :attributes].collect do |x|
        next if !tmp[x] && query.size == 0
        query.unshift(tmp[x])
      end

      tmp[:query] = query.join('?')

      return super(tmp)
    end

    #
    # == Description
    #
    # Creates a new URI::LDAP object from generic URI components as per
    # RFC 2396. No LDAP-specific syntax checking is performed.
    #
    # Arguments are +scheme+, +userinfo+, +host+, +port+, +registry+, +path+,
    # +opaque+, +query+, and +fragment+, in that order.
    #
    # Example:
    #
    #     uri = URI::LDAP.new("ldap", nil, "ldap.example.com", nil, nil,
    #       "/dc=example;dc=com", nil, "query", nil)
    #
    # See also URI::Generic.new.
    #
    def initialize(*arg)
      super(*arg)

      if @fragment
        raise InvalidURIError, 'bad LDAP URL'
      end

      parse_dn
      parse_query
    end

    # Private method to cleanup +dn+ from using the +path+ component attribute.
    def parse_dn
      raise InvalidURIError, 'bad LDAP URL' unless @path
      @dn = @path[1..-1]
    end
    private :parse_dn

    # Private method to cleanup +attributes+, +scope+, +filter+, and +extensions+
    # from using the +query+ component attribute.
    def parse_query
      @attributes = nil
      @scope      = nil
      @filter     = nil
      @extensions = nil

      if @query
        attrs, scope, filter, extensions = @query.split('?')

        @attributes = attrs if attrs && attrs.size > 0
        @scope      = scope if scope && scope.size > 0
        @filter     = filter if filter && filter.size > 0
        @extensions = extensions if extensions && extensions.size > 0
      end
    end
    private :parse_query

    # Private method to assemble +query+ from +attributes+, +scope+, +filter+, and +extensions+.
    def build_path_query
      @path = '/' + @dn

      query = []
      [@extensions, @filter, @scope, @attributes].each do |x|
        next if !x && query.size == 0
        query.unshift(x)
      end
      @query = query.join('?')
    end
    private :build_path_query

    # Returns dn.
    def dn
      @dn
    end

    # Private setter for dn +val+.
    def set_dn(val)
      @dn = val
      build_path_query
      @dn
    end
    protected :set_dn

    # Setter for dn +val+.
    def dn=(val)
      set_dn(val)
      val
    end

    # Returns attributes.
    def attributes
      @attributes
    end

    # Private setter for attributes +val+.
    def set_attributes(val)
      @attributes = val
      build_path_query
      @attributes
    end
    protected :set_attributes

    # Setter for attributes +val+.
    def attributes=(val)
      set_attributes(val)
      val
    end

    # Returns scope.
    def scope
      @scope
    end

    # Private setter for scope +val+.
    def set_scope(val)
      @scope = val
      build_path_query
      @scope
    end
    protected :set_scope

    # Setter for scope +val+.
    def scope=(val)
      set_scope(val)
      val
    end

    # Returns filter.
    def filter
      @filter
    end

    # Private setter for filter +val+.
    def set_filter(val)
      @filter = val
      build_path_query
      @filter
    end
    protected :set_filter

    # Setter for filter +val+.
    def filter=(val)
      set_filter(val)
      val
    end

    # Returns extensions.
    def extensions
      @extensions
    end

    # Private setter for extensions +val+.
    def set_extensions(val)
      @extensions = val
      build_path_query
      @extensions
    end
    protected :set_extensions

    # Setter for extensions +val+.
    def extensions=(val)
      set_extensions(val)
      val
    end

    # Checks if URI has a path.
    # For URI::LDAP this will return +false+.
    def hierarchical?
      false
    end
  end

  register_scheme 'LDAP', LDAP
end
                                                                                                                                                                                                                                 ruby/uri/generic.rb                                                                                 0000644                 00000112064 15040313432 0010264 0                                                                                                    ustar 00                                                                                                                                                                                                                                                       # frozen_string_literal: true

# = uri/generic.rb
#
# Author:: Akira Yamada <akira@ruby-lang.org>
# License:: You can redistribute it and/or modify it under the same term as Ruby.
#
# See URI for general documentation
#

require_relative 'common'
autoload :IPSocket, 'socket'
autoload :IPAddr, 'ipaddr'

module URI

  #
  # Base class for all URI classes.
  # Implements generic URI syntax as per RFC 2396.
  #
  class Generic
    include URI

    #
    # A Default port of nil for URI::Generic.
    #
    DEFAULT_PORT = nil

    #
    # Returns default port.
    #
    def self.default_port
      self::DEFAULT_PORT
    end

    #
    # Returns default port.
    #
    def default_port
      self.class.default_port
    end

    #
    # An Array of the available components for URI::Generic.
    #
    COMPONENT = [
      :scheme,
      :userinfo, :host, :port, :registry,
      :path, :opaque,
      :query,
      :fragment
    ].freeze

    #
    # Components of the URI in the order.
    #
    def self.component
      self::COMPONENT
    end

    USE_REGISTRY = false # :nodoc:

    def self.use_registry # :nodoc:
      self::USE_REGISTRY
    end

    #
    # == Synopsis
    #
    # See ::new.
    #
    # == Description
    #
    # At first, tries to create a new URI::Generic instance using
    # URI::Generic::build. But, if exception URI::InvalidComponentError is raised,
    # then it does URI::Escape.escape all URI components and tries again.
    #
    def self.build2(args)
      begin
        return self.build(args)
      rescue InvalidComponentError
        if args.kind_of?(Array)
          return self.build(args.collect{|x|
            if x.is_a?(String)
              URI::RFC2396_PARSER.escape(x)
            else
              x
            end
          })
        elsif args.kind_of?(Hash)
          tmp = {}
          args.each do |key, value|
            tmp[key] = if value
                URI::RFC2396_PARSER.escape(value)
              else
                value
              end
          end
          return self.build(tmp)
        end
      end
    end

    #
    # == Synopsis
    #
    # See ::new.
    #
    # == Description
    #
    # Creates a new URI::Generic instance from components of URI::Generic
    # with check.  Components are: scheme, userinfo, host, port, registry, path,
    # opaque, query, and fragment. You can provide arguments either by an Array or a Hash.
    # See ::new for hash keys to use or for order of array items.
    #
    def self.build(args)
      if args.kind_of?(Array) &&
          args.size == ::URI::Generic::COMPONENT.size
        tmp = args.dup
      elsif args.kind_of?(Hash)
        tmp = ::URI::Generic::COMPONENT.collect do |c|
          if args.include?(c)
            args[c]
          else
            nil
          end
        end
      else
        component = self.class.component rescue ::URI::Generic::COMPONENT
        raise ArgumentError,
        "expected Array of or Hash of components of #{self.class} (#{component.join(', ')})"
      end

      tmp << nil
      tmp << true
      return self.new(*tmp)
    end

    #
    # == Args
    #
    # +scheme+::
    #   Protocol scheme, i.e. 'http','ftp','mailto' and so on.
    # +userinfo+::
    #   User name and password, i.e. 'sdmitry:bla'.
    # +host+::
    #   Server host name.
    # +port+::
    #   Server port.
    # +registry+::
    #   Registry of naming authorities.
    # +path+::
    #   Path on server.
    # +opaque+::
    #   Opaque part.
    # +query+::
    #   Query data.
    # +fragment+::
    #   Part of the URI after '#' character.
    # +parser+::
    #   Parser for internal use [URI::DEFAULT_PARSER by default].
    # +arg_check+::
    #   Check arguments [false by default].
    #
    # == Description
    #
    # Creates a new URI::Generic instance from ``generic'' components without check.
    #
    def initialize(scheme,
                   userinfo, host, port, registry,
                   path, opaque,
                   query,
                   fragment,
                   parser = DEFAULT_PARSER,
                   arg_check = false)
      @scheme = nil
      @user = nil
      @password = nil
      @host = nil
      @port = nil
      @path = nil
      @query = nil
      @opaque = nil
      @fragment = nil
      @parser = parser == DEFAULT_PARSER ? nil : parser

      if arg_check
        self.scheme = scheme
        self.userinfo = userinfo
        self.hostname = host
        self.port = port
        self.path = path
        self.query = query
        self.opaque = opaque
        self.fragment = fragment
      else
        self.set_scheme(scheme)
        self.set_userinfo(userinfo)
        self.set_host(host)
        self.set_port(port)
        self.set_path(path)
        self.query = query
        self.set_opaque(opaque)
        self.fragment=(fragment)
      end
      if registry
        raise InvalidURIError,
          "the scheme #{@scheme} does not accept registry part: #{registry} (or bad hostname?)"
      end

      @scheme&.freeze
      self.set_path('') if !@path && !@opaque # (see RFC2396 Section 5.2)
      self.set_port(self.default_port) if self.default_port && !@port
    end

    #
    # Returns the scheme component of the URI.
    #
    #   URI("http://foo/bar/baz").scheme #=> "http"
    #
    attr_reader :scheme

    # Returns the host component of the URI.
    #
    #   URI("http://foo/bar/baz").host #=> "foo"
    #
    # It returns nil if no host component exists.
    #
    #   URI("mailto:foo@example.org").host #=> nil
    #
    # The component does not contain the port number.
    #
    #   URI("http://foo:8080/bar/baz").host #=> "foo"
    #
    # Since IPv6 addresses are wrapped with brackets in URIs,
    # this method returns IPv6 addresses wrapped with brackets.
    # This form is not appropriate to pass to socket methods such as TCPSocket.open.
    # If unwrapped host names are required, use the #hostname method.
    #
    #   URI("http://[::1]/bar/baz").host     #=> "[::1]"
    #   URI("http://[::1]/bar/baz").hostname #=> "::1"
    #
    attr_reader :host

    # Returns the port component of the URI.
    #
    #   URI("http://foo/bar/baz").port      #=> 80
    #   URI("http://foo:8080/bar/baz").port #=> 8080
    #
    attr_reader :port

    def registry # :nodoc:
      nil
    end

    # Returns the path component of the URI.
    #
    #   URI("http://foo/bar/baz").path #=> "/bar/baz"
    #
    attr_reader :path

    # Returns the query component of the URI.
    #
    #   URI("http://foo/bar/baz?search=FooBar").query #=> "search=FooBar"
    #
    attr_reader :query

    # Returns the opaque part of the URI.
    #
    #   URI("mailto:foo@example.org").opaque #=> "foo@example.org"
    #   URI("http://foo/bar/baz").opaque     #=> nil
    #
    # The portion of the path that does not make use of the slash '/'.
    # The path typically refers to an absolute path or an opaque part.
    # (See RFC2396 Section 3 and 5.2.)
    #
    attr_reader :opaque

    # Returns the fragment component of the URI.
    #
    #   URI("http://foo/bar/baz?search=FooBar#ponies").fragment #=> "ponies"
    #
    attr_reader :fragment

    # Returns the parser to be used.
    #
    # Unless a 