+,
  #   using {File.chown}[rdoc-ref:File.chown].
  # - <tt>preserve: true</tt> - preserve timestamps
  #   using {File.utime}[rdoc-ref:File.utime].
  # - <tt>verbose: true</tt> - prints an equivalent command:
  #
  #     FileUtils.install('src0.txt', 'dest0.txt', noop: true, verbose: true)
  #     FileUtils.install('src1.txt', 'dest1.txt', noop: true, verbose: true)
  #     FileUtils.install('src2.txt', 'dest2', noop: true, verbose: true)
  #
  #   Output:
  #
  #     install -c src0.txt dest0.txt
  #     install -c src1.txt dest1.txt
  #     install -c src2.txt dest2
  #
  # Related: {methods for copying}[rdoc-ref:FileUtils@Copying].
  #
  def install(src, dest, mode: nil, owner: nil, group: nil, preserve: nil,
              noop: nil, verbose: nil)
    if verbose
      msg = +"install -c"
      msg << ' -p' if preserve
      msg << ' -m ' << mode_to_s(mode) if mode
      msg << " -o #{owner}" if owner
      msg << " -g #{group}" if group
      msg << ' ' << [src,dest].flatten.join(' ')
      fu_output_message msg
    end
    return if noop
    uid = fu_get_uid(owner)
    gid = fu_get_gid(group)
    fu_each_src_dest(src, dest) do |s, d|
      st = File.stat(s)
      unless File.exist?(d) and compare_file(s, d)
        remove_file d, true
        copy_file s, d
        File.utime st.atime, st.mtime, d if preserve
        File.chmod fu_mode(mode, st), d if mode
        File.chown uid, gid, d if uid or gid
      end
    end
  end
  module_function :install

  def user_mask(target)  #:nodoc:
    target.each_char.inject(0) do |mask, chr|
      case chr
      when "u"
        mask | 04700
      when "g"
        mask | 02070
      when "o"
        mask | 01007
      when "a"
        mask | 07777
      else
        raise ArgumentError, "invalid `who' symbol in file mode: #{chr}"
      end
    end
  end
  private_module_function :user_mask

  def apply_mask(mode, user_mask, op, mode_mask)   #:nodoc:
    case op
    when '='
      (mode & ~user_mask) | (user_mask & mode_mask)
    when '+'
      mode | (user_mask & mode_mask)
    when '-'
      mode & ~(user_mask & mode_mask)
    end
  end
  private_module_function :apply_mask

  def symbolic_modes_to_i(mode_sym, path)  #:nodoc:
    path = File.stat(path) unless File::Stat === path
    mode = path.mode
    mode_sym.split(/,/).inject(mode & 07777) do |current_mode, clause|
      target, *actions = clause.split(/([=+-])/)
      raise ArgumentError, "invalid file mode: #{mode_sym}" if actions.empty?
      target = 'a' if target.empty?
      user_mask = user_mask(target)
      actions.each_slice(2) do |op, perm|
        need_apply = op == '='
        mode_mask = (perm || '').each_char.inject(0) do |mask, chr|
          case chr
          when "r"
            mask | 0444
          when "w"
            mask | 0222
          when "x"
            mask | 0111
          when "X"
            if path.directory?
              mask | 0111
            else
              mask
            end
          when "s"
            mask | 06000
          when "t"
            mask | 01000
          when "u", "g", "o"
            if mask.nonzero?
              current_mode = apply_mask(current_mode, user_mask, op, mask)
            end
            need_apply = false
            copy_mask = user_mask(chr)
            (current_mode & copy_mask) / (copy_mask & 0111) * (user_mask & 0111)
          else
            raise ArgumentError, "invalid `perm' symbol in file mode: #{chr}"
          end
        end

        if mode_mask.nonzero? || need_apply
          current_mode = apply_mask(current_mode, user_mask, op, mode_mask)
        end
      end
      current_mode
    end
  end
  private_module_function :symbolic_modes_to_i

  def fu_mode(mode, path)  #:nodoc:
    mode.is_a?(String) ? symbolic_modes_to_i(mode, path) : mode
  end
  private_module_function :fu_mode

  def mode_to_s(mode)  #:nodoc:
    mode.is_a?(String) ? mode : "%o" % mode
  end
  private_module_function :mode_to_s

  # Changes permissions on the entries at the paths given in +list+
  # (a single path or an array of paths)
  # to the permissions given by +mode+;
  # returns +list+ if it is an array, <tt>[list]</tt> otherwise:
  #
  # - Modifies each entry that is a regular file using
  #   {File.chmod}[rdoc-ref:File.chmod].
  # - Modifies each entry that is a symbolic link using
  #   {File.lchmod}[rdoc-ref:File.lchmod].
  #
  # Argument +list+ or its elements
  # should be {interpretable as paths}[rdoc-ref:FileUtils@Path+Arguments].
  #
  # Argument +mode+ may be either an integer or a string:
  #
  # - \Integer +mode+: represents the permission bits to be set:
  #
  #     FileUtils.chmod(0755, 'src0.txt')
  #     FileUtils.chmod(0644, ['src0.txt', 'src0.dat'])
  #
  # - \String +mode+: represents the permissions to be set:
  #
  #   The string is of the form <tt>[targets][[operator][perms[,perms]]</tt>, where:
  #
  #   - +targets+ may be any combination of these letters:
  #
  #     - <tt>'u'</tt>: permissions apply to the file's owner.
  #     - <tt>'g'</tt>: permissions apply to users in the file's group.
  #     - <tt>'o'</tt>: permissions apply to other users not in the file's group.
  #     - <tt>'a'</tt> (the default): permissions apply to all users.
  #
  #   - +operator+ may be one of these letters:
  #
  #     - <tt>'+'</tt>: adds permissions.
  #     - <tt>'-'</tt>: removes permissions.
  #     - <tt>'='</tt>: sets (replaces) permissions.
  #
  #   - +perms+ (may be repeated, with separating commas)
  #     may be any combination of these letters:
  #
  #     - <tt>'r'</tt>: Read.
  #     - <tt>'w'</tt>: Write.
  #     - <tt>'x'</tt>: Execute (search, for a directory).
  #     - <tt>'X'</tt>: Search (for a directories only;
  #       must be used with <tt>'+'</tt>)
  #     - <tt>'s'</tt>: Uid or gid.
  #     - <tt>'t'</tt>: Sticky bit.
  #
  #   Examples:
  #
  #     FileUtils.chmod('u=wrx,go=rx', 'src1.txt')
  #     FileUtils.chmod('u=wrx,go=rx', '/usr/bin/ruby')
  #
  # Keyword arguments:
  #
  # - <tt>noop: true</tt> - does not change permissions; returns +nil+.
  # - <tt>verbose: true</tt> - prints an equivalent command:
  #
  #     FileUtils.chmod(0755, 'src0.txt', noop: true, verbose: true)
  #     FileUtils.chmod(0644, ['src0.txt', 'src0.dat'], noop: true, verbose: true)
  #     FileUtils.chmod('u=wrx,go=rx', 'src1.txt', noop: true, verbose: true)
  #     FileUtils.chmod('u=wrx,go=rx', '/usr/bin/ruby', noop: true, verbose: true)
  #
  #   Output:
  #
  #     chmod 755 src0.txt
  #     chmod 644 src0.txt src0.dat
  #     chmod u=wrx,go=rx src1.txt
  #     chmod u=wrx,go=rx /usr/bin/ruby
  #
  # Related: FileUtils.chmod_R.
  #
  def chmod(mode, list, noop: nil, verbose: nil)
    list = fu_list(list)
    fu_output_message sprintf('chmod %s %s', mode_to_s(mode), list.join(' ')) if verbose
    return if noop
    list.each do |path|
      Entry_.new(path).chmod(fu_mode(mode, path))
    end
  end
  module_function :chmod

  # Like FileUtils.chmod, but changes permissions recursively.
  #
  def chmod_R(mode, list, noop: nil, verbose: nil, force: nil)
    list = fu_list(list)
    fu_output_message sprintf('chmod -R%s %s %s',
                              (force ? 'f' : ''),
                              mode_to_s(mode), list.join(' ')) if verbose
    return if noop
    list.each do |root|
      Entry_.new(root).traverse do |ent|
        begin
          ent.chmod(fu_mode(mode, ent.path))
        rescue
          raise unless force
        end
      end
    end
  end
  module_function :chmod_R

  # Changes the owner and group on the entries at the paths given in +list+
  # (a single path or an array of paths)
  # to the given +user+ and +group+;
  # returns +list+ if it is an array, <tt>[list]</tt> otherwise:
  #
  # - Modifies each entry that is a regular file using
  #   {File.chown}[rdoc-ref:File.chown].
  # - Modifies each entry that is a symbolic link using
  #   {File.lchown}[rdoc-ref:File.lchown].
  #
  # Argument +list+ or its elements
  # should be {interpretable as paths}[rdoc-ref:FileUtils@Path+Arguments].
  #
  # User and group:
  #
  # - Argument +user+ may be a user name or a user id;
  #   if +nil+ or +-1+, the user is not changed.
  # - Argument +group+ may be a group name or a group id;
  #   if +nil+ or +-1+, the group is not changed.
  # - The user must be a member of the group.
  #
  # Examples:
  #
  #   # One path.
  #   # User and group as string names.
  #   File.stat('src0.txt').uid # => 1004
  #   File.stat('src0.txt').gid # => 1004
  #   FileUtils.chown('user2', 'group1', 'src0.txt')
  #   File.stat('src0.txt').uid # => 1006
  #   File.stat('src0.txt').gid # => 1005
  #
  #   # User and group as uid and gid.
  #   FileUtils.chown(1004, 1004, 'src0.txt')
  #   File.stat('src0.txt').uid # => 1004
  #   File.stat('src0.txt').gid # => 1004
  #
  #   # Array of paths.
  #   FileUtils.chown(1006, 1005, ['src0.txt', 'src0.dat'])
  #
  #   # Directory (not recursive).
  #   FileUtils.chown('user2', 'group1', '.')
  #
  # Keyword arguments:
  #
  # - <tt>noop: true</tt> - does not change permissions; returns +nil+.
  # - <tt>verbose: true</tt> - prints an equivalent command:
  #
  #     FileUtils.chown('user2', 'group1', 'src0.txt', noop: true, verbose: true)
  #     FileUtils.chown(1004, 1004, 'src0.txt', noop: true, verbose: true)
  #     FileUtils.chown(1006, 1005, ['src0.txt', 'src0.dat'], noop: true, verbose: true)
  #     FileUtils.chown('user2', 'group1', path, noop: true, verbose: true)
  #     FileUtils.chown('user2', 'group1', '.', noop: true, verbose: true)
  #
  #   Output:
  #
  #     chown user2:group1 src0.txt
  #     chown 1004:1004 src0.txt
  #     chown 1006:1005 src0.txt src0.dat
  #     chown user2:group1 src0.txt
  #     chown user2:group1 .
  #
  # Related: FileUtils.chown_R.
  #
  def chown(user, group, list, noop: nil, verbose: nil)
    list = fu_list(list)
    fu_output_message sprintf('chown %s %s',
                              (group ? "#{user}:#{group}" : user || ':'),
                              list.join(' ')) if verbose
    return if noop
    uid = fu_get_uid(user)
    gid = fu_get_gid(group)
    list.each do |path|
      Entry_.new(path).chown uid, gid
    end
  end
  module_function :chown

  # Like FileUtils.chown, but changes owner and group recursively.
  #
  def chown_R(user, group, list, noop: nil, verbose: nil, force: nil)
    list = fu_list(list)
    fu_output_message sprintf('chown -R%s %s %s',
                              (force ? 'f' : ''),
                              (group ? "#{user}:#{group}" : user || ':'),
                              list.join(' ')) if verbose
    return if noop
    uid = fu_get_uid(user)
    gid = fu_get_gid(group)
    list.each do |root|
      Entry_.new(root).traverse do |ent|
        begin
          ent.chown uid, gid
        rescue
          raise unless force
        end
      end
    end
  end
  module_function :chown_R

  def fu_get_uid(user)   #:nodoc:
    return nil unless user
    case user
    when Integer
      user
    when /\A\d+\z/
      user.to_i
    else
      require 'etc'
      Etc.getpwnam(user) ? Etc.getpwnam(user).uid : nil
    end
  end
  private_module_function :fu_get_uid

  def fu_get_gid(group)   #:nodoc:
    return nil unless group
    case group
    when Integer
      group
    when /\A\d+\z/
      group.to_i
    else
      require 'etc'
      Etc.getgrnam(group) ? Etc.getgrnam(group).gid : nil
    end
  end
  private_module_function :fu_get_gid

  # Updates modification times (mtime) and access times (atime)
  # of the entries given by the paths in +list+
  # (a single path or an array of paths);
  # returns +list+ if it is an array, <tt>[list]</tt> otherwise.
  #
  # By default, creates an empty file for any path to a non-existent entry;
  # use keyword argument +nocreate+ to raise an exception instead.
  #
  # Argument +list+ or its elements
  # should be {interpretable as paths}[rdoc-ref:FileUtils@Path+Arguments].
  #
  # Examples:
  #
  #   # Single path.
  #   f = File.new('src0.txt') # Existing file.
  #   f.atime # => 2022-06-10 11:11:21.200277 -0700
  #   f.mtime # => 2022-06-10 11:11:21.200277 -0700
  #   FileUtils.touch('src0.txt')
  #   f = File.new('src0.txt')
  #   f.atime # => 2022-06-11 08:28:09.8185343 -0700
  #   f.mtime # => 2022-06-11 08:28:09.8185343 -0700
  #
  #   # Array of paths.
  #   FileUtils.touch(['src0.txt', 'src0.dat'])
  #
  # Keyword arguments:
  #
  # - <tt>mtime: <i>time</i></tt> - sets the entry's mtime to the given time,
  #   instead of the current time.
  # - <tt>nocreate: true</tt> - raises an exception if the entry does not exist.
  # - <tt>noop: true</tt> - does not touch entries; returns +nil+.
  # - <tt>verbose: true</tt> - prints an equivalent command:
  #
  #     FileUtils.touch('src0.txt', noop: true, verbose: true)
  #     FileUtils.touch(['src0.txt', 'src0.dat'], noop: true, verbose: true)
  #     FileUtils.touch(path, noop: true, verbose: true)
  #
  #   Output:
  #
  #     touch src0.txt
  #     touch src0.txt src0.dat
  #     touch src0.txt
  #
  # Related: FileUtils.uptodate?.
  #
  def touch(list, noop: nil, verbose: nil, mtime: nil, nocreate: nil)
    list = fu_list(list)
    t = mtime
    if verbose
      fu_output_message "touch #{nocreate ? '-c ' : ''}#{t ? t.strftime('-t %Y%m%d%H%M.%S ') : ''}#{list.join ' '}"
    end
    return if noop
    list.each do |path|
      created = nocreate
      begin
        File.utime(t, t, path)
      rescue Errno::ENOENT
        raise if created
        File.open(path, 'a') {
          ;
        }
        created = true
        retry if t
      end
    end
  end
  module_function :touch

  private

  module StreamUtils_
    private

    case (defined?(::RbConfig) ? ::RbConfig::CONFIG['host_os'] : ::RUBY_PLATFORM)
    when /mswin|mingw/
      def fu_windows?; true end
    else
      def fu_windows?; false end
    end

    def fu_copy_stream0(src, dest, blksize = nil)   #:nodoc:
      IO.copy_stream(src, dest)
    end

    def fu_stream_blksize(*streams)
      streams.each do |s|
        next unless s.respond_to?(:stat)
        size = fu_blksize(s.stat)
        return size if size
      end
      fu_default_blksize()
    end

    def fu_blksize(st)
      s = st.blksize
      return nil unless s
      return nil if s == 0
      s
    end

                               ruby/reline.rb                                                                                      0000644                 00000035665 15040313430 0007340 0                                                                                                    ustar 00                                                                                                                                                                                                                                                       require 'io/console'
require 'forwardable'
require 'reline/version'
require 'reline/config'
require 'reline/key_actor'
require 'reline/key_stroke'
require 'reline/line_editor'
require 'reline/history'
require 'reline/io'
require 'reline/face'
require 'rbconfig'

module Reline
  # NOTE: For making compatible with the rb-readline gem
  FILENAME_COMPLETION_PROC = nil
  USERNAME_COMPLETION_PROC = nil

  class ConfigEncodingConversionError < StandardError; end

  # EOF key: { char: nil, method_symbol: nil }
  # Other key: { char: String, method_symbol: Symbol }
  Key = Struct.new(:char, :method_symbol, :unused_boolean) do
    # For dialog_proc `key.match?(dialog.name)`
    def match?(sym)
      method_symbol && method_symbol == sym
    end
  end
  CursorPos = Struct.new(:x, :y)
  DialogRenderInfo = Struct.new(
    :pos,
    :contents,
    :face,
    :bg_color, # For the time being, this line should stay here for the compatibility with IRB.
    :width,
    :height,
    :scrollbar,
    keyword_init: true
  )

  class Core
    ATTR_READER_NAMES = %i(
      completion_append_character
      basic_word_break_characters
      completer_word_break_characters
      basic_quote_characters
      completer_quote_characters
      filename_quote_characters
      special_prefixes
      completion_proc
      output_modifier_proc
      prompt_proc
      auto_indent_proc
      pre_input_hook
      dig_perfect_match_proc
    ).each(