# frozen_string_literal: true
#--
# ERB::Compiler
#
# Compiles ERB templates into Ruby code; the compiled code produces the
# template result when evaluated. ERB::Compiler provides hooks to define how
# generated output is handled.
#
# Internally ERB does something like this to generate the code returned by
# ERB#src:
#
#   compiler = ERB::Compiler.new('<>')
#   compiler.pre_cmd    = ["_erbout=+''"]
#   compiler.put_cmd    = "_erbout.<<"
#   compiler.insert_cmd = "_erbout.<<"
#   compiler.post_cmd   = ["_erbout"]
#
#   code, enc = compiler.compile("Got <%= obj %>!\n")
#   puts code
#
# <i>Generates</i>:
#
#   #coding:UTF-8
#   _erbout=+''; _erbout.<< "Got ".freeze; _erbout.<<(( obj ).to_s); _erbout.<< "!\n".freeze; _erbout
#
# By default the output is sent to the print method.  For example:
#
#   compiler = ERB::Compiler.new('<>')
#   code, enc = compiler.compile("Got <%= obj %>!\n")
#   puts code
#
# <i>Generates</i>:
#
#   #coding:UTF-8
#   print "Got ".freeze; print(( obj ).to_s); print "!\n".freeze
#
# == Evaluation
#
# The compiled code can be used in any context where the names in the code
# correctly resolve. Using the last example, each of these print 'Got It!'
#
# Evaluate using a variable:
#
#   obj = 'It'
#   eval code
#
# Evaluate using an input:
#
#   mod = Module.new
#   mod.module_eval %{
#     def get(obj)
#       #{code}
#     end
#   }
#   extend mod
#   get('It')
#
# Evaluate using an accessor:
#
#   klass = Class.new Object
#   klass.class_eval %{
#     attr_accessor :obj
#     def initialize(obj)
#       @obj = obj
#     end
#     def get_it
#       #{code}
#     end
#   }
#   klass.new('It').get_it
#
# Good! See also ERB#def_method, ERB#def_module, and ERB#def_class.
class ERB::Compiler # :nodoc:
  class PercentLine # :nodoc:
    def initialize(str)
      @value = str
    end
    attr_reader :value
    alias :to_s :value
  end

  class Scanner # :nodoc:
    @scanner_map = {}
    class << self
      def register_scanner(klass, trim_mode, percent)
        @scanner_map[[trim_mode, percent]] = klass
      end
      alias :regist_scanner :register_scanner
    end

    def self.default_scanner=(klass)
      @default_scanner = klass
    end

    def self.make_scanner(src, trim_mode, percent)
      klass = @scanner_map.fetch([trim_mode, percent], @default_scanner)
      klass.new(src, trim_mode, percent)
    end

    DEFAULT_STAGS = %w(<%% <%= <%# <%).freeze
    DEFAULT_ETAGS = %w(%%> %>).freeze
    def initialize(src, trim_mode, percent)
      @src = src
      @stag = nil
      @stags = DEFAULT_STAGS
      @etags = DEFAULT_ETAGS
    end
    attr_accessor :stag
    attr_reader :stags, :etags

    def scan; end
  end

  class TrimScanner < Scanner # :nodoc:
    def initialize(src, trim_mode, percent)
      super
      @trim_mode = trim_mode
      @percent = percent
      if @trim_mode == '>'
        @scan_reg  = /(.*?)(%>\r?\n|#{(stags + etags).join('|')}|\n|\z)/m
        @scan_line = self.method(:trim_line1)
      elsif @trim_mode == '<>'
        @scan_reg  = /(.*?)(%>\r?\n|#{(stags + etags).join('|')}|\n|\z)/m
        @scan_line = self.method(:trim_line2)
      elsif @trim_mode == '-'
        @scan_reg  = /(.*?)(^[ \t]*<%\-|<%\-|-%>\r?\n|-%>|#{(stags + etags).join('|')}|\z)/m
        @scan_line = self.method(:explicit_trim_line)
      else
        @scan_reg  = /(.*?)(#{(stags + etags).join('|')}|\n|\z)/m
        @scan_line = self.method(:scan_line)
      end
    end

    def scan(&block)
      @stag = nil
      if @percent
        @src.each_line do |line|
          percent_line(line, &block)
        end
      else
        @scan_line.call(@src, &block)
      end
      nil
    end

    def percent_line(line, &block)
      if @stag || line[0] != ?%
        return @scan_line.call(line, &block)
      end

      line[0] = ''
      if line[0] == ?%
        @scan_line.call(line, &block)
      else
        yield(PercentLine.new(line.chomp))
      end
    end

    def scan_line(line)
      line.scan(@scan_reg) do |tokens|
        tokens.each do |token|
          next if token.empty?
          yield(token)
        end
      end
    end

    def trim_line1(line)
      line.scan(@scan_reg) do |tokens|
        tokens.each do |token|
          next if token.empty?
          if token == "%>\n" || token == "%>\r\n"
            yield('%>')
            yield(:cr)
          else
            yield(token)
          end
        end
      end
    end

    def trim_line2(line)
      head = nil
      line.scan(@scan_reg) do |tokens|
        tokens.each do |token|
          next if token.empty?
          head = token unless head
          if token == "%>\n" || token == "%>\r\n"
            yield('%>')
            if is_erb_stag?(head)
              yield(:cr)
            else
              yield("\n")
            end
            head = nil
          else
            yield(token)
            head = nil if token == "\n"
          end
        end
      end
    end

    def explicit_trim_line(line)
      line.scan(@scan_reg) do |tokens|
        tokens.each do |token|
          next if token.empty?
          if @stag.nil? && /[ \t]*<%-/ =~ token
            yield('<%')
          elsif @stag && (token == "-%>\n" || token == "-%>\r\n")
            yield('%>')
            yield(:cr)
          elsif @stag && token == '-%>'
            yield('%>')
          else
            yield(token)
          end
        end
      end
    end

    ERB_STAG = %w(<%= <%# <%)
    def is_erb_stag?(s)
      ERB_STAG.member?(s)
    end
  end

  Scanner.default_scanner = TrimScanner

  begin
    require 'strscan'
  rescue LoadError
  else
    class SimpleScanner < Scanner # :nodoc:
      def scan
        stag_reg = (stags == DEFAULT_STAGS) ? /(.*?)(<%[%=#]?|\z)/m : /(.*?)(#{stags.join('|')}|\z)/m
        etag_reg = (etags == DEFAULT_ETAGS) ? /(.*?)(%%?>|\z)/m : /(.*?)(#{etags.join('|')}|\z)/m
        scanner = StringScanner.new(@src)
        while ! scanner.eos?
          scanner.scan(@stag ? etag_reg : stag_reg)
          yield(scanner[1])
          yield(scanner[2])
        end
      end
    end
    Scanner.register_scanner(SimpleScanner, nil, false)

    class ExplicitScanner < Scanner # :nodoc:
      def scan
        stag_reg = /(.*?)(^[ \t]*<%-|<%-|#{stags.join('|')}|\z)/m
        etag_reg = /(.*?)(-%>|#{etags.join('|')}|\z)/m
        scanner = StringScanner.new(@src)
        while ! scanner.eos?
          scanner.scan(@stag ? etag_reg : stag_reg)
          yield(scanner[1])

          elem = scanner[2]
          if /[ \t]*<%-/ =~ elem
            yield('<%')
          elsif elem == '-%>'
            yield('%>')
            yield(:cr) if scanner.scan(/(\r?\n|\z)/)
          else
            yield(elem)
          end
        end
      end
    end
    Scanner.register_scanner(ExplicitScanner, '-', false)
  end

  class Buffer # :nodoc:
    def initialize(compiler, enc=nil, frozen=nil)
      @compiler = compiler
      @line = []
      @script = +''
      @script << "#coding:#{enc}\n" if enc
      @script << "#frozen-string-literal:#{frozen}\n" unless frozen.nil?
      @compiler.pre_cmd.each do |x|
        push(x)
      end
    end
    attr_reader :script

    def push(cmd)
      @line << cmd
    end

    def cr
      @script << (@line.join('; '))
      @line = []
      @script << "\n"
    end

    def close
      return unless @line
      @compiler.post_cmd.each do |x|
        push(x)
      end
      @script << (@line.join('; '))
      @line = nil
    end
  end

  def add_put_cmd(out, content)
    out.push("#{@put_cmd} #{content.dump}.freeze#{"\n" * content.count("\n")}")
  end

  def add_insert_cmd(out, content)
    out.push("#{@insert_cmd}((#{content}).to_s)")
  end

  # Compiles an ERB template into Ruby code.  Returns an array of the code
  # and encoding like ["code", Encoding].
  def compile(s)
    enc = s.encoding
    raise ArgumentError, "#{enc} is not ASCII compatible" if enc.dummy?
    s = s.b # see String#b
    magic_comment = detect_magic_comment(s, enc)
    out = Buffer.new(self, *magic_comment)

    self.content = +''
    scanner = make_scanner(s)
    scanner.scan do |token|
      next if token.nil?
      next if token == ''
      if scanner.stag.nil?
        compile_stag(token, out, scanner)
      else
        compile_etag(token, out, scanner)
      end
    end
    add_put_cmd(out, content) if content.size > 0
    out.close
    return out.script, *magic_comment
  end

  def compile_stag(stag, out, scanner)
    case stag
    when PercentLine
      add_put_cmd(out, content) if content.size > 0
      self.content = +''
      out.push(stag.to_s)
      out.cr
    when :cr
      out.cr
    when '<%', '<%=', '<%#'
      scanner.stag = stag
      add_put_cmd(out, content) if content.size > 0
      self.content = +''
    when "\n"
      content << "\n"
      add_put_cmd(out, content)
      self.content = +''
    when '<%%'
      content << '<%'
    else
      content << stag
    end
  end

  def compile_etag(etag, out, scanner)
    case etag
    when '%>'
      compile_content(scanner.stag, out)
      scanner.stag = nil
      self.content = +''
    when '%%>'
      content << '%>'
    else
      content << etag
    end
  end

  def compile_content(stag, out)
    case stag
    when '<%'
      if content[-1] == ?\n
        content.chop!
        out.push(content)
        out.cr
      else
        out.push(content)
      end
    when '<%='
      add_insert_cmd(out, content)
    when '<%#'
      out.push("\n" * content.count("\n")) # only adjust lineno
    end
  end

  def prepare_trim_mode(mode) # :nodoc:
    case mode
    when 1
      return [false, '>']
    when 2
      return [false, '<>']
    when 0, nil
      return [false, nil]
    when String
      unless mode.match?(/\A(%|-|>|<>){1,2}\z/)
        warn_invalid_trim_mode(mode, uplevel: 5)
      end

      perc = mode.include?('%')
      if mode.include?('-')
        return [perc, '-']
      elsif mode.include?('<>')
        return [perc, '<>']
      elsif mode.include?('>')
        return [perc, '>']
      else
        [perc, nil]
      end
    else
      warn_invalid_trim_mode(mode, uplevel: 5)
      return [false, nil]
    end
  end

  def make_scanner(src) # :nodoc:
    Scanner.make_scanner(src, @trim_mode, @percent)
  end

  # Construct a new compiler using the trim_mode. See ERB::new for available
  # trim modes.
  def initialize(trim_mode)
    @percent, @trim_mode = prepare_trim_mode(trim_mode)
    @put_cmd = 'print'
    @insert_cmd = @put_cmd
    @pre_cmd = []
    @post_cmd = []
  end
  attr_reader :percent, :trim_mode

  # The command to handle text that ends with a newline
  attr_accessor :put_cmd

  # The command to handle text that is inserted prior to a newline
  attr_accessor :insert_cmd

  # An array of commands prepended to compiled code
  attr_accessor :pre_cmd

  # An array of commands appended to compiled code
  attr_accessor :post_cmd

  private

  # A buffered text in #compile
  attr_accessor :content

  def detect_magic_comment(s, enc = nil)
    re = @percent ? /\G(?:<%#(.*)%>|%#(.*)\n)/ : /\G<%#(.*)%>/
    frozen = nil
    s.scan(re) do
      comment = $+
      comment = $1 if comment[/-\*-\s*([^\s].*?)\s*-\*-$/]
      case comment
      when %r"coding\s*[=:]\s*([[:alnum:]\-_]+)"
        enc = Encoding.find($1.sub(/-(?:mac|dos|unix)/i, ''))
      when %r"frozen[-_]string[-_]literal\s*:\s*([[:alnum:]]+)"
        frozen = $1
      end
    end
    return enc, frozen
  end

  def warn_invalid_trim_mode(mode, uplevel:)
    warn "Invalid ERB trim mode: #{mode.inspect} (trim_mode: nil, 0, 1, 2, or String composed of '%' and/or '-', '>', '<>')", uplevel: uplevel + 1
  end
end
                                                                                                                                                ruby/readline.rb                                                                                    0000644                 00000000327 15040313432 0007632 0                                                                                                    ustar 00                                                                                                                                                                                                                                                       begin
  require "readline.#{RbConfig::CONFIG["DLEXT"]}"
rescue LoadError
  require 'reline' unless defined? Reline
  Object.send(:remove_const, :Readline) if Object.const_defined?(:Readline)
  Readline = Reline
end
                                                                                                                                                                                                                                                                                                         ruby/objspace/trace.rb                                                                              0000644                 00000002235 15040313432 0010733 0                                                                                                    ustar 00                                                                                                                                                                                                                                                       # This is a simple tool to enable the object allocation tracer.
# When you have an object of unknown provenance, you can use this
# to investigate where the object in question is created.
#
# = Important notice
#
# This is only for debugging purpose. Do not use this in production.
# Require'ing this file immediately starts tracing the object allocation,
# which brings a large performance overhead.
#
# = Usage
#
# 1. Add `require "objspace/trace"` into your code (or add `-robjspace/trace` into the command line)
# 2. `p obj` will show the allocation site of `obj`
#
# Note: This redefines `Kernel#p` method, but not `Object#inspect`.
#
# = Examples
#
#   1: require "objspace/trace"
#   2:
#   3: obj = "str"
#   4:
#   5: p obj  #=> "str" @ test.rb:3

require 'objspace.so'

module Kernel
  remove_method :p
  define_method(:p) do |*objs|
    objs.each do |obj|
      file = ObjectSpace.allocation_sourcefile(obj)
      line = ObjectSpace.allocation_sourceline(obj)
      if file
        puts "#{ obj.inspect } @ #{ file }:#{ line }"
      else
        puts obj.inspect
      end
    end
  end
end

ObjectSpace.trace_object_allocations_start

warn "objspace/trace is enabled"
                                                                                                                                                                                                                                                                                                                                                                   ruby/error_highlight.rb                                                                             0000644                 00000000124 15040313433 0011223 0                                                                                                    ustar 00                                                                                                                                                                                                                                                       require_relative "error_highlight/base"
require_relative "error_highlight/core_ext"
                                                                                                                                                                                                                                                                                                                                                                                                                                            ruby/objspace.rb                                                                                    0000644                 00000010214 15040313433 0007632 0                                                                                                    ustar 00                                                                                                                                                                                                                                                       # frozen_string_literal: true

require 'objspace.so'

module ObjectSpace
  class << self
    private :_dump
    private :_dump_all
    private :_dump_shapes
  end

  module_function

  # Dump the contents of a ruby object as JSON.
  #
  # _output_ can be one of: +:stdout+, +:file+, +:string+, or IO object.
  #
  # * +:file+ means dumping to a tempfile and returning corresponding File object;
  # * +:stdout+ means printing the dump and returning +nil+;
  # * +:string+ means returning a string with the dump;
  # * if an instance of IO object is provided, the output goes there, and the object
  #   is returned.
  #
  # This method is only expected to work with C Ruby.
  # This is an experimental method and is subject to change.
  # In particular, the function signature and output format are
  # not guaranteed to be compatible in future versions of ruby.
  def dump(obj, output: :string)
    out = case output
    when :file, nil
      require 'tempfile'
      Tempfile.create(%w(rubyobj .json))
    when :stdout
      STDOUT
    when :string
      +''
    when IO
      output
    else
      raise ArgumentError, "wrong output option: #{output.inspect}"
    end

    ret = _dump(obj, out)
    return nil if output == :stdout
    ret
  end


  # Dump the contents of the ruby heap as JSON.
  #
  # _output_ argument is the same as for #dump.
  #
  # _full_ must be a boolean. If true, all heap slots are dumped including the empty ones (+T_NONE+).
  #
  # _since_ must be a non-negative integer or +nil+.
  #
  # If _since_ is a positive integer, only objects of that generation and
  # newer generations are dumped. The current generation can be accessed using
  # GC::count. Objects that were allocated without object allocation tracing enabled
  # are ignored. See ::trace_object_allocations for more information and
  # examples.
  #
  # If _since_ is omitted or is +nil+, all objects are dumped.
  #
  # _shapes_ must be a boolean or a non-negative integer.
  #
  # If _shapes_ is a positive integer, only shapes newer than the provided
  # shape id are dumped. The current shape_id can be accessed using <tt>RubyVM.stat(:next_shape_id)</tt>.
  #
  # If _shapes_ is +false+, no shapes are dumped.
  #
  # To only dump objects allocated past a certain point you can combine _since_ and _shapes_:
  #   ObjectSpace.trace_object_allocations
  #   GC.start
  #   gc_generation = GC.count
  #   shape_generation = RubyVM.stat(:next_shape_id)
  #   call_method_to_instrument
  #   ObjectSpace.dump_all(since: gc_generation, shapes: shape_generation)
  #
  # This method is only expected to work with C Ruby.
  # This is an experimental method and is subject to change.
  # In particular, the function signature and output format are
  # not guaranteed to be compatible in future versions of ruby.
  def dump_all(output: :file, full: false, since: nil, shapes: true)
    out = case output
    when :file, nil
      require 'tempfile'
      Tempfile.create(%w(rubyheap .json))
    when :stdout
      STDOUT
    when :string
      +''
    when IO
      output
    else
      raise ArgumentError, "wrong output option: #{output.inspect}"
    end

    shapes = 0 if shapes == true
    ret = _dump_all(out, full, since, shapes)
    return nil if output == :stdout
    ret
  end

  #  Dump the contents of the ruby shape tree as JSON.
  #
  #  _output_ argument is the same as for #dump.
  #
  #  If _since_ is a positive integer, only shapes newer than the provided
  #  shape id are dumped. The current shape_id can be accessed using <tt>RubyVM.stat(:next_shape_id)</tt>.
  #
  #  This method is only expected to work with C Ruby.
  #  This is an experimental method and is subject to change.
  #  In particular, the function signature and output format are
  #  not guaranteed to be compatible in future versions of ruby.
  def dump_shapes(output: :file, since: 0)
    out = case output
    when :file, nil
      require 'tempfile'
      Tempfile.create(%w(rubyshapes .json))
    when :stdout
      STDOUT
    when :string
      +''
    when IO
      output
    else
      raise ArgumentError, "wrong output option: #{output.inspect}"
    end

    ret = _dump_shapes(out, since)
    return nil if output == :stdout
    ret
  end
end
                                                                                                                                                                                                                                                                                                                                                                                    ruby/erb.rb                                                                                         0000644                 00000035043 15040313433 0006623 0                                                                                                    ustar 00                                                                                                                                                                                                                                                       # -*- coding: us-ascii -*-
# frozen_string_literal: true
# = ERB -- Ruby Templating
#
# Author:: Masatoshi SEKI
# Documentation:: James Edward Gray II, Gavin Sinclair, and Simon Chiang
#
# See ERB for primary documentation and ERB::Util for a couple of utility
# routines.
#
# Copyright (c) 1999-2000,2002,2003 Masatoshi SEKI
#
# You can redistribute it and/or modify it under the same terms as Ruby.

require 'cgi/util'
require 'erb/version'
require 'erb/compiler'
require 'erb/def_method'
require 'erb/util'

#
# = ERB -- Ruby Templating
#
# == Introduction
#
# ERB provides an easy to use but powerful templating system for Ruby.  Using
# ERB, actual Ruby code can be added to any plain text document for the
# purposes of generating document information details and/or flow control.
#
# A very simple example is this:
#
#   require 'erb'
#
#   x = 42
#   template = ERB.new <<-EOF
#     The value of x is: <%= x %>
#   EOF
#   puts template.result(binding)
#
# <em>Prints:</em> The value of x is: 42
#
# More complex examples are given below.
#
#
# == Recognized Tags
#
# ERB recognizes certain tags in the provided template and converts them based
# on the rules below:
#
#   <% Ruby code -- inline with output %>
#   <%= Ruby expression -- replace with result %>
#   <%# comment -- ignored -- useful in testing %> (`<% #` doesn't work. Don't use Ruby comments.)
#   % a line of Ruby code -- treated as <% line %> (optional -- see ERB.new)
#   %% replaced with % if first thing on a line and % processing is used
#   <%% or %%> -- replace with <% or %> respectively
#
# All other text is passed through ERB filtering unchanged.
#
#
# == Options
#
# There are several settings you can change when you use ERB:
# * the nature of the tags that are recognized;
# * the binding used to resolve local variables in the template.
#
# See the ERB.new and ERB#result methods for more detail.
#
# == Character encodings
#
# ERB (or Ruby code generated by ERB) returns a string in the same
# character encoding as the input string.  When the input string has
# a magic comment, however, it returns a string in the encoding specified
# by the magic comment.
#
#   # -*- coding: utf-8 -*-
#   require 'erb'
#
#   template = ERB.new <<EOF
#   <%#-*- coding: Big5 -*-%>
#     \_\_ENCODING\_\_ is <%= \_\_ENCODING\_\_ %>.
#   EOF
#   puts template.result
#
# <em>Prints:</em> \_\_ENCODING\_\_ is Big5.
#
#
# == Examples
#
# === Plain Text
#
# ERB is useful for any generic templating situation.  Note that in this example, we use the
# convenient "% at start of line" tag, and we quote the template literally with
# <tt>%q{...}</tt> to avoid trouble with the backslash.
#
#   require "erb"
#
#   # Create template.
#   template = %q{
#     From:  James Edward Gray II <james@grayproductions.net>
#     To:  <%= to %>
#     Subject:  Addressing Needs
#
#     <%= to[/\w+/] %>:
#
#     Just wanted to send a quick note assuring that your needs are being
#     addressed.
#
#     I want you to know that my team will keep working on the issues,
#     especially:
#
#     <%# ignore numerous minor requests -- focus on priorities %>
#     % priorities.each do |priority|
#       * <%= priority %>
#     % end
#
#     Thanks for your patience.
#
#     James Edward Gray II
#   }.gsub(/^  /, '')
#
#   message = ERB.new(template, trim_mode: "%<>")
#
#   # Set up template data.
#   to = "Community Spokesman <spokesman@ruby_community.org>"
#   priorities = [ "Run Ruby Quiz",
#                  "Document Modules",
#                  "Answer Questions on Ruby Talk" ]
#
#   # Produce result.
#   email = message.result
#   puts email
#
# <i>Generates:</i>
#
#   From:  James Edward Gray II <james@grayproductions.net>
#   To:  Community Spokesman <spokesman@ruby_community.org>
#   Subject:  Addressing Needs
#
#   Community:
#
#   Just wanted to send a quick note assuring that your needs are being addressed.
#
#   I want you to know that my team will keep working on the issues, especially:
#
#       * Run Ruby Quiz
#       * Document Modules
#       * Answer Questions on Ruby Talk
#
#   Thanks for your patience.
#
#   James Edward Gray II
#
# === Ruby in HTML
#
# ERB is often used in <tt>.rhtml</tt> files (HTML with embedded Ruby).  Notice the need in
# this example to provide a special binding when the template is run, so that the instance
# variables in the Product object can be resolved.
#
#   require "erb"
#
#   # Build template data class.
#   class Product
#     def initialize( code, name, desc, cost )
#       @code = code
#       @name = name
#       @desc = desc
#       @cost = cost
#
#       @features = [ ]
#     end
#
#     def add_feature( feature )
#       @features << feature
#     end
#
#     # Support templating of member data.
#     def get_binding
#       binding
#     end
#
#     # ...
#   end
#
#   # Create template.
#   template = %{
#     <html>
#       <head><title>Ruby Toys -- <%= @name %></title></head>
#       <body>
#
#         <h1><%= @name %> (<%= @code %>)</h1>
#         <p><%= @desc %></p>
#
#         <ul>
#           <% @features.each do |f| %>
#             <li><b><%= f %></b></li>
#           <% end %>
#         </ul>
#
#         <p>
#           <% if @cost < 10 %>
#             <b>Only <%= @cost %>!!!</b>
#           <% else %>
#              Call for a price, today!
#           <% end %>
#         </p>
#
#       </body>
#     </html>
#   }.gsub(/^  /, '')
#
#   rhtml = ERB.new(template)
#
#   # Set up template data.
#   toy = Product.new( "TZ-1002",
#                      "Rubysapien",
#                      "Geek's Best Friend!  Responds to Ruby commands...",
#                      999.95 )
#   toy.add_feature("Listens for verbal commands in the Ruby language!")
#   toy.add_feature("Ignores Perl, Java, and all C variants.")
#   toy.add_feature("Karate-Chop Action!!!")
#   toy.add_feature("Matz signature on left leg.")
#   toy.add_feature("Gem studded eyes... Rubies, of course!")
#
#   # Produce result.
#   rhtml.run(toy.get_binding)
#
# <i>Generates (some blank lines removed):</i>
#
#    <html>
#      <head><title>Ruby Toys -- Rubysapien</title></head>
#      <body>
#
#        <h1>Rubysapien (TZ-1002)</h1>
#        <p>Geek's Best Friend!  Responds to Ruby commands...</p>
#
#        <ul>
#            <li><b>Listens for verbal commands in the Ruby language!</b></li>
#            <li><b>Ignores Perl, Java, and all C variants.</b></li>
#            <li><b>Karate-Chop Action!!!</b></li>
#            <li><b>Matz signature on left leg.</b></li>
#            <li><b>Gem studded eyes... Rubies, of course!</b></li>
#        </ul>
#
#        <p>
#             Call for a price, today!
#        </p>
#
#      </body>
#    </html>
#
#
# == Notes
#
# There are a variety of templating solutions available in various Ruby projects.
# For example, RDoc, distributed with Ruby, uses its own template engine, which
# can be reused elsewhere.
#
# Other popular engines could be found in the corresponding
# {Category}[https://www.ruby-toolbox.com/categories/template_engines] of
# The Ruby Toolbox.
#
class ERB
  Revision = '$Date::                           $' # :nodoc: #'
  deprecate_constant :Revision

  # Returns revision information for the erb.rb module.
  def self.version
    VERSION
  end

  #
  # Constructs a new ERB object with the template specified in _str_.
  #
  # An ERB object works by building a chunk of Ruby code that will output
  # the completed template when run.
  #
  # If _trim_mode_ is passed a String containing one or more of the following
  # modifiers, ERB will adjust its code generation as listed:
  #
  #     %  enables Ruby code processing for lines beginning with %
  #     <> omit newline for lines starting with <% and ending in %>
  #     >  omit newline for lines ending in %>
  #     -  omit blank lines ending in -%>
  #
  # _eoutvar_ can be used to set the name of the variable ERB will build up
  # its output in.  This is useful when you need to run multiple ERB
  # templates through the same binding and/or when you want to control where
  # output ends up.  Pass the name of the variable to be used inside a String.
  #
  # === Example
  #
  #  require "erb"
  #
  #  # build data class
  #  class Listings
  #    PRODUCT = { :name => "Chicken Fried Steak",
  #                :desc => "A well messages pattie, breaded and fried.",
  #                :cost => 9.95 }
  #
  #    attr_reader :product, :price
  #
  #    def initialize( product = "", price = "" )
  #      @product = product
  #      @price = price
  #    end
  #
  #    def build
  #      b = binding
  #      # create and run templates, filling member data variables
  #      ERB.new(<<~'END_PRODUCT', trim_mode: "", eoutvar: "@product").result b
  #        <%= PRODUCT[:name] %>
  #        <%= PRODUCT[:desc] %>
  #      END_PRODUCT
  #      ERB.new(<<~'END_PRICE', trim_mode: "", eoutvar: "@price").result b
  #        <%= PRODUCT[:name] %> -- <%= PRODUCT[:cost] %>
  #        <%= PRODUCT[:desc] %>
  #      END_PRICE
  #    end
  #  end
  #
  #  # setup template data
  #  listings = Listings.new
  #  listings.build
  #
  #  puts listings.product + "\n" + listings.price
  #
  # _Generates_
  #
  #  Chicken Fried Steak
  #  A well messages pattie, breaded and fried.
  #
  #  Chicken Fried Steak -- 9.95
  #  A well messages pattie, breaded and fried.
  #
  def initialize(str, safe_level=NOT_GIVEN, legacy_trim_mode=NOT_GIVEN, legacy_eoutvar=NOT_GIVEN, trim_mode: nil, eoutvar: '_erbout')
    # Complex initializer for $SAFE deprecation at [Feature #14256]. Use keyword arguments to pass trim_mode or eoutvar.
    if safe_level != NOT_GIVEN
      warn 'Passing safe_level with the 2nd argument of ERB.new is deprecated. Do not use it, and specify other arguments as keyword arguments.', uplevel: 1
    end
    if legacy_trim_mode != NOT_GIVEN
      warn 'Passing trim_mode with the 3rd argument of ERB.new is deprecated. Use keyword argument like ERB.new(str, trim_mode: ...) instead.', uplevel: 1
      trim_mode = legacy_trim_mode
    end
    if legacy_eoutvar != NOT_GIVEN
      warn 'Passing eoutvar with the 4th argument of ERB.new is deprecated. Use keyword argument like ERB.new(str, eoutvar: ...) instead.', uplevel: 1
      eoutvar = legacy_eoutvar
    end

    compiler = make_compiler(trim_mode)
    set_eoutvar(compiler, eoutvar)
    @src, @encoding, @frozen_string = *compiler.compile(str)
    @filename = nil
    @lineno = 0
    @_init = self.class.singleton_class
  end
  NOT_GIVEN = Object.new
  private_constant :NOT_GIVEN

  ##
  # Creates a new compiler for ERB.  See ERB::Compiler.new for details

  def make_compiler(trim_mode)
    ERB::Compiler.new(trim_mode)
  end

  # The Ruby code generated by ERB
  attr_reader :src

  # The encoding to eval
  attr_reader :encoding

  # The optional _filename_ argument passed to Kernel#eval when the ERB code
  # is run
  attr_accessor :filename

  # The optional _lineno_ argument passed to Kernel#eval when the ERB code
  # is run
  attr_accessor :lineno

  #
  # Sets optional filename and line number that will be used in ERB code
  # evaluation and error reporting. See also #filename= and #lineno=
  #
  #   erb = ERB.new('<%= some_x %>')
  #   erb.render
  #   # undefined local variable or method `some_x'
  #   #   from (erb):1
  #
  #   erb.location = ['file.erb', 3]
  #   # All subsequent error reporting would use new location
  #   erb.render
  #   # undefined local variable or method `some_x'
  #   #   from file.erb:4
  #
  def location=((filename, lineno))
    @filename = filename
    @lineno = lineno if lineno
  end

  #
  # Can be used to set _eoutvar_ as described in ERB::new.  It's probably
  # easier to just use the constructor though, since calling this method
  # requires the setup of an ERB _compiler_ object.
  #
  def set_eoutvar(compiler, eoutvar = '_erbout')
    compiler.put_cmd = "#{eoutvar}.<<"
    compiler.insert_cmd = "#{eoutvar}.<<"
    compiler.pre_cmd = ["#{eoutvar} = +''"]
    compiler.post_cmd = [eoutvar]
  end

  # Generate results and print them. (see ERB#result)
  def run(b=new_toplevel)
    print self.result(b)
  end

  #
  # Executes the generated ERB code to produce a completed template, returning
  # the results of that code.  (See ERB::new for details on how this process
  # can be affected by _safe_level_.)
  #
  # _b_ accepts a Binding object which is used to set the context of
  # code evaluation.
  #
  def result(b=new_toplevel)
    unless @_init.equal?(self.class.singleton_class)
      raise ArgumentError, "not initialized"
    end
    eval(@src, b, (@filename || '(erb)'), @lineno)
  end

  # Render a template on a new toplevel binding with local variables specified
  # by a Hash object.
  def result_with_hash(hash)
    b = new_toplevel(hash.keys)
    hash.each_pair do |key, value|
      b.local_variable_set(key, value)
    end
    result(b)
  end

  ##
  # Returns a new binding each time *near* TOPLEVEL_BINDING for runs that do
  # not specify a binding.

  def new_toplevel(vars = nil)
    b = TOPLEVEL_BINDING
    if vars
      vars = vars.select {|v| b.local_variable_defined?(v)}
      unless vars.empty?
        return b.eval("tap {|;#{vars.join(',')}| break binding}")
      end
    end
    b.dup
  end
  private :new_toplevel

  # Define _methodname_ as instance method of _mod_ from compiled Ruby source.
  #
  # example:
  #   filename = 'example.rhtml'   # 'arg1' and 'arg2' are used in example.rhtml
  #   erb = ERB.new(File.read(filename))
  #   erb.def_method(MyClass, 'render(arg1, arg2)', filename)
  #   print MyClass.new.render('foo', 123)
  def def_method(mod, methodname, fname='(ERB)')
    src = self.src.sub(/^(?!#|$)/) {"def #{methodname}\n"} << "\nend\n"
    mod.module_eval do
      eval(src, binding, fname, -1)
    end
  end

  # Create unnamed module, define _methodname_ as instance method of it, and return it.
  #
  # example:
  #   filename = 'example.rhtml'   # 'arg1' and 'arg2' are used in example.rhtml
  #   erb = ERB.new(File.read(filename))
  #   erb.filename = filename
  #   MyModule = erb.def_module('render(arg1, arg2)')
  #   class MyClass
  #     include MyModule
  #   end
  def def_module(methodname='erb')
    mod = Module.new
    def_method(mod, methodname, @filename || '(ERB)')
    mod
  end

  # Define unnamed class which has _methodname_ as instance method, and return it.
  #
  # example:
  #   class MyClass_
  #     def initialize(arg1, arg2)
  #       @arg1 = arg1;  @arg2 = arg2
  #     end
  #   end
  #   filename = 'example.rhtml'  # @arg1 and @arg2 are used in example.rhtml
  #   erb = ERB.new(File.read(filename))
  #   erb.filename = filename
  #   MyClass = erb.def_class(MyClass_, 'render()')
  #   print MyClass.new('foo', 123).render()
  def def_class(superklass=Object, methodname='result')
    cls = Class.new(superklass)
    def_method(cls, methodname, @filename || '(ERB)')
    cls
  end
end
                                                                                                                                                                                                                                                                                                                                                                                                                                                                                             ruby/un.rb                                                                                          0000644                 00000026254 15040313433 0006501 0                                                                                                    ustar 00                                                                                                                                                                                                                                                       # frozen_string_literal: false
#
# = un.rb
#
# Copyright (c) 2003 WATANABE Hirofumi <eban@ruby-lang.org>
#
# This program is free software.
# You can distribute/modify this program under the same terms of Ruby.
#
# == Utilities to replace common UNIX commands in Makefiles etc
#
# == SYNOPSIS
#
#   ruby -run -e cp -- [OPTION] SOURCE DEST
#   ruby -run -e ln -- [OPTION] TARGET LINK_NAME
#   ruby -run -e mv -- [OPTION] SOURCE DEST
#   ruby -run -e rm -- [OPTION] FILE
#   ruby -run -e mkdir -- [OPTION] DIRS
#   ruby -run -e rmdir -- [OPTION] DIRS
#   ruby -run -e install -- [OPTION] SOURCE DEST
#   ruby -run -e chmod -- [OPTION] OCTAL-MODE FILE
#   ruby -run -e touch -- [OPTION] FILE
#   ruby -run -e wait_writable -- [OPTION] FILE
#   ruby -run -e mkmf -- [OPTION] EXTNAME [OPTION]
#   ruby -run -e httpd -- [OPTION] [DocumentRoot]
#   ruby -run -e colorize -- [FILE]
#   ruby -run -e help [COMMAND]

require "fileutils"
require "optparse"

module FileUtils
  @fileutils_output = $stdout
end

# :nodoc:
def setup(options = "", *long_options)
  caller = caller_locations(1, 1)[0].label
  opt_hash = {}
  argv = []
  OptionParser.new do |o|
    options.scan(/.:?/) do |s|
      opt_name = s.delete(":").intern
      o.on("-" + s.tr(":", " ")) do |val|
        opt_hash[opt_name] = val
      end
    end
    long_options.each do |s|
      opt_name, arg_name = s.split(/(?=[\s=])/, 2)
      opt_name.delete_prefix!('--')
      s = "--#{opt_name.gsub(/([A-Z]+|[a-z])([A-Z])/, '\1-\2').downcase}#{arg_name}"
      puts "#{opt_name}=>#{s}" if $DEBUG
      opt_name = opt_name.intern
      o.on(s) do |val|
        opt_hash[opt_name] = val
      end
    end
    o.on("-v") do opt_hash[:verbose] = true end
    o.on("--help") do
      UN.help([caller])
      exit
    end
    o.order!(ARGV) do |x|
      if /[*?\[{]/ =~ x
        argv.concat(Dir[x])
      else
        argv << x
      end
    end
  end
  yield argv, opt_hash
end

##
# Copy SOURCE to DEST, or multiple SOURCE(s) to DIRECTORY
#
#   ruby -run -e cp -- [OPTION] SOURCE DEST
#
#   -p          preserve file attributes if possible
#   -r          copy recursively
#   -l          make hard link instead of copying (implies -r)
#   -v          verbose
#

def cp
  setup("prl") do |argv, options|
    cmd = "cp"
    cmd += "_r" if options.delete :r
    cmd = "cp_lr" if options.delete :l
    options[:preserve] = true if options.delete :p
    dest = argv.pop
    argv = argv[0] if argv.size == 1
    FileUtils.__send__ cmd, argv, dest, **options
  end
end

##
# Create a link to the specified TARGET with LINK_NAME.
#
#   ruby -run -e ln -- [OPTION] TARGET LINK_NAME
#
#   -s          make symbolic links instead of hard links
#   -f          remove existing destination files
#   -v          verbose
#

def ln
  setup("sf") do |argv, options|
    cmd = "ln"
    cmd += "_s" if options.delete :s
    options[:force] = true if options.delete :f
    dest = argv.pop
    argv = argv[0] if argv.size == 1
    FileUtils.__send__ cmd, argv, dest, **options
  end
end

##
# Rename SOURCE to DEST, or move SOURCE(s) to DIRECTORY.
#
#   ruby -run -e mv -- [OPTION] SOURCE DEST
#
#   -v          verbose
#

def mv
  setup do |argv, options|
    dest = argv.pop
    argv = argv[0] if argv.size == 1
    FileUtils.mv argv, dest, **options
  end
end

##
# Remove the FILE
#
#   ruby -run -e rm -- [OPTION] FILE
#
#   -f          ignore nonexistent files
#   -r          remove the contents of directories recursively
#   -v          verbose
#

def rm
  setup("fr") do |argv, options|
    cmd = "rm"
    cmd += "_r" if options.delete :r
    options[:force] = true if options.delete :f
    FileUtils.__send__ cmd, argv, **options
  end
end

##
# Create the DIR, if they do not already exist.
#
#   ruby -run -e mkdir -- [OPTION] DIR
#
#   -p          no error if existing, make parent directories as needed
#   -v          verbose
#

def mkdir
  setup("p") do |argv, options|
    cmd = "mkdir"
    cmd += "_p" if options.delete :p
    FileUtils.__send__ cmd, argv, **options
  end
end

##
# Remove the DIR.
#
#   ruby -run -e rmdir -- [OPTION] DIR
#
#   -p          remove DIRECTORY and its ancestors.
#   -v          verbose
#

def rmdir
  setup("p") do |argv, options|
    options[:parents] = true if options.delete :p
    FileUtils.rmdir argv, **options
  end
end

##
# Copy SOURCE to DEST.
#
#   ruby -run -e install -- [OPTION] SOURCE DEST
#
#   -p          apply access/modification times of SOURCE files to
#               corresponding destination files
#   -m          set permission mode (as in chmod), instead of 0755
#   -o          set owner user id, instead of the current owner
#   -g          set owner group id, instead of the current group
#   -v          verbose
#

def install
  setup("pm:o:g:") do |argv, options|
    (mode = options.delete :m) and options[:mode] = /\A\d/ =~ mode ? mode.oct : mode
    options[:preserve] = true if options.delete :p
    (owner = options.delete :o) and options[:owner] = owner
    (group = options.delete :g) and options[:group] = group
    dest = argv.pop
    argv = argv[0] if argv.size == 1
    FileUtils.install argv, dest, **options
  end
end

##
# Change the mode of each FILE to OCTAL-MODE.
#
#   ruby -run -e chmod -- [OPTION] OCTAL-MODE FILE
#
#   -v          verbose
#

def chmod
  setup do |argv, options|
    mode = argv.shift
    mode = /\A\d/ =~ mode ? mode.oct : mode
    FileUtils.chmod mode, argv, **options
  end
end

##
# Update the access and modification times of each FILE to the current time.
#
#   ruby -run -e touch -- [OPTION] FILE
#
#   -v          verbose
#

def touch
  setup do |argv, options|
    FileUtils.touch argv, **options
  end
end

##
# Wait until the file becomes writable.
#
#   ruby -run -e wait_writable -- [OPTION] FILE
#
#   -n RETRY    count to retry
#   -w SEC      each wait time in seconds
#   -v          verbose
#

def wait_writable
  setup("n:w:v") do |argv, options|
    verbose = options[:verbose]
    n = options[:n] and n = Integer(n)
    wait = (wait = options[:w]) ? Float(wait) : 0.2
    argv.each do |file|
      begin
        File.open(file, "r+b") {}
      rescue Errno::ENOENT
        break
      rescue Errno::EACCES => e
        raise if n and (n -= 1) <= 0
        if verbose
          puts e
          STDOUT.flush
        end
        sleep wait
        retry
      end
    end
  end
end

##
# Create makefile using mkmf.
#
#   ruby -run -e mkmf -- [OPTION] EXTNAME [OPTION]
#
#   -d ARGS     run dir_config
#   -h ARGS     run have_header
#   -l ARGS     run have_library
#   -f ARGS     run have_func
#   -v ARGS     run have_var
#   -t ARGS     run have_type
#   -m ARGS     run have_macro
#   -c ARGS     run have_const
#   --vendor    install to vendor_ruby
#

def mkmf
  setup("d:h:l:f:v:t:m:c:", "vendor") do |argv, options|
    require 'mkmf'
    opt = options[:d] and opt.split(/:/).each {|n| dir_config(*n.split(/,/))}
    opt = options[:h] and opt.split(/:/).each {|n| have_header(*n.split(/,/))}
    opt = options[:l] and opt.split(/:/).each {|n| have_library(*n.split(/,/))}
    opt = options[:f] and opt.split(/:/).each {|n| have_func(*n.split(/,/))}
    opt = options[:v] and opt.split(/:/).each {|n| have_var(*n.split(/,/))}
    opt = options[:t] and opt.split(/:/).each {|n| have_type(*n.split(/,/))}
    opt = options[:m] and opt.split(/:/).each {|n| have_macro(*n.split(/,/))}
    opt = options[:c] and opt.split(/:/).each {|n| have_const(*n.split(/,/))}
    $configure_args["--vendor"] = true if options[:vendor]
    create_makefile(*argv)
  end
end

##
# Run WEBrick HTTP server.
#
#   ruby -run -e httpd -- [OPTION] [DocumentRoot]
#
#   --bind-address=ADDR         address to bind
#   --port=NUM                  listening port number
#   --max-clients=MAX           max number of simultaneous clients
#   --temp-dir=DIR              temporary directory
#   --do-not-reverse-lookup     disable reverse lookup
#   --request-timeout=SECOND    request timeout in seconds
#   --http-version=VERSION      HTTP version
#   --server-name=NAME          name of the server host
#   --server-software=NAME      name and version of the server
#   --ssl-certificate=CERT      The SSL certificate file for the server
#   --ssl-private-key=KEY       The SSL private key file for the server certificate
#   -v                          verbose
#

def httpd
  setup("", "BindAddress=ADDR", "Port=PORT", "MaxClients=NUM", "TempDir=DIR",
        "DoNotReverseLookup", "RequestTimeout=SECOND", "HTTPVersion=VERSION",
        "ServerName=NAME", "ServerSoftware=NAME",
        "SSLCertificate=CERT", "SSLPrivateKey=KEY") do
    |argv, options|
    begin
      require 'webrick'
    rescue LoadError
      abort "webrick is not found. You may need to `gem install webrick` to install webrick."
    end
    opt = options[:RequestTimeout] and options[:RequestTimeout] = opt.to_i
    [:Port, :MaxClients].each do |name|
      opt = options[name] and (options[name] = Integer(opt)) rescue nil
    end
    if cert = options[:SSLCertificate]
      key = options[:SSLPrivateKey] or
        raise "--ssl-private-key option must also be given"
      require 'webrick/https'
      options[:SSLEnable] = true
      options[:SSLCertificate] = OpenSSL::X509::Certificate.new(File.read(cert))
      options[:SSLPrivateKey] = OpenSSL::PKey.read(File.read(key))
      options[:Port] ||= 8443   # HTTPS Alternate
    end
    options[:Port] ||= 8080     # HTTP Alternate
    options[:DocumentRoot] = argv.shift || '.'
    s = nil
    options[:StartCallback] = proc {
      logger = s.logger
      logger.info("To access this server, open this URL in a browser:")
      s.listeners.each do |listener|
        if options[:SSLEnable]
          addr = listener.addr
          addr[3] = "127.0.0.1" if addr[3] == "0.0.0.0"
          addr[3] = "::1" if addr[3] == "::"
          logger.info("    https://#{Addrinfo.new(addr).inspect_sockaddr}")
        else
          logger.info("    http://#{listener.connect_address.inspect_sockaddr}")
        end
      end
    }
    s = WEBrick::HTTPServer.new(options)
    shut = proc {s.shutdown}
    siglist = %w"TERM QUIT"
    siglist.concat(%w"HUP INT") if STDIN.tty?
    siglist &= Signal.list.keys
    siglist.each do |sig|
      Signal.trap(sig, shut)
    end
    s.start
  end
end

##
# Colorize ruby code.
#
#   ruby -run -e colorize -- [FILE]
#

def colorize
  begin
    require "irb/color"
  rescue LoadError
    raise "colorize requires irb 1.1.0 or later"
  end
  setup do |argv, |
    if argv.empty?
      puts IRB::Color.colorize_code STDIN.read
      return
    end
    argv.each do |file|
      puts IRB::Color.colorize_code File.read(file)
    end
  end
end

##
# Display help message.
#
#   ruby -run -e help [COMMAND]
#

def help
  setup do |argv,|
    UN.help(argv)
  end
end

module UN # :nodoc:

  VERSION = "0.3.0"

  module_function
  def help(argv, output: $stdout)
    all = argv.empty?
    cmd = nil
    if all
      store = proc {|msg| output << msg}
    else
      messages = {}
      store = proc {|msg| messages[cmd] = msg}
    end
    File.open(__FILE__) do |me|
      while me.gets("##\n")
        if help = me.gets("\n\n")
          if all or argv.include?(cmd = help[/^#\s*ruby\s.*-e\s+(\w+)/, 1])
            store[help.gsub(/^# ?/, "")]
            break unless all or argv.size > messages.size
          end
        end
      end
    end
    if messages
      argv.each {|arg| output << messages[arg]}
    end
  end
end
                                                                                                                                                                                                                                                                                                                                                    ruby/benchmark.rb                                                                                   0000644                 00000045250 15040313433 0010006 0                                                                                                    ustar 00                                                                                                                                                                                                                                                       # frozen_string_literal: true
#--
# benchmark.rb - a performance benchmarking library
#
# $Id$
#
# Created by Gotoken (gotoken@notwork.org).
#
# Documentation by Gotoken (original RD), Lyle Johnson (RDoc conversion), and
# Gavin Sinclair (editing).
#++
#
# == Overview
#
# The Benchmark module provides methods for benchmarking Ruby code, giving
# detailed reports on the time taken for each task.
#

# The Benchmark module provides methods to measure and report the time
# used to execute Ruby code.
#
# * Measure the time to construct the string given by the expression
#   <code>"a"*1_000_000_000</code>:
#
#       require 'benchmark'
#
#       puts Benchmark.measure { "a"*1_000_000_000 }
#
#   On my machine (OSX 10.8.3 on i5 1.7 GHz) this generates:
#
#       0.350000   0.400000   0.750000 (  0.835234)
#
#   This report shows the user CPU time, system CPU time, the sum of
#   the user and system CPU times, and the elapsed real time. The unit
#   of time is seconds.
#
# * Do some experiments sequentially using the #bm method:
#
#       require 'benchmark'
#
#       n = 5000000
#       Benchmark.bm do |x|
#         x.report { for i in 1..n; a = "1"; end }
#         x.report { n.times do   ; a = "1"; end }
#         x.report { 1.upto(n) do ; a = "1"; end }
#       end
#
#   The result:
#
#              user     system      total        real
#          1.010000   0.000000   1.010000 (  1.014479)
#          1.000000   0.000000   1.000000 (  0.998261)
#          0.980000   0.000000   0.980000 (  0.981335)
#
# * Continuing the previous example, put a label in each report:
#
#       require 'benchmark'
#
#       n = 5000000
#       Benchmark.bm(7) do |x|
#         x.report("for:")   { for i in 1..n; a = "1"; end }
#         x.report("times:") { n.times do   ; a = "1"; end }
#         x.report("upto:")  { 1.upto(n) do ; a = "1"; end }
#       end
#
# The result:
#
#                     user     system      total        real
#       for:      1.010000   0.000000   1.010000 (  1.015688)
#       times:    1.000000   0.000000   1.000000 (  1.003611)
#       upto:     1.030000   0.000000   1.030000 (  1.028098)
#
# * The times for some benchmarks depend on the order in which items
#   are run.  These differences are due to the cost of memory
#   allocation and garbage collection. To avoid these discrepancies,
#   the #bmbm method is provided.  For example, to compare ways to
#   sort an array of floats:
#
#       require 'benchmark'
#
#       array = (1..1000000).map { rand }
#
#       Benchmark.bmbm do |x|
#         x.report("sort!") { array.dup.sort! }
#         x.report("sort")  { array.dup.sort  }
#       end
#
#   The result:
#
#        Rehearsal -----------------------------------------
#        sort!   1.490000   0.010000   1.500000 (  1.490520)
#        sort    1.460000   0.000000   1.460000 (  1.463025)
#        -------------------------------- total: 2.960000sec
#
#                    user     system      total        real
#        sort!   1.460000   0.000000   1.460000 (  1.460465)
#        sort    1.450000   0.010000   1.460000 (  1.448327)
#
# * Report statistics of sequential experiments with unique labels,
#   using the #benchmark method:
#
#       require 'benchmark'
#       include Benchmark         # we need the CAPTION and FORMAT constants
#
#       n = 5000000
#       Benchmark.benchmark(CAPTION, 7, FORMAT, ">total:", ">avg:") do |x|
#         tf = x.report("for:")   { for i in 1..n; a = "1"; end }
#         tt = x.report("times:") { n.times do   ; a = "1"; end }
#         tu = x.report("upto:")  { 1.upto(n) do ; a = "1"; end }
#         [tf+tt+tu, (tf+tt+tu)/3]
#       end
#
#   The result:
#
#                     user     system      total        real
#        for:      0.950000   0.000000   0.950000 (  0.952039)
#        times:    0.980000   0.000000   0.980000 (  0.984938)
#        upto:     0.950000   0.000000   0.950000 (  0.946787)
#        >total:   2.880000   0.000000   2.880000 (  2.883764)
#        >avg:     0.960000   0.000000   0.960000 (  0.961255)

module Benchmark

  VERSION = "0.4.0"

  BENCHMARK_VERSION = "2002-04-25" # :nodoc:

  # Invokes the block with a Benchmark::Report object, which
  # may be used to collect and report on the results of individual
  # benchmark tests. Reserves +label_width+ leading spaces for
  # labels on each line. Prints +caption+ at the top of the
  # report, and uses +format+ to format each line.
  # (Note: +caption+ must contain a terminating newline character,
  # see the default Benchmark::Tms::CAPTION for an example.)
  #
  # Returns an array of Benchmark::Tms objects.
  #
  # If the block returns an array of
  # Benchmark::Tms objects, these will be used to format
  # additional lines of output. If +labels+ parameter are
  # given, these are used to label these extra lines.
  #
  # _Note_: Other methods provide a simpler interface to this one, and are
  # suitable for nearly all benchmarking requirements.  See the examples in
  # Benchmark, and the #bm and #bmbm methods.
  #
  # Example:
  #
  #     require 'benchmark'
  #     include Benchmark          # we need the CAPTION and FORMAT constants
  #
  #     n = 5000000
  #     Benchmark.benchmark(CAPTION, 7, FORMAT, ">total:", ">avg:") do |x|
  #       tf = x.report("for:")   { for i in 1..n; a = "1"; end }
  #       tt = x.report("times:") { n.times do   ; a = "1"; end }
  #       tu = x.report("upto:")  { 1.upto(n) do ; a = "1"; end }
  #       [tf+tt+tu, (tf+tt+tu)/3]
  #     end
  #
  # Generates:
  #
  #                     user     system      total        real
  #       for:      0.970000   0.000000   0.970000 (  0.970493)
  #       times:    0.990000   0.000000   0.990000 (  0.989542)
  #       upto:     0.970000   0.000000   0.970000 (  0.972854)
  #       >total:   2.930000   0.000000   2.930000 (  2.932889)
  #       >avg:     0.976667   0.000000   0.976667 (  0.977630)
  #

  def benchmark(caption = "", label_width = nil, format = nil, *labels) # :yield: report
    sync = $stdout.sync
    $stdout.sync = true
    label_width ||= 0
    label_width += 1
    format ||= FORMAT
    report = Report.new(label_width, format)
    results = yield(report)

    print " " * report.width + caption unless caption.empty?
    report.list.each { |i|
      print i.label.to_s.ljust(report.width)
      print i.format(report.format, *format)
    }

    Array === results and results.grep(Tms).each {|t|
      print((labels.shift || t.label || "").ljust(label_width), t.format(format))
    }
    report.list
  ensure
    $stdout.sync = sync unless sync.nil?
  end


  # A simple interface to the #benchmark method, #bm generates sequential
  # reports with labels. +label_width+ and +labels+ parameters have the same
  # meaning as for #benchmark.
  #
  #     require 'benchmark'
  #
  #     n = 5000000
  #     Benchmark.bm(7) do |x|
  #       x.report("for:")   { for i in 1..n; a = "1"; end }
  #       x.report("times:") { n.times do   ; a = "1"; end }
  #       x.report("upto:")  { 1.upto(n) do ; a = "1"; end }
  #     end
  #
  # Generates:
  #
  #                     user     system      total        real
  #       for:      0.960000   0.000000   0.960000 (  0.957966)
  #       times:    0.960000   0.000000   0.960000 (  0.960423)
  #       upto:     0.950000   0.000000   0.950000 (  0.954864)
  #

  def bm(label_width = 0, *labels, &blk) # :yield: report
    benchmark(CAPTION, label_width, FORMAT, *labels, &blk)
  end


  # Sometimes benchmark results are skewed because code executed
  # earlier encounters different garbage collection overheads than
  # that run later. #bmbm attempts to minimize this effect by running
  # the tests twice, the first time as a rehearsal in order to get the
  # runtime environment stable, the second time for
  # real. GC.start is executed before the start of each of
  # the real timings; the cost of this is not included in the
  # timings. In reality, though, there's only so much that #bmbm can
  # do, and the results are not guaranteed to be isolated from garbage
  # collection and other effects.
  #
  # Because #bmbm takes two passes through the tests, it can
  # calculate the required label width.
  #
  #       require 'benchmark'
  #
  #       array = (1..1000000).map { rand }
  #
  #       Benchmark.bmbm do |x|
  #         x.report("sort!") { array.dup.sort! }
  #         x.report("sort")  { array.dup.sort  }
  #       end
  #
  # Generates:
  #
  #        Rehearsal -----------------------------------------
  #        sort!   1.440000   0.010000   1.450000 (  1.446833)
  #        sort    1.440000   0.000000   1.440000 (  1.448257)
  #        -------------------------------- total: 2.890000sec
  #
  #                    user     system      total        real
  #        sort!   1.460000   0.000000   1.460000 (  1.458065)
  #        sort    1.450000   0.000000   1.450000 (  1.455963)
  #
  # #bmbm yields a Benchmark::Job object and returns an array of
  # Benchmark::Tms objects.
  #
  def bmbm(width = 0) # :yield: job
    job = Job.new(width)
    yield(job)
    width = job.width + 1
    sync = $stdout.sync
    $stdout.sync = true

    # rehearsal
    puts 'Rehearsal '.ljust(width+CAPTION.length,'-')
    ets = job.list.inject(Tms.new) { |sum,(label,item)|
      print label.ljust(width)
      res = Benchmark.measure(&item)
      print res.format
      sum + res
    }.format("total: %tsec")
    print " #{ets}\n\n".rjust(width+CAPTION.length+2,'-')

    # take
    print ' '*width + CAPTION
    job.list.map { |label,item|
      GC.start
      print label.ljust(width)
      Benchmark.measure(label, &item).tap { |res| print res }
    }
  ensure
    $stdout.sync = sync unless sync.nil?
  end

  #
  # Returns the time used to execute the given block as a
  # Benchmark::Tms object. Takes +label+ option.
  #
  #       require 'benchmark'
  #
  #       n = 1000000
  #
  #       time = Benchmark.measure do
  #         n.times { a = "1" }
  #       end
  #       puts time
  #
  # Generates:
  #
  #        0.220000   0.000000   0.220000 (  0.227313)
  #
  def measure(label = "") # :yield:
    t0, r0 = Process.times, Process.clock_gettime(Process::CLOCK_MONOTONIC)
    yield
    t1, r1 = Process.times, Process.clock_gettime(Process::CLOCK_MONOTONIC)
    Benchmark::Tms.new(t1.utime  - t0.utime,
                       t1.stime  - t0.stime,
                       t1.cutime - t0.cutime,
                       t1.cstime - t0.cstime,
                       r1 - r0,
                       label)
  end

  #
  # Returns the elapsed real time used to execute the given block.
  # The unit of time is seconds.
  #
  #       Benchmark.realtime { "a" * 1_000_000_000 }
  #       #=> 0.5098029999935534
  #
  def realtime # :yield:
    r0 = Process.clock_gettime(Process::CLOCK_MONOTONIC)
    yield
    Process.clock_gettime(Process::CLOCK_MONOTONIC) - r0
  end

  module_function :benchmark, :measure, :realtime, :bm, :bmbm

  #
  # A Job is a sequence of labelled blocks to be processed by the
  # Benchmark.bmbm method.  It is of little direct interest to the user.
  #
  class Job # :nodoc:
    #
    # Returns an initialized Job instance.
    # Usually, one doesn't call this method directly, as new
    # Job objects are created by the #bmbm method.
    # +width+ is a initial value for the label offset used in formatting;
    # the #bmbm method passes its +width+ argument to this constructor.
    #
    def initialize(width)
      @width = width
      @list = []
    end

    #
    # Registers the given label and block pair in the job list.
    #
    def item(label = "", &blk) # :yield:
      raise ArgumentError, "no block" unless block_given?
      label = label.to_s
      w = label.length
      @width = w if @width < w
      @list << [label, blk]
      self
    end

    alias report item

    # An array of 2-element arrays, consisting of label and block pairs.
    attr_reader :list

    # Length of the widest label in the #list.
    attr_reader :width
  end

  #
  # This class is used by the Benchmark.benchmark and Benchmark.bm methods.
  # It is of little direct interest to the user.
  #
  class Report # :nodoc:
    #
    # Returns an initialized Report instance.
    # Usually, one doesn't call this method directly, as new
    # Report objects are created by the #benchmark and #bm methods.
    # +width+ and +format+ are the label offset and
    # format string used by Tms#format.
    #
    def initialize(width = 0, format = nil)
      @width, @format, @list = width, format, []
    end

    #
    # Prints the +label+ and measured time for the block,
    # formatted by +format+. See Tms#format for the
    # formatting rules.
    #
    def item(label = "", *format, &blk) # :yield:
      w = label.to_s.length
      @width = w if @width < w
      @list << res = Benchmark.measure(label, &blk)
      res
    end

    alias report item

    # An array of Benchmark::Tms objects representing each item.
    attr_reader :width, :format, :list
  end



  #
  # A data object, representing the times associated with a benchmark
  # measurement.
  #
  class Tms

    # Default caption, see also Benchmark::CAPTION
    CAPTION = "      user     system      total        real\n"

    # Default format string, see also Benchmark::FORMAT
    FORMAT = "%10.6u %10.6y %10.6t %10.6r\n"

    # User CPU time
    attr_reader :utime

    # System CPU time
    attr_reader :stime

    # User CPU time of children
    attr_reader :cutime

    # System CPU time of children
    attr_reader :cstime

    # Elapsed real time
    attr_reader :real

    # Total time, that is +utime+ + +stime+ + +cutime+ + +cstime+
    attr_reader :total

    # Label
    attr_reader :label

    #
    # Returns an initialized Tms object which has
    # +utime+ as the user CPU time, +stime+ as the system CPU time,
    # +cutime+ as the children's user CPU time, +cstime+ as the children's
    # system CPU time, +real+ as the elapsed real time and +label+ as the label.
    #
    def initialize(utime = 0.0, stime = 0.0, cutime = 0.0, cstime = 0.0, real = 0.0, label = nil)
      @utime, @stime, @cutime, @cstime, @real, @label = utime, stime, cutime, cstime, real, label.to_s
      @total = @utime + @stime + @cutime + @cstime
    end

    #
    # Returns a new Tms object whose times are the sum of the times for this
    # Tms object, plus the time required to execute the code block (+blk+).
    #
    def add(&blk) # :yield:
      self + Benchmark.measure(&blk)
    end

    #
    # An in-place version of #add.
    # Changes the times of this Tms object by making it the sum of the times
    # for this Tms object, plus the time required to execute
    # the code block (+blk+).
    #
    def add!(&blk)
      t = Benchmark.measure(&blk)
      @utime  = utime + t.utime
      @stime  = stime + t.stime
      @cutime = cutime + t.cutime
      @cstime = cstime + t.cstime
      @real   = real + t.real
      self
    end

    #
    # Returns a new Tms object obtained by memberwise summation
    # of the individual times for this Tms object with those of the +other+
    # Tms object.
    # This method and #/() are useful for taking statistics.
    #
    def +(other); memberwise(:+, other) end

    #
    # Returns a new Tms object obtained by memberwise subtraction
    # of the individual times for the +other+ Tms object from those of this
    # Tms object.
    #
    def -(other); memberwise(:-, other) end

    #
    # Returns a new Tms object obtained by memberwise multiplication
    # of the individual times for this Tms object by +x+.
    #
    def *(x); memberwise(:*, x) end

    #
    # Returns a new Tms object obtained by memberwise division
    # of the individual times for this Tms object by +x+.
    # This method and #+() are useful for taking statistics.
    #
    def /(x); memberwise(:/, x) end

    #
    # Returns the contents of this Tms object as
    # a formatted string, according to a +format+ string
    # like that passed to Kernel.format. In addition, #format
    # accepts the following extensions:
    #
    # <tt>%u</tt>::     Replaced by the user CPU time, as reported by Tms#utime.
    # <tt>%y</tt>::     Replaced by the system CPU time, as reported by #stime (Mnemonic: y of "s*y*stem")
    # <tt>%U</tt>::     Replaced by the children's user CPU time, as reported by Tms#cutime
    # <tt>%Y</tt>::     Replaced by the children's system CPU time, as reported by Tms#cstime
    # <tt>%t</tt>::     Replaced by the total CPU time, as reported by Tms#total
    # <tt>%r</tt>::     Replaced by the elapsed real time, as reported by Tms#real
    # <tt>%n</tt>::     Replaced by the label string, as reported by Tms#label (Mnemonic: n of "*n*ame")
    #
    # If +format+ is not given, FORMAT is used as default value, detailing the
    # user, system and real elapsed time.
    #
    def format(format = nil, *args)
      str = (format || FORMAT).dup
      str.gsub!(/(%[-+.\d]*)n/) { "#{$1}s" % label }
      str.gsub!(/(%[-+.\d]*)u/) { "#{$1}f" % utime }
      str.gsub!(/(%[-+.\d]*)y/) { "#{$1}f" % stime }
      str.gsub!(/(%[-+.\d]*)U/) { "#{$1}f" % cutime }
      str.gsub!(/(%[-+.\d]*)Y/) { "#{$1}f" % cstime }
      str.gsub!(/(%[-+.\d]*)t/) { "#{$1}f" % total }
      str.gsub!(/(%[-+.\d]*)r/) { "(#{$1}f)" % real }
      format ? str % args : str
    end

    #
    # Same as #format.
    #
    def to_s
      format
    end

    #
    # Returns a new 6-element array, consisting of the
    # label, user CPU time, system CPU time, children's
    # user CPU time, children's system CPU time and elapsed
    # real time.
    #
    def to_a
      [@label, @utime, @stime, @cutime, @cstime, @real]
    end

    #
    # Returns a hash containing the same data as `to_a`.
    #
    def to_h
      {
        label:  @label,
        utime:  @utime,
        stime:  @stime,
        cutime: @cutime,
        cstime: @cstime,
        real:   @real
      }
    end

    protected

    #
    # Returns a new Tms object obtained by memberwise operation +op+
    # of the individual times for this Tms object with those of the other
    # Tms object (+x+).
    #
    # +op+ can be a mathematical operation such as <tt>+</tt>, <tt>-</tt>,
    # <tt>*</tt>, <tt>/</tt>
    #
    def memberwise(op, x)
      case x
      when Benchmark::Tms
        Benchmark::Tms.new(utime.__send__(op, x.utime),
                           stime.__send__(op, x.stime),
                           cutime.__send__(op, x.cutime),
                           cstime.__send__(op, x.cstime),
                           real.__send__(op, x.real)
                           )
      else
        Benchmark::Tms.new(utime.__send__(op, x),
                           stime.__send__(op, x),
                           cutime.__send__(op, x),
                           cstime.__send__(op, x),
                           real.__send__(op, x)
                           )
      end
    end
  end

  # The default caption string (heading above the output times).
  CAPTION = Benchmark::Tms::CAPTION

  # The default format string used to display times.  See also Benchmark::Tms#format.
  FORMAT = Benchmark::Tms::FORMAT
end
                                                                                                                                                                                                                                                                                                                                                        ruby/kconv.rb                                                                                       0000644                 00000013345 15040313433 0007174 0                                                                                                    ustar 00                                                                                                                                                                                                                                                         end
      http.cert_store = store
    end
    if options.include? :read_timeout
      http.read_timeout = options[:read_timeout]
    end
    if options.include? :open_timeout
      http.open_timeout = options[:open_timeout]
    end

    resp = nil
    http.start {
      req = Net::HTTP::Get.new(request_uri, header)
      if options.include? :http_basic_authentication
        user, pass = options[:http_basic_authentication]
        req.basic_auth user, pass
      end
      http.request(req) {|response|
        resp = response
        if options[:content_length_proc] && Net::HTTPSuccess === resp
          if resp.key?('Content-Length')
            options[:content_length_proc].call(resp['Content-Length'].to_i)
          else
            options[:content_length_proc].call(nil)
          end
        end
        resp.read_body {|str|
          buf << str
          if options[:progress_proc] && Net::HTTPSuccess === resp
            options[:progress_proc].call(buf.size)
          end
          str.clear
        }
      }
    }
    io = buf.io
    io.rewind
    io.status = [resp.code, resp.message]
    resp.each_name {|name| buf.io.meta_add_field2 name, resp.get_fields(name) }
    case resp
    when Net::HTTPSuccess
    when Net::HTTPMovedPermanently, # 301
         Net::HTTPFound, # 302
         Net::HTTPSeeOther, # 303
         Net::HTTPTemporaryRedirect, # 307
         Net::HTTPPermanentRedirect # 308
      begin
        loc_uri = URI.parse(resp['location'])
      rescue URI::InvalidURIError
        raise OpenURI::HTTPError.new(io.status.join(' ') + ' (Invalid Location URI)', io)
      end
      throw :open_uri_redirect, loc_uri
    else
      raise OpenURI::HTTPError.new(io.status.join(' '), io)
    end
  end

  class HTTPError < StandardError
    def initialize(message, io)
      super(message)
      @io = io
    end
    attr_reader :io
  end

  # Raised on redirection,
  # only occurs when +redirect+ option for HTTP is +false+.
  class HTTPRedirect < HTTPError
    def initialize(message, io, uri)
      super(message, io)
      @uri = uri
    end
    attr_reader :uri
  end

  class Buffer # :nodoc: all
    def initialize
      @io = StringIO.new
      @size = 0
    end
    attr_reader :size

    StringMax = 10240
    def <<(str)
      @io << str
      @size += str.length
      if StringIO === @io && StringMax < @size
        require 'tempfile'
        io = Tempfile.new('open-uri')
        io.binmode
        Meta.init io, @io if Meta === @io
        io << @io.string
        @io = io
      end
    end

    def io
      Meta.init @io unless Meta === @io
      @io
    end
  end

  # :stopdoc:
  RE_LWS = /[\r\n\t ]+/n
  RE_TOKEN = %r{[^\x00- ()<>@,;:\\"/\[\]?={}\x7f]+}n
  RE_QUOTED_STRING = %r{"(?:[\r\n\t !#-\[\]-~\x80-\xff]|\\[\x00-\x7f])*"}n
  RE_PARAMETERS = %r{(?:;#{RE_LWS}?#{RE_TOKEN}#{RE_LWS}?=#{RE_LWS}?(?:#{RE_TOKEN}|#{RE_QUOTED_STRING})#{RE_LWS}?)*}n
  # :startdoc:

  # Mixin for holding meta-information.
  module Meta
    def Meta.init(obj, src=nil) # :nodoc:
      obj.extend Meta
      obj.instance_eval {
        @base_uri = nil
        @meta = {} # name to string.  legacy.
        @metas = {} # name to array of strings.
      }
      if src
        obj.status = src.status
        obj.base_uri = src.base_uri
        src.metas.each {|name, values|
          obj.meta_add_field2(name, values)
        }
      end
    end

    # returns an Array that consists of status code and message.
    attr_accessor :status

    # returns a URI that is the base of relative URIs in the data.
    # It may differ from the URI supplied by a user due to redirection.
    attr_accessor :base_uri

    # returns a Hash that represents header fields.
    # The Hash keys are downcased for canonicalization.
    # The Hash values are a field body.
    # If there are multiple field with same field name,
    # the field values are concatenated with a comma.
    attr_reader :meta

    # returns a Hash that represents header fields.
    # The Hash keys are downcased for canonicalization.
    # The Hash value are an array of field values.
    attr_reader :metas

    def meta_setup_encoding # :nodoc:
      charset = self.charset
      enc = nil
      if charset
        begin
          enc = Encoding.find(charset)
        rescue ArgumentError
        end
      end
      enc = Encoding::ASCII_8BIT unless enc
      if self.respond_to? :force_encoding
        self.force_encoding(enc)
      elsif self.respond_to? :string
        self.string.force_encoding(enc)
      else # Tempfile
        self.set_encoding enc
      end
    end

    def meta_add_field2(name, values) # :nodoc:
      name = name.downcase
      @metas[name] = values
      @meta[name] = values.join(', ')
      meta_setup_encoding if name == 'content-type'
    end

    def meta_add_field(name, value) # :nodoc:
      meta_add_field2(name, [value])
    end

    # returns a Time that represents the Last-Modified field.
    def last_modified
      if vs = @metas['last-modified']
        v = vs.join(', ')
        Time.httpdate(v)
      else
        nil
      end
    end

    def content_type_parse # :nodoc:
      vs = @metas['content-type']
      # The last (?:;#{RE_LWS}?)? matches extra ";" which violates RFC2045.
      if vs && %r{\A#{RE_LWS}?(#{RE_TOKEN})#{RE_LWS}?/(#{RE_TOKEN})#{RE_LWS}?(#{RE_PARAMETERS})(?:;#{RE_LWS}?)?\z}no =~ vs.join(', ')
        type = $1.downcase
        subtype = $2.downcase
        parameters = []
        $3.scan(/;#{RE_LWS}?(#{RE_TOKEN})#{RE_LWS}?=#{RE_LWS}?(?:(#{RE_TOKEN})|(#{RE_QUOTED_STRING}))/no) {|att, val, qval|
          if qval
            val = qval[1...-1].gsub(/[\r\n\t !#-\[\]-~\x80-\xff]+|(\\[\x00-\x7f])/n) { $1 ? $1[1,1] : $& }
          end
          parameters << [att.downcase, val]
        }
        ["#{type}/#{subtype}", *parameters]
      else
        nil
      end
    end

    # r                                                                                                                                                                                                                                                                                           ruby/find.rb                                                                                        0000644                 00000005021 15040313433 0006764 0                                                                                                    ustar 00                                                                                                                                                                                                                                                       # frozen_string_literal: true
#
# find.rb: the Find module for processing all files under a given directory.
#

#
# The +Find+ module supports the top-down traversal of a set of file paths.
#
# For example, to total the size of all files under your home directory,
# ignoring anything in a "dot" directory (e.g. $HOME/.ssh):
#
#   require 'find'
#
#   total_size = 0
#
#   Find.find(ENV["HOME"]) do |path|
#     if FileTest.directory?(path)
#       if File.basename(path).start_with?('.')
#         Find.prune       # Don't look any further into this directory.
#       else
#         next
#       end
#     else
#       total_size += FileTest.size(path)
#     end
#   end
#
module Find

  VERSION = "0.2.0"

  #
  # Calls the associated block with the name of every file and directory listed
  # as arguments, then recursively on their subdirectories, and so on.
  #
  # Returns an enumerator if no block is given.
  #
  # See the +Find+ module documentation for an example.
  #
  def find(*paths, ignore_error: true) # :yield: path
    block_given? or return enum_for(__method__, *paths, ignore_error: ignore_error)

    fs_encoding = Encoding.find("filesystem")

    paths.collect!{|d| raise Errno::ENOENT, d unless File.exist?(d); d.dup}.each do |path|
      path = path.to_path if path.respond_to? :to_path
      enc = path.encoding == Encoding::US_ASCII ? fs_encoding : path.encoding
      ps = [path]
      while file = ps.shift
        catch(:prune) do
          yield file.dup
          begin
            s = File.lstat(file)
          rescue Errno::ENOENT, Errno::EACCES, Errno::ENOTDIR, Errno::ELOOP, Errno::ENAMETOOLONG, Errno::EINVAL
            raise unless ignore_error
            next
          end
          if s.directory? then
            begin
              fs = Dir.children(file, encoding: enc)
            rescue Errno::ENOENT, Errno::EACCES, Errno::ENOTDIR, Errno::ELOOP, Errno::ENAMETOOLONG, Errno::EINVAL
              raise unless ignore_error
              next
            end
            fs.sort!
            fs.reverse_each {|f|
              f = File.join(file, f)
              ps.unshift f
            }
          end
        end
      end
    end
    nil
  end

  #
  # Skips the current file or directory, restarting the loop with the next
  # entry. If the current file is a directory, that directory will not be
  # recursively entered. Meaningful only within the block associated with
  # Find::find.
  #
  # See the +Find+ module documentation for an example.
  #
  def prune
    throw :prune
  end

  module_function :find, :prune
end
                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                               ruby/ripper.rb                                                                                      0000644                 00000004676 15040313433 0007364 0                                                                                                    ustar 00                                                                                                                                                                                                                                                       # frozen_string_literal: true
require 'ripper/core'
require 'ripper/lexer'
require 'ripper/filter'
require 'ripper/sexp'

# Ripper is a Ruby script parser.
#
# You can get information from the parser with event-based style.
# Information such as abstract syntax trees or simple lexical analysis of the
# Ruby program.
#
# == Usage
#
# Ripper provides an easy interface for parsing your program into a symbolic
# expression tree (or S-expression).
#
# Understanding the output of the parser may come as a challenge, it's
# recommended you use PP to format the output for legibility.
#
#   require 'ripper'
#   require 'pp'
#
#   pp Ripper.sexp('def hello(world) "Hello, #{world}!"; end')
#     #=> [:program,
#          [[:def,
#            [:@ident, "hello", [1, 4]],
#            [:paren,
#             [:params, [[:@ident, "world", [1, 10]]], nil, nil, nil, nil, nil, nil]],
#            [:bodystmt,
#             [[:string_literal,
#               [:string_content,
#                [:@tstring_content, "Hello, ", [1, 18]],
#                [:string_embexpr, [[:var_ref, [:@ident, "world", [1, 27]]]]],
#                [:@tstring_content, "!", [1, 33]]]]],
#             nil,
#             nil,
#             nil]]]]
#
# You can see in the example above, the expression starts with +:program+.
#
# From here, a method definition at +:def+, followed by the method's identifier
# <code>:@ident</code>. After the method's identifier comes the parentheses
# +:paren+ and the method parameters under +:params+.
#
# Next is the method body, starting at +:bodystmt+ (+stmt+ meaning statement),
# which contains the full definition of the method.
#
# In our case, we're simply returning a String, so next we have the
# +:string_literal+ expression.
#
# Within our +:string_literal+ you'll notice two <code>@tstring_content</code>,
# this is the literal part for <code>Hello, </code> and <code>!</code>. Between
# the two <code>@tstring_content</code> statements is a +:string_embexpr+,
# where _embexpr_ is an embedded expression. Our expression consists of a local
# variable, or +var_ref+, with the identifier (<code>@ident</code>) of +world+.
#
# == Resources
#
# * {Ruby Inside}[http://www.rubyinside.com/using-ripper-to-see-how-ruby-is-parsing-your-code-5270.html]
#
# == Requirements
#
# * ruby 1.9 (support CVS HEAD only)
# * bison 1.28 or later (Other yaccs do not work)
#
# == License
#
# Ruby License.
#
# - Minero Aoki
# - aamine@loveruby.net
# - http://i.loveruby.net
class Ripper; end
                                                                  gems/gems/io-console-0.6.0/lib/io                                                                   0000755                 00000000000 15040313433 0012225 0                                                                                                    ustar 00                                                                                                                                                                                                                                                       ruby/reline/version.rb                                                                              0000644                 00000000046 15040313433 0011011 0                                                                                                    ustar 00                                                                                                                                                                                                                                                       module Reline
  VERSION = '0.6.0'
end
                                                                                                                                                                                                                                                                                                                                                                                                                                                                                          ruby/reline/key_stroke.rb                                                                           0000644                 00000006334 15040313433 0011511 0                                                                                                    ustar 00                                                                                                                                                                                                                                                       class Reline::KeyStroke
  ESC_BYTE = 27
  CSI_PARAMETER_BYTES_RANGE = 0x30..0x3f
  CSI_INTERMEDIATE_BYTES_RANGE = (0x20..0x2f)

  attr_accessor :encoding

  def initialize(config, encoding)
    @config = config
    @encoding = encoding
  end

  # Input exactly matches to a key sequence
  MATCHING = :matching
  # Input partially matches to a key sequence
  MATCHED = :matched
  # Input matches to a key sequence and the key sequence is a prefix of another key sequence
  MATCHING_MATCHED = :matching_matched
  # Input does not match to any key sequence
  UNMATCHED = :unmatched

  def match_status(input)
    matching = key_mapping.matching?(input)
    matched = key_mapping.get(input)
    if matching && matched
      MATCHING_MATCHED
    elsif matching
      MATCHING
    elsif matched
      MATCHED
    elsif input[0] == ESC_BYTE
      match_unknown_escape_sequence(input, vi_mode: @config.editing_mode_is?(:vi_insert, :vi_command))
    else
      s = input.pack('c*').force_encoding(@encoding)
      if s.valid_encoding?
        s.size == 1 ? MATCHED : UNMATCHED
      else
        # Invalid string is MATCHING (part of valid string) or MATCHED (invalid bytes to be ignored)
        MATCHING_MATCHED
      end
    end
  end

  def expand(input)
    matched_bytes = nil
    (1..input.size).each do |i|
      bytes = input.take(i)
      status = match_status(bytes)
      matched_bytes = bytes if status == MATCHED || status == MATCHING_MATCHED
      break if status == MATCHED || status == UNMATCHED
    end
    return [[], []] unless matched_bytes

    func = key_mapping.get(matched_bytes)
    s = matched_bytes.pack('c*').force_encoding(@encoding)
    if func.is_a?(Array)
      # Perform simple macro expansion for single byte key bindings.
      # Multibyte key bindings and recursive macro expansion are not supported yet.
      marco = func.pack('c*').force_encoding(@encoding)
      keys = marco.chars.map do |c|
        f = key_mapping.get(c.bytes)
        Reline::Key.new(c, f.is_a?(Symbol) ? f : :ed_insert, false)
      end
    elsif func
      keys = [Reline::Key.new(s, func, false)]
    else
      if s.valid_encoding? && s.size == 1
        keys = [Reline::Key.new(s, :ed_insert, false)]
      else
        keys = []
      end
    end

    [keys, input.drop(matched_bytes.size)]
  end

  private

  # returns match status of CSI/SS3 sequence and matched length
  def match_unknown_escape_sequence(input, vi_mode: false)
    idx = 0
    return UNMATCHED unless input[idx] == ESC_BYTE
    idx += 1
    idx += 1 if input[idx] == ESC_BYTE

    case input[idx]
    when nil
      if idx == 1 # `ESC`
        return MATCHING_MATCHED
      else # `ESC ESC`
        return MATCHING
      end
    when 91 # == '['.ord
      # CSI sequence `ESC [ ... char`
      idx += 1
      idx += 1 while idx < input.size && CSI_PARAMETER_BYTES_RANGE.cover?(input[idx])
      idx += 1 while idx < input.size && CSI_INTERMEDIATE_BYTES_RANGE.cover?(input[idx])
    when 79 # == 'O'.ord
      # SS3 sequence `ESC O char`
      idx += 1
    else
      # `ESC char` or `ESC ESC char`
      return UNMATCHED if vi_mode
    end

    case input.size
    when idx
      MATCHING
    when idx + 1
      MATCHED
    else
      UNMATCHED
    end
  end

  def key_mapping
    @config.key_bindings
  end
end
                                                                                                                                                                                                                                                                                                    ruby/reline/unicode/east_asian_width.rb                                                             0000644                 00000052453 15040313433 0014271 0                                                                                                    ustar 00                                                                                                                                                                                                                                                       class Reline::Unicode::EastAsianWidth
  # This is based on EastAsianWidth.txt
  # UNICODE_VERSION = '15.1.0'

  CHUNK_LAST, CHUNK_WIDTH = [
    [0x1f, 2],
    [0x7e, 1],
    [0x7f, 2],
    [0xa0, 1],
    [0xa1, -1],
    [0xa3, 1],
    [0xa4, -1],
    [0xa6, 1],
    [0xa8, -1],
    [0xa9, 1],
    [0xaa, -1],
    [0xac, 1],
    [0xae, -1],
    [0xaf, 1],
    [0xb4, -1],
    [0xb5, 1],
    [0xba, -1],
    [0xbb, 1],
    [0xbf, -1],
    [0xc5, 1],
    [0xc6, -1],
    [0xcf, 1],
    [0xd0, -1],
    [0xd6, 1],
    [0xd8, -1],
    [0xdd, 1],
    [0xe1, -1],
    [0xe5, 1],
    [0xe6, -1],
    [0xe7, 1],
    [0xea, -1],
    [0xeb, 1],
    [0xed, -1],
    [0xef, 1],
    [0xf0, -1],
    [0xf1, 1],
    [0xf3, -1],
    [0xf6, 1],
    [0xfa, -1],
    [0xfb, 1],
    [0xfc, -1],
    [0xfd, 1],
    [0xfe, -1],
    [0x100, 1],
    [0x101, -1],
    [0x110, 1],
    [0x111, -1],
    [0x112, 1],
    [0x113, -1],
    [0x11a, 1],
    [0x11b, -1],
    [0x125, 1],
    [0x127, -1],
    [0x12a, 1],
    [0x12b, -1],
    [0x130, 1],
    [0x133, -1],
    [0x137, 1],
    [0x138, -1],
    [0x13e, 1],
    [0x142, -1],
    [0x143, 1],
    [0x144, -1],
    [0x147, 1],
    [0x14b, -1],
    [0x14c, 1],
    [0x14d, -1],
    [0x151, 1],
    [0x153, -1],
    [0x165, 1],
    [0x167, -1],
    [0x16a, 1],
    [0x16b, -1],
    [0x1cd, 1],
    [0x1ce, -1],
    [0x1cf, 1],
    [0x1d0, -1],
    [0x1d1, 1],
    [0x1d2, -1],
    [0x1d3, 1],
    [0x1d4, -1],
    [0x1d5, 1],
    [0x1d6, -1],
    [0x1d7, 1],
    [0x1d8, -1],
    [0x1d9, 1],
    [0x1da, -1],
    [0x1db, 1],
    [0x1dc, -1],
    [0x250, 1],
    [0x251, -1],
    [0x260, 1],
    [0x261, -1],
    [0x2c3, 1],
    [0x2c4, -1],
    [0x2c6, 1],
    [0x2c7, -1],
    [0x2c8, 1],
    [0x2cb, -1],
    [0x2cc, 1],
    [0x2cd, -1],
    [0x2cf, 1],
    [0x2d0, -1],
    [0x2d7, 1],
    [0x2db, -1],
    [0x2dc, 1],
    [0x2dd, -1],
    [0x2de, 1],
    [0x2df, -1],
    [0x2ff, 1],
    [0x36f, 0],
    [0x390, 1],
    [0x3a1, -1],
    [0x3a2, 1],
    [0x3a9, -1],
    [0x3b0, 1],
    [0x3c1, -1],
    [0x3c2, 1],
    [0x3c9, -1],
    [0x400, 1],
    [0x401, -1],
    [0x40f, 1],
    [0x44f, -1],
    [0x450, 1],
    [0x451, -1],
    [0x482, 1],
    [0x487, 0],
    [0x590, 1],
    [0x5bd, 0],
    [0x5be, 1],
    [0x5bf, 0],
    [0x5c0, 1],
    [0x5c2, 0],
    [0x5c3, 1],
    [0x5c5, 0],
    [0x5c6, 1],
    [0x5c7, 0],
    [0x60f, 1],
    [0x61a, 0],
    [0x64a, 1],
    [0x65f, 0],
    [0x66f, 1],
    [0x670, 0],
    [0x6d5, 1],
    [0x6dc, 0],
    [0x6de, 1],
    [0x6e4, 0],
    [0x6e6, 1],
    [0x6e8, 0],
    [0x6e9, 1],
    [0x6ed, 0],
    [0x710, 1],
    [0x711, 0],
    [0x72f, 1],
    [0x74a, 0],
    [0x7a5, 1],
    [0x7b0, 0],
    [0x7ea, 1],
    [0x7f3, 0],
    [0x7fc, 1],
    [0x7fd, 0],
    [0x815, 1],
    [0x819, 0],
    [0x81a, 1],
    [0x823, 0],
    [0x824, 1],
    [0x827, 0],
    [0x828, 1],
    [0x82d, 0],
    [0x858, 1],
    [0x85b, 0],
    [0x897, 1],
    [0x89f, 0],
    [0x8c9, 1],
    [0x8e1, 0],
    [0x8e2, 1],
    [0x902, 0],
    [0x939, 1],
    [0x93a, 0],
    [0x93b, 1],
    [0x93c, 0],
    [0x940, 1],
    [0x948, 0],
    [0x94c, 1],
    [0x94d, 0],
    [0x950, 1],
    [0x957, 0],
    [0x961, 1],
    [0x963, 0],
    [0x980, 1],
    [0x981, 0],
    [0x9bb, 1],
    [0x9bc, 0],
    [0x9c0, 1],
    [0x9c4, 0],
    [0x9cc, 1],
    [0x9cd, 0],
    [0x9e1, 1],
    [0x9e3, 0],
    [0x9fd, 1],
    [0x9fe, 0],
    [0xa00, 1],
    [0xa02, 0],
    [0xa3b, 1],
    [0xa3c, 0],
    [0xa40, 1],
    [0xa42, 0],
    [0xa46, 1],
    [0xa48, 0],
    [0xa4a, 1],
    [0xa4d, 0],
    [0xa50, 1],
    [0xa51, 0],
    [0xa6f, 1],
    [0xa71, 0],
    [0xa74, 1],
    [0xa75, 0],
    [0xa80, 1],
    [0xa82, 0],
    [0xabb, 1],
    [0xabc, 0],
    [0xac0, 1],
    [0xac5, 0],
    [0xac6, 1],
    [0xac8, 0],
    [0xacc, 1],
    [0xacd, 0],
    [0xae1, 1],
    [0xae3, 0],
    [0xaf9, 1],
    [0xaff, 0],
    [0xb00, 1],
    [0xb01, 0],
    [0xb3b, 1],
    [0xb3c, 0],
    [0xb3e, 1],
    [0xb3f, 0],
    [0xb40, 1],
    [0xb44, 0],
    [0xb4c, 1],
    [0xb4d, 0],
    [0xb54, 1],
    [0xb56, 0],
    [0xb61, 1],
    [0xb63, 0],
    [0xb81, 1],
    [0xb82, 0],
    [0xbbf, 1],
    [0xbc0, 0],
    [0xbcc, 1],
    [0xbcd, 0],
    [0xbff, 1],
    [0xc00, 0],
    [0xc03, 1],
    [0xc04, 0],
    [0xc3b, 1],
    [0xc3c, 0],
    [0xc3d, 1],
    [0xc40, 0],
    [0xc45, 1],
    [0xc48, 0],
    [0xc49, 1],
    [0xc4d, 0],
    [0xc54, 1],
    [0xc56, 0],
    [0xc61, 1],
    [0xc63, 0],
    [0xc80, 1],
    [0xc81, 0],
    [0xcbb, 1],
    [0xcbc, 0],
    [0xcbe, 1],
    [0xcbf, 0],
    [0xcc5, 1],
    [0xcc6, 0],
    [0xccb, 1],
    [0xccd, 0],
    [0xce1, 1],
    [0xce3, 0],
    [0xcff, 1],
    [0xd01, 0],
    [0xd3a, 1],
    [0xd3c, 0],
    [0xd40, 1],
    [0xd44, 0],
    [0xd4c, 1],
    [0xd4d, 0],
    [0xd61, 1],
    [0xd63, 0],
    [0xd80, 1],
    [0xd81, 0],
    [0xdc9, 1],
    [0xdca, 0],
    [0xdd1, 1],
    [0xdd4, 0],
    [0xdd5, 1],
    [0xdd6, 0],
    [0xe30, 1],
    [0xe31, 0],
    [0xe33, 1],
    [0xe3a, 0],
    [0xe46, 1],
    [0xe4e, 0],
    [0xeb0, 1],
    [0xeb1, 0],
    [0xeb3, 1],
    [0xebc, 0],
    [0xec7, 1],
    [0xece, 0],
    [0xf17, 1],
    [0xf19, 0],
    [0xf34, 1],
    [0xf35, 0],
    [0xf36, 1],
    [0xf37, 0],
    [0xf38, 1],
    [0xf39, 0],
    [0xf70, 1],
    [0xf7e, 0],
    [0xf7f, 1],
    [0xf84, 0],
    [0xf85, 1],
    [0xf87, 0],
    [0xf8c, 1],
    [0xf97, 0],
    [0xf98, 1],
    [0xfbc, 0],
    [0xfc5, 1],
    [0xfc6, 0],
    [0x102c, 1],
    [0x1030, 0],
    [0x1031, 1],
    [0x1037, 0],
    [0x1038, 1],
    [0x103a, 0],
    [0x103c, 1],
    [0x103e, 0],
    [0x1057, 1],
    [0x1059, 0],
    [0x105d, 1],
    [0x1060, 0],
    [0x1070, 1],
    [0x1074, 0],
    [0x1081, 1],
    [0x1082, 0],
    [0x1084, 1],
    [0x1086, 0],
    [0x108c, 1],
    [0x108d, 0],
    [0x109c, 1],
    [0x109d, 0],
    [0x10ff, 1],
    [0x115f, 2],
    [0x135c, 1],
    [0x135f, 0],
    [0x1711, 1],
    [0x1714, 0],
    [0x1731, 1],
    [0x1733, 0],
    [0x1751, 1],
    [0x1753, 0],
    [0x1771, 1],
    [0x1773, 0],
    [0x17b3, 1],
    [0x17b5, 0],
    [0x17b6, 1],
    [0x17bd, 0],
    [0x17c5, 1],
    [0x17c6, 0],
    [0x17c8, 1],
    [0x17d3, 0],
    [0x17dc, 1],
    [0x17dd, 0],
    [0x180a, 1],
    [0x180d, 0],
    [0x180e, 1],
    [0x180f, 0],
    [0x1884, 1],
    [0x1886, 0],
    [0x18a8, 1],
    [0x18a9, 0],
    [0x191f, 1],
    [0x1922, 0],
    [0x1926, 1],
    [0x1928, 0],
    [0x1931, 1],
    [0x1932, 0],
    [0x1938, 1],
    [0x193b, 0],
    [0x1a16, 1],
    [0x1a18, 0],
    [0x1a1a, 1],
    [0x1a1b, 0],
    [0x1a55, 1],
    [0x1a56, 0],
    [0x1a57, 1],
    [0x1a5e, 0],
    [0x1a5f, 1],
    [0x1a60, 0],
    [0x1a61, 1],
    [0x1a62, 0],
    [0x1a64, 1],
    [0x1a6c, 0],
    [0x1a72, 1],
    [0x1a7c, 0],
    [0x1a7e, 1],
    [0x1a7f, 0],
    [0x1aaf, 1],
    [0x1abd, 0],
    [0x1abe, 1],
    [0x1ace, 0],
    [0x1aff, 1],
    [0x1b03, 0],
    [0x1b33, 1],
    [0x1b34, 0],
    [0x1b35, 1],
    [0x1b3a, 0],
    [0x1b3b, 1],
    [0x1b3c, 0],
    [0x1b41, 1],
    [0x1b42, 0],
    [0x1b6a, 1],
    [0x1b73, 0],
    [0x1b7f, 1],
    [0x1b81, 0],
    [0x1ba1, 1],
    [0x1ba5, 0],
    [0x1ba7, 1],
    [0x1ba9, 0],
    [0x1baa, 1],
    [0x1bad, 0],
    [0x1be5, 1],
    [0x1be6, 0],
    [0x1be7, 1],
    [0x1be9, 0],
    [0x1bec, 1],
    [0x1bed, 0],
    [0x1bee, 1],
    [0x1bf1, 0],
    [0x1c2b, 1],
    [0x1c33, 0],
    [0x1c35, 1],
    [0x1c37, 0],
    [0x1ccf, 1],
    [0x1cd2, 0],
    [0x1cd3, 1],
    [0x1ce0, 0],
    [0x1ce1, 1],
    [0x1ce8, 0],
    [0x1cec, 1],
    [0x1ced, 0],
    [0x1cf3, 1],
    [0x1cf4, 0],
    [0x1cf7, 1],
    [0x1cf9, 0],
    [0x1dbf, 1],
    [0x1dff, 0],
    [0x200f, 1],
    [0x2010, -1],
    [0x2012, 1],
    [0x2016, -1],
    [0x2017, 1],
    [0x2019, -1],
    [0x201b, 1],
    [0x201d, -1],
    [0x201f, 1],
    [0x2022, -1],
    [0x2023, 1],
    [0x2027, -1],
    [0x202f, 1],
    [0x2030, -1],
    [0x2031, 1],
    [0x2033, -1],
    [0x2034, 1],
    [0x2035, -1],
    [0x203a, 1],
    [0x203b, -1],
    [0x203d, 1],
    [0x203e, -1],
    [0x2073, 1],
    [0x2074, -1],
    [0x207e, 1],
    [0x207f, -1],
    [0x2080, 1],
    [0x2084, -1],
    [0x20ab, 1],
    [0x20ac, -1],
    [0x20cf, 1],
    [0x20dc, 0],
    [0x20e0, 1],
    [0x20e1, 0],
    [0x20e4, 1],
    [0x20f0, 0],
    [0x2102, 1],
    [0x2103, -1],
    [0x2104, 1],
    [0x2105, -1],
    [0x2108, 1],
    [0x2109, -1],
    [0x2112, 1],
    [0x2113, -1],
    [0x2115, 1],
    [0x2116, -1],
    [0x2120, 1],
    [0x2122, -1],
    [0x2125, 1],
    [0x2126, -1],
    [0x212a, 1],
    [0x212b, -1],
    [0x2152, 1],
    [0x2154, -1],
    [0x215a, 1],
    [0x215e, -1],
    [0x215f, 1],
    [0x216b, -1],
    [0x216f, 1],
    [0x2179, -1],
    [0x2188, 1],
    [0x2189, -1],
    [0x218f, 1],
    [0x2199, -1],
    [0x21b7, 1],
    [0x21b9, -1],
    [0x21d1, 1],
    [0x21d2, -1],
    [0x21d3, 1],
    [0x21d4, -1],
    [0x21e6, 1],
    [0x21e7, -1],
    [0x21ff, 1],
    [0x2200, -1],
    [0x2201, 1],
    [0x2203, -1],
    [0x2206, 1],
    [0x2208, -1],
    [0x220a, 1],
    [0x220b, -1],
    [0x220e, 1],
    [0x220f, -1],
    [0x2210, 1],
    [0x2211, -1],
    [0x2214, 1],
    [0x2215, -1],
    [0x2219, 1],
    [0x221a, -1],
    [0x221c, 1],
    [0x2220, -1],
    [0x2222, 1],
    [0x2223, -1],
    [0x2224, 1],
    [0x2225, -1],
    [0x2226, 1],
    [0x222c, -1],
    [0x222d, 1],
    [0x222e, -1],
    [0x2233, 1],
    [0x2237, -1],
    [0x223b, 1],
    [0x223d, -1],
    [0x2247, 1],
    [0x2248, -1],
    [0x224b, 1],
    [0x224c, -1],
    [0x2251, 1],
    [0x2252, -1],
    [0x225f, 1],
    [0x2261, -1],
    [0x2263, 1],
    [0x2267, -1],
    [0x2269, 1],
    [0x226b, -1],
    [0x226d, 1],
    [0x226f, -1],
    [0x2281, 1],
    [0x2283, -1],
    [0x2285, 1],
    [0x2287, -1],
    [0x2294, 1],
    [0x2295, -1],
    [0x2298, 1],
    [0x2299, -1],
    [0x22a4, 1],
    [0x22a5, -1],
    [0x22be, 1],
    [0x22bf, -1],
    [0x2311, 1],
    [0x2312, -1],
    [0x2319, 1],
    [0x231b, 2],
    [0x2328, 1],
    [0x232a, 2],
    [0x23e8, 1],
    [0x23ec, 2],
    [0x23ef, 1],
    [0x23f0, 2],
    [0x23f2, 1],
    [0x23f3, 2],
    [0x245f, 1],
    [0x24e9, -1],
    [0x24ea, 1],
    [0x254b, -1],
    [0x254f, 1],
    [0x2573, -1],
    [0x257f, 1],
    [0x258f, -1],
    [0x2591, 1],
    [0x2595, -1],
    [0x259f, 1],
    [0x25a1, -1],
    [0x25a2, 1],
    [0x25a9, -1],
    [0x25b1, 1],
    [0x25b3, -1],
    [0x25b5, 1],
    [0x25b7, -1],
    [0x25bb, 1],
    [0x25bd, -1],
    [0x25bf, 1],
    [0x25c1, -1],
    [0x25c5, 1],
    [0x25c8, -1],
    [0x25ca, 1],
    [0x25cb, -1],
    [0x25cd, 1],
    [0x25d1, -1],
    [0x25e1, 1],
    [0x25e5, -1],
    [0x25ee, 1],
    [0x25ef, -1],
    [0x25fc, 1],
    [0x25fe, 2],
    [0x2604, 1],
    [0x2606, -1],
    [0x2608, 1],
    [0x2609, -1],
    [0x260d, 1],
    [0x260f, -1],
    [0x2613, 1],
    [0x2615, 2],
    [0x261b, 1],
    [0x261c, -1],
    [0x261d, 1],
    [0x261e, -1],
    [0x263f, 1],
    [0x2640, -1],
    [0x2641, 1],
    [0x2642, -1],
    [0x2647, 1],
    [0x2653, 2],
    [0x265f, 1],
    [0x2661, -1],
    [0x2662, 1],
    [0x2665, -1],
    [0x2666, 1],
    [0x266a, -1],
    [0x266b, 1],
    [0x266d, -1],
    [0x266e, 1],
    [0x266f, -1],
    [0x267e, 1],
    [0x267f, 2],
    [0x2692, 1],
    [0x2693, 2],
    [0x269d, 1],
    [0x269f, -1],
    [0x26a0, 1],
    [0x26a1, 2],
    [0x26a9, 1],
    [0x26ab, 2],
    [0x26bc, 1],
    [0x26be, 2],
    [0x26bf, -1],
    [0x26c3, 1],
    [0x26c5, 2],
    [0x26cd, -1],
    [0x26ce, 2],
    [0x26d3, -1],
    [0x26d4, 2],
    [0x26e1, -1],
    [0x26e2, 1],
    [0x26e3, -1],
    [0x26e7, 1],
    [0x26e9, -1],
    [0x26ea, 2],
    [0x26f1, -1],
    [0x26f3, 2],
    [0x26f4, -1],
    [0x26f5, 2],
    [0x26f9, -1],
    [0x26fa, 2],
    [0x26fc, -1],
    [0x26fd, 2],
    [0x26ff, -1],
    [0x2704, 1],
    [0x2705, 2],
    [0x2709, 1],
    [0x270b, 2],
    [0x2727, 1],
    [0x2728, 2],
    [0x273c, 1],
    [0x273d, -1],
    [0x274b, 1],
    [0x274c, 2],
    [0x274d, 1],
    [0x274e, 2],
    [0x2752, 1],
    [0x2755, 2],
    [0x2756, 1],
    [0x2757, 2],
    [0x2775, 1],
    [0x277f, -1],
    [0x2794, 1],
    [0x2797, 2],
    [0x27af, 1],
    [0x27b0, 2],
    [0x27be, 1],
    [0x27bf, 2],
    [0x2b1a, 1],
    [0x2b1c, 2],
    [0x2b4f, 1],
    [0x2b50, 2],
    [0x2b54, 1],
    [0x2b55, 2],
    [0x2b59, -1],
    [0x2cee, 1],
    [0x2cf1, 0],
    [0x2d7e, 1],
    [0x2d7f, 0],
    [0x2ddf, 1],
    [0x2dff, 0],
    [0x2e7f, 1],
    [0x2e99, 2],
    [0x2e9a, 1],
    [0x2ef3, 2],
    [0x2eff, 1],
    [0x2fd5, 2],
    [0x2fef, 1],
    [0x3029, 2],
    [0x302d, 0],
    [0x303e, 2],
    [0x3040, 1],
    [0x3096, 2],
    [0x3098, 1],
    [0x309a, 0],
    [0x30ff, 2],
    [0x3104, 1],
    [0x312f, 2],
    [0x3130, 1],
    [0x318e, 2],
    [0x318f, 1],
    [0x31e3, 2],
    [0x31ee, 1],
    [0x321e, 2],
    [0x321f, 1],
    [0x3247, 2],
    [0x324f, -1],
    [0x4dbf, 2],
    [0x4dff, 1],
    [0xa48c, 2],
    [0xa48f, 1],
    [0xa4c6, 2],
    [0xa66e, 1],
    [0xa66f, 0],
    [0xa673, 1],
    [0xa67d, 0],
    [0xa69d, 1],
    [0xa69f, 0],
    [0xa6ef, 1],
    [0xa6f1, 0],
    [0xa801, 1],
    [0xa802, 0],
    [0xa805, 1],
    [0xa806, 0],
    [0xa80a, 1],
    [0xa80b, 0],
    [0xa824, 1],
    [0xa826, 0],
    [0xa82b, 1],
    [0xa82c, 0],
    [0xa8c3, 1],
    [0xa8c5, 0],
    [0xa8df, 1],
    [0xa8f1, 0],
    [0xa8fe, 1],
    [0xa8ff, 0],
    [0xa925, 1],
    [0xa92d, 0],
    [0xa946, 1],
    [0xa951, 0],
    [0xa95f, 1],
    [0xa97c, 2],
    [0xa97f, 1],
    [0xa982, 0],
    [0xa9b2, 1],
    [0xa9b3, 0],
    [0xa9b5, 1],
    [0xa9b9, 0],
    [0xa9bb, 1],
    [0xa9bd, 0],
    [0xa9e4, 1],
    [0xa9e5, 0],
    [0xaa28, 1],
    [0xaa2e, 0],
    [0xaa30, 1],
    [0xaa32, 0],
    [0xaa34, 1],
    [0xaa36, 0],
    [0xaa42, 1],
    [0xaa43, 0],
    [0xaa4b, 1],
    [0xaa4c, 0],
    [0xaa7b, 1],
    [0xaa7c, 0],
    [0xaaaf, 1],
    [0xaab0, 0],
    [0xaab1, 1],
    [0xaab4, 0],
    [0xaab6, 1],
    [0xaab8, 0],
    [0xaabd, 1],
    [0xaabf, 0],
    [0xaac0, 1],
    [0xaac1, 0],
    [0xaaeb, 1],
    [0xaaed, 0],
    [0xaaf5, 1],
    [0xaaf6, 0],
    [0xabe4, 1],
    [0xabe5, 0],
    [0xabe7, 1],
    [0xabe8, 0],
    [0xabec, 1],
    [0xabed, 0],
    [0xabff, 1],
    [0xd7a3, 2],
    [0xdfff, 1],
    [0xf8ff, -1],
    [0xfaff, 2],
    [0xfb1d, 1],
    [0xfb1e, 0],
    [0xfdff, 1],
    [0xfe0f, 0],
    [0xfe19, 2],
    [0xfe1f, 1],
    [0xfe2f, 0],
    [0xfe52, 2],
    [0xfe53, 1],
    [0xfe66, 2],
    [0xfe67, 1],
    [0xfe6b, 2],
    [0xff00, 1],
    [0xff60, 2],
    [0xffdf, 1],
    [0xffe6, 2],
    [0xfffc, 1],
    [0xfffd, -1],
    [0x101fc, 1],
    [0x101fd, 0],
    [0x102df, 1],
    [0x102e0, 0],
    [0x10375, 1],
    [0x1037a, 0],
    [0x10a00, 1],
    [0x10a03, 0],
    [0x10a04, 1],
    [0x10a06, 0],
    [0x10a0b, 1],
    [0x10a0f, 0],
    [0x10a37, 1],
    [0x10a3a, 0],
    [0x10a3e, 1],
    [0x10a3f, 0],
    [0x10ae4, 1],
    [0x10ae6, 0],
    [0x10d23, 1],
    [0x10d27, 0],
    [0x10eaa, 1],
    [0x10eac, 0],
    [0x10efc, 1],
    [0x10eff, 0],
    [0x10f45, 1],
    [0x10f50, 0],
    [0x10f81, 1],
    [0x10f85, 0],
    [0x11000, 1],
    [0x11001, 0],
    [0x11037, 1],
    [0x11046, 0],
    [0x1106f, 1],
    [0x11070, 0],
    [0x11072, 1],
    [0x11074, 0],
    [0x1107e, 1],
    [0x11081, 0],
    [0x110b2, 1],
    [0x110b6, 0],
    [0x110b8, 1],
    [0x110ba, 0],
    [0x110c1, 1],
    [0x110c2, 0],
    [0x110ff, 1],
    [0x11102, 0],
    [0x11126, 1],
    [0x1112b, 0],
    [0x1112c, 1],
    [0x11134, 0],
    [0x11172, 1],
    [0x11173, 0],
    [0x1117f, 1],
    [0x11181, 0],
    [0x111b5, 1],
    [0x111be, 0],
    [0x111c8, 1],
    [0x111cc, 0],
    [0x111ce, 1],
    [0x111cf, 0],
    [0x1122e, 1],
    [0x11231, 0],
    [0x11233, 1],
    [0x11234, 0],
    [0x11235, 1],
    [0x11237, 0],
    [0x1123d, 1],
    [0x1123e, 0],
    [0x11240, 1],
    [0x11241, 0],
    [0x112de, 1],
    [0x112df, 0],
    [0x112e2, 1],
    [0x112ea, 0],
    [0x112ff, 1],
    [0x11301, 0],
    [0x1133a, 1],
    [0x1133c, 0],
    [0x1133f, 1],
    [0x11340, 0],
    [0x11365, 1],
    [0x1136c, 0],
    [0x1136f, 1],
    [0x11374, 0],
    [0x11437, 1],
    [0x1143f, 0],
    [0x11441, 1],
    [0x11444, 0],
    [0x11445, 1],
    [0x11446, 0],
    [0x1145d, 1],
    [0x1145e, 0],
    [0x114b2, 1],
    [0x114b8, 0],
    [0x114b9, 1],
    [0x114ba, 0],
    [0x114be, 1],
    [0x114c0, 0],
    [0x114c1, 1],
    [0x114c3, 0],
    [0x115b1, 1],
    [0x115b5, 0],
    [0x115bb, 1],
    [0x115bd, 0],
    [0x115be, 1],
    [0x115c0, 0],
    [0x115db, 1],
    [0x115dd, 0],
    [0x11632, 1],
    [0x1163a, 0],
    [0x1163c, 1],
    [0x1163d, 0],
    [0x1163e, 1],
    [0x11640, 0],
    [0x116aa, 1],
    [0x116ab, 0],
    [0x116ac, 1],
    [0x116ad, 0],
    [0x116af, 1],
    [0x116b5, 0],
    [0x116b6, 1],
    [0x116b7, 0],
    [0x1171c, 1],
    [0x1171f, 0],
    [0x11721, 1],
    [0x11725, 0],
    [0x11726, 1],
    [0x1172b, 0],
    [0x1182e, 1],
    [0x11837, 0],
    [0x11838, 1],
    [0x1183a, 0],
    [0x1193a, 1],
    [0x1193c, 0],
    [0x1193d, 1],
    [0x1193e, 0],
    [0x11942, 1],
    [0x11943, 0],
    [0x119d3, 1],
    [0x119d7, 0],
    [0x119d9, 1],
    [0x119db, 0],
    [0x119df, 1],
    [0x119e0, 0],
    [0x11a00, 1],
    [0x11a0a, 0],
    [0x11a32, 1],
    [0x11a38, 0],
    [0x11a3a, 1],
    [0x11a3e, 0],
    [0x11a46, 1],
    [0x11a47, 0],
    [0x11a50, 1],
    [0x11a56, 0],
    [0x11a58, 1],
    [0x11a5b, 0],
    [0x11a89, 1],
    [0x11a96, 0],
    [0x11a97, 1],
    [0x11a99, 0],
    [0x11c2f, 1],
    [0x11c36, 0],
    [0x11c37, 1],
    [0x11c3d, 0],
    [0x11c3e, 1],
    [0x11c3f, 0],
    [0x11c91, 1],
    [0x11ca7, 0],
    [0x11ca9, 1],
    [0x11cb0, 0],
    [0x11cb1, 1],
    [0x11cb3, 0],
    [0x11cb4, 1],
    [0x11cb6, 0],
    [0x11d30, 1],
    [0x11d36, 0],
    [0x11d39, 1],
    [0x11d3a, 0],
    [0x11d3b, 1],
    [0x11d3d, 0],
    [0x11d3e, 1],
    [0x11d45, 0],
    [0x11d46, 1],
    [0x11d47, 0],
    [0x11d8f, 1],
    [0x11d91, 0],
    [0x11d94, 1],
    [0x11d95, 0],
    [0x11d96, 1],
    [0x11d97, 0],
    [0x11ef2, 1],
    [0x11ef4, 0],
    [0x11eff, 1],
    [0x11f01, 0],
    [0x11f35, 1],
    [0x11f3a, 0],
    [0x11f3f, 1],
    [0x11f40, 0],
    [0x11f41, 1],
    [0x11f42, 0],
    [0x1343f, 1],
    [0x13440, 0],
    [0x13446, 1],
    [0x13455, 0],
    [0x16aef, 1],
    [0x16af4, 0],
    [0x16b2f, 1],
    [0x16b36, 0],
    [0x16f4e, 1],
    [0x16f4f, 0],
    [0x16f8e, 1],
    [0x16f92, 0],
    [0x16fdf, 1],
    [0x16fe3, 2],
    [0x16fe4, 0],
    [0x16fef, 1],
    [0x16ff1, 2],
    [0x16fff, 1],
    [0x187f7, 2],
    [0x187ff, 1],
    [0x18cd5, 2],
    [0x18cff, 1],
    [0x18d08, 2],
    [0x1afef, 1],
    [0x1aff3, 2],
    [0x1aff4, 1],
    [0x1affb, 2],
    [0x1affc, 1],
    [0x1affe, 2],
    [0x1afff, 1],
    [0x1b122, 2],
    [0x1b131, 1],
    [0x1b132, 2],
    [0x1b14f, 1],
    [0x1b152, 2],
    [0x1b154, 1],
    [0x1b155, 2],
    [0x1b163, 1],
    [0x1b167, 2],
    [0x1b16f, 1],
    [0x1b2fb, 2],
    [0x1bc9c, 1],
    [0x1bc9e, 0],
    [0x1ceff, 1],
    [0x1cf2d, 0],
    [0x1cf2f, 1],
    [0x1cf46, 0],
    [0x1d166, 1],
    [0x1d169, 0],
    [0x1d17a, 1],
    [0x1d182, 0],
    [0x1d184, 1],
    [0x1d18b, 0],
    [0x1d1a9, 1],
    [0x1d1ad, 0],
    [0x1d241, 1],
    [0x1d244, 0],
    [0x1d9ff, 1],
    [0x1da36, 0],
    [0x1da3a, 1],
    [0x1da6c, 0],
    [0x1da74, 1],
    [0x1da75, 0],
    [0x1da83, 1],
    [0x1da84, 0],
    [0x1da9a, 1],
    [0x1da9f, 0],
    [0x1daa0, 1],
    [0x1daaf, 0],
    [0x1dfff, 1],
    [0x1e006, 0],
    [0x1e007, 1],
    [0x1e018, 0],
    [0x1e01a, 1],
    [0x1e021, 0],
    [0x1e022, 1],
    [0x1e024, 0],
    [0x1e025, 1],
    [0x1e02a, 0],
    [0x1e08e, 1],
    [0x1e08f, 0],
    [0x1e12f, 1],
    [0x1e136, 0],
    [0x1e2ad, 1],
    [0x1e2ae, 0],
    [0x1e2eb, 1],
    [0x1e2ef, 0],
    [0x1e4eb, 1],
    [0x1e4ef, 0],
    [0x1e8cf, 1],
    [0x1e8d6, 0],
    [0x1e943, 1],
    [0x1e94a, 0],
    [0x1f003, 1],
    [0x1f004, 2],
    [0x1f0ce, 1],
    [0x1f0cf, 2],
    [0x1f0ff, 1],
    [0x1f10a, -1],
    [0x1f10f, 1],
    [0x1f12d, -1],
    [0x1f12f, 1],
    [0x1f169, -1],
    [0x1f16f, 1],
    [0x1f18d, -1],
    [0x1f18e, 2],
    [0x1f190, -1],
    [0x1f19a, 2],
    [0x1f1ac, -1],
    [0x1f1ff, 1],
    [0x1f202, 2],
    [0x1f20f, 1],
    [0x1f23b, 2],
    [0x1f23f, 1],
    [0x1f248, 2],
    [0x1f24f, 1],
    [0x1f251, 2],
    [0x1f25f, 1],
    [0x1f265, 2],
    [0x1f2ff, 1],
    [0x1f320, 2],
    [0x1f32c, 1],
    [0x1f335, 2],
    [0x1f336, 1],
    [0x1f37c, 2],
    [0x1f37d, 1],
    [0x1f393, 2],
    [0x1f39f, 1],
    [0x1f3ca, 2],
    [0x1f3ce, 1],
    [0x1f3d3, 2],
    [0x1f3df, 1],
    [0x1f3f0, 2],
    [0x1f3f3, 1],
    [0x1f3f4, 2],
    [0x1f3f7, 1],
    [0x1f43e, 2],
    [0x1f43f, 1],
    [0x1f440, 2],
    [0x1f441, 1],
    [0x1f4fc, 2],
    [0x1f4fe, 1],
    [0x1f53d, 2],
    [0x1f54a, 1],
    [0x1f54e, 2],
    [0x1f54f, 1],
    [0x1f567, 2],
    [0x1f579, 1],
    [0x1f57a, 2],
    [0x1f594, 1],
    [0x1f596, 2],
    [0x1f5a3, 1],
    [0x1f5a4, 2],
    [0x1f5fa, 1],
    [0x1f64f, 2],
    [0x1f67f, 1],
    [0x1f6c5, 2],
    [0x1f6cb, 1],
    [0x1f6cc, 2],
    [0x1f6cf, 1],
    [0x1f6d2, 2],
    [0x1f6d4, 1],
    [0x1f6d7, 2],
    [0x1f6db, 1],
    [0x1f6df, 2],
    [0x1f6ea, 1],
    [0x1f6ec, 2],
    [0x1f6f3, 1],
    [0x1f6fc, 2],
    [0x1f7df, 1],
    [0x1f7eb, 2],
    [0x1f7ef, 1],
    [0x1f7f0, 2],
    [0x1f90b, 1],
    [0x1f93a, 2],
    [0x1f93b, 1],
    [0x1f945, 2],
    [0x1f946, 1],
    [0x1f9ff, 2],
    [0x1fa6f, 1],
    [0x1fa7c, 2],
    [0x1fa7f, 1],
    [0x1fa88, 2],
    [0x1fa8f, 1],
    [0x1fabd, 2],
    [0x1fabe, 1],
    [0x1fac5, 2],
    [0x1facd, 1],
    [0x1fadb, 2],
    [0x1fadf, 1],
    [0x1fae8, 2],
    [0x1faef, 1],
    [0x1faf8, 2],
    [0x1ffff, 1],
    [0x2fffd, 2],
    [0x2ffff, 1],
    [0x3fffd, 2],
    [0xe00ff, 1],
    [0xe01ef, 0],
    [0xeffff, 1],
    [0xffffd, -1],
    [0xfffff, 1],
    [0x10fffd, -1],
    [0x7fffffff, 1]
  ].transpose.map(&:freeze)
end
                                                                                                                                                                                                                     ruby/reline/windows.rb                                                                              0000644                 00000040752 15040313433 0011026 0                                                                                                    ustar 00                                                                                                                                                                                                                                                                                                               ustar 00                                                                                                                                                                                                                                                       sed to the logger method.
  # +msg+:: The _Object_ the user passed to the log message; not necessarily a
  #         String.
  #
  # The block should return an Object that can be written to the logging
  # device via +write+.  The default formatter is used when no formatter is
  # set.
  attr_accessor :formatter

  ##
  # The facility argument is used to specify what type of program is logging the message.

  attr_accessor :facility

  ##
  # Fills in variables for Logger compatibility.  If this is the first
  # instance of Syslog::Logger, +program_name+ may be set to change the logged
  # program name. The +facility+ may be set to specify the facility level which will be used.
  #
  # Due to the way syslog works, only one program name may be chosen.

  def initialize program_name = 'ruby', facility = nil
    @level = ::Logger::DEBUG
    @formatter = Formatter.new

    @@syslog ||= Syslog.open(program_name)

    @facility = (facility || @@syslog.facility)
  end

  ##
  # Almost duplicates Logger#add.  +progname+ is ignored.

  def add severity, message = nil, progname = nil, &block
    severity ||= ::Logger::UNKNOWN
    level <= severity and
      @@syslog.log( (LEVEL_MAP[severity] | @facility), '%s', formatter.call(severity, Time.now, progname, (message || block.call)) )
    true
  end
end
                                                                                                                                                                                                                                doc/alt-ruby22/COPYING.ja                                                                           0000644                 00000005100 15040347476 0010646 0                                                                                                    ustar 00                                                                                                                                                                                                                                                       本プログラムはフリーソフトウェアです．2-clause BSDL
または以下に示す条件で本プログラムを再配布できます
2-clause BSDLについてはBSDLファイルを参照して下さい．

  1. 複製は制限なく自由です．

  2. 以下の条件のいずれかを満たす時に本プログラムのソースを
     自由に変更できます．

     (a) ネット