# frozen_string_literal: true

module SyntaxSuggest
  # Scans up/down from the given block
  #
  # You can try out a change, stash it, or commit it to save for later
  #
  # Example:
  #
  #   scanner = ScanHistory.new(code_lines: code_lines, block: block)
  #   scanner.scan(
  #     up: ->(_, _, _) { true },
  #     down: ->(_, _, _) { true }
  #   )
  #   scanner.changed? # => true
  #   expect(scanner.lines).to eq(code_lines)
  #
  #   scanner.stash_changes
  #
  #   expect(scanner.lines).to_not eq(code_lines)
  class ScanHistory
    attr_reader :before_index, :after_index

    def initialize(code_lines:, block:)
      @code_lines = code_lines
      @history = [block]
      refresh_index
    end

    def commit_if_changed
      if changed?
        @history << CodeBlock.new(lines: @code_lines[before_index..after_index])
      end

      self
    end

    # Discards any changes that have not been committed
    def stash_changes
      refresh_index
      self
    end

    # Discard changes that have not been committed and revert the last commit
    #
    # Cannot revert the first commit
    def revert_last_commit
      if @history.length > 1
        @history.pop
        refresh_index
      end

      self
    end

    def changed?
      @before_index != current.lines.first.index ||
        @after_index != current.lines.last.index
    end

    # Iterates up and down
    #
    # Returns line, kw_count, end_count for each iteration
    def scan(up:, down:)
      kw_count = 0
      end_count = 0

      up_index = befor