
  def arg_config(config, default=nil, &block)
    $arg_config << [config, default]
    defaults = []
    if default
      defaults << default
    elsif !block
      defaults << nil
    end
    $configure_args.fetch(config.tr('_', '-'), *defaults, &block)
  end

  # :startdoc:

  # Tests for the presence of a <tt>--with-</tt>_config_ or
  # <tt>--without-</tt>_config_ option.  Returns +true+ if the with option is
  # given, +false+ if the without option is given, and the default value
  # otherwise.
  #
  # This can be useful for adding custom definitions, such as debug
  # information.
  #
  # Example:
  #
  #    if with_config("debug")
  #       $defs.push("-DOSSL_DEBUG") unless $defs.include? "-DOSSL_DEBUG"
  #    end
  #
  def with_config(config, default=nil)
    config = config.sub(/^--with[-_]/, '')
    val = arg_config("--with-"+config) do
      if arg_config("--without-"+config)
        false
      elsif block_given?
        yield(config, default)
      else
        break default
      end
    end
    case val
    when "yes"
      true
    when "no"
      false
    else
      val
    end
  end

  # Tests for the presence of an <tt>--enable-</tt>_config_ or
  # <tt>--disable-</tt>_config_ option. Returns +true+ if the enable option is
  # given, +false+ if the disable option is given, and the default value
  # otherwise.
  #
  # This can be useful for adding custom definitions, such as debug
  # information.
  #
  # Example:
  #
  #    if enable_config("debug")
  #       $defs.push("-DOSSL_DEBUG") unless $defs.include? "-DOSSL_DEBUG"
  #    end
  #
  def enable_config(config, default=nil)
    if arg_config("--enable-"+config)
      true
    elsif arg_config("--disable-"+config)
      false
    elsif block_given?
      yield(config, default)
    else
      return default
    end
  end

  # Generates a header file consisting of the various macro definitions
  # generated by other methods such as have_func and have_header. These are
  # then wrapped in a custom <code>#ifndef</code> based on the +header+ file
  # name, which defaults to "extconf.h".
  #
  # For example:
  #
  #   # extconf.rb
  #   require 'mkmf'
  #   have_func('realpath')
  #   have_header('sys/utime.h')
  #   create_header
  #   create_makefile('foo')
  #
  # The above script would generate the following extconf.h file:
  #
  #   #ifndef EXTCONF_H
  #   #define EXTCONF_H
  #   #define HAVE_REALPATH 1
  #   #define HAVE_SYS_UTIME_H 1
  #   #endif
  #
  # Given that the create_header method generates a file based on definitions
  # set earlier in your extconf.rb file, you will probably want to make this
  # one of the last methods you call in your script.
  #
  def create_header(header = "extconf.h")
    message "creating %s\n", header
    sym = header.tr_cpp
    hdr = ["#ifndef #{sym}\n#define #{sym}\n"]
    for line in $defs
      case line
      when /^-D([^=]+)(?:=(.*))?/
        hdr << "#define #$1 #{$2 ? Shellwords.shellwords($2)[0].gsub(/(?=\t+)/, "\\\n") : 1}