
      v = parser.ver or abort("#{parser.program_name}: version unknown")
      puts v
      exit
    end
  end

  # :startdoc:

  #
  # Class methods
  #

  #
  # Initializes a new instance and evaluates the optional block in context
  # of the instance. Arguments +args+ are passed to #new, see there for
  # description of parameters.
  #
  # This method is *deprecated*, its behavior corresponds to the older #new
  # method.
  #
  def self.with(*args, &block)
    opts = new(*args)
    opts.instance_eval(&block)
    opts
  end

  #
  # Returns an incremented value of +default+ according to +arg+.
  #
  def self.inc(arg, default = nil)
    case arg
    when Integer
      arg.nonzero?
    when nil
      default.to_i + 1
    end
  end

  #
  # See self.inc
  #
  def inc(*args)
    self.class.inc(*args)
  end

  #
  # Initializes the instance and yields itself if called with a block.
  #
  # +banner+:: Banner message.
  # +width+::  Summary width.
  # +indent+:: Summary indent.
  #
  def initialize(banner = nil, width = 32, indent = ' ' * 4)
    @stack = [DefaultList, List.new, List.new]
    @program_name = nil
    @banner = banner
    @summary_width = width
    @summary_indent = indent
    @default_argv = ARGV
    @require_exact = false
    @raise_unknown = true
    add_officious
    yield self if block_given?
  end

  def add_officious  # :nodoc:
    list = base()
    Officious.each do |opt, block|
      list.long[opt] ||= block.call(self)
    end
  end

  #
  # Terminates option parsing. Optional parameter +arg+ is a string pushed
  # back to be the first non-option argument.
  #
  def terminate(arg = nil)
    self.class.terminate(arg)
  end
  #
  # See #terminate.
  #
  def self.terminate(arg = nil)
    throw :terminate, arg
  end

  @stack = [DefaultList]
  #
  # Returns the global top option list.
  #
  # Do not use directly.
  #
  def self.top() DefaultList end

  #
  # Directs to accept specified class +t+. The argument string is passed to
  # the block in which it should be converted to the desired class.
  #
  # +t+::   Argument class specifier, any object including Class.
  # +pat+:: Pattern for argument, defaults to +t+ if it responds to match.
  #
  #   accept(t, pat, &block)
  #
  def accept(*args, &blk) top.accept(*args, &blk) end
  #
  # See #accept.
  #
  def self.accept(*args, &blk) top.accept(*args, &blk) end

  #
  # Directs to reject specified class argument.
  #
  # +type+:: Argument class specifier, any object including Class.
  #
  #   reject(type)
  #
  def reject(*args, &blk) top.reject(*args, &blk) end
  #
  # See #reject.
  #
  def self.reject(*args, &blk) top.reject(*args, &blk) end

  #
  # Instance methods
  #

  # Heading banner preceding summary.
  attr_writer :banner

  # Program name to be emitted in error message and default banner,
  # defaults to $0.
  attr_writer :program_name

  # Width for option list portion of summary. Must be Numeric.
  attr_accessor :summary_width

  # Indentation for summary. Must be String (or have + String method).
  attr_accessor :summary_indent

  # Strings to be parsed in default.
  attr_accessor :default_argv

  # Whether to require that options match exactly (disallows providing
  # abbreviated long option as short option).
  attr_accessor :require_exact

  # Whether to raise at unknown option.
  attr_accessor :raise_unknown

  #
  # Heading banner preceding summary.
  #
  def banner
    unless @banner
      @banner = +"Usage: #{program_name} [options]"
      visit(:add_banner, @banner)
    end
    @banner
  end

  #
  # Program name to be emitted in error message and default banner, defaults
  # to $0.
  #
  def program_name
    @program_name || File.basename($0, '.*')
  end

  # for experimental cascading :-)
  alias set_banner banner=
  alias set_program_name program_name=
  alias set_summary_width summary_width=
  alias set_summary_indent summary_indent=

  # Version
  attr_writer :version
  # Release code
  attr_writer :release

  #
  # Version
  #
  def version
    (defined?(@version) && @version) || (defined?(::Version) && ::Version)
  end

  #
  # Release code
  #
  def release
    (defined?(@release) && @release) || (defined?(::Release) && ::Release) || (defined?(::RELEASE) && ::RELEASE)
  end

  #
  # Returns version string from program_name, version and release.
  #
  def ver
    if v = version
      str = +"#{program_name} #{[v].join('.')}"
      str << " (#{v})" if v = release
      str
    end
  end

  #
  # Shows warning message with the program name
  #
  # +mesg+:: Message, defaulted to +$!+.
  #
  # See Kernel#warn.
  #
  def warn(mesg = $!)
    super("#{program_name}: #{mesg}")
  end

  #
  # Shows message with the program name then aborts.
  #
  # +mesg+:: Message, defaulted to +$!+.
  #
  # See Kernel#abort.
  #
  def abort(mesg = $!)
    super("#{program_name}: #{mesg}")
  end

  #
  # Subject of #on / #on_head, #accept / #reject
  #
  def top
    @stack[-1]
  end

  #
  # Subject of #on_tail.
  #
  def base
    @stack[1]
  end

  #
  # Pushes a new List.
  #
  # If a block is given, yields +self+ and returns the result of the
  # block, otherwise returns +self+.
  #
  def new
    @stack.push(List.new)
    if block_given?
      yield self
    else
      self
    end
  end

  #
  # Removes the last List.
  #
  def remove
    @stack.pop
  end

  #
  # Puts option summary into +to+ and returns +to+. Yields each line if
  # a block is given.
  #
  # +to+:: Output destination, which must have method <<. Defaults to [].
  # +width+:: Width of left side, defaults to @summary_width.
  # +max+:: Maximum length allowed for left side, defaults to +width+ - 1.
  # +indent+:: Indentation, defaults to @summary_indent.
  #
  def summarize(to = [], width = @summary_width, max = width - 1, indent = @summary_indent, &blk)
    nl = "\n"
    blk ||= proc {|l| to << (l.index(nl, -1) ? l : l + nl)}
    visit(:summarize, {}, {}, width, max, indent, &blk)
    to
  end

  #
  # Returns option summary string.
  #
  def help; summarize("#{banner}".sub(/\n?\z/, "\n")) end
  alias to_s help

  def pretty_print(q)           # :nodoc:
    q.object_group(self) do
      first = true
      if @stack.size > 2
        @stack.each_with_index do |s, i|
          next if i < 2
          next if s.list.empty?
          if first
            first = false
            q.text ":"
          end
          q.breakable
          s.pretty_print(q)
        end
      end
    end
  end

  def inspect                   # :nodoc:
    require 'pp'
    pretty_print_inspect
  end

  #
  # Returns option summary list.
  #
  def to_a; summarize("#{banner}".split(/^/)) end

  #
  # Checks if an argument is given twice, in which case an ArgumentError is
  # raised. Called from OptionParser#switch only.
  #
  # +obj+:: New argument.
  # +prv+:: Previously specified argument.
  # +msg+:: Exception message.
  #
  def notwice(obj, prv, msg) # :nodoc:
    unless !prv or prv == obj
      raise(ArgumentError, "argument #{msg} given twice: #{obj}",
            ParseError.filter_backtrace(caller(2)))
    end
    obj
  end
  private :notwice

  SPLAT_PROC = proc {|*a| a.length <= 1 ? a.first : a} # :nodoc:

  # :call-seq:
  #   make_switch(params, block = nil)
  #
  # :include: ../doc/optparse/creates_option.rdoc
  #
  def make_switch(opts, block = nil)
    short, long, nolong, style, pattern, conv, not_pattern, not_conv, not_style = [], [], []
    ldesc, sdesc, desc, arg = [], [], []
    default_style = Switch::NoArgument
    default_pattern = nil
    klass = nil
    q, a = nil
    has_arg = false

    opts.each do |o|
      # argument class
      next if search(:atype, o) do |pat, c|
        klass = notwice(o, klass, 'type')
        if not_style and not_style != Switch::NoArgument
          not_pattern, not_conv = pat, c
        else
          default_pattern, conv = pat, c
        end
      end

      # directly specified pattern(any object possible to match)
      if (!(String === o || Symbol === o)) and o.respond_to?(:match)
        pattern = notwice(o, pattern, 'pattern')
        if pattern.respond_to?(:convert)
          conv = pattern.method(:convert).to_proc
        else
          conv = SPLAT_PROC
        end
        next
      end

      # anything others
      case o
      when Proc, Method
        block = notwice(o, block, 'block')
      when Array, Hash
        case pattern
        when CompletingHash
        when nil
          pattern = CompletingHash.new
          conv = pattern.method(:convert).to_proc if pattern.respond_to?(:convert)
        else
          raise ArgumentError, "argument pattern given twice"
        end
        o.each {|pat, *v| pattern[pat] = v.fetch(0) {pat}}
      when Module
        raise ArgumentError, "unsupported argument type: #{o}", ParseError.filter_backtrace(caller(4))
      when *ArgumentStyle.keys
        style = notwice(ArgumentStyle[o], style, 'style')
      when /^--no-([^\[\]=\s]*)(.+)?/
        q, a = $1, $2
        o = notwice(a ? Object : TrueClass, klass, 'type')
        not_pattern, not_conv = search(:atype, o) unless not_style
        not_style = (not_style || default_style).guess(arg = a) if a
        default_style = Switch::NoArgument
        default_pattern, conv = search(:atype, FalseClass) unless default_pattern
        ldesc << "--no-#{q}"
        (q = q.downcase).tr!('_', '-')
        long << "no-#{q}"
        nolong << q
      when /^--\[no-\]([^\[\]=\s]*)(.+)?/
        q, a = $1, $2
        o = notwice(a ? Object : TrueClass, klass, 'type')
        if a
          default_style = default_style.guess(arg = a)
          default_pattern, conv = search(:atype, o) unless default_pattern
        end
        ldesc << "--[no-]#{q}"
        (o = q.downcase).tr!('_', '-')
        long << o
        not_pattern, not_conv = search(:atype, FalseClass) unless not_style
        not_style = Switch::NoArgument
        nolong << "no-#{o}"
      when /^--([^\[\]=\s]*)(.+)?/
        q, a = $1, $2
        if a
          o = notwice(NilClass, klass, 'type')
          default_style = default_style.guess(arg = a)
          default_pattern, conv = search(:atype, o) unless default_pattern
        end
        ldesc << "--#{q}"
        (o = q.downcase).tr!('_', '-')
        long << o
      when /^-(\[\^?\]?(?:[^\\\]]|\\.)*\])(.+)?/
        q, a = $1, $2
        o = notwice(Object, klass, 'type')
        if a
          default_style = default_style.guess(arg = a)
          default_pattern, conv = search(:atype, o) unless default_pattern
        else
          has_arg = true
        end
        sdesc << "-#{q}"
        short << Regexp.new(q)
      when /^-(.)(.+)?/
        q, a = $1, $2
        if a
          o = notwice(NilClass, klass, 'type')
          default_style = default_style.guess(arg = a)
          default_pattern, conv = search(:atype, o) unless default_pattern
        end
        sdesc << "-#{q}"
        short << q
      when /^=/
        style = notwice(default_style.guess(arg = o), style, 'style')
        default_pattern, conv = search(:atype, Object) unless default_pattern
      else
        desc.push(o) if o && !o.empty?
      end
    end

    default_pattern, conv = search(:atype, default_style.pattern) unless default_pattern
    if !(short.empty? and long.empty?)
      if has_arg and default_style == Switch::NoArgument
        default_style = Switch::RequiredArgument
      end
      s = (style || default_style).new(pattern || default_pattern,
                                       conv, sdesc, ldesc, arg, desc, block)
    elsif !block
      if style or pattern
        raise ArgumentError, "no switch given", ParseError.filter_backtrace(caller)
      end
      s = desc
    else
      short << pattern
      s = (style || default_style).new(pattern,
                                       conv, nil, nil, arg, desc, block)
    end
    return s, short, long,
      (not_style.new(not_pattern, not_conv, sdesc, ldesc, nil, desc, block) if not_style),
      nolong
  end

  # ----
  # Option definition phase methods
  #
  # These methods are used to define options, or to construct an
  # OptionParser instance in other words.

  # :call-seq:
  #   define(*params, &block)
  #
  # :include: ../doc/optparse/creates_option.rdoc
  #
  def define(*opts, &block)
    top.append(*(sw = make_switch(opts, block)))
    sw[0]
  end

  # :call-seq:
  #   on(*params, &block)
  #
  # :include: ../doc/optparse/creates_option.rdoc
  #
  def on(*opts, &block)
    define(*opts, &block)
    self
  end
  alias def_option define

  # :call-seq:
  #   define_head(*params, &block)
  #
  # :include: ../doc/optparse/creates_option.rdoc
  #
  def define_head(*opts, &block)
    top.prepend(*(sw = make_switch(opts, block)))
    sw[0]
  end

  # :call-seq:
  #   on_head(*params, &block)
  #
  # :include: ../doc/optparse/creates_option.rdoc
  #
  # The new option is added at the head of the summary.
  #
  def on_head(*opts, &block)
    define_head(*opts, &block)
    self
  end
  alias def_head_option define_head

  # :call-seq:
  #   define_tail(*params, &block)
  #
  # :include: ../doc/optparse/creates_option.rdoc
  #
  def define_tail(*opts, &block)
    base.append(*(sw = make_switch(opts, block)))
    sw[0]
  end

  #
  # :call-seq:
  #   on_tail(*params, &block)
  #
  # :include: ../doc/optparse/creates_option.rdoc
  #
  # The new option is added at the tail of the summary.
  #
  def on_tail(*opts, &block)
    define_tail(*opts, &block)
    self
  end
  alias def_tail_option define_tail

  #
  # Add separator in summary.
  #
  def separator(string)
    top.append(string, nil, nil)
  end

  # ----
  # Arguments parse phase methods
  #
  # These methods parse +argv+, convert, and store the results by
  # calling handlers.  As these methods do not modify +self+, +self+
  # can be frozen.

  #
  # Parses command line arguments +argv+ in order. When a block is given,
  # each non-option argument is yielded. When optional +into+ keyword
  # argument is provided, the parsed option values are stored there via
  # <code>[]=</code> method (so it can be Hash, or OpenStruct, or other
  # similar object).
  #
  # Returns the rest of +argv+ left unparsed.
  #
  def order(*argv, **keywords, &nonopt)
    argv = argv[0].dup if argv.size == 1 and Array === argv[0]
    order!(argv, **keywords, &nonopt)
  end

  #
  # Same as #order, but removes switches destructively.
  # Non-option arguments remain in +argv+.
  #
  def order!(argv = default_argv, into: nil, **keywords, &nonopt)
    setter = ->(name, val) {into[name.to_sym] = val} if into
    parse_in_order(argv, setter, **keywords, &nonopt)
  end

  def parse_in_order(argv = default_argv, setter = nil, exact: require_exact, **, &nonopt)  # :nodoc:
    opt, arg, val, rest = nil
    nonopt ||= proc {|a| throw :terminate, a}
    argv.unshift(arg) if arg = catch(:terminate) {
      while arg = argv.shift
        case arg
        # long option
        when /\A--([^=]*)(?:=(.*))?/m
          opt, rest = $1, $2
          opt.tr!('_', '-')
          begin
            if exact
              sw, = search(:long, opt)
            else
              sw, = complete(:long, opt, true)
            end
          rescue ParseError
            throw :terminate, arg unless raise_unknown
            raise $!.set_option(arg, true)
          else
            unless sw
              throw :terminate, arg unless raise_unknown
              raise InvalidOption, arg
            end
          end
          begin
            opt, cb, val = sw.parse(rest, argv) {|*exc| raise(*exc)}
            val = callback!(cb, 1, val) if cb
            callback!(setter, 2, sw.switch_name, val) if setter
          rescue ParseError
            raise $!.set_option(arg, rest)
          end

        # short option
        when /\A-(.)((=).*|.+)?/m
          eq, rest, opt = $3, $2, $1
          has_arg, val = eq, rest
          begin
            sw, = search(:short, opt)
            unless sw
              begin
                sw, = complete(:short, opt)
                # short option matched.
                val = arg.delete_prefix('-')
                has_arg = true
              rescue InvalidOption
                raise if exact
                # if no short options match, try completion with long
                # options.
                sw, = complete(:long, opt)
                eq ||= !rest
              end
            end
          rescue ParseError
            throw :terminate, arg unless raise_unknown
            raise $!.set_option(arg, true)
          end
          begin
            opt, cb, val = sw.parse(val, argv) {|*exc| raise(*exc) if eq}
          rescue ParseError
            raise $!.set_option(arg, arg.length > 2)
          else
            raise InvalidOption, arg if has_arg and !eq and arg == "-#{opt}"
          end
          begin
            argv.unshift(opt) if opt and (!rest or (opt = opt.sub(/\A-*/, '-')) != '-')
            val = callback!(cb, 1, val) if cb
            callback!(setter, 2, sw.switch_name, val) if setter
          rescue ParseError
            raise $!.set_option(arg, arg.length > 2)
          end

        # non-option argument
        else
          catch(:prune) do
            visit(:each_option) do |sw0|
              sw = sw0
              sw.block.call(arg) if Switch === sw and sw.match_nonswitch?(arg)
            end
            nonopt.call(arg)
          end
        end
      end

      nil
    }

    visit(:search, :short, nil) {|sw| sw.block.call(*argv) if !sw.pattern}

    argv
  end
  private :parse_in_order

  # Calls callback with _val_.
  def callback!(cb, max_arity, *args) # :nodoc:
    args.compact!

    if (size = args.size) < max_arity and cb.to_proc.lambda?
      (arity = cb.arity) < 0 and arity = (1-arity)
      arity = max_arity if arity > max_arity
      args[arity - 1] = nil if arity > size
    end
    cb.call(*args)
  end
  private :callback!

  #
  # Parses command line arguments +argv+ in permutation mode and returns
  # list of non-option arguments. When optional +into+ keyword
  # argument is provided, the parsed option values are stored there via
  # <code>[]=</code> method (so it can be Hash, or OpenStruct, or other
  # similar object).
  #
  def permute(*argv, **keywords)
    argv = argv[0].dup if argv.size == 1 and Array === argv[0]
    permute!(argv, **keywords)
  end

  #
  # Same as #permute, but removes switches destructively.
  # Non-option arguments remain in +argv+.
  #
  def permute!(argv = default_argv, **keywords)
    nonopts = []
    order!(argv, **keywords, &nonopts.method(:<<))
    argv[0, 0] = nonopts
    argv
  end

  #
  # Parses command line arguments +argv+ in order when environment variable
  # POSIXLY_CORRECT is set, and in permutation mode otherwise.
  # When optional +into+ keyword argument is provided, the parsed option
  # values are stored there via <code>[]=</code> method (so it can be Hash,
  # or OpenStruct, or other similar object).
  #
  def parse(*argv, **keywords)
    argv = argv[0].dup if argv.size == 1 and Array === argv[0]
    parse!(argv, **keywords)
  end

  #
  # Same as #parse, but removes switches destructively.
  # Non-option arguments remain in +argv+.
  #
  def parse!(argv = default_argv, **keywords)
    if ENV.include?('POSIXLY_CORRECT')
      order!(argv, **keywords)
    else
      permute!(argv, **keywords)
    end
  end

  #
  # Wrapper method for getopts.rb.
  #
  #   params = ARGV.getopts("ab:", "foo", "bar:", "zot:Z;zot option")
  #   # params["a"] = true   # -a
  #   # params["b"] = "1"    # -b1
  #   # params["foo"] = "1"  # --foo
  #   # params["bar"] = "x"  # --bar x
  #   # params["zot"] = "z"  # --zot Z
  #
  # Option +symbolize_names+ (boolean) specifies whether returned Hash keys should be Symbols; defaults to +false+ (use Strings).
  #
  #   params = ARGV.getopts("ab:", "foo", "bar:", "zot:Z;zot option", symbolize_names: true)
  #   # params[:a] = true   # -a
  #   # params[:b] = "1"    # -b1
  #   # params[:foo] = "1"  # --foo
  #   # params[:bar] = "x"  # --bar x
  #   # params[:zot] = "z"  # --zot Z
  #
  def getopts(*args, symbolize_names: false, **keywords)
    argv = Array === args.first ? args.shift : default_argv
    single_options, *long_options = *args

    result = {}

    single_options.scan(/(.)(:)?/) do |opt, val|
      if val
        result[opt] = nil
        define("-#{opt} VAL")
      else
        result[opt] = false
        define("-#{opt}")
      end
    end if single_options

    long_options.each do |arg|
      arg, desc = arg.split(';', 2)
      opt, val = arg.split(':', 2)
      if val
        result[opt] = val.empty? ? nil : val
        define("--#{opt}=#{result[opt] || "VAL"}", *[desc].compact)
      else
        result[opt] = false
        define("--#{opt}", *[desc].compact)
      end
    end

    parse_in_order(argv, result.method(:[]=), **keywords)
    symbolize_names ? result.transform_keys(&:to_sym) : result
  end

  #
  # See #getopts.
  #
  def self.getopts(*args, symbolize_names: false)
    new.getopts(*args, symbolize_names: symbolize_names)
  end

  #
  # Traverses @stack, sending each element method +id+ with +args+ and
  # +block+.
  #
  def visit(id, *args, &block) # :nodoc:
    @stack.reverse_each do |el|
      el.__send__(id, *args, &block)
    end
    nil
  end
  private :visit

  #
  # Searches +key+ in @stack for +id+ hash and returns or yields the result.
  #
  def search(id, key) # :nodoc:
    block_given = block_given?
    visit(:search, id, key) do |k|
      return block_given ? yield(k) : k
    end
  end
  private :search

  #
  # Completes shortened long style option switch and returns pair of
  # canonical switch and switch descriptor OptionParser::Switch.
  #
  # +typ+::   Searching table.
  # +opt+::   Searching key.
  # +icase+:: Search case insensitive if true.
  # +pat+::   Optional pattern for completion.
  #
  def complete(typ, opt, icase = false, *pat) # :nodoc:
    if pat.empty?
      search(typ, opt) {|sw| return [sw, opt]} # exact match or...
    end
    ambiguous = catch(:ambiguous) {
      visit(:complete, typ, opt, icase, *pat) {|o, *sw| return sw}
    }
    exc = ambiguous ? AmbiguousOption : InvalidOption
    raise exc.new(opt, additional: self.method(:additional_message).curry[typ])
  end
  private :complete

  #
  # Returns additional info.
  #
  def additional_message(typ, opt)
    return unless typ and opt and defined?(DidYouMean::SpellChecker)
    all_candidates = []
    visit(:get_candidates, typ) do |candidates|
      all_candidates.concat(candidates)
    end
    all_candidates.select! {|cand| cand.is_a?(String) }
    checker = DidYouMean::SpellChecker.new(dictionary: all_candidates)
    DidYouMean.formatter.message_for(all_candidates & checker.correct(opt))
  end

  #
  # Return candidates for +word+.
  #
  def candidate(word)
    list = []
    case word
    when '-'
      long = short = true
    when /\A--/
      word, arg = word.split(/=/, 2)
      argpat = Completion.regexp(arg, false) if arg and !arg.empty?
      long = true
    when /\A-/
      short = true
    end
    pat = Completion.regexp(word, long)
    visit(:each_option) do |opt|
      next unless Switch === opt
      opts = (long ? opt.long : []) + (short ? opt.short : [])
      opts = Completion.candidate(word, true, pat, &opts.method(:each)).map(&:first) if pat
      if /\A=/ =~ opt.arg
        opts.map! {|sw| sw + "="}
        if arg and CompletingHash === opt.pattern
          if opts = opt.pattern.candidate(arg, false, argpat)
            opts.map!(&:last)
          end
        end
      end
      list.concat(opts)
    end
    list
  end

  #
  # Loads options from file names as +filename+. Does nothing when the file
  # is not present. Returns whether successfully loaded.
  #
  # +filename+ defaults to basename of the program without suffix in a
  # directory ~/.options, then the basename with '.options' suffix
  # under XDG and Haiku standard places.
  #
  # The optional +into+ keyword argument works exactly like that accepted in
  # method #parse.
  #
  def load(filename = nil, **keywords)
    unless filename
      basename = File.basename($0, '.*')
      return true if load(File.expand_path(basename, '~/.options'), **keywords) rescue nil
      basename << ".options"
      return [
        # XDG
        ENV['XDG_CONFIG_HOME'],
        '~/.config',
        *ENV['XDG_CONFIG_DIRS']&.split(File::PATH_SEPARATOR),

        # Haiku
        '~/config/settings',
      ].any? {|dir|
        next if !dir or dir.empty?
        load(File.expand_path(basename, dir), **keywords) rescue nil
      }
    end
    begin
      parse(*File.readlines(filename, chomp: true), **keywords)
      true
    rescue Errno::ENOENT, Errno::ENOTDIR
      false
    end
  end

  #
  # Parses environment variable +env+ or its uppercase with splitting like a
  # shell.
  #
  # +env+ defaults to the basename of the program.
  #
  def environment(env = File.basename($0, '.*'), **keywords)
    env = ENV[env] || ENV[env.upcase] or return
    require 'shellwords'
    parse(*Shellwords.shellwords(env), **keywords)
  end

  #
  # Acceptable argument classes
  #

  #
  # Any string and no conversion. This is fall-back.
  #
  accept(Object) {|s,|s or s.nil?}

  accept(NilClass) {|s,|s}

  #
  # Any non-empty string, and no conversion.
  #
  accept(String, /.+/m) {|s,*|s}

  #
  # Ruby/C-like integer, octal for 0-7 sequence, binary for 0b, hexadecimal
  # for 0x, and decimal for others; with optional sign prefix. Converts to
  # Integer.
  #
  decimal = '\d+(?:_\d+)*'
  binary = 'b[01]+(?:_[01]+)*'
  hex = 'x[\da-f]+(?:_[\da-f]+)*'
  octal = "0(?:[0-7]+(?:_[0-7]+)*|#{binary}|#{hex})?"
  integer = "#{octal}|#{decimal}"

  accept(Integer, %r"\A[-+]?(?:#{integer})\z"io) {|s,|
    begin
      Integer(s)
    rescue ArgumentError
      raise OptionParser::InvalidArgument, s
    end if s
  }

  #
  # Float number format, and converts to Float.
  #
  float = "(?:#{decimal}(?=(.)?)(?:\\.(?:#{decimal})?)?|\\.#{decimal})(?:E[-+]?#{decimal})?"
  floatpat = %r"\A[-+]?#{float}\z"io
  accept(Float, floatpat) {|s,| s.to_f if s}

  #
  # Generic numeric format, converts to Integer for integer format, Float
  # for float format, and Rational for rational format.
  #
  real = "[-+]?(?:#{octal}|#{float})"
  accept(Numeric, /\A(#{real})(?:\/(#{real}))?\z/io) {|s, d, f, n,|
    if n
      Rational(d, n)
    elsif f
      Float(s)
    else
      Integer(s)
    end
  }

  #
  # Decimal integer format, to be converted to Integer.
  #
  DecimalInteger = /\A[-+]?#{decimal}\z/io
  accept(DecimalInteger, DecimalInteger) {|s,|
    begin
      Integer(s, 10)
    rescue ArgumentError
      raise OptionParser::InvalidArgument, s
    end if s
  }

  #
  # Ruby/C like octal/hexadecimal/binary integer format, to be converted to
  # Integer.
  #
  OctalInteger = /\A[-+]?(?:[0-7]+(?:_[0-7]+)*|0(?:#{binary}|#{hex}))\z/io
  accept(OctalInteger, OctalInteger) {|s,|
    begin
      Integer(s, 8)
    rescue ArgumentError
      raise OptionParser::InvalidArgument, s
    end if s
  }

  #
  # Decimal integer/float number format, to be converted to Integer for
  # integer format, Float for float format.
  #
  DecimalNumeric = floatpat     # decimal integer is allowed as float also.
  accept(DecimalNumeric, floatpat) {|s, f|
    begin
      if f
        Float(s)
      else
        Integer(s)
      end
    rescue ArgumentError
      raise OptionParser::InvalidArgument, s
    end if s
  }

  #
  # Boolean switch, which means whether it is present or not, whether it is
  # absent or not with prefix no-, or it takes an argument
  # yes/no/true/false/+/-.
  #
  yesno = CompletingHash.new
  %w[- no false].each {|el| yesno[el] = false}
  %w[+ yes true].each {|el| yesno[el] = true}
  yesno['nil'] = false          # should be nil?
  accept(TrueClass, yesno) {|arg, val| val == nil or val}
  #
  # Similar to TrueClass, but defaults to false.
  #
  accept(FalseClass, yesno) {|arg, val| val != nil and val}

  #
  # List of strings separated by ",".
  #
  accept(Array) do |s, |
    if s
      s = s.split(',').collect {|ss| ss unless ss.empty?}
    end
    s
  end

  #
  # Regular expression with options.
  #
  accept(Regexp, %r"\A/((?:\\.|[^\\])*)/([[:alpha:]]+)?\z|.*") do |all, s, o|
    f = 0
    if o
      f |= Regexp::IGNORECASE if /i/ =~ o
      f |= Regexp::MULTILINE if /m/ =~ o
      f |= Regexp::EXTENDED if /x/ =~ o
      case o = o.delete("imx")
      when ""
      when "u"
        s = s.encode(Encoding::UTF_8)
      when "e"
        s = s.encode(Encoding::EUC_JP)
      when "s"
        s = s.encode(Encoding::SJIS)
      when "n"
        f |= Regexp::NOENCODING
      else
        raise OptionParser::InvalidArgument, "unknown regexp option - #{o}"
      end
    else
      s ||= all
    end
    Regexp.new(s, f)
  end

  #
  # Exceptions
  #

  #
  # Base class of exceptions from OptionParser.
  #
  class ParseError < RuntimeError
    # Reason which caused the error.
    Reason = 'parse error'

    # :nodoc:
    def initialize(*args, additional: nil)
      @additional = additional
      @arg0, = args
      @args = args
      @reason = nil
    end

    attr_reader :args
    attr_writer :reason
    attr_accessor :additional

    #
    # Pushes back erred argument(s) to +argv+.
    #
    def recover(argv)
      argv[0, 0] = @args
      argv
    end

    def self.filter_backtrace(array)
      unless $DEBUG
        array.delete_if(&%r"\A#{Regexp.quote(__FILE__)}:"o.method(:=~))
      end
      array
    end

    def set_backtrace(array)
      super(self.class.filter_backtrace(array))
    end

    def set_option(opt, eq)
      if eq
        @args[0] = opt
      else
        @args.unshift(opt)
      end
      self
    end

    #
    # Returns error reason. Override this for I18N.
    #
    def reason
      @reason || self.class::Reason
    end

    def inspect
      "#<#{self.class}: #{args.join(' ')}>"
    end

    #
    # Default stringizing method to emit standard error message.
    #
    def message
      "#{reason}: #{args.join(' ')}#{additional[@arg0] if additional}"
    end

    alias to_s message
  end

  #
  # Raises when ambiguously completable string is encountered.
  #
  class AmbiguousOption < ParseError
    const_set(:Reason, 'ambiguous option')
  end

  #
  # Raises when there is an argument for a switch which takes no argument.
  #
  class NeedlessArgument < ParseError
    const_set(:Reason, 'needless argument')
  end

  #
  # Raises when a switch with mandatory argument has no argument.
  #
  class MissingArgument < ParseError
    const_set(:Reason, 'missing argument')
  end

  #
  # Raises when switch is undefined.
  #
  class InvalidOption < ParseError
    const_set(:Reason, 'invalid option')
  end

  #
  # Raises when the given argument does not match required format.
  #
  class InvalidArgument < ParseError
    const_set(:Reason, 'invalid argument')
  end

  #
  # Raises when the given argument word can't be completed uniquely.
  #
  class AmbiguousArgument < InvalidArgument
    const_set(:Reason, 'ambiguous argument')
  end

  #
  # Miscellaneous
  #

  #
  # Extends command line arguments array (ARGV) to parse itself.
  #
  module Arguable

    #
    # Sets OptionParser object, when +opt+ is +false+ or +nil+, methods
    # OptionParser::Arguable#options and OptionParser::Arguable#options= are
    # undefined. Thus, there is no ways to access the OptionParser object
    # via the receiver object.
    #
    def options=(opt)
      unless @optparse = opt
        class << self
          undef_method(:options)
          undef_method(:options=)
        end
      end
    end

    #
    # Actual OptionParser object, automatically created if nonexistent.
    #
    # If called with a block, yields the OptionParser object and returns the
    # result of the block. If an OptionParser::ParseError exception occurs
    # in the block, it is rescued, a error message printed to STDERR and
    # +nil+ returned.
    #
    def options
      @optparse ||= OptionParser.new
      @optparse.default_argv = self
      block_given? or return @optparse
      begin
        yield @optparse
      rescue ParseError
        @optparse.warn $!
        nil
      end
    end

    #
    # Parses +self+ destructively in order and returns +self+ containing the
    # rest arguments left unparsed.
    #
    def order!(**keywords, &blk) options.order!(self, **keywords, &blk) end

    #
    # Parses +self+ destructively in permutation mode and returns +self+
    # containing the rest arguments left unparsed.
    #
    def permute!(**keywords) options.permute!(self, **keywords) end

    #
    # Parses +self+ destructively and returns +self+ containing the
    # rest arguments left unparsed.
    #
    def parse!(**keywords) options.parse!(self, **keywords) end

    #
    # Substitution of getopts is possible as follows. Also see
    # OptionParser#getopts.
    #
    #   def getopts(*args)
    #     ($OPT = ARGV.getopts(*args)).each do |opt, val|
    #       eval "$OPT_#{opt.gsub(/[^A-Za-z0-9_]/, '_')} = val"
    #     end
    #   rescue OptionParser::ParseError
    #   end
    #
    def getopts(*args, symbolize_names: false, **keywords)
      options.getopts(self, *args, symbolize_names: symbolize_names, **keywords)
    end

    #
    # Initializes instance variable.
    #
    def self.extend_object(obj)
      super
      obj.instance_eval {@optparse = nil}
    end

    def initialize(*args)       # :nodoc:
      super
      @optparse = nil
    end
  end

  #
  # Acceptable argument classes. Now contains DecimalInteger, OctalInteger
  # and DecimalNumeric. See Acceptable argument classes (in source code).
  #
  module Acceptables
    const_set(:DecimalInteger, OptionParser::DecimalInteger)
    const_set(:OctalInteger, OptionParser::OctalInteger)
    const_set(:DecimalNumeric, OptionParser::DecimalNumeric)
  end
end

# ARGV is arguable by OptionParser
ARGV.extend(OptionParser::Arguable)

# An alias for OptionParser.
OptParse = OptionParser  # :nodoc:
                                                                                                                                                                                                                                                                                                                                             ruby/yaml.rb                                                                                        0000644                 00000004210 15040313432 0007004 0                                                                                                    ustar 00                                                                                                                                                                                                                                                       # frozen_string_literal: false

begin
  require 'psych'
rescue LoadError
  case RUBY_ENGINE
  when 'jruby'
    warn "The Psych YAML extension failed to load.\n" \
      "Check your env for conflicting versions of SnakeYAML\n" \
      "See https://github.com/jruby/jruby/wiki/FAQs#why-does-the-psych-yaml-extension-fail-to-load-in-my-environment",
         uplevel: 1
  else
    warn "It seems your ruby installation is missing psych (for YAML output).\n" \
      "To eliminate this warning, please install libyaml and reinstall your ruby.\n",
         uplevel: 1
  end
  raise
end

YAML = Psych # :nodoc:

# YAML Ain't Markup Language
#
# This module provides a Ruby interface for data serialization in YAML format.
#
# The YAML module is an alias of Psych, the YAML engine for Ruby.
#
# == Usage
#
# Working with YAML can be very simple, for example:
#
#     require 'yaml'
#     # Parse a YAML string
#     YAML.load("--- foo") #=> "foo"
#
#     # Emit some YAML
#     YAML.dump("foo")     # => "--- foo\n...\n"
#     { :a => 'b'}.to_yaml  # => "---\n:a: b\n"
#
# As the implementation is provided by the Psych library, detailed documentation
# can be found in that library's docs (also part of standard library).
#
# == Security
#
# Do not use YAML to load untrusted data. Doing so is unsafe and could allow
# malicious input to execute arbitrary code inside your application. Please see
# doc/security.rdoc for more information.
#
# == History
#
# Syck was the original YAML implementation in Ruby's standard library
# developed by why the lucky stiff.
#
# You can still use Syck, if you prefer, for parsing and emitting YAML, but you
# must install the 'syck' gem now in order to use it.
#
# In older Ruby versions, ie. <= 1.9, Syck is still provided, however it was
# completely removed with the release of Ruby 2.0.0.
#
# == More info
#
# For more advanced details on the implementation see Psych, and also check out
# http://yaml.org for spec details and other helpful information.
#
# Psych is maintained by Aaron Patterson on github: https://github.com/ruby/psych
#
# Syck can also be found on github: https://github.com/ruby/syck
module YAML
  LOADER_VERSION = "0.4.0"
end
                                                                                                                                                                                                                                                                                                                                                                                        ruby/fileutils.rb                                                                                   0000644                 00000235463 15040313432 0010062 0                                                                                                    ustar 00                                                                                                                                                                                                                                                       # frozen_string_literal: true

begin
  require 'rbconfig'
rescue LoadError
  # for make rjit-headers
end

# Namespace for file utility methods for copying, moving, removing, etc.
#
# == What's Here
#
# First, what’s elsewhere. \Module \FileUtils:
#
# - Inherits from {class Object}[rdoc-ref:Object].
# - Supplements {class File}[rdoc-ref:File]
#   (but is not included or extended there).
#
# Here, module \FileUtils provides methods that are useful for:
#
# - {Creating}[rdoc-ref:FileUtils@Creating].
# - {Deleting}[rdoc-ref:FileUtils@Deleting].
# - {Querying}[rdoc-ref:FileUtils@Querying].
# - {Setting}[rdoc-ref:FileUtils@Setting].
# - {Comparing}[rdoc-ref:FileUtils@Comparing].
# - {Copying}[rdoc-ref:FileUtils@Copying].
# - {Moving}[rdoc-ref:FileUtils@Moving].
# - {Options}[rdoc-ref:FileUtils@Options].
#
# === Creating
#
# - ::mkdir: Creates directories.
# - ::mkdir_p, ::makedirs, ::mkpath: Creates directories,
#   also creating ancestor directories as needed.
# - ::link_entry: Creates a hard link.
# - ::ln, ::link: Creates hard links.
# - ::ln_s, ::symlink: Creates symbolic links.
# - ::ln_sf: Creates symbolic links, overwriting if necessary.
# - ::ln_sr: Creates symbolic links relative to targets
#
# === Deleting
#
# - ::remove_dir: Removes a directory and its descendants.
# - ::remove_entry: Removes an entry, including its descendants if it is a directory.
# - ::remove_entry_secure: Like ::remove_entry, but removes securely.
# - ::remove_file: Removes a file entry.
# - ::rm, ::remove: Removes entries.
# - ::rm_f, ::safe_unlink: Like ::rm, but removes forcibly.
# - ::rm_r: Removes entries and their descendants.
# - ::rm_rf, ::rmtree: Like ::rm_r, but removes forcibly.
# - ::rmdir: Removes directories.
#
# === Querying
#
# - ::pwd, ::getwd: Returns the path to the working directory.
# - ::uptodate?: Returns whether a given entry is newer than given other entries.
#
# === Setting
#
# - ::cd, ::chdir: Sets the working directory.
# - ::chmod: Sets permissions for an entry.
# - ::chmod_R: Sets permissions for an entry and its descendants.
# - ::chown: Sets the owner and group for entries.
# - ::chown_R: Sets the owner and group for entries and their descendants.
# - ::touch: Sets modification and access times for entries,
#   creating if necessary.
#
# === Comparing
#
# - ::compare_file, ::cmp, ::identical?: Returns whether two entries are identical.
# - ::compare_stream: Returns whether two streams are identical.
#
# === Copying
#
# - ::copy_entry: Recursively copies an entry.
# - ::copy_file: Copies an entry.
# - ::copy_stream: Copies a stream.
# - ::cp, ::copy: Copies files.
# - ::cp_lr: Recursively creates hard links.
# - ::cp_r: Recursively copies files, retaining mode, owner, and group.
# - ::install: Recursively copies files, optionally setting mode,
#   owner, and group.
#
# === Moving
#
# - ::mv, ::move: Moves entries.
#
# === Options
#
# - ::collect_method: Returns the names of methods that accept a given option.
# - ::commands: Returns the names of methods that accept options.
# - ::have_option?: Returns whether a given method accepts a given option.
# - ::options: Returns all option names.
# - ::options_of: Returns the names of the options for a given method.
#
# == Path Arguments
#
# Some methods in \FileUtils accept _path_ arguments,
# which are interpreted as paths to filesystem entries:
#
# - If the argument is a string, that value is the path.
# - If the argument has method +:to_path+, it is converted via that method.
# - If the argument has method +:to_str+, it is converted via that method.
#
# == About the Examples
#
# Some examples here involve trees of file entries.
# For these, we sometimes display trees using the
# {tree command-line utility}[https://en.wikipedia.org/wiki/Tree_(command)],
# which is a recursive directory-listing utility that produces
# a depth-indented listing of files and directories.
#
# We use a helper method to launch the command and control the format:
#
#   def tree(dirpath = '.')
#     command = "tree --noreport --charset=ascii #{dirpath}"
#     system(command)
#   end
#
# To illustrate:
#
#   tree('src0')
#   # => src0
#   #    |-- sub0
#   #    |   |-- src0.txt
#   #    |   `-- src1.txt
#   #    `-- sub1
#   #        |-- src2.txt
#   #        `-- src3.txt
#
# == Avoiding the TOCTTOU Vulnerability
#
# For certain methods that recursively remove entries,
# there is a potential vulnerability called the
# {Time-of-check to time-of-use}[https://en.wikipedia.org/wiki/Time-of-check_to_time-of-use],
# or TOCTTOU, vulnerability that can exist when:
#
# - An ancestor directory of the entry at the target path is world writable;
#   such directories include <tt>/tmp</tt>.
# - The directory tree at the target path includes:
#
#   - A world-writable descendant directory.
#   - A symbolic link.
#
# To avoid that vulnerability, you can use this method to remove entries:
#
# - FileUtils.remove_entry_secure: removes recursively
#   if the target path points to a directory.
#
# Also available are these methods,
# each of which calls \FileUtils.remove_entry_secure:
#
# - FileUtils.rm_r with keyword argument <tt>secure: true</tt>.
# - FileUtils.rm_rf with keyword argument <tt>secure: true</tt>.
#
# Finally, this method for moving entries calls \FileUtils.remove_entry_secure
# if the source and destination are on different file systems
# (which means that the "move" is really a copy and remove):
#
# - FileUtils.mv with keyword argument <tt>secure: true</tt>.
#
# \Method \FileUtils.remove_entry_secure removes securely
# by applying a special pre-process:
#
# - If the target path points to a directory, this method uses methods
#   {File#chown}[rdoc-ref:File#chown]
#   and {File#chmod}[rdoc-ref:File#chmod]
#   in removing directories.
# - The owner of the target directory should be either the current process
#   or the super user (root).
#
# WARNING: You must ensure that *ALL* parent directories cannot be
# moved by other untrusted users.  For example, parent directories
# should not be owned by untrusted users, and should not be world
# writable except when the sticky bit is set.
#
# For details of this security vulnerability, see Perl cases:
#
# - {CVE-2005-0448}[https://cve.mitre.org/cgi-bin/cvename.cgi?name=CAN-2005-0448].
# - {CVE-2004-0452}[https://cve.mitre.org/cgi-bin/cvename.cgi?name=CAN-2004-0452].
#
module FileUtils
  # The version number.
  VERSION = "1.7.3"

  def self.private_module_function(name)   #:nodoc:
    module_function name
    private_class_method name
  end

  #
  # Returns a string containing the path to the current directory:
  #
  #   FileUtils.pwd # => "/rdoc/fileutils"
  #
  # Related: FileUtils.cd.
  #
  def pwd
    Dir.pwd
  end
  module_function :pwd

  alias getwd pwd
  module_function :getwd

  # Changes the working directory to the given +dir+, which
  # should be {interpretable as a path}[rdoc-ref:FileUtils@Path+Arguments]:
  #
  # With no block given,
  # changes the current directory to the directory at +dir+; returns zero:
  #
  #   FileUtils.pwd # => "/rdoc/fileutils"
  #   FileUtils.cd('..')
  #   FileUtils.pwd # => "/rdoc"
  #   FileUtils.cd('fileutils')
  #
  # With a block given, changes the current directory to the directory
  # at +dir+, calls the block with argument +dir+,
  # and restores the original current directory; returns the block's value:
  #
  #   FileUtils.pwd                                     # => "/rdoc/fileutils"
  #   FileUtils.cd('..') { |arg| [arg, FileUtils.pwd] } # => ["..", "/rdoc"]
  #   FileUtils.pwd                                     # => "/rdoc/fileutils"
  #
  # Keyword arguments:
  #
  # - <tt>verbose: true</tt> - prints an equivalent command:
  #
  #     FileUtils.cd('..')
  #     FileUtils.cd('fileutils')
  #
  #   Output:
  #
  #     cd ..
  #     cd fileutils
  #
  # Related: FileUtils.pwd.
  #
  def cd(dir, verbose: nil, &block) # :yield: dir
    fu_output_message "cd #{dir}" if verbose
    result = Dir.chdir(dir, &block)
    fu_output_message 'cd -' if verbose and block
    result
  end
  module_function :cd

  alias chdir cd
  module_function :chdir

  #
  # Returns +true+ if the file at path +new+
  # is newer than all the files at paths in array +old_list+;
  # +false+ otherwise.
  #
  # Argument +new+ and the elements of +old_list+
  # should be {interpretable as paths}[rdoc-ref:FileUtils@Path+Arguments]:
  #
  #   FileUtils.uptodate?('Rakefile', ['Gemfile', 'README.md']) # => true
  #   FileUtils.uptodate?('Gemfile', ['Rakefile', 'README.md']) # => false
  #
  # A non-existent file is considered to be infinitely old.
  #
  # Related: FileUtils.touch.
  #
  def uptodate?(new, old_list)
    return false unless File.exist?(new)
    new_time = File.mtime(new)
    old_list.each do |old|
      if File.exist?(old)
        return false unless new_time > File.mtime(old)
      end
    end
    true
  end
  module_function :uptodate?

  def remove_trailing_slash(dir)   #:nodoc:
    dir == '/' ? dir : dir.chomp(?/)
  end
  private_module_function :remove_trailing_slash

  #
  # Creates directories at the paths in the given +list+
  # (a single path or an array of paths);
  # returns +list+ if it is an array, <tt>[list]</tt> otherwise.
  #
  # Argument +list+ or its elements
  # should be {interpretable as paths}[rdoc-ref:FileUtils@Path+Arguments].
  #
  # With no keyword arguments, creates a directory at each +path+ in +list+
  # by calling: <tt>Dir.mkdir(path, mode)</tt>;
  # see {Dir.mkdir}[rdoc-ref:Dir.mkdir]:
  #
  #   FileUtils.mkdir(%w[tmp0 tmp1]) # => ["tmp0", "tmp1"]
  #   FileUtils.mkdir('tmp4')        # => ["tmp4"]
  #
  # Keyword arguments:
  #
  # - <tt>mode: <i>mode</i></tt> - also calls <tt>File.chmod(mode, path)</tt>;
  #   see {File.chmod}[rdoc-ref:File.chmod].
  # - <tt>noop: true</tt> - does not create directories.
  # - <tt>verbose: true</tt> - prints an equivalent command:
  #
  #     FileUtils.mkdir(%w[tmp0 tmp1], verbose: true)
  #     FileUtils.mkdir(%w[tmp2 tmp3], mode: 0700, verbose: true)
  #
  #   Output:
  #
  #     mkdir tmp0 tmp1
  #     mkdir -m 700 tmp2 tmp3
  #
  # Raises an exception if any path points to an existing
  # file or directory, or if for any reason a directory cannot be created.
  #
  # Related: FileUtils.mkdir_p.
  #
  def mkdir(list, mode: nil, noop: nil, verbose: nil)
    list = fu_list(list)
    fu_output_message "mkdir #{mode ? ('-m %03o ' % mode) : ''}#{list.join ' '}" if verbose
    return if noop

    list.each do |dir|
      fu_mkdir dir, mode
    end
  end
  module_function :mkdir

  #
  # Creates directories at the paths in the given +list+
  # (a single path or an array of paths),
  # also creating ancestor directories as needed;
  # returns +list+ if it is an array, <tt>[list]</tt> otherwise.
  #
  # Argument +list+ or its elements
  # should be {interpretable as paths}[rdoc-ref:FileUtils@Path+Arguments].
  #
  # With no keyword arguments, creates a directory at each +path+ in +list+,
  # along with any needed ancestor directories,
  # by calling: <tt>Dir.mkdir(path, mode)</tt>;
  # see {Dir.mkdir}[rdoc-ref:Dir.mkdir]:
  #
  #   FileUtils.mkdir_p(%w[tmp0/tmp1 tmp2/tmp3]) # => ["tmp0/tmp1", "tmp2/tmp3"]
  #   FileUtils.mkdir_p('tmp4/tmp5')             # => ["tmp4/tmp5"]
  #
  # Keyword arguments:
  #
  # - <tt>mode: <i>mode</i></tt> - also calls <tt>File.chmod(mode, path)</tt>;
  #   see {File.chmod}[rdoc-ref:File.chmod].
  # - <tt>noop: true</tt> - does not create directories.
  # - <tt>verbose: true</tt> - prints an equivalent command:
  #
  #     FileUtils.mkdir_p(%w[tmp0 tmp1], verbose: true)
  #     FileUtils.mkdir_p(%w[tmp2 tmp3], mode: 0700, verbose: true)
  #
  #   Output:
  #
  #     mkdir -p tmp0 tmp1
  #     mkdir -p -m 700 tmp2 tmp3
  #
  # Raises an exception if for any reason a directory cannot be created.
  #
  # FileUtils.mkpath and FileUtils.makedirs are aliases for FileUtils.mkdir_p.
  #
  # Related: FileUtils.mkdir.
  #
  def mkdir_p(list, mode: nil, noop: nil, verbose: nil)
    list = fu_list(list)
    fu_output_message "mkdir -p #{mode ? ('-m %03o ' % mode) : ''}#{list.join ' '}" if verbose
    return *list if noop

    list.each do |item|
      path = remove_trailing_slash(item)

      stack = []
      until File.directory?(path) || File.dirname(path) == path
        stack.push path
        path = File.dirname(path)
      end
      stack.reverse_each do |dir|
        begin
          fu_mkdir dir, mode
        rescue SystemCallError
          raise unless File.directory?(dir)
        end
      end
    end

    return *list
  end
  module_function :mkdir_p

  alias mkpath    mkdir_p
  alias makedirs  mkdir_p
  module_function :mkpath
  module_function :makedirs

  def fu_mkdir(path, mode)   #:nodoc:
    path = remove_trailing_slash(path)
    if mode
      Dir.mkdir path, mode
      File.chmod mode, path
    else
      Dir.mkdir path
    end
  end
  private_module_function :fu_mkdir

  #
  # Removes directories at the paths in the given +list+
  # (a single path or an array of paths);
  # returns +list+, if it is an array, <tt>[list]</tt> otherwise.
  #
  # Argument +list+ or its elements
  # should be {interpretable as paths}[rdoc-ref:FileUtils@Path+Arguments].
  #
  # With no keyword arguments, removes the directory at each +path+ in +list+,
  # by calling: <tt>Dir.rmdir(path)</tt>;
  # see {Dir.rmdir}[rdoc-ref:Dir.rmdir]:
  #
  #   FileUtils.rmdir(%w[tmp0/tmp1 tmp2/tmp3]) # => ["tmp0/tmp1", "tmp2/tmp3"]
  #   FileUtils.rmdir('tmp4/tmp5')             # => ["tmp4/tmp5"]
  #
  # Keyword arguments:
  #
  # - <tt>parents: true</tt> - removes successive ancestor directories
  #   if empty.
  # - <tt>noop: true</tt> - does not remove directories.
  # - <tt>verbose: true</tt> - prints an equivalent command:
  #
  #     FileUtils.rmdir(%w[tmp0/tmp1 tmp2/tmp3], parents: true, verbose: true)
  #     FileUtils.rmdir('tmp4/tmp5', parents: true, verbose: true)
  #
  #   Output:
  #
  #     rmdir -p tmp0/tmp1 tmp2/tmp3
  #     rmdir -p tmp4/tmp5
  #
  # Raises an exception if a directory does not exist
  # or if for any reason a directory cannot be removed.
  #
  # Related: {methods for deleting}[rdoc-ref:FileUtils@Deleting].
  #
  def rmdir(list, parents: nil, noop: nil, verbose: nil)
    list = fu_list(list)
    fu_output_message "rmdir #{parents ? '-p ' : ''}#{list.join ' '}" if verbose
    return if noop
    list.each do |dir|
      Dir.rmdir(dir = remove_trailing_slash(dir))
      if parents
        begin
          until (parent = File.dirname(dir)) == '.' or parent == dir
            dir = parent
            Dir.rmdir(dir)
          end
        rescue Errno::ENOTEMPTY, Errno::EEXIST, Errno::ENOENT
        end
      end
    end
  end
  module_function :rmdir

  # Creates {hard links}[https://en.wikipedia.org/wiki/Hard_link].
  #
  # Arguments +src+ (a single path or an array of paths)
  # and +dest+ (a single path)
  # should be {interpretable as paths}[rdoc-ref:FileUtils@Path+Arguments].
  #
  # When +src+ is the path to an existing file
  # and +dest+ is the path to a non-existent file,
  # creates a hard link at +dest+ pointing to +src+; returns zero:
  #
  #   Dir.children('tmp0/')                    # => ["t.txt"]
  #   Dir.children('tmp1/')                    # => []
  #   FileUtils.ln('tmp0/t.txt', 'tmp1/t.lnk') # => 0
  #   Dir.children('tmp1/')                    # => ["t.lnk"]
  #
  # When +src+ is the path to an existing file
  # and +dest+ is the path to an existing directory,
  # creates a hard link at <tt>dest/src</tt> pointing to +src+; returns zero:
  #
  #   Dir.children('tmp2')               # => ["t.dat"]
  #   Dir.children('tmp3')               # => []
  #   FileUtils.ln('tmp2/t.dat', 'tmp3') # => 0
  #   Dir.children('tmp3')               # => ["t.dat"]
  #
  # When +src+ is an array of paths to existing files
  # and +dest+ is the path to an existing directory,
  # then for each path +target+ in +src+,
  # creates a hard link at <tt>dest/target</tt> pointing to +target+;
  # returns +src+:
  #
  #   Dir.children('tmp4/')                               # => []
  #   FileUtils.ln(['tmp0/t.txt', 'tmp2/t.dat'], 'tmp4/') # => ["tmp0/t.txt", "tmp2/t.dat"]
  #   Dir.children('tmp4/')                               # => ["t.dat", "t.txt"]
  #
  # Keyword arguments:
  #
  # - <tt>force: true</tt> - overwrites +dest+ if it exists.
  # - <tt>noop: true</tt> - does not create links.
  # - <tt>verbose: true</tt> - prints an equivalent command:
  #
  #     FileUtils.ln('tmp0/t.txt', 'tmp1/t.lnk', verbose: true)
  #     FileUtils.ln('tmp2/t.dat', 'tmp3', verbose: true)
  #     FileUtils.ln(['tmp0/t.txt', 'tmp2/t.dat'], 'tmp4/', verbose: true)
  #
  #   Output:
  #
  #     ln tmp0/t.txt tmp1/t.lnk
  #     ln tmp2/t.dat tmp3
  #     ln tmp0/t.txt tmp2/t.dat tmp4/
  #
  # Raises an exception if +dest+ is the path to an existing file
  # and keyword argument +force+ is not +true+.
  #
  # Related: FileUtils.link_entry (has different options).
  #
  def ln(src, dest, force: nil, noop: nil, verbose: nil)
    fu_output_message "ln#{force ? ' -f' : ''} #{[src,dest].flatten.join ' '}" if verbose
    return if noop
    fu_each_src_dest0(src, dest) do |s,d|
      remove_file d, true if force
      File.link s, d
    end
  end
  module_function :ln

  alias link ln
  module_function :link

  # Creates {hard links}[https://en.wikipedia.org/wiki/Hard_link].
  #
  # Arguments +src+ (a single path or an array of paths)
  # and +dest+ (a single path)
  # should be {interpretable as paths}[rdoc-ref:FileUtils@Path+Arguments].
  #
  # If +src+ is the path to a directory and +dest+ does not exist,
  # creates links +dest+ and descendents pointing to +src+ and its descendents:
  #
  #   tree('src0')
  #   # => src0
  #   #    |-- sub0
  #   #    |   |-- src0.txt
  #   #    |   `-- src1.txt
  #   #    `-- sub1
  #   #        |-- src2.txt
  #   #        `-- src3.txt
  #   File.exist?('dest0') # => false
  #   FileUtils.cp_lr('src0', 'dest0')
  #   tree('dest0')
  #   # => dest0
  #   #    |-- sub0
  #   #    |   |-- src0.txt
  #   #    |   `-- src1.txt
  #   #    `-- sub1
  #   #        |-- src2.txt
  #   #        `-- src3.txt
  #
  # If +src+ and +dest+ are both paths to directories,
  # creates links <tt>dest/src</tt> and descendents
  # pointing to +src+ and its descendents:
  #
  #   tree('src1')
  #   # => src1
  #   #    |-- sub0
  #   #    |   |-- src0.txt
  #   #    |   `-- src1.txt
  #   #    `-- sub1
  #   #        |-- src2.txt
  #   #        `-- src3.txt
  #   FileUtils.mkdir('dest1')
  #   FileUtils.cp_lr('src1', 'dest1')
  #   tree('dest1')
  #   # => dest1
  #   #    `-- src1
  #   #        |-- sub0
  #   #        |   |-- src0.txt
  #   #        |   `-- src1.txt
  #   #        `-- sub1
  #   #            |-- src2.txt
  #   #            `-- src3.txt
  #
  # If +src+ is an array of paths to entries and +dest+ is the path to a directory,
  # for each path +filepath+ in +src+, creates a link at <tt>dest/filepath</tt>
  # pointing to that path:
  #
  #   tree('src2')
  #   # => src2
  #   #    |-- sub0
  #   #    |   |-- src0.txt
  #   #    |   `-- src1.txt
  #   #    `-- sub1
  #   #        |-- src2.txt
  #   #        `-- src3.txt
  #   FileUtils.mkdir('dest2')
  #   FileUtils.cp_lr(['src2/sub0', 'src2/sub1'], 'dest2')
  #   tree('dest2')
  #   # => dest2
  #   #    |-- sub0
  #   #    |   |-- src0.txt
  #   #    |   `-- src1.txt
  #   #    `-- sub1
  #   #        |-- src2.txt
  #   #        `-- src3.txt
  #
  # Keyword arguments:
  #
  # - <tt>dereference_root: false</tt> - if +src+ is a symbolic link,
  #   does not dereference it.
  # - <tt>noop: true</tt> - does not create links.
  # - <tt>remove_destination: true</tt> - removes +dest+ before creating links.
  # - <tt>verbose: true</tt> - prints an equivalent command:
  #
  #     FileUtils.cp_lr('src0', 'dest0', noop: true, verbose: true)
  #     FileUtils.cp_lr('src1', 'dest1', noop: true, verbose: true)
  #     FileUtils.cp_lr(['src2/sub0', 'src2/sub1'], 'dest2', noop: true, verbose: true)
  #
  #   Output:
  #
  #     cp -lr src0 dest0
  #     cp -lr src1 dest1
  #     cp -lr src2/sub0 src2/sub1 dest2
  #
  # Raises an exception if +dest+ is the path to an existing file or directory
  # and keyword argument <tt>remove_destination: true</tt> is not given.
  #
  # Related: {methods for copying}[rdoc-ref:FileUtils@Copying].
  #
  def cp_lr(src, dest, noop: nil, verbose: nil,
            dereference_root: true, remove_destination: false)
    fu_output_message "cp -lr#{remove_destination ? ' --remove-destination' : ''} #{[src,dest].flatten.join ' '}" if verbose
    return if noop
    fu_each_src_dest(src, dest) do |s, d|
      link_entry s, d, dereference_root, remove_destination
    end
  end
  module_function :cp_lr

  # Creates {symbolic links}[https://en.wikipedia.org/wiki/Symbolic_link].
  #
  # Arguments +src+ (a single path or an array of paths)
  # and +dest+ (a single path)
  # should be {interpretable as paths}[rdoc-ref:FileUtils@Path+Arguments].
  #
  # If +src+ is the path to an existing file:
  #
  # - When +dest+ is the path to a non-existent file,
  #   creates a symbolic link at +dest+ pointing to +src+:
  #
  #     FileUtils.touch('src0.txt')
  #     File.exist?('dest0.txt')   # => false
  #     FileUtils.ln_s('src0.txt', 'dest0.txt')
  #     File.symlink?('dest0.txt') # => true
  #
  # - When +dest+ is the path to an existing file,
  #   creates a symbolic link at +dest+ pointing to +src+
  #   if and only if keyword argument <tt>force: true</tt> is given
  #   (raises an exception otherwise):
  #
  #     FileUtils.touch('src1.txt')
  #     FileUtils.touch('dest1.txt')
  #     FileUtils.ln_s('src1.txt', 'dest1.txt', force: true)
  #     FileTest.symlink?('dest1.txt') # => true
  #
  #     FileUtils.ln_s('src1.txt', 'dest1.txt') # Raises Errno::EEXIST.
  #
  # If +dest+ is the path to a directory,
  # creates a symbolic link at <tt>dest/src</tt> pointing to +src+:
  #
  #   FileUtils.touch('src2.txt')
  #   FileUtils.mkdir('destdir2')
  #   FileUtils.ln_s('src2.txt', 'destdir2')
  #   File.symlink?('destdir2/src2.txt') # => true
  #
  # If +src+ is an array of paths to existing files and +dest+ is a directory,
  # for each child +child+ in +src+ creates a symbolic link <tt>dest/child</tt>
  # pointing to +child+:
  #
  #   FileUtils.mkdir('srcdir3')
  #   FileUtils.touch('srcdir3/src0.txt')
  #   FileUtils.touch('srcdir3/src1.txt')
  #   FileUtils.mkdir('destdir3')
  #   FileUtils.ln_s(['srcdir3/src0.txt', 'srcdir3/src1.txt'], 'destdir3')
  #   File.symlink?('destdir3/src0.txt') # => true
  #   File.symlink?('destdir3/src1.txt') # => true
  #
  # Keyword arguments:
  #
  # - <tt>force: true</tt> - overwrites +dest+ if it exists.
  # - <tt>relative: false</tt> - create links relative to +dest+.
  # - <tt>noop: true</tt> - does not create links.
  # - <tt>verbose: true</tt> - prints an equivalent command:
  #
  #     FileUtils.ln_s('src0.txt', 'dest0.txt', noop: true, verbose: true)
  #     FileUtils.ln_s('src1.txt', 'destdir1', noop: true, verbose: true)
  #     FileUtils.ln_s('src2.txt', 'dest2.txt', force: true, noop: true, verbose: true)
  #     FileUtils.ln_s(['srcdir3/src0.txt', 'srcdir3/src1.txt'], 'destdir3', noop: true, verbose: true)
  #
  #   Output:
  #
  #     ln -s src0.txt dest0.txt
  #     ln -s src1.txt destdir1
  #     ln -sf src2.txt dest2.txt
  #     ln -s srcdir3/src0.txt srcdir3/src1.txt destdir3
  #
  # Related: FileUtils.ln_sf.
  #
  def ln_s(src, dest, force: nil, relative: false, target_directory: true, noop: nil, verbose: nil)
    if relative
      return ln_sr(src, dest, force: force, noop: noop, verbose: verbose)
    end
    fu_output_message "ln -s#{force ? 'f' : ''} #{[src,dest].flatten.join ' '}" if verbose
    return if noop
    fu_each_src_dest0(src, dest) do |s,d|
      remove_file d, true if force
      File.symlink s, d
    end
  end
  module_function :ln_s

  alias symlink ln_s
  module_function :symlink

  # Like FileUtils.ln_s, but always with keyword argument <tt>force: true</tt> given.
  #
  def ln_sf(src, dest, noop: nil, verbose: nil)
    ln_s src, dest, force: true, noop: noop, verbose: verbose
  end
  module_function :ln_sf

  # Like FileUtils.ln_s, but create links relative to +dest+.
  #
  def ln_sr(src, dest, target_directory: true, force: nil, noop: nil, verbose: nil)
    options = "#{force ? 'f' : ''}#{target_directory ? '' : 'T'}"
    dest = File.path(dest)
    srcs = Array(src)
    link = proc do |s, target_dir_p = true|
      s = File.path(s)
      if target_dir_p
        d = File.join(destdirs = dest, File.basename(s))
      else
        destdirs = File.dirname(d = dest)
      end
      destdirs = fu_split_path(File.realpath(destdirs))
      if fu_starting_path?(s)
        srcdirs = fu_split_path((File.realdirpath(s) rescue File.expand_path(s)))
        base = fu_relative_components_from(srcdirs, destdirs)
        s = File.join(*base)
      else
        srcdirs = fu_clean_components(*fu_split_path(s))
        base = fu_relative_components_from(fu_split_path(Dir.pwd), destdirs)
        while srcdirs.first&. == ".." and base.last&.!=("..") and !fu_starting_path?(base.last)
          srcdirs.shift
          base.pop
        end
        s = File.join(*base, *srcdirs)
      end
      fu_output_message "ln -s#{options} #{s} #{d}" if verbose
      next if noop
      remove_file d, true if force
      File.symlink s, d
    end
    case srcs.size
    when 0
    when 1
      link[srcs[0], target_directory && File.directory?(dest)]
    else
      srcs.each(&link)
    end
  end
  module_function :ln_sr

  # Creates {hard links}[https://en.wikipedia.org/wiki/Hard_link]; returns +nil+.
  #
  # Arguments +src+ and +dest+
  # should be {interpretable as paths}[rdoc-ref:FileUtils@Path+Arguments].
  #
  # If +src+ is the path to a file and +dest+ does not exist,
  # creates a hard link at +dest+ pointing to +src+:
  #
  #   FileUtils.touch('src0.txt')
  #   File.exist?('dest0.txt') # => false
  #   FileUtils.link_entry('src0.txt', 'dest0.txt')
  #   File.file?('dest0.txt')  # => true
  #
  # If +src+ is the path to a directory and +dest+ does not exist,
  # recursively creates hard links at +dest+ pointing to paths in +src+:
  #
  #   FileUtils.mkdir_p(['src1/dir0', 'src1/dir1'])
  #   src_file_paths = [
  #     'src1/dir0/t0.txt',
  #     'src1/dir0/t1.txt',
  #     'src1/dir1/t2.txt',
  #     'src1/dir1/t3.txt',
  #     ]
  #   FileUtils.touch(src_file_paths)
  #   File.directory?('dest1')        # => true
  #   FileUtils.link_entry('src1', 'dest1')
  #   File.file?('dest1/dir0/t0.txt') # => true
  #   File.file?('dest1/dir0/t1.txt') # => true
  #   File.file?('dest1/dir1/t2.txt') # => true
  #   File.file?('dest1/dir1/t3.txt') # => true
  #
  # Keyword arguments:
  #
  # - <tt>dereference_root: true</tt> - dereferences +src+ if it is a symbolic link.
  # - <tt>remove_destination: true</tt> - removes +dest+ before creating links.
  #
  # Raises an exception if +dest+ is the path to an existing file or directory
  # and keyword argument <tt>remove_destination: true</tt> is not given.
  #
  # Related: FileUtils.ln (has different options).
  #
  def link_entry(src, dest, dereference_root = false, remove_destination = false)
    Entry_.new(src, nil, dereference_root).traverse do |ent|
      destent = Entry_.new(dest, ent.rel, false)
      File.unlink destent.path if remove_destination && File.file?(destent.path)
      ent.link destent.path
    end
  end
  module_function :link_entry

  # Copies files.
  #
  # Arguments +src+ (a single path or an array of paths)
  # and +dest+ (a single path)
  # should be {interpretable as paths}[rdoc-ref:FileUtils@Path+Arguments].
  #
  # If +src+ is the path to a file and +dest+ is not the path to a directory,
  # copies +src+ to +dest+:
  #
  #   FileUtils.touch('src0.txt')
  #   File.exist?('dest0.txt') # => false
  #   FileUtils.cp('src0.txt', 'dest0.txt')
  #   File.file?('dest0.txt')  # => true
  #
  # If +src+ is the path to a file and +dest+ is the path to a directory,
  # copies +src+ to <tt>dest/src</tt>:
  #
  #   FileUtils.touch('src1.txt')
  #   FileUtils.mkdir('dest1')
  #   FileUtils.cp('src1.txt', 'dest1')
  #   File.file?('dest1/src1.txt') # => true
  #
  # If +src+ is an array of paths to files and +dest+ is the path to a directory,
  # copies from each +src+ to +dest+:
  #
  #   src_file_paths = ['src2.txt', 'src2.dat']
  #   FileUtils.touch(src_file_paths)
  #   FileUtils.mkdir('dest2')
  #   FileUtils.cp(src_file_paths, 'dest2')
  #   File.file?('dest2/src2.txt') # => true
  #   File.file?('dest2/src2.dat') # => true
  #
  # Keyword arguments:
  #
  # - <tt>preserve: true</tt> - preserves file times.
  # - <tt>noop: true</tt> - does not copy files.
  # - <tt>verbose: true</tt> - prints an equivalent command:
  #
  #     FileUtils.cp('src0.txt', 'dest0.txt', noop: true, verbose: true)
  #     FileUtils.cp('src1.txt', 'dest1', noop: true, verbose: true)
  #     FileUtils.cp(src_file_paths, 'dest2', noop: true, verbose: true)
  #
  #   Output:
  #
  #     cp src0.txt dest0.txt
  #     cp src1.txt dest1
  #     cp src2.txt src2.dat dest2
  #
  # Raises an exception if +src+ is a directory.
  #
  # Related: {methods for copying}[rdoc-ref:FileUtils@Copying].
  #
  def cp(src, dest, preserve: nil, noop: nil, verbose: nil)
    fu_output_message "cp#{preserve ? ' -p' : ''} #{[src,dest].flatten.join ' '}" if verbose
    return if noop
    fu_each_src_dest(src, dest) do |s, d|
      copy_file s, d, preserve
    end
  end
  module_function :cp

  alias copy cp
  module_function :copy

  # Recursively copies files.
  #
  # Arguments +src+ (a single path or an array of paths)
  # and +dest+ (a single path)
  # should be {interpretable as paths}[rdoc-ref:FileUtils@Path+Arguments].
  #
  # The mode, owner, and group are retained in the copy;
  # to change those, use FileUtils.install instead.
  #
  # If +src+ is the path to a file and +dest+ is not the path to a directory,
  # copies +src+ to +dest+:
  #
  #   FileUtils.touch('src0.txt')
  #   File.exist?('dest0.txt') # => false
  #   FileUtils.cp_r('src0.txt', 'dest0.txt')
  #   File.file?('dest0.txt')  # => true
  #
  # If +src+ is the path to a file and +dest+ is the path to a directory,
  # copies +src+ to <tt>dest/src</tt>:
  #
  #   FileUtils.touch('src1.txt')
  #   FileUtils.mkdir('dest1')
  #   FileUtils.cp_r('src1.txt', 'dest1')
  #   File.file?('dest1/src1.txt') # => true
  #
  # If +src+ is the path to a directory and +dest+ does not exist,
  # recursively copies +src+ to +dest+:
  #
  #   tree('src2')
  #   # => src2
  #   #    |-- dir0
  #   #    |   |-- src0.txt
  #   #    |   `-- src1.txt
  #   #    `-- dir1
  #   #    |-- src2.txt
  #   #    `-- src3.txt
  #   FileUtils.exist?('dest2') # => false
  #   FileUtils.cp_r('src2', 'dest2')
  #   tree('dest2')
  #   # => dest2
  #   #    |-- dir0
  #   #    |   |-- src0.txt
  #   #    |   `-- src1.txt
  #   #    `-- dir1
  #   #    |-- src2.txt
  #   #    `-- src3.txt
  #
  # If +src+ and +dest+ are paths to directories,
  # recursively copies +src+ to <tt>dest/src</tt>:
  #
  #   tree('src3')
  #   # => src3
  #   #    |-- dir0
  #   #    |   |-- src0.txt
  #   #    |   `-- src1.txt
  #   #    `-- dir1
  #   #    |-- src2.txt
  #   #    `-- src3.txt
  #   FileUtils.mkdir('dest3')
  #   FileUtils.cp_r('src3', 'dest3')
  #   tree('dest3')
  #   # => dest3
  #   #    `-- src3
  #   #      |-- dir0
  #   #      |   |-- src0.txt
  #   #      |   `-- src1.txt
  #   #      `-- dir1
  #   #          |-- src2.txt
  #   #          `-- src3.txt
  #
  # If +src+ is an array of paths and +dest+ is a directory,
  # recursively copies from each path in +src+ to +dest+;
  # the paths in +src+ may point to files and/or directories.
  #
  # Keyword arguments:
  #
  # - <tt>dereference_root: false</tt> - if +src+ is a symbolic link,
  #   does not dereference it.
  # - <tt>noop: true</tt> - does not copy files.
  # - <tt>preserve: true</tt> - preserves file times.
  # - <tt>remove_destination: true</tt> - removes +dest+ before copying files.
  # - <tt>verbose: true</tt> - prints an equivalent command:
  #
  #     FileUtils.cp_r('src0.txt', 'dest0.txt', noop: true, verbose: true)
  #     FileUtils.cp_r('src1.txt', 'dest1', noop: true, verbose: true)
  #     FileUtils.cp_r('src2', 'dest2', noop: true, verbose: true)
  #     FileUtils.cp_r('src3', 'dest3', noop: true, verbose: true)
  #
  #   Output:
  #
  #     cp -r src0.txt dest0.txt
  #     cp -r src1.txt dest1
  #     cp -r src2 dest2
  #     cp -r src3 dest3
  #
  # Raises an exception of +src+ is the path to a directory
  # and +dest+ is the path to a file.
  #
  # Related: {methods for copying}[rdoc-ref:FileUtils@Copying].
  #
  def cp_r(src, dest, preserve: nil, noop: nil, verbose: nil,
           dereference_root: true, remove_destination: nil)
    fu_output_message "cp -r#{preserve ? 'p' : ''}#{remove_destination ? ' --remove-destination' : ''} #{[src,dest].flatten.join ' '}" if verbose
    return if noop
    fu_each_src_dest(src, dest) do |s, d|
      copy_entry s, d, preserve, dereference_root, remove_destination
    end
  end
  module_function :cp_r

  # Recursively copies files from +src+ to +dest+.
  #
  # Arguments +src+ and +dest+
  # should be {interpretable as paths}[rdoc-ref:FileUtils@Path+Arguments].
  #
  # If +src+ is the path to a file, copies +src+ to +dest+:
  #
  #   FileUtils.touch('src0.txt')
  #   File.exist?('dest0.txt') # => false
  #   FileUtils.copy_entry('src0.txt', 'dest0.txt')
  #   File.file?('dest0.txt')  # => true
  #
  # If +src+ is a directory, recursively copies +src+ to +dest+:
  #
  #   tree('src1')
  #   # => src1
  #   #    |-- dir0
  #   #    |   |-- src0.txt
  #   #    |   `-- src1.txt
  #   #    `-- dir1
  #   #        |-- src2.txt
  #   #        `-- src3.txt
  #   FileUtils.copy_entry('src1', 'dest1')
  #   tree('dest1')
  #   # => dest1
  #   #    |-- dir0
  #   #    |   |-- src0.txt
  #   #    |   `-- src1.txt
  #   #    `-- dir1
  #   #        |-- src2.txt
  #   #        `-- src3.txt
  #
  # The recursive copying preserves file types for regular files,
  # directories, and symbolic links;
  # other file types (FIFO streams, device files, etc.) are not supported.
  #
  # Keyword arguments:
  #
  # - <tt>dereference_root: true</tt> - if +src+ is a symbolic link,
  #   follows the link.
  # - <tt>preserve: true</tt> - preserves file times.
  # - <tt>remove_destination: true</tt> - removes +dest+ before copying files.
  #
  # Related: {methods for copying}[rdoc-ref:FileUtils@Copying].
  #
  def copy_entry(src, dest, preserve = false, dereference_root = false, remove_destination = false)
    if dereference_root
      src = File.realpath(src)
    end

    Entry_.new(src, nil, false).wrap_traverse(proc do |ent|
      destent = Entry_.new(dest, ent.rel, false)
      File.unlink destent.path if remove_destination && (File.file?(destent.path) || File.symlink?(destent.path))
      ent.copy destent.path
    end, proc do |ent|
      destent = Entry_.new(dest, ent.rel, false)
      ent.copy_metadata destent.path if preserve
    end)
  end
  module_function :copy_entry

  # Copies file from +src+ to +dest+, which should not be directories.
  #
  # Arguments +src+ and +dest+
  # should be {interpretable as paths}[rdoc-ref:FileUtils@Path+Arguments].
  #
  # Examples:
  #
  #   FileUtils.touch('src0.txt')
  #   FileUtils.copy_file('src0.txt', 'dest0.txt')
  #   File.file?('dest0.txt') # => true
  #
  # Keyword arguments:
  #
  # - <tt>dereference: false</tt> - if +src+ is a symbolic link,
  #   does not follow the link.
  # - <tt>preserve: true</tt> - preserves file times.
  # - <tt>remove_destination: true</tt> - removes +dest+ before copying files.
  #
  # Related: {methods for copying}[rdoc-ref:FileUtils@Copying].
  #
  def copy_file(src, dest, preserve = false, dereference = true)
    ent = Entry_.new(src, nil, dereference)
    ent.copy_file dest
    ent.copy_metadata dest if preserve
  end
  module_function :copy_file

  # Copies \IO stream +src+ to \IO stream +dest+ via
  # {IO.copy_stream}[rdoc-ref:IO.copy_stream].
  #
  # Related: {methods for copying}[rdoc-ref:FileUtils@Copying].
  #
  def copy_stream(src, dest)
    IO.copy_stream(src, dest)
  end
  module_function :copy_stream

  # Moves entries.
  #
  # Arguments +src+ (a single path or an array of paths)
  # and +dest+ (a single path)
  # should be {interpretable as paths}[rdoc-ref:FileUtils@Path+Arguments].
  #
  # If +src+ and +dest+ are on different file systems,
  # first copies, then removes +src+.
  #
  # May cause a local vulnerability if not called with keyword argument
  # <tt>secure: true</tt>;
  # see {Avoiding the TOCTTOU Vulnerability}[rdoc-ref:FileUtils@Avoiding+the+TOCTTOU+Vulnerability].
  #
  # If +src+ is the path to a single file or directory and +dest+ does not exist,
  # moves +src+ to +dest+:
  #
  #   tree('src0')
  #   # => src0
  #   #    |-- src0.txt
  #   #    `-- src1.txt
  #   File.exist?('dest0') # => false
  #   FileUtils.mv('src0', 'dest0')
  #   File.exist?('src0')  # => false
  #   tree('dest0')
  #   # => dest0
  #   #    |-- src0.txt
  #   #    `-- src1.txt
  #
  # If +src+ is an array of paths to files and directories
  # and +dest+ is the path to a directory,
  # copies from each path in the array to +dest+:
  #
  #   File.file?('src1.txt') # => true
  #   tree('src1')
  #   # => src1
  #   #    |-- src.dat
  #   #    `-- src.txt
  #   Dir.empty?('dest1')    # => true
  #   FileUtils.mv(['src1.txt', 'src1'], 'dest1')
  #   tree('dest1')
  #   # => dest1
  #   #    |-- src1
  #   #    |   |-- src.dat
  #   #    |   `-- src.txt
  #   #    `-- src1.txt
  #
  # Keyword arguments:
  #
  # - <tt>force: true</tt> - if the move includes removing +src+
  #   (that is, if +src+ and +dest+ are on different file systems),
  #   ignores raised exceptions of StandardError and its descendants.
  # - <tt>noop: true</tt> - does not move files.
  # - <tt>secure: true</tt> - removes +src+ securely;
  #   see details at FileUtils.remove_entry_secure.
  # - <tt>verbose: true</tt> - prints an equivalent command:
  #
  #     FileUtils.mv('src0', 'dest0', noop: true, verbose: true)
  #     FileUtils.mv(['src1.txt', 'src1'], 'dest1', noop: true, verbose: true)
  #
  #   Output:
  #
  #     mv src0 dest0
  #     mv src1.txt src1 dest1
  #
  def mv(src, dest, force: nil, noop: nil, verbose: nil, secure: nil)
    fu_output_message "mv#{force ? ' -f' : ''} #{[src,dest].flatten.join ' '}" if verbose
    return if noop
    fu_each_src_dest(src, dest) do |s, d|
      destent = Entry_.new(d, nil, true)
      begin
        if destent.exist?
          if destent.directory?
            raise Errno::EEXIST, d
          end
        end
        begin
          File.rename s, d
        rescue Errno::EXDEV,
               Errno::EPERM # move from unencrypted to encrypted dir (ext4)
          copy_entry s, d, true
          if secure
            remove_entry_secure s, force
          else
            remove_entry s, force
          end
        end
      rescue SystemCallError
        raise unless force
      end
    end
  end
  module_function :mv

  alias move mv
  module_function :move

  # Removes entries at the paths in the given +list+
  # (a single path or an array of paths)
  # returns +list+, if it is an array, <tt>[list]</tt> otherwise.
  #
  # Argument +list+ or its elements
  # should be {interpretable as paths}[rdoc-ref:FileUtils@Path+Arguments].
  #
  # With no keyword arguments, removes files at the paths given in +list+:
  #
  #   FileUtils.touch(['src0.txt', 'src0.dat'])
  #   FileUtils.rm(['src0.dat', 'src0.txt']) # => ["src0.dat", "src0.txt"]
  #
  # Keyword arguments:
  #
  # - <tt>force: true</tt> - ignores raised exceptions of StandardError
  #   and its descendants.
  # - <tt>noop: true</tt> - does not remove files; returns +nil+.
  # - <tt>verbose: true</tt> - prints an equivalent command:
  #
  #     FileUtils.rm(['src0.dat', 'src0.txt'], noop: true, verbose: true)
  #
  #   Output:
  #
  #     rm src0.dat src0.txt
  #
  # Related: {methods for deleting}[rdoc-ref:FileUtils@Deleting].
  #
  def rm(list, force: nil, noop: nil, verbose: nil)
    list = fu_list(list)
    fu_output_message "rm#{force ? ' -f' : ''} #{list.join ' '}" if verbose
    return if noop

    list.each do |path|
      remove_file path, force
    end
  end
  module_function :rm

  alias remove rm
  module_function :remove

  # Equivalent to:
  #
  #   FileUtils.rm(list, force: true, **kwargs)
  #
  # Argument +list+ (a single path or an array of paths)
  # should be {interpretable as paths}[rdoc-ref:FileUtils@Path+Arguments].
  #
  # See FileUtils.rm for keyword arguments.
  #
  # Related: {methods for deleting}[rdoc-ref:FileUtils@Deleting].
  #
  def rm_f(list, noop: nil, verbose: nil)
    rm list, force: true, noop: noop, verbose: verbose
  end
  module_function :rm_f

  alias safe_unlink rm_f
  module_function :safe_unlink

  # Removes entries at the paths in the given +list+
  # (a single path or an array of paths);
  # returns +list+, if it is an array, <tt>[list]</tt> otherwise.
  #
  # Argument +list+ or its elements
  # should be {interpretable as paths}[rdoc-ref:FileUtils@Path+Arguments].
  #
  # May cause a local vulnerability if not called with keyword argument
  # <tt>secure: true</tt>;
  # see {Avoiding the TOCTTOU Vulnerability}[rdoc-ref:FileUtils@Avoiding+the+TOCTTOU+Vulnerability].
  #
  # For each file path, removes the file at that path:
  #
  #   FileUtils.touch(['src0.txt', 'src0.dat'])
  #   FileUtils.rm_r(['src0.dat', 'src0.txt'])
  #   File.exist?('src0.txt') # => false
  #   File.exist?('src0.dat') # => false
  #
  # For each directory path, recursively removes files and directories:
  #
  #   tree('src1')
  #   # => src1
  #   #    |-- dir0
  #   #    |   |-- src0.txt
  #   #    |   `-- src1.txt
  #   #    `-- dir1
  #   #        |-- src2.txt
  #   #        `-- src3.txt
  #   FileUtils.rm_r('src1')
  #   File.exist?('src1') # => false
  #
  # Keyword arguments:
  #
  # - <tt>force: true</tt> - ignores raised exceptions of StandardError
  #   and its descendants.
  # - <tt>noop: true</tt> - does not remove entries; returns +nil+.
  # - <tt>secure: true</tt> - removes +src+ securely;
  #   see details at FileUtils.remove_entry_secure.
  # - <tt>verbose: true</tt> - prints an equivalent command:
  #
  #     FileUtils.rm_r(['src0.dat', 'src0.txt'], noop: true, verbose: true)
  #     FileUtils.rm_r('src1', noop: true, verbose: true)
  #
  #   Output:
  #
  #     rm -r src0.dat src0.txt
  #     rm -r src1
  #
  # Related: {methods for deleting}[rdoc-ref:FileUtils@Deleting].
  #
  def rm_r(list, force: nil, noop: nil, verbose: nil, secure: nil)
    list = fu_list(list)
    fu_output_message "rm -r#{force ? 'f' : ''} #{list.join ' '}" if verbose
    return if noop
    list.each do |path|
      if secure
        remove_entry_secure path, force
      else
        remove_entry path, force
      end
    end
  end
  module_function :rm_r

  # Equivalent to:
  #
  #   FileUtils.rm_r(list, force: true, **kwargs)
  #
  # Argument +list+ or its elements
  # should be {interpretable as paths}[rdoc-ref:FileUtils@Path+Arguments].
  #
  # May cause a local vulnerability if not called with keyword argument
  # <tt>secure: true</tt>;
  # see {Avoiding the TOCTTOU Vulnerability}[rdoc-ref:FileUtils@Avoiding+the+TOCTTOU+Vulnerability].
  #
  # See FileUtils.rm_r for keyword arguments.
  #
  # Related: {methods for deleting}[rdoc-ref:FileUtils@Deleting].
  #
  def rm_rf(list, noop: nil, verbose: nil, secure: nil)
    rm_r list, force: true, noop: noop, verbose: verbose, secure: secure
  end
  module_function :rm_rf

  alias rmtree rm_rf
  module_function :rmtree

  # Securely removes the entry given by +path+,
  # which should be the entry for a regular file, a symbolic link,
  # or a directory.
  #
  # Argument +path+
  # should be {interpretable as a path}[rdoc-ref:FileUtils@Path+Arguments].
  #
  # Avoids a local vulnerability that can exist in certain circumstances;
  # see {Avoiding the TOCTTOU Vulnerability}[rdoc-ref:FileUtils@Avoiding+the+TOCTTOU+Vulnerability].
  #
  # Optional argument +force+ specifies whether to ignore
  # raised exceptions of StandardError and its descendants.
  #
  # Related: {methods for deleting}[rdoc-ref:FileUtils@Deleting].
  #
  def remove_entry_secure(path, force = false)
    unless fu_have_symlink?
      remove_entry path, force
      return
    end
    fullpath = File.expand_path(path)
    st = File.lstat(fullpath)
    unless st.directory?
      File.unlink fullpath
      return
    end
    # is a directory.
    parent_st = File.stat(File.dirname(fullpath))
    unless parent_st.world_writable?
      remove_entry path, force
      return
    end
    unless parent_st.sticky?
      raise ArgumentError, "parent directory is world writable, FileUtils#remove_entry_secure does not work; abort: #{path.inspect} (parent directory mode #{'%o' % parent_st.mode})"
    end

    # freeze tree root
    euid = Process.euid
    dot_file = fullpath + "/."
    begin
      File.open(dot_file) {|f|
        unless fu_stat_identical_entry?(st, f.stat)
          # symlink (TOC-to-TOU attack?)
          File.unlink fullpath
          return
        end
        f.chown euid, -1
        f.chmod 0700
      }
    rescue Errno::EISDIR # JRuby in non-native mode can't open files as dirs
      File.lstat(dot_file).tap {|fstat|
        unless fu_stat_identical_entry?(st, fstat)
          # symlink (TOC-to-TOU attack?)
          File.unlink fullpath
          return
        end
        File.chown euid, -1, dot_file
        File.chmod 0700, dot_file
      }
    end

    unless fu_stat_identical_entry?(st, File.lstat(fullpath))
      # TOC-to-TOU attack?
      File.unlink fullpath
      return
    end

    # ---- tree root is frozen ----
    root = Entry_.new(path)
    root.preorder_traverse do |ent|
      if ent.directory?
        ent.chown euid, -1
        ent.chmod 0700
      end
    end
    root.postorder_traverse do |ent|
      begin
        ent.remove
      rescue
        raise unless force
      end
    end
  rescue
    raise unless force
  end
  module_function :remove_entry_secure

  def fu_have_symlink?   #:nodoc:
    File.symlink nil, nil
  rescue NotImplementedError
    return false
  rescue TypeError
    return true
  end
  private_module_function :fu_have_symlink?

  def fu_stat_identical_entry?(a, b)   #:nodoc:
    a.dev == b.dev and a.ino == b.ino
  end
  private_module_function :fu_stat_identical_entry?

  # Removes the entry given by +path+,
  # which should be the entry for a regular file, a symbolic link,
  # or a directory.
  #
  # Argument +path+
  # should be {interpretable as a path}[rdoc-ref:FileUtils@Path+Arguments].
  #
  # Optional argument +force+ specifies whether to ignore
  # raised exceptions of StandardError and its descendants.
  #
  # Related: FileUtils.remove_entry_secure.
  #
  def remove_entry(path, force = false)
    Entry_.new(path).postorder_traverse do |ent|
      begin
        ent.remove
      rescue
        raise unless force
      end
    end
  rescue
    raise unless force
  end
  module_function :remove_entry

  # Removes the file entry given by +path+,
  # which should be the entry for a regular file or a symbolic link.
  #
  # Argument +path+
  # should be {interpretable as a path}[rdoc-ref:FileUtils@Path+Arguments].
  #
  # Optional argument +force+ specifies whether to ignore
  # raised exceptions of StandardError and its descendants.
  #
  # Related: {methods for deleting}[rdoc-ref:FileUtils@Deleting].
  #
  def remove_file(path, force = false)
    Entry_.new(path).remove_file
  rescue
    raise unless force
  end
  module_function :remove_file

  # Recursively removes the directory entry given by +path+,
  # which should be the entry for a regular file, a symbolic link,
  # or a directory.
  #
  # Argument +path+
  # should be {interpretable as a path}[rdoc-ref:FileUtils@Path+Arguments].
  #
  # Optional argument +force+ specifies whether to ignore
  # raised exceptions of StandardError and its descendants.
  #
  # Related: {methods for deleting}[rdoc-ref:FileUtils@Deleting].
  #
  def remove_dir(path, force = false)
    remove_entry path, force   # FIXME?? check if it is a directory
  end
  module_function :remove_dir

  # Returns +true+ if the contents of files +a+ and +b+ are identical,
  # +false+ otherwise.
  #
  # Arguments +a+ and +b+
  # should be {interpretable as a path}[rdoc-ref:FileUtils@Path+Arguments].
  #
  # FileUtils.identical? and FileUtils.cmp are aliases for FileUtils.compare_file.
  #
  # Related: FileUtils.compare_stream.
  #
  def compare_file(a, b)
    return false unless File.size(a) == File.size(b)
    File.open(a, 'rb') {|fa|
      File.open(b, 'rb') {|fb|
        return compare_stream(fa, fb)
      }
    }
  end
  module_function :compare_file

  alias identical? compare_file
  alias cmp compare_file
  module_function :identical?
  module_function :cmp

  # Returns +true+ if the contents of streams +a+ and +b+ are identical,
  # +false+ otherwise.
  #
  # Arguments +a+ and +b+
  # should be {interpretable as a path}[rdoc-ref:FileUtils@Path+Arguments].
  #
  # Related: FileUtils.compare_file.
  #
  def compare_stream(a, b)
    bsize = fu_stream_blksize(a, b)

    sa = String.new(capacity: bsize)
    sb = String.new(capacity: bsize)

    begin
      a.read(bsize, sa)
      b.read(bsize, sb)
      return true if sa.empty? && sb.empty?
    end while sa == sb
    false
  end
  module_function :compare_stream

  # Copies a file entry.
  # See {install(1)}[https://man7.org/linux/man-pages/man1/install.1.html].
  #
  # Arguments +src+ (a single path or an array of paths)
  # and +dest+ (a single path)
  # should be {interpretable as paths}[rdoc-ref:FileUtils@Path+Arguments];
  #
  # If the entry at +dest+ does not exist, copies from +src+ to +dest+:
  #
  #   File.read('src0.txt')    # => "aaa\n"
  #   File.exist?('dest0.txt') # => false
  #   FileUtils.install('src0.txt', 'dest0.txt')
  #   File.read('dest0.txt')   # => "aaa\n"
  #
  # If +dest+ is a file entry, copies from +src+ to +dest+, overwriting:
  #
  #   File.read('src1.txt')  # => "aaa\n"
  #   File.read('dest1.txt') # => "bbb\n"
  #   FileUtils.install('src1.txt', 'dest1.txt')
  #   File.read('dest1.txt') # => "aaa\n"
  #
  # If +dest+ is a directory entry, copies from +src+ to <tt>dest/src</tt>,
  # overwriting if necessary:
  #
  #   File.read('src2.txt')       # => "aaa\n"
  #   File.read('dest2/src2.txt') # => "bbb\n"
  #   FileUtils.install('src2.txt', 'dest2')
  #   File.read('dest2/src2.txt') # => "aaa\n"
  #
  # If +src+ is an array of paths and +dest+ points to a directory,
  # copies each path +path+ in +src+ to <tt>dest/path</tt>:
  #
  #   File.file?('src3.txt') # => true
  #   File.file?('src3.dat') # => true
  #   FileUtils.mkdir('dest3')
  #   FileUtils.install(['src3.txt', 'src3.dat'], 'dest3')
  #   File.file?('dest3/src3.txt') # => true
  #   File.file?('dest3/src3.dat') # => true
  #
  # Keyword arguments:
  #
  # - <tt>group: <i>group</i></tt> - changes the group if not +nil+,
  #   using {File.chown}[rdoc-ref:File.chown].
  # - <tt>mode: <i>permissions</i></tt> - changes the permissions.
  #   using {File.chmod}[rdoc-ref:File.chmod].
  # - <tt>noop: true</tt> - does not copy entries; returns +nil+.
  # - <tt>owner: <i>owner</i></tt> - changes the owner if not +nil+,
  #   using {File.chown}[rdoc-ref:File.chown].
  # - <tt>preserve: true</tt> - preserve timestamps
  #   using {File.utime}[rdoc-ref:File.utime].
  # - <tt>verbose: true</tt> - prints an equivalent command:
  #
  #     FileUtils.install('src0.txt', 'dest0.txt', noop: true, verbose: true)
  #     FileUtils.install('src1.txt', 'dest1.txt', noop: true, verbose: true)
  #     FileUtils.install('src2.txt', 'dest2', noop: true, verbose: true)
  #
  #   Output:
  #
  #     install -c src0.txt dest0.txt
  #     install -c src1.txt dest1.txt
  #     install -c src2.txt dest2
  #
  # Related: {methods for copying}[rdoc-ref:FileUtils@Copying].
  #
  def install(src, dest, mode: nil, owner: nil, group: nil, preserve: nil,
              noop: nil, verbose: nil)
    if verbose
      msg = +"install -c"
      msg << ' -p' if preserve
      msg << ' -m ' << mode_to_s(mode) if mode
      msg << " -o #{owner}" if owner
      msg << " -g #{group}" if group
      msg << ' ' << [src,dest].flatten.join(' ')
      fu_output_message msg
    end
    return if noop
    uid = fu_get_uid(owner)
    gid = fu_get_gid(group)
    fu_each_src_dest(src, dest) do |s, d|
      st = File.stat(s)
      unless File.exist?(d) and compare_file(s, d)
        remove_file d, true
        if d.end_with?('/')
          mkdir_p d
          copy_file s, d + File.basename(s)
        else
          mkdir_p File.expand_path('..', d)
          copy_file s, d
        end
        File.utime st.atime, st.mtime, d if preserve
        File.chmod fu_mode(mode, st), d if mode
        File.chown uid, gid, d if uid or gid
      end
    end
  end
  module_function :install

  def user_mask(target)  #:nodoc:
    target.each_char.inject(0) do |mask, chr|
      case chr
      when "u"
        mask | 04700
      when "g"
        mask | 02070
      when "o"
        mask | 01007
      when "a"
        mask | 07777
      else
        raise ArgumentError, "invalid 'who' symbol in file mode: #{chr}"
      end
    end
  end
  private_module_function :user_mask

  def apply_mask(mode, user_mask, op, mode_mask)   #:nodoc:
    case op
    when '='
      (mode & ~user_mask) | (user_mask & mode_mask)
    when '+'
      mode | (user_mask & mode_mask)
    when '-'
      mode & ~(user_mask & mode_mask)
    end
  end
  private_module_function :apply_mask

  def symbolic_modes_to_i(mode_sym, path)  #:nodoc:
    path = File.stat(path) unless File::Stat === path
    mode = path.mode
    mode_sym.split(/,/).inject(mode & 07777) do |current_mode, clause|
      target, *actions = clause.split(/([=+-])/)
      raise ArgumentError, "invalid file mode: #{mode_sym}" if actions.empty?
      target = 'a' if target.empty?
      user_mask = user_mask(target)
      actions.each_slice(2) do |op, perm|
        need_apply = op == '='
        mode_mask = (perm || '').each_char.inject(0) do |mask, chr|
          case chr
          when "r"
            mask | 0444
          when "w"
            mask | 0222
          when "x"
            mask | 0111
          when "X"
            if path.directory?
              mask | 0111
            else
              mask
            end
          when "s"
            mask | 06000
          when "t"
            mask | 01000
          when "u", "g", "o"
            if mask.nonzero?
              current_mode = apply_mask(current_mode, user_mask, op, mask)
            end
            need_apply = false
            copy_mask = user_mask(chr)
            (current_mode & copy_mask) / (copy_mask & 0111) * (user_mask & 0111)
          else
            raise ArgumentError, "invalid 'perm' symbol in file mode: #{chr}"
          end
        end

        if mode_mask.nonzero? || need_apply
          current_mode = apply_mask(current_mode, user_mask, op, mode_mask)
        end
      end
      current_mode
    end
  end
  private_module_function :symbolic_modes_to_i

  def fu_mode(mode, path)  #:nodoc:
    mode.is_a?(String) ? symbolic_modes_to_i(mode, path) : mode
  end
  private_module_function :fu_mode

  def mode_to_s(mode)  #:nodoc:
    mode.is_a?(String) ? mode : "%o" % mode
  end
  private_module_function :mode_to_s

  # Changes permissions on the entries at the paths given in +list+
  # (a single path or an array of paths)
  # to the permissions given by +mode+;
  # returns +list+ if it is an array, <tt>[list]</tt> otherwise:
  #
  # - Modifies each entry that is a regular file using
  #   {File.chmod}[rdoc-ref:File.chmod].
  # - Modifies each entry that is a symbolic link using
  #   {File.lchmod}[rdoc-ref:File.lchmod].
  #
  # Argument +list+ or its elements
  # should be {interpretable as paths}[rdoc-ref:FileUtils@Path+Arguments].
  #
  # Argument +mode+ may be either an integer or a string:
  #
  # - \Integer +mode+: represents the permission bits to be set:
  #
  #     FileUtils.chmod(0755, 'src0.txt')
  #     FileUtils.chmod(0644, ['src0.txt', 'src0.dat'])
  #
  # - \String +mode+: represents the permissions to be set:
  #
  #   The string is of the form <tt>[targets][[operator][perms[,perms]]</tt>, where:
  #
  #   - +targets+ may be any combination of these letters:
  #
  #     - <tt>'u'</tt>: permissions apply to the file's owner.
  #     - <tt>'g'</tt>: permissions apply to users in the file's group.
  #     - <tt>'o'</tt>: permissions apply to other users not in the file's group.
  #     - <tt>'a'</tt> (the default): permissions apply to all users.
  #
  #   - +operator+ may be one of these letters:
  #
  #     - <tt>'+'</tt>: adds permissions.
  #     - <tt>'-'</tt>: removes permissions.
  #     - <tt>'='</tt>: sets (replaces) permissions.
  #
  #   - +perms+ (may be repeated, with separating commas)
  #     may be any combination of these letters:
  #
  #     - <tt>'r'</tt>: Read.
  #     - <tt>'w'</tt>: Write.
  #     - <tt>'x'</tt>: Execute (search, for a directory).
  #     - <tt>'X'</tt>: Search (for a directories only;
  #       must be used with <tt>'+'</tt>)
  #     - <tt>'s'</tt>: Uid or gid.
  #     - <tt>'t'</tt>: Sticky bit.
  #
  #   Examples:
  #
  #     FileUtils.chmod('u=wrx,go=rx', 'src1.txt')
  #     FileUtils.chmod('u=wrx,go=rx', '/usr/bin/ruby')
  #
  # Keyword arguments:
  #
  # - <tt>noop: true</tt> - does not change permissions; returns +nil+.
  # - <tt>verbose: true</tt> - prints an equivalent command:
  #
  #     FileUtils.chmod(0755, 'src0.txt', noop: true, verbose: true)
  #     FileUtils.chmod(0644, ['src0.txt', 'src0.dat'], noop: true, verbose: true)
  #     FileUtils.chmod('u=wrx,go=rx', 'src1.txt', noop: true, verbose: true)
  #     FileUtils.chmod('u=wrx,go=rx', '/usr/bin/ruby', noop: true, verbose: true)
  #
  #   Output:
  #
  #     chmod 755 src0.txt
  #     chmod 644 src0.txt src0.dat
  #     chmod u=wrx,go=rx src1.txt
  #     chmod u=wrx,go=rx /usr/bin/ruby
  #
  # Related: FileUtils.chmod_R.
  #
  def chmod(mode, list, noop: nil, verbose: nil)
    list = fu_list(list)
    fu_output_message sprintf('chmod %s %s', mode_to_s(mode), list.join(' ')) if verbose
    return if noop
    list.each do |path|
      Entry_.new(path).chmod(fu_mode(mode, path))
    end
  end
  module_function :chmod

  # Like FileUtils.chmod, but changes permissions recursively.
  #
  def chmod_R(mode, list, noop: nil, verbose: nil, force: nil)
    list = fu_list(list)
    fu_output_message sprintf('chmod -R%s %s %s',
                              (force ? 'f' : ''),
                              mode_to_s(mode), list.join(' ')) if verbose
    return if noop
    list.each do |root|
      Entry_.new(root).traverse do |ent|
        begin
          ent.chmod(fu_mode(mode, ent.path))
        rescue
          raise unless force
        end
      end
    end
  end
  module_function :chmod_R

  # Changes the owner and group on the entries at the paths given in +list+
  # (a single path or an array of paths)
  # to the given +user+ and +group+;
  # returns +list+ if it is an array, <tt>[list]</tt> otherwise:
  #
  # - Modifies each entry that is a regular file using
  #   {File.chown}[rdoc-ref:File.chown].
  # - Modifies each entry that is a symbolic link using
  #   {File.lchown}[rdoc-ref:File.lchown].
  #
  # Argument +list+ or its elements
  # should be {interpretable as paths}[rdoc-ref:FileUtils@Path+Arguments].
  #
  # User and group:
  #
  # - Argument +user+ may be a user name or a user id;
  #   if +nil+ or +-1+, the user is not changed.
  # - Argument +group+ may be a group name or a group id;
  #   if +nil+ or +-1+, the group is not changed.
  # - The user must be a member of the group.
  #
  # Examples:
  #
  #   # One path.
  #   # User and group as string names.
  #   File.stat('src0.txt').uid # => 1004
  #   File.stat('src0.txt').gid # => 1004
  #   FileUtils.chown('user2', 'group1', 'src0.txt')
  #   File.stat('src0.txt').uid # => 1006
  #   File.stat('src0.txt').gid # => 1005
  #
  #   # User and group as uid and gid.
  #   FileUtils.chown(1004, 1004, 'src0.txt')
  #   File.stat('src0.txt').uid # => 1004
  #   File.stat('src0.txt').gid # => 1004
  #
  #   # Array of paths.
  #   FileUtils.chown(1006, 1005, ['src0.txt', 'src0.dat'])
  #
  #   # Directory (not recursive).
  #   FileUtils.chown('user2', 'group1', '.')
  #
  # Keyword arguments:
  #
  # - <tt>noop: true</tt> - does not change permissions; returns +nil+.
  # - <tt>verbose: true</tt> - prints an equivalent command:
  #
  #     FileUtils.chown('user2', 'group1', 'src0.txt', noop: true, verbose: true)
  #     FileUtils.chown(1004, 1004, 'src0.txt', noop: true, verbose: true)
  #     FileUtils.chown(1006, 1005, ['src0.txt', 'src0.dat'], noop: true, verbose: true)
  #     FileUtils.chown('user2', 'group1', path, noop: true, verbose: true)
  #     FileUtils.chown('user2', 'group1', '.', noop: true, verbose: true)
  #
  #   Output:
  #
  #     chown user2:group1 src0.txt
  #     chown 1004:1004 src0.txt
  #     chown 1006:1005 src0.txt src0.dat
  #     chown user2:group1 src0.txt
  #     chown user2:group1 .
  #
  # Related: FileUtils.chown_R.
  #
  def chown(user, group, list, noop: nil, verbose: nil)
    list = fu_list(list)
    fu_output_message sprintf('chown %s %s',
                              (group ? "#{user}:#{group}" : user || ':'),
                              list.join(' ')) if verbose
    return if noop
    uid = fu_get_uid(user)
    gid = fu_get_gid(group)
    list.each do |path|
      Entry_.new(path).chown uid, gid
    end
  end
  module_function :chown

  # Like FileUtils.chown, but changes owner and group recursively.
  #
  def chown_R(user, group, list, noop: nil, verbose: nil, force: nil)
    list = fu_list(list)
    fu_output_message sprintf('chown -R%s %s %s',
                              (force ? 'f' : ''),
                              (group ? "#{user}:#{group}" : user || ':'),
                              list.join(' ')) if verbose
    return if noop
    uid = fu_get_uid(user)
    gid = fu_get_gid(group)
    list.each do |root|
      Entry_.new(root).traverse do |ent|
        begin
          ent.chown uid, gid
        rescue
          raise unless force
        end
      end
    end
  end
  module_function :chown_R

  def fu_get_uid(user)   #:nodoc:
    return nil unless user
    case user
    when Integer
      user
    when /\A\d+\z/
      user.to_i
    else
      require 'etc'
      Etc.getpwnam(user) ? Etc.getpwnam(user).uid : nil
    end
  end
  private_module_function :fu_get_uid

  def fu_get_gid(group)   #:nodoc:
    return nil unless group
    case group
    when Integer
      group
    when /\A\d+\z/
      group.to_i
    else
      require 'etc'
      Etc.getgrnam(group) ? Etc.getgrnam(group).gid : nil
    end
  end
  private_module_function :fu_get_gid

  # Updates modification times (mtime) and access times (atime)
  # of the entries given by the paths in +list+
  # (a single path or an array of paths);
  # returns +list+ if it is an array, <tt>[list]</tt> otherwise.
  #
  # By default, creates an empty file for any path to a non-existent entry;
  # use keyword argument +nocreate+ to raise an exception instead.
  #
  # Argument +list+ or its elements
  # should be {interpretable as paths}[rdoc-ref:FileUtils@Path+Arguments].
  #
  # Examples:
  #
  #   # Single path.
  #   f = File.new('src0.txt') # Existing file.
  #   f.atime # => 2022-06-10 11:11:21.200277 -0700
  #   f.mtime # => 2022-06-10 11:11:21.200277 -0700
  #   FileUtils.touch('src0.txt')
  #   f = File.new('src0.txt')
  #   f.atime # => 2022-06-11 08:28:09.8185343 -0700
  #   f.mtime # => 2022-06-11 08:28:09.8185343 -0700
  #
  #   # Array of paths.
  #   FileUtils.touch(['src0.txt', 'src0.dat'])
  #
  # Keyword arguments:
  #
  # - <tt>mtime: <i>time</i></tt> - sets the entry's mtime to the given time,
  #   instead of the current time.
  # - <tt>nocreate: true</tt> - raises an exception if the entry does not exist.
  # - <tt>noop: true</tt> - does not touch entries; returns +nil+.
  # - <tt>verbose: true</tt> - prints an equivalent command:
  #
  #     FileUtils.touch('src0.txt', noop: true, verbose: true)
  #     FileUtils.touch(['src0.txt', 'src0.dat'], noop: true, verbose: true)
  #     FileUtils.touch(path, noop: true, verbose: true)
  #
  #   Output:
  #
  #     touch src0.txt
  #     touch src0.txt src0.dat
  #     touch src0.txt
  #
  # Related: FileUtils.uptodate?.
  #
  def touch(list, noop: nil, verbose: nil, mtime: nil, nocreate: nil)
    list = fu_list(list)
    t = mtime
    if verbose
      fu_output_message "touch #{nocreate ? '-c ' : ''}#{t ? t.strftime('-t %Y%m%d%H%M.%S ') : ''}#{list.join ' '}"
    end
    return if noop
    list.each do |path|
      created = nocreate
      begin
        File.utime(t, t, path)
      rescue Errno::ENOENT
        raise if created
        File.open(path, 'a') {
          ;
        }
        created = true
        retry if t
      end
    end
  end
  module_function :touch

  private

  module StreamUtils_ # :nodoc:

    private

    case (defined?(::RbConfig) ? ::RbConfig::CONFIG['host_os'] : ::RUBY_PLATFORM)
    when /mswin|mingw/
      def fu_windows?; true end #:nodoc:
    else
      def fu_windows?; false end #:nodoc:
    end

    def fu_copy_stream0(src, dest, blksize = nil)   #:nodoc:
      IO.copy_stream(src, dest)
    end

    def fu_stream_blksize(*streams) #:nodoc:
      streams.each do |s|
        next unless s.respond_to?(:stat)
        size = fu_blksize(s.stat)
        return size if size
      end
      fu_default_blksize()
    end

    def fu_blksize(st) #:nodoc:
      s = st.blksize
      return nil unless s
      return nil if s == 0
      s
    end

    def fu_default_blksize #:nodoc:
      1024
    end
  end

  include StreamUtils_
  extend StreamUtils_

  class Entry_   #:nodoc: internal use only
    include StreamUtils_

    def initialize(a, b = nil, deref = false)
      @prefix = @rel = @path = nil
      if b
        @prefix = a
        @rel = b
      else
        @path = a
      end
      @deref = deref
      @stat = nil
      @lstat = nil
    end

    def inspect
      "\#<#{self.class} #{path()}>"
    end

    def path
      if @path
        File.path(@path)
      else
        join(@prefix, @rel)
      end
    end

    def prefix
      @prefix || @path
    end

    def rel
      @rel
    end

    def dereference?
      @deref
    end

    def exist?
      begin
        lstat
        true
      rescue Errno::ENOENT
        false
      end
    end

    def file?
      s = lstat!
      s and s.file?
    end

    def directory?
      s = lstat!
      s and s.directory?
    end

    def symlink?
      s = lstat!
      s and s.symlink?
    end

    def chardev?
      s = lstat!
      s and s.chardev?
    end

    def blockdev?
      s = lstat!
      s and s.blockdev?
    end

    def socket?
      s = lstat!
      s and s.socket?
    end

    def pipe?
      s = lstat!
      s and s.pipe?
    end

    S_IF_DOOR = 0xD000

    def door?
      s = lstat!
      s and (s.mode & 0xF000 == S_IF_DOOR)
    end

    def entries
      opts = {}
      opts[:encoding] = fu_windows? ? ::Encoding::UTF_8 : path.encoding

      files = Dir.children(path, **opts)

      untaint = RUBY_VERSION < '2.7'
      files.map {|n| Entry_.new(prefix(), join(rel(), untaint ? n.untaint : n)) }
    end

    def stat
      return @stat if @stat
      if lstat() and lstat().symlink?
        @stat = File.stat(path())
      else
        @stat = lstat()
      end
      @stat
    end

    def stat!
      return @stat if @stat
      if lstat! and lstat!.symlink?
        @stat = File.stat(path())
      else
        @stat = lstat!
      end
      @stat
    rescue SystemCallError
      nil
    end

    def lstat
      if dereference?
        @lstat ||= File.stat(path())
      else
        @lstat ||= File.lstat(path())
      end
    end

    def lstat!
      lstat()
    rescue SystemCallError
      nil
    end

    def chmod(mode)
      if symlink?
        File.lchmod mode, path() if have_lchmod?
      else
        File.chmod mode, path()
      end
    rescue Errno::EOPNOTSUPP
    end

    def chown(uid, gid)
      if symlink?
        File.lchown uid, gid, path() if have_lchown?
      else
        File.chown uid, gid, path()
      end
    end

    def link(dest)
      case
      when directory?
        if !File.exist?(dest) and descendant_directory?(dest, path)
          raise ArgumentError, "cannot link directory %s to itself %s" % [path, dest]
        end
        begin
          Dir.mkdir dest
        rescue
          raise unless File.directory?(dest)
        end
      else
        File.link path(), dest
      end
    end

    def copy(dest)
      lstat
      case
      when file?
        copy_file dest
      when directory?
        if !File.exist?(dest) and descendant_directory?(dest, path)
          raise ArgumentError, "cannot copy directory %s to itself %s" % [path, dest]
        end
        begin
          Dir.mkdir dest
        rescue
          raise unless File.directory?(dest)
        end
      when symlink?
        File.symlink File.readlink(path()), dest
      when chardev?, blockdev?
        raise "cannot handle device file"
      when socket?
        begin
          require 'socket'
        rescue LoadError
          raise "cannot handle socket"
        else
          raise "cannot handle socket" unless defined?(UNIXServer)
        end
        UNIXServer.new(dest).close
        File.chmod lstat().mode, dest
      when pipe?
        raise "cannot handle FIFO" unless File.respond_to?(:mkfifo)
        File.mkfifo dest, lstat().mode
      when door?
        raise "cannot handle door: #{path()}"
      else
        raise "unknown file type: #{path()}"
      end
    end

    def copy_file(dest)
      File.open(path()) do |s|
        File.open(dest, 'wb', s.stat.mode) do |f|
          IO.copy_stream(s, f)
        end
      end
    end

    def copy_metadata(path)
      st = lstat()
      if !st.symlink?
        File.utime st.atime, st.mtime, path
      end
      mode = st.mode
      begin
        if st.symlink?
          begin
            File.lchown st.uid, st.gid, path
          rescue NotImplementedError
          end
        else
          File.chown st.uid, st.gid, path
        end
      rescue Errno::EPERM, Errno::EACCES
        # clear setuid/setgid
        mode &= 01777
      end
      if st.symlink?
        begin
          File.lchmod mode, path
        rescue NotImplementedError, Errno::EOPNOTSUPP
        end
      else
        File.chmod mode, path
      end
    end

    def remove
      if directory?
        remove_dir1
      else
        remove_file
      end
    end

    def remove_dir1
      platform_support {
        Dir.rmdir path().chomp(?/)
      }
    end

    def remove_file
      platform_support {
        File.unlink path
      }
    end

    def platform_support
      return yield unless fu_windows?
      first_time_p = true
      begin
        yield
      rescue Errno::ENOENT
        raise
      rescue => err
        if first_time_p
          first_time_p = false
          begin
            File.chmod 0700, path()   # Windows does not have symlink
            retry
          rescue SystemCallError
          end
        end
        raise err
      end
    end

    def preorder_traverse
      stack = [self]
      while ent = stack.pop
        yield ent
        stack.concat ent.entries.reverse if ent.directory?
      end
    end

    alias traverse preorder_traverse

    def postorder_traverse
      if directory?
        begin
          children = entries()
        rescue Errno::EACCES
          # Failed to get the list of children.
          # Assuming there is no children, try to process the parent directory.
          yield self
          return
        end

        children.each do |ent|
          ent.postorder_traverse do |e|
            yield e
          end
        end
      end
      yield self
    end

    def wrap_traverse(pre, post)
      pre.call self
      if directory?
        entries.each do |ent|
          ent.wrap_traverse pre, post
        end
      end
      post.call self
    end

    private

    @@fileutils_rb_have_lchmod = nil

    def have_lchmod?
      # This is not MT-safe, but it does not matter.
      if @@fileutils_rb_have_lchmod == nil
        @@fileutils_rb_have_lchmod = check_have_lchmod?
      end
      @@fileutils_rb_have_lchmod
    end

    def check_have_lchmod?
      return false unless File.respond_to?(:lchmod)
      File.lchmod 0
      return true
    rescue NotImplementedError
      return false
    end

    @@fileutils_rb_have_lchown = nil

    def have_lchown?
      # This is not MT-safe, but it does not matter.
      if @@fileutils_rb_have_lchown == nil
        @@fileutils_rb_have_lchown = check_have_lchown?
      end
      @@fileutils_rb_have_lchown
    end

    def check_have_lchown?
      return false unless File.respond_to?(:lchown)
      File.lchown nil, nil
      return true
    rescue NotImplementedError
      return false
    end

    def join(dir, base)
      return File.path(dir) if not base or base == '.'
      return File.path(base) if not dir or dir == '.'
      begin
        File.join(dir, base)
      rescue EncodingError
        if fu_windows?
          File.join(dir.encode(::Encoding::UTF_8), base.encode(::Encoding::UTF_8))
        else
          raise
        end
      end
    end

    if File::ALT_SEPARATOR
      DIRECTORY_TERM = "(?=[/#{Regexp.quote(File::ALT_SEPARATOR)}]|\\z)"
    else
      DIRECTORY_TERM = "(?=/|\\z)"
    end

    def descendant_directory?(descendant, ascendant)
      if File::FNM_SYSCASE.nonzero?
        File.expand_path(File.dirname(descendant)).casecmp(File.expand_path(ascendant)) == 0
      else
        File.expand_path(File.dirname(descendant)) == File.expand_path(ascendant)
      end
    end
  end   # class Entry_

  def fu_list(arg)   #:nodoc:
    [arg].flatten.map {|path| File.path(path) }
  end
  private_module_function :fu_list

  def fu_each_src_dest(src, dest)   #:nodoc:
    fu_each_src_dest0(src, dest) do |s, d|
      raise ArgumentError, "same file: #{s} and #{d}" if fu_same?(s, d)
      yield s, d
    end
  end
  private_module_function :fu_each_src_dest

  def fu_each_src_dest0(src, dest, target_directory = true)   #:nodoc:
    if tmp = Array.try_convert(src)
      tmp.each do |s|
        s = File.path(s)
        yield s, (target_directory ? File.join(dest, File.basename(s)) : dest)
      end
    else
      src = File.path(src)
      if target_directory and File.directory?(dest)
        yield src, File.join(dest, File.basename(src))
      else
        yield src, File.path(dest)
      end
    end
  end
  private_module_function :fu_each_src_dest0

  def fu_same?(a, b)   #:nodoc:
    File.identical?(a, b)
  end
  private_module_function :fu_same?

  def fu_output_message(msg)   #:nodoc:
    output = @fileutils_output if defined?(@fileutils_output)
    output ||= $stdout
    if defined?(@fileutils_label)
      msg = @fileutils_label + msg
    end
    output.puts msg
  end
  private_module_function :fu_output_message

  def fu_split_path(path) #:nodoc:
    path = File.path(path)
    list = []
    until (parent, base = File.split(path); parent == path or parent == ".")
      list << base
      path = parent
    end
    list << path
    list.reverse!
  end
  private_module_function :fu_split_path

  def fu_relative_components_from(target, base) #:nodoc:
    i = 0
    while target[i]&.== base[i]
      i += 1
    end
    Array.new(base.size-i, '..').concat(target[i..-1])
  end
  private_module_function :fu_relative_components_from

  def fu_clean_components(*comp) #:nodoc:
    comp.shift while comp.first == "."
    return comp if comp.empty?
    clean = [comp.shift]
    path = File.join(*clean, "") # ending with File::SEPARATOR
    while c = comp.shift
      if c == ".." and clean.last != ".." and !(fu_have_symlink? && File.symlink?(path))
        clean.pop
        path.chomp!(%r((?<=\A|/)[^/]+/\z), "")
      else
        clean << c
        path << c << "/"
      end
    end
    clean
  end
  private_module_function :fu_clean_components

  if fu_windows?
    def fu_starting_path?(path) #:nodoc:
      path&.start_with?(%r(\w:|/))
    end
  else
    def fu_starting_path?(path) #:nodoc:
      path&.start_with?("/")
    end
  end
  private_module_function :fu_starting_path?

  # This hash table holds command options.
  OPT_TABLE = {}    #:nodoc: internal use only
  (private_instance_methods & methods(false)).inject(OPT_TABLE) {|tbl, name|
    (tbl[name.to_s] = instance_method(name).parameters).map! {|t, n| n if t == :key}.compact!
    tbl
  }

  public

  # Returns an array of the string names of \FileUtils methods
  # that accept one or more keyword arguments:
  #
  #   FileUtils.commands.sort.take(3) # => ["cd", "chdir", "chmod"]
  #
  def self.commands
    OPT_TABLE.keys
  end

  # Returns an array of the string keyword names:
  #
  #   FileUtils.options.take(3) # => ["noop", "verbose", "force"]
  #
  def self.options
    OPT_TABLE.values.flatten.uniq.map {|sym| sym.to_s }
  end

  # Returns +true+ if method +mid+ accepts the given option +opt+, +false+ otherwise;
  # the arguments may be strings or symbols:
  #
  #   FileUtils.have_option?(:chmod, :noop) # => true
  #   FileUtils.have_option?('chmod', 'secure') # => false
  #
  def self.have_option?(mid, opt)
    li = OPT_TABLE[mid.to_s] or raise ArgumentError, "no such method: #{mid}"
    li.include?(opt)
  end

  # Returns an array of the string keyword name for method +mid+;
  # the argument may be a string or a symbol:
  #
  #   FileUtils.options_of(:rm) # => ["force", "noop", "verbose"]
  #   FileUtils.options_of('mv') # => ["force", "noop", "verbose", "secure"]
  #
  def self.options_of(mid)
    OPT_TABLE[mid.to_s].map {|sym| sym.to_s }
  end

  # Returns an array of the string method names of the methods
  # that accept the given keyword option +opt+;
  # the argument must be a symbol:
  #
  #   FileUtils.collect_method(:preserve) # => ["cp", "copy", "cp_r", "install"]
  #
  def self.collect_method(opt)
    OPT_TABLE.keys.select {|m| OPT_TABLE[m].include?(opt) }
  end

  private

  LOW_METHODS = singleton_methods(false) - collect_method(:noop).map(&:intern) # :nodoc:
  module LowMethods # :nodoc: internal use only
    private
    def _do_nothing(*)end
    ::FileUtils::LOW_METHODS.map {|name| alias_method name, :_do_nothing}
  end

  METHODS = singleton_methods() - [:private_module_function,                  # :nodoc:
      :commands, :options, :have_option?, :options_of, :collect_method]

  #
  # This module has all methods of FileUtils module, but it outputs messages
  # before acting.  This equates to passing the <tt>:verbose</tt> flag to
  # methods in FileUtils.
  #
  module Verbose
    include FileUtils
    names = ::FileUtils.collect_method(:verbose)
    names.each do |name|
      module_eval(<<-EOS, __FILE__, __LINE__ + 1)
        def #{name}(*args, **options)
          super(*args, **options, verbose: true)
        end
      EOS
    end
    private(*names)
    extend self
    class << self
      public(*::FileUtils::METHODS)
    end
  end

  #
  # This module has all methods of FileUtils module, but never changes
  # files/directories.  This equates to passing the <tt>:noop</tt> flag
  # to methods in FileUtils.
  #
  module NoWrite
    include FileUtils
    include LowMethods
    names = ::FileUtils.collect_method(:noop)
    names.each do |name|
      module_eval(<<-EOS, __FILE__, __LINE__ + 1)
        def #{name}(*args, **options)
          super(*args, **options, noop: true)
        end
      EOS
    end
    private(*names)
    extend self
    class << self
      public(*::FileUtils::METHODS)
    end
  end

  #
  # This module has all methods of FileUtils module, but never changes
  # files/directories, with printing message before acting.
  # This equates to passing the <tt>:noop</tt> and <tt>:verbose</tt> flag
  # to methods in FileUtils.
  #
  module DryRun
    include FileUtils
    include LowMethods
    names = ::FileUtils.collect_method(:noop)
    names.each do |name|
      module_eval(<<-EOS, __FILE__, __LINE__ + 1)
        def #{name}(*args, **options)
          super(*args, **options, noop: true, verbose: true)
        end
      EOS
    end
    private(*names)
    extend self
    class << self
      public(*::FileUtils::METHODS)
    end
  end

end
                                                                                                                                                                                                             ruby/unicode_normalize/tables.rb                                                                    0000644                 00000666566 15040313432 0013056 0                                                                                                    ustar 00                                                                                                                                                                                                                                                       # coding: us-ascii
# frozen_string_literal: true

# automatically generated by template/unicode_norm_gen.tmpl

module UnicodeNormalize  # :nodoc:
  accents = "" \
    "[\u0300-\u034E" \
    "\u0350-\u036F" \
    "\u0483-\u0487" \
    "\u0591-\u05BD" \
    "\u05BF" \
    "\u05C1\u05C2" \
    "\u05C4\u05C5" \
    "\u05C7" \
    "\u0610-\u061A" \
    "\u064B-\u065F" \
    "\u0670" \
    "\u06D6-\u06DC" \
    "\u06DF-\u06E4" \
    "\u06E7\u06E8" \
    "\u06EA-\u06ED" \
    "\u0711" \
    "\u0730-\u074A" \
    "\u07EB-\u07F3" \
    "\u07FD" \
    "\u0816-\u0819" \
    "\u081B-\u0823" \
    "\u0825-\u0827" \
    "\u0829-\u082D" \
    "\u0859-\u085B" \
    "\u0898-\u089F" \
    "\u08CA-\u08E1" \
    "\u08E3-\u08FF" \
    "\u093C" \
    "\u094D" \
    "\u0951-\u0954" \
    "\u09BC" \
    "\u09BE" \
    "\u09CD" \
    "\u09D7" \
    "\u09FE" \
    "\u0A3C" \
    "\u0A4D" \
    "\u0ABC" \
    "\u0ACD" \
    "\u0B3C" \
    "\u0B3E" \
    "\u0B4D" \
    "\u0B56\u0B57" \
    "\u0BBE" \
    "\u0BCD" \
    "\u0BD7" \
    "\u0C3C" \
    "\u0C4D" \
    "\u0C55\u0C56" \
    "\u0CBC" \
    "\u0CC2" \
    "\u0CCD" \
    "\u0CD5\u0CD6" \
    "\u0D3B\u0D3C" \
    "\u0D3E" \
    "\u0D4D" \
    "\u0D57" \
    "\u0DCA" \
    "\u0DCF" \
    "\u0DDF" \
    "\u0E38-\u0E3A" \
    "\u0E48-\u0E4B" \
    "\u0EB8-\u0EBA" \
    "\u0EC8-\u0ECB" \
    "\u0F18\u0F19" \
    "\u0F35" \
    "\u0F37" \
    "\u0F39" \
    "\u0F71\u0F72" \
    "\u0F74" \
    "\u0F7A-\u0F7D" \
    "\u0F80" \
    "\u0F82-\u0F84" \
    "\u0F86\u0F87" \
    "\u0FC6" \
    "\u102E" \
    "\u1037" \
    "\u1039\u103A" \
    "\u108D" \
    "\u135D-\u135F" \
    "\u1714\u1715" \
    "\u1734" \
    "\u17D2" \
    "\u17DD" \
    "\u18A9" \
    "\u1939-\u193B" \
    "\u1A17\u1A18" \
    "\u1A60" \
    "\u1A75-\u1A7C" \
    "\u1A7F" \
    "\u1AB0-\u1ABD" \
    "\u1ABF-\u1ACE" \
    "\u1B34\u1B35" \
    "\u1B44" \
    "\u1B6B-\u1B73" \
    "\u1BAA\u1BAB" \
    "\u1BE6" \
    "\u1BF2\u1BF3" \
    "\u1C37" \
    "\u1CD0-\u1CD2" \
    "\u1CD4-\u1CE0" \
    "\u1CE2-\u1CE8" \
    "\u1CED" \
    "\u1CF4" \
    "\u1CF8\u1CF9" \
    "\u1DC0-\u1DFF" \
    "\u20D0-\u20DC" \
    "\u20E1" \
    "\u20E5-\u20F0" \
    "\u2CEF-\u2CF1" \
    "\u2D7F" \
    "\u2DE0-\u2DFF" \
    "\u302A-\u302F" \
    "\u3099\u309A" \
    "\uA66F" \
    "\uA674-\uA67D" \
    "\uA69E\uA69F" \
    "\uA6F0\uA6F1" \
    "\uA806" \
    "\uA82C" \
    "\uA8C4" \
    "\uA8E0-\uA8F1" \
    "\uA92B-\uA92D" \
    "\uA953" \
    "\uA9B3" \
    "\uA9C0" \
    "\uAAB0" \
    "\uAAB2-\uAAB4" \
    "\uAAB7\uAAB8" \
    "\uAABE\uAABF" \
    "\uAAC1" \
    "\uAAF6" \
    "\uABED" \
    "\uFB1E" \
    "\uFE20-\uFE2F" \
    "\u{101FD}" \
    "\u{102E0}" \
    "\u{10376}-\u{1037A}" \
    "\u{10A0D}" \
    "\u{10A0F}" \
    "\u{10A38}-\u{10A3A}" \
    "\u{10A3F}" \
    "\u{10AE5}\u{10AE6}" \
    "\u{10D24}-\u{10D27}" \
    "\u{10EAB}\u{10EAC}" \
    "\u{10EFD}-\u{10EFF}" \
    "\u{10F46}-\u{10F50}" \
    "\u{10F82}-\u{10F85}" \
    "\u{11046}" \
    "\u{11070}" \
    "\u{1107F}" \
    "\u{110B9}\u{110BA}" \
    "\u{11100}-\u{11102}" \
    "\u{11127}" \
    "\u{11133}\u{11134}" \
    "\u{11173}" \
    "\u{111C0}" \
    "\u{111CA}" \
    "\u{11235}\u{11236}" \
    "\u{112E9}\u{112EA}" \
    "\u{1133B}\u{1133C}" \
    "\u{1133E}" \
    "\u{1134D}" \
    "\u{11357}" \
    "\u{11366}-\u{1136C}" \
    "\u{11370}-\u{11374}" \
    "\u{11442}" \
    "\u{11446}" \
    "\u{1145E}" \
    "\u{114B0}" \
    "\u{114BA}" \
    "\u{114BD}" \
    "\u{114C2}\u{114C3}" \
    "\u{115AF}" \
    "\u{115BF}\u{115C0}" \
    "\u{1163F}" \
    "\u{116B6}\u{116B7}" \
    "\u{1172B}" \
    "\u{11839}\u{1183A}" \
    "\u{11930}" \
    "\u{1193D}\u{1193E}" \
    "\u{11943}" \
    "\u{119E0}" \
    "\u{11A34}" \
    "\u{11A47}" \
    "\u{11A99}" \
    "\u{11C3F}" \
    "\u{11D42}" \
    "\u{11D44}\u{11D45}" \
    "\u{11D97}" \
    "\u{11F41}\u{11F42}" \
    "\u{16AF0}-\u{16AF4}" \
    "\u{16B30}-\u{16B36}" \
    "\u{16FF0}\u{16FF1}" \
    "\u{1BC9E}" \
    "\u{1D165}-\u{1D169}" \
    "\u{1D16D}-\u{1D172}" \
    "\u{1D17B}-\u{1D182}" \
    "\u{1D185}-\u{1D18B}" \
    "\u{1D1AA}-\u{1D1AD}" \
    "\u{1D242}-\u{1D244}" \
    "\u{1E000}-\u{1E006}" \
    "\u{1E008}-\u{1E018}" \
    "\u{1E01B}-\u{1E021}" \
    "\u{1E023}\u{1E024}" \
    "\u{1E026}-\u{1E02A}" \
    "\u{1E08F}" \
    "\u{1E130}-\u{1E136}" \
    "\u{1E2AE}" \
    "\u{1E2EC}-\u{1E2EF}" \
    "\u{1E4EC}-\u{1E4EF}" \
    "\u{1E8D0}-\u{1E8D6}" \
    "\u{1E944}-\u{1E94A}" \
    "]"
  ACCENTS = accents
  REGEXP_D_STRING = "#{''  # composition starters and composition exclusions
    }" \
    "[\u00C0-\u00C5" \
    "\u00C7-\u00CF" \
    "\u00D1-\u00D6" \
    "\u00D9-\u00DD" \
    "\u00E0-\u00E5" \
    "\u00E7-\u00EF" \
    "\u00F1-\u00F6" \
    "\u00F9-\u00FD" \
    "\u00FF-\u010F" \
    "\u0112-\u0125" \
    "\u0128-\u0130" \
    "\u0134-\u0137" \
    "\u0139-\u013E" \
    "\u0143-\u0148" \
    "\u014C-\u0151" \
    "\u0154-\u0165" \
    "\u0168-\u017E" \
    "\u01A0\u01A1" \
    "\u01AF\u01B0" \
    "\u01CD-\u01DC" \
    "\u01DE-\u01E3" \
    "\u01E6-\u01F0" \
    "\u01F4\u01F5" \
    "\u01F8-\u021B" \
    "\u021E\u021F" \
    "\u0226-\u0233" \
    "\u0340\u0341" \
    "\u0343\u0344" \
    "\u0374" \
    "\u037E" \
    "\u0385-\u038A" \
    "\u038C" \
    "\u038E-\u0390" \
    "\u03AA-\u03B0" \
    "\u03CA-\u03CE" \
    "\u03D3\u03D4" \
    "\u0400\u0401" \
    "\u0403" \
    "\u0407" \
    "\u040C-\u040E" \
    "\u0419" \
    "\u0439" \
    "\u0450\u0451" \
    "\u0453" \
    "\u0457" \
    "\u045C-\u045E" \
    "\u0476\u0477" \
    "\u04C1\u04C2" \
    "\u04D0-\u04D3" \
    "\u04D6\u04D7" \
    "\u04DA-\u04DF" \
    "\u04E2-\u04E7" \
    "\u04EA-\u04F5" \
    "\u04F8\u04F9" \
    "\u0622-\u0626" \
    "\u06C0" \
    "\u06C2" \
    "\u06D3" \
    "\u0929" \
    "\u0931" \
    "\u0934" \
    "\u0958-\u095F" \
    "\u09CB\u09CC" \
    "\u09DC\u09DD" \
    "\u09DF" \
    "\u0A33" \
    "\u0A36" \
    "\u0A59-\u0A5B" \
    "\u0A5E" \
    "\u0B48" \
    "\u0B4B\u0B4C" \
    "\u0B5C\u0B5D" \
    "\u0B94" \
    "\u0BCA-\u0BCC" \
    "\u0C48" \
    "\u0CC0" \
    "\u0CC7\u0CC8" \
    "\u0CCA\u0CCB" \
    "\u0D4A-\u0D4C" \
    "\u0DDA" \
    "\u0DDC-\u0DDE" \
    "\u0F43" \
    "\u0F4D" \
    "\u0F52" \
    "\u0F57" \
    "\u0F5C" \
    "\u0F69" \
    "\u0F73" \
    "\u0F75\u0F76" \
    "\u0F78" \
    "\u0F81" \
    "\u0F93" \
    "\u0F9D" \
    "\u0FA2" \
    "\u0FA7" \
    "\u0FAC" \
    "\u0FB9" \
    "\u1026" \
    "\u1B06" \
    "\u1B08" \
    "\u1B0A" \
    "\u1B0C" \
    "\u1B0E" \
    "\u1B12" \
    "\u1B3B" \
    "\u1B3D" \
    "\u1B40\u1B41" \
    "\u1B43" \
    "\u1E00-\u1E99" \
    "\u1E9B" \
    "\u1EA0-\u1EF9" \
    "\u1F00-\u1F15" \
    "\u1F18-\u1F1D" \
    "\u1F20-\u1F45" \
    "\u1F48-\u1F4D" \
    "\u1F50-\u1F57" \
    "\u1F59" \
    "\u1F5B" \
    "\u1F5D" \
    "\u1F5F-\u1F7D" \
    "\u1F80-\u1FB4" \
    "\u1FB6-\u1FBC" \
    "\u1FBE" \
    "\u1FC1-\u1FC4" \
    "\u1FC6-\u1FD3" \
    "\u1FD6-\u1FDB" \
    "\u1FDD-\u1FEF" \
    "\u1FF2-\u1FF4" \
    "\u1FF6-\u1FFD" \
    "\u2000\u2001" \
    "\u2126" \
    "\u212A\u212B" \
    "\u219A\u219B" \
    "\u21AE" \
    "\u21CD-\u21CF" \
    "\u2204" \
    "\u2209" \
    "\u220C" \
    "\u2224" \
    "\u2226" \
    "\u2241" \
    "\u2244" \
    "\u2247" \
    "\u2249" \
    "\u2260" \
    "\u2262" \
    "\u226D-\u2271" \
    "\u2274\u2275" \
    "\u2278\u2279" \
    "\u2280\u2281" \
    "\u2284\u2285" \
    "\u2288\u2289" \
    "\u22AC-\u22AF" \
    "\u22E0-\u22E3" \
    "\u22EA-\u22ED" \
    "\u2329\u232A" \
    "\u2ADC" \
    "\u304C" \
    "\u304E" \
    "\u3050" \
    "\u3052" \
    "\u3054" \
    "\u3056" \
    "\u3058" \
    "\u305A" \
    "\u305C" \
    "\u305E" \
    "\u3060" \
    "\u3062" \
    "\u3065" \
    "\u3067" \
    "\u3069" \
    "\u3070\u3071" \
    "\u3073\u3074" \
    "\u3076\u3077" \
    "\u3079\u307A" \
    "\u307C\u307D" \
    "\u3094" \
    "\u309E" \
    "\u30AC" \
    "\u30AE" \
    "\u30B0" \
    "\u30B2" \
    "\u30B4" \
    "\u30B6" \
    "\u30B8" \
    "\u30BA" \
    "\u30BC" \
    "\u30BE" \
    "\u30C0" \
    "\u30C2" \
    "\u30C5" \
    "\u30C7" \
    "\u30C9" \
    "\u30D0\u30D1" \
    "\u30D3\u30D4" \
    "\u30D6\u30D7" \
    "\u30D9\u30DA" \
    "\u30DC\u30DD" \
    "\u30F4" \
    "\u30F7-\u30FA" \
    "\u30FE" \
    "\uF900-\uFA0D" \
    "\uFA10" \
    "\uFA12" \
    "\uFA15-\uFA1E" \
    "\uFA20" \
    "\uFA22" \
    "\uFA25\uFA26" \
    "\uFA2A-\uFA6D" \
    "\uFA70-\uFAD9" \
    "\uFB1D" \
    "\uFB1F" \
    "\uFB2A-\uFB36" \
    "\uFB38-\uFB3C" \
    "\uFB3E" \
    "\uFB40\uFB41" \
    "\uFB43\uFB44" \
    "\uFB46-\uFB4E" \
    "\u{1109A}" \
    "\u{1109C}" \
    "\u{110AB}" \
    "\u{1112E}\u{1112F}" \
    "\u{1134B}\u{1134C}" \
    "\u{114BB}\u{114BC}" \
    "\u{114BE}" \
    "\u{115BA}\u{115BB}" \
    "\u{11938}" \
    "\u{1D15E}-\u{1D164}" \
    "\u{1D1BB}-\u{1D1C0}" \
    "\u{2F800}-\u{2FA1D}" \
    "]#{accents}*" \
    "|#{''  # characters that can be the result of a composition, except composition starters
    }" \
    "[<->" \
    "A-P" \
    "R-Z" \
    "a-p" \
    "r-z" \
    "\u00A8" \
    "\u00C6" \
    "\u00D8" \
    "\u00E6" \
    "\u00F8" \
    "\u017F" \
    "\u01B7" \
    "\u0292" \
    "\u0391" \
    "\u0395" \
    "\u0397" \
    "\u0399" \
    "\u039F" \
    "\u03A1" \
    "\u03A5" \
    "\u03A9" \
    "\u03B1" \
    "\u03B5" \
    "\u03B7" \
    "\u03B9" \
    "\u03BF" \
    "\u03C1" \
    "\u03C5" \
    "\u03C9" \
    "\u03D2" \
    "\u0406" \
    "\u0410" \
    "\u0413" \
    "\u0415-\u0418" \
    "\u041A" \
    "\u041E" \
    "\u0423" \
    "\u0427" \
    "\u042B" \
    "\u042D" \
    "\u0430" \
    "\u0433" \
    "\u0435-\u0438" \
    "\u043A" \
    "\u043E" \
    "\u0443" \
    "\u0447" \
    "\u044B" \
    "\u044D" \
    "\u0456" \
    "\u0474\u0475" \
    "\u04D8\u04D9" \
    "\u04E8\u04E9" \
    "\u0627" \
    "\u0648" \
    "\u064A" \
    "\u06C1" \
    "\u06D2" \
    "\u06D5" \
    "\u0928" \
    "\u0930" \
    "\u0933" \
    "\u09C7" \
    "\u0B47" \
    "\u0B92" \
    "\u0BC6\u0BC7" \
    "\u0C46" \
    "\u0CBF" \
    "\u0CC6" \
    "\u0D46\u0D47" \
    "\u0DD9" \
    "\u1025" \
    "\u1B05" \
    "\u1B07" \
    "\u1B09" \
    "\u1B0B" \
    "\u1B0D" \
    "\u1B11" \
    "\u1B3A" \
    "\u1B3C" \
    "\u1B3E\u1B3F" \
    "\u1B42" \
    "\u1FBF" \
    "\u1FFE" \
    "\u2190" \
    "\u2192" \
    "\u2194" \
    "\u21D0" \
    "\u21D2" \
    "\u21D4" \
    "\u2203" \
    "\u2208" \
    "\u220B" \
    "\u2223" \
    "\u2225" \
    "\u223C" \
    "\u2243" \
    "\u2245" \
    "\u2248" \
    "\u224D" \
    "\u2261" \
    "\u2264\u2265" \
    "\u2272\u2273" \
    "\u2276\u2277" \
    "\u227A-\u227D" \
    "\u2282\u2283" \
    "\u2286\u2287" \
    "\u2291\u2292" \
    "\u22A2" \
    "\u22A8\u22A9" \
    "\u22AB" \
    "\u22B2-\u22B5" \
    "\u3046" \
    "\u304B" \
    "\u304D" \
    "\u304F" \
    "\u3051" \
    "\u3053" \
    "\u3055" \
    "\u3057" \
    "\u3059" \
    "\u305B" \
    "\u305D" \
    "\u305F" \
    "\u3061" \
    "\u3064" \
    "\u3066" \
    "\u3068" \
    "\u306F" \
    "\u3072" \
    "\u3075" \
    "\u3078" \
    "\u307B" \
    "\u309D" \
    "\u30A6" \
    "\u30AB" \
    "\u30AD" \
    "\u30AF" \
    "\u30B1" \
    "\u30B3" \
    "\u30B5" \
    "\u30B7" \
    "\u30B9" \
    "\u30BB" \
    "\u30BD" \
    "\u30BF" \
    "\u30C1" \
    "\u30C4" \
    "\u30C6" \
    "\u30C8" \
    "\u30CF" \
    "\u30D2" \
    "\u30D5" \
    "\u30D8" \
    "\u30DB" \
    "\u30EF-\u30F2" \
    "\u30FD" \
    "\u{11099}" \
    "\u{1109B}" \
    "\u{110A5}" \
    "\u{11131}\u{11132}" \
    "\u{11347}" \
    "\u{114B9}" \
    "\u{115B8}\u{115B9}" \
    "\u{11935}" \
    "]?#{accents}+" \
    "|#{''  # precomposed Hangul syllables
    }" \
    "[\u{AC00}-\u{D7A4}]"
  REGEXP_C_STRING = "#{''  # composition exclusions
    }" \
    "[\u0340\u0341" \
    "\u0343\u0344" \
    "\u0374" \
    "\u037E" \
    "\u0387" \
    "\u0958-\u095F" \
    "\u09DC\u09DD" \
    "\u09DF" \
    "\u0A33" \
    "\u0A36" \
    "\u0A59-\u0A5B" \
    "\u0A5E" \
    "\u0B5C\u0B5D" \
    "\u0F43" \
    "\u0F4D" \
    "\u0F52" \
    "\u0F57" \
    "\u0F5C" \
    "\u0F69" \
    "\u0F73" \
    "\u0F75\u0F76" \
    "\u0F78" \
    "\u0F81" \
    "\u0F93" \
    "\u0F9D" \
    "\u0FA2" \
    "\u0FA7" \
    "\u0FAC" \
    "\u0FB9" \
    "\u1F71" \
    "\u1F73" \
    "\u1F75" \
    "\u1F77" \
    "\u1F79" \
    "\u1F7B" \
    "\u1F7D" \
    "\u1FBB" \
    "\u1FBE" \
    "\u1FC9" \
    "\u1FCB" \
    "\u1FD3" \
    "\u1FDB" \
    "\u1FE3" \
    "\u1FEB" \
    "\u1FEE\u1FEF" \
    "\u1FF9" \
    "\u1FFB" \
    "\u1FFD" \
    "\u2000\u2001" \
    "\u2126" \
    "\u212A\u212B" \
    "\u2329\u232A" \
    "\u2ADC" \
    "\uF900-\uFA0D" \
    "\uFA10" \
    "\uFA12" \
    "\uFA15-\uFA1E" \
    "\uFA20" \
    "\uFA22" \
    "\uFA25\uFA26" \
    "\uFA2A-\uFA6D" \
    "\uFA70-\uFAD9" \
    "\uFB1D" \
    "\uFB1F" \
    "\uFB2A-\uFB36" \
    "\uFB38-\uFB3C" \
    "\uFB3E" \
    "\uFB40\uFB41" \
    "\uFB43\uFB44" \
    "\uFB46-\uFB4E" \
    "\u{1D15E}-\u{1D164}" \
    "\u{1D1BB}-\u{1D1C0}" \
    "\u{2F800}-\u{2FA1D}" \
    "]#{accents}*" \
    "|#{''  # composition starters and characters that can be the result of a composition
    }" \
    "[<->" \
    "A-P" \
    "R-Z" \
    "a-p" \
    "r-z" \
    "\u00A8" \
    "\u00C0-\u00CF" \
    "\u00D1-\u00D6" \
    "\u00D8-\u00DD" \
    "\u00E0-\u00EF" \
    "\u00F1-\u00F6" \
    "\u00F8-\u00FD" \
    "\u00FF-\u010F" \
    "\u0112-\u0125" \
    "\u0128-\u0130" \
    "\u0134-\u0137" \
    "\u0139-\u013E" \
    "\u0143-\u0148" \
    "\u014C-\u0151" \
    "\u0154-\u0165" \
    "\u0168-\u017F" \
    "\u01A0\u01A1" \
    "\u01AF\u01B0" \
    "\u01B7" \
    "\u01CD-\u01DC" \
    "\u01DE-\u01E3" \
    "\u01E6-\u01F0" \
    "\u01F4\u01F5" \
    "\u01F8-\u021B" \
    "\u021E\u021F" \
    "\u0226-\u0233" \
    "\u0292" \
    "\u0385\u0386" \
    "\u0388-\u038A" \
    "\u038C" \
    "\u038E-\u0391" \
    "\u0395" \
    "\u0397" \
    "\u0399" \
    "\u039F" \
    "\u03A1" \
    "\u03A5" \
    "\u03A9-\u03B1" \
    "\u03B5" \
    "\u03B7" \
    "\u03B9" \
    "\u03BF" \
    "\u03C1" \
    "\u03C5" \
    "\u03C9-\u03CE" \
    "\u03D2-\u03D4" \
    "\u0400\u0401" \
    "\u0403" \
    "\u0406\u0407" \
    "\u040C-\u040E" \
    "\u0410" \
    "\u0413" \
    "\u0415-\u041A" \
    "\u041E" \
    "\u0423" \
    "\u0427" \
    "\u042B" \
    "\u042D" \
    "\u0430" \
    "\u0433" \
    "\u0435-\u043A" \
    "\u043E" \
    "\u0443" \
    "\u0447" \
    "\u044B" \
    "\u044D" \
    "\u0450\u0451" \
    "\u0453" \
    "\u0456\u0457" \
    "\u045C-\u045E" \
    "\u0474-\u0477" \
    "\u04C1\u04C2" \
    "\u04D0-\u04D3" \
    "\u04D6-\u04DF" \
    "\u04E2-\u04F5" \
    "\u04F8\u04F9" \
    "\u0622-\u0627" \
    "\u0648" \
    "\u064A" \
    "\u06C0-\u06C2" \
    "\u06D2\u06D3" \
    "\u06D5" \
    "\u0928\u0929" \
    "\u0930\u0931" \
    "\u0933\u0934" \
    "\u09C7" \
    "\u09CB\u09CC" \
    "\u0B47\u0B48" \
    "\u0B4B\u0B4C" \
    "\u0B92" \
    "\u0B94" \
    "\u0BC6\u0BC7" \
    "\u0BCA-\u0BCC" \
    "\u0C46" \
    "\u0C48" \
    "\u0CBF\u0CC0" \
    "\u0CC6-\u0CC8" \
    "\u0CCA\u0CCB" \
    "\u0D46\u0D47" \
    "\u0D4A-\u0D4C" \
    "\u0DD9\u0DDA" \
    "\u0DDC-\u0DDE" \
    "\u1025\u1026" \
    "\u1B05-\u1B0E" \
    "\u1B11\u1B12" \
    "\u1B3A-\u1B43" \
    "\u1E00-\u1E99" \
    "\u1E9B" \
    "\u1EA0-\u1EF9" \
    "\u1F00-\u1F15" \
    "\u1F18-\u1F1D" \
    "\u1F20-\u1F45" \
    "\u1F48-\u1F4D" \
    "\u1F50-\u1F57" \
    "\u1F59" \
    "\u1F5B" \
    "\u1F5D" \
    "\u1F5F-\u1F70" \
    "\u1F72" \
    "\u1F74" \
    "\u1F76" \
    "\u1F78" \
    "\u1F7A" \
    "\u1F7C" \
    "\u1F80-\u1FB4" \
    "\u1FB6-\u1FBA" \
    "\u1FBC" \
    "\u1FBF" \
    "\u1FC1-\u1FC4" \
    "\u1FC6-\u1FC8" \
    "\u1FCA" \
    "\u1FCC-\u1FD2" \
    "\u1FD6-\u1FDA" \
    "\u1FDD-\u1FE2" \
    "\u1FE4-\u1FEA" \
    "\u1FEC\u1FED" \
    "\u1FF2-\u1FF4" \
    "\u1FF6-\u1FF8" \
    "\u1FFA" \
    "\u1FFC" \
    "\u1FFE" \
    "\u2190" \
    "\u2192" \
    "\u2194" \
    "\u219A\u219B" \
    "\u21AE" \
    "\u21CD-\u21D0" \
    "\u21D2" \
    "\u21D4" \
    "\u2203\u2204" \
    "\u2208\u2209" \
    "\u220B\u220C" \
    "\u2223-\u2226" \
    "\u223C" \
    "\u2241" \
    "\u2243-\u2245" \
    "\u2247-\u2249" \
    "\u224D" \
    "\u2260-\u2262" \
    "\u2264\u2265" \
    "\u226D-\u227D" \
    "\u2280-\u2289" \
    "\u2291\u2292" \
    "\u22A2" \
    "\u22A8\u22A9" \
    "\u22AB-\u22AF" \
    "\u22B2-\u22B5" \
    "\u22E0-\u22E3" \
    "\u22EA-\u22ED" \
    "\u3046" \
    "\u304B-\u3062" \
    "\u3064-\u3069" \
    "\u306F-\u307D" \
    "\u3094" \
    "\u309D\u309E" \
    "\u30A6" \
    "\u30AB-\u30C2" \
    "\u30C4-\u30C9" \
    "\u30CF-\u30DD" \
    "\u30EF-\u30F2" \
    "\u30F4" \
    "\u30F7-\u30FA" \
    "\u30FD\u30FE" \
    "\u{11099}-\u{1109C}" \
    "\u{110A5}" \
    "\u{110AB}" \
    "\u{1112E}\u{1112F}" \
    "\u{11131}\u{11132}" \
    "\u{11347}" \
    "\u{1134B}\u{1134C}" \
    "\u{114B9}" \
    "\u{114BB}\u{114BC}" \
    "\u{114BE}" \
    "\u{115B8}-\u{115BB}" \
    "\u{11935}" \
    "\u{11938}" \
    "]?#{accents}+" \
    "|#{''  # Hangul syllables with separate trailer
    }" \
    "[\uAC00" \
    "\uAC1C" \
    "\uAC38" \
    "\uAC54" \
    "\uAC70" \
    "\uAC8C" \
    "\uACA8" \
    "\uACC4" \
    "\uACE0" \
    "\uACFC" \
    "\uAD18" \
    "\uAD34" \
    "\uAD50" \
    "\uAD6C" \
    "\uAD88" \
    "\uADA4" \
    "\uADC0" \
    "\uADDC" \
    "\uADF8" \
    "\uAE14" \
    "\uAE30" \
    "\uAE4C" \
    "\uAE68" \
    "\uAE84" \
    "\uAEA0" \
    "\uAEBC" \
    "\uAED8" \
    "\uAEF4" \
    "\uAF10" \
    "\uAF2C" \
    "\uAF48" \
    "\uAF64" \
    "\uAF80" \
    "\uAF9C" \
    "\uAFB8" \
    "\uAFD4" \
    "\uAFF0" \
    "\uB00C" \
    "\uB028" \
    "\uB044" \
    "\uB060" \
    "\uB07C" \
    "\uB098" \
    "\uB0B4" \
    "\uB0D0" \
    "\uB0EC" \
    "\uB108" \
    "\uB124" \
    "\uB140" \
    "\uB15C" \
    "\uB178" \
    "\uB194" \
    "\uB1B0" \
    "\uB1CC" \
    "\uB1E8" \
    "\uB204" \
    "\uB220" \
    "\uB23C" \
    "\uB258" \
    "\uB274" \
    "\uB290" \
    "\uB2AC" \
    "\uB2C8" \
    "\uB2E4" \
    "\uB300" \
    "\uB31C" \
    "\uB338" \
    "\uB354" \
    "\uB370" \
    "\uB38C" \
    "\uB3A8" \
    "\uB3C4" \
    "\uB3E0" \
    "\uB3FC" \
    "\uB418" \
    "\uB434" \
    "\uB450" \
    "\uB46C" \
    "\uB488" \
    "\uB4A4" \
    "\uB4C0" \
    "\uB4DC" \
    "\uB4F8" \
    "\uB514" \
    "\uB530" \
    "\uB54C" \
    "\uB568" \
    "\uB584" \
    "\uB5A0" \
    "\uB5BC" \
    "\uB5D8" \
    "\uB5F4" \
    "\uB610" \
    "\uB62C" \
    "\uB648" \
    "\uB664" \
    "\uB680" \
    "\uB69C" \
    "\uB6B8" \
    "\uB6D4" \
    "\uB6F0" \
    "\uB70C" \
    "\uB728" \
    "\uB744" \
    "\uB760" \
    "\uB77C" \
    "\uB798" \
    "\uB7B4" \
    "\uB7D0" \
    "\uB7EC" \
    "\uB808" \
    "\uB824" \
    "\uB840" \
    "\uB85C" \
    "\uB878" \
    "\uB894" \
    "\uB8B0" \
    "\uB8CC" \
    "\uB8E8" \
    "\uB904" \
    "\uB920" \
    "\uB93C" \
    "\uB958" \
    "\uB974" \
    "\uB990" \
    "\uB9AC" \
    "\uB9C8" \
    "\uB9E4" \
    "\uBA00" \
    "\uBA1C" \
    "\uBA38" \
    "\uBA54" \
    "\uBA70" \
    "\uBA8C" \
    "\uBAA8" \
    "\uBAC4" \
    "\uBAE0" \
    "\uBAFC" \
    "\uBB18" \
    "\uBB34" \
    "\uBB50" \
    "\uBB6C" \
    "\uBB88" \
    "\uBBA4" \
    "\uBBC0" \
    "\uBBDC" \
    "\uBBF8" \
    "\uBC14" \
    "\uBC30" \
    "\uBC4C" \
    "\uBC68" \
    "\uBC84" \
    "\uBCA0" \
    "\uBCBC" \
    "\uBCD8" \
    "\uBCF4" \
    "\uBD10" \
    "\uBD2C" \
    "\uBD48" \
    "\uBD64" \
    "\uBD80" \
    "\uBD9C" \
    "\uBDB8" \
    "\uBDD4" \
    "\uBDF0" \
    "\uBE0C" \
    "\uBE28" \
    "\uBE44" \
    "\uBE60" \
    "\uBE7C" \
    "\uBE98" \
    "\uBEB4" \
    "\uBED0" \
    "\uBEEC" \
    "\uBF08" \
    "\uBF24" \
    "\uBF40" \
    "\uBF5C" \
    "\uBF78" \
    "\uBF94" \
    "\uBFB0" \
    "\uBFCC" \
    "\uBFE8" \
    "\uC004" \
    "\uC020" \
    "\uC03C" \
    "\uC058" \
    "\uC074" \
    "\uC090" \
    "\uC0AC" \
    "\uC0C8" \
    "\uC0E4" \
    "\uC100" \
    "\uC11C" \
    "\uC138" \
    "\uC154" \
    "\uC170" \
    "\uC18C" \
    "\uC1A8" \
    "\uC1C4" \
    "\uC1E0" \
    "\uC1FC" \
    "\uC218" \
    "\uC234" \
    "\uC250" \
    "\uC26C" \
    "\uC288" \
    "\uC2A4" \
    "\uC2C0" \
    "\uC2DC" \
    "\uC2F8" \
    "\uC314" \
    "\uC330" \
    "\uC34C" \
    "\uC368" \
    "\uC384" \
    "\uC3A0" \
    "\uC3BC" \
    "\uC3D8" \
    "\uC3F4" \
    "\uC410" \
    "\uC42C" \
    "\uC448" \
    "\uC464" \
    "\uC480" \
    "\uC49C" \
    "\uC4B8" \
    "\uC4D4" \
    "\uC4F0" \
    "\uC50C" \
    "\uC528" \
    "\uC544" \
    "\uC560" \
    "\uC57C" \
    "\uC598" \
    "\uC5B4" \
    "\uC5D0" \
    "\uC5EC" \
    "\uC608" \
    "\uC624" \
    "\uC640" \
    "\uC65C" \
    "\uC678" \
    "\uC694" \
    "\uC6B0" \
    "\uC6CC" \
    "\uC6E8" \
    "\uC704" \
    "\uC720" \
    "\uC73C" \
    "\uC758" \
    "\uC774" \
    "\uC790" \
    "\uC7AC" \
    "\uC7C8" \
    "\uC7E4" \
    "\uC800" \
    "\uC81C" \
    "\uC838" \
    "\uC854" \
    "\uC870" \
    "\uC88C" \
    "\uC8A8" \
    "\uC8C4" \
    "\uC8E0" \
    "\uC8FC" \
    "\uC918" \
    "\uC934" \
    "\uC950" \
    "\uC96C" \
    "\uC988" \
    "\uC9A4" \
    "\uC9C0" \
    "\uC9DC" \
    "\uC9F8" \
    "\uCA14" \
    "\uCA30" \
    "\uCA4C" \
    "\uCA68" \
    "\uCA84" \
    "\uCAA0" \
    "\uCABC" \
    "\uCAD8" \
    "\uCAF4" \
    "\uCB10" \
    "\uCB2C" \
    "\uCB48" \
    "\uCB64" \
    "\uCB80" \
    "\uCB9C" \
    "\uCBB8" \
    "\uCBD4" \
    "\uCBF0" \
    "\uCC0C" \
    "\uCC28" \
    "\uCC44" \
    "\uCC60" \
    "\uCC7C" \
    "\uCC98" \
    "\uCCB4" \
    "\uCCD0" \
    "\uCCEC" \
    "\uCD08" \
    "\uCD24" \
    "\uCD40" \
    "\uCD5C" \
    "\uCD78" \
    "\uCD94" \
    "\uCDB0" \
    "\uCDCC" \
    "\uCDE8" \
    "\uCE04" \
    "\uCE20" \
    "\uCE3C" \
    "\uCE58" \
    "\uCE74" \
    "\uCE90" \
    "\uCEAC" \
    "\uCEC8" \
    "\uCEE4" \
    "\uCF00" \
    "\uCF1C" \
    "\uCF38" \
    "\uCF54" \
    "\uCF70" \
    "\uCF8C" \
    "\uCFA8" \
    "\uCFC4" \
    "\uCFE0" \
    "\uCFFC" \
    "\uD018" \
    "\uD034" \
    "\uD050" \
    "\uD06C" \
    "\uD088" \
    "\uD0A4" \
    "\uD0C0" \
    "\uD0DC" \
    "\uD0F8" \
    "\uD114" \
    "\uD130" \
    "\uD14C" \
    "\uD168" \
    "\uD184" \
    "\uD1A0" \
    "\uD1BC" \
    "\uD1D8" \
    "\uD1F4" \
    "\uD210" \
    "\uD22C" \
    "\uD248" \
    "\uD264" \
    "\uD280" \
    "\uD29C" \
    "\uD2B8" \
    "\uD2D4" \
    "\uD2F0" \
    "\uD30C" \
    "\uD328" \
    "\uD344" \
    "\uD360" \
    "\uD37C" \
    "\uD398" \
    "\uD3B4" \
    "\uD3D0" \
    "\uD3EC" \
    "\uD408" \
    "\uD424" \
    "\uD440" \
    "\uD45C" \
    "\uD478" \
    "\uD494" \
    "\uD4B0" \
    "\uD4CC" \
    "\uD4E8" \
    "\uD504" \
    "\uD520" \
    "\uD53C" \
    "\uD558" \
    "\uD574" \
    "\uD590" \
    "\uD5AC" \
    "\uD5C8" \
    "\uD5E4" \
    "\uD600" \
    "\uD61C" \
    "\uD638" \
    "\uD654" \
    "\uD670" \
    "\uD68C" \
    "\uD6A8" \
    "\uD6C4" \
    "\uD6E0" \
    "\uD6FC" \
    "\uD718" \
    "\uD734" \
    "\uD750" \
    "\uD76C" \
    "\uD788" \
    "][\u11A8-\u11C2]" \
    "|#{''  # decomposed Hangul syllables
    }" \
    "[\u1100-\u1112][\u1161-\u1175][\u11A8-\u11C2]?"
  REGEXP_K_STRING = "" \
    "[\u00A0" \
    "\u00A8" \
    "\u00AA" \
    "\u00AF" \
    "\u00B2-\u00B5" \
    "\u00B8-\u00BA" \
    "\u00BC-\u00BE" \
    "\u0132\u0133" \
    "\u013F\u0140" \
    "\u0149" \
    "\u017F" \
    "\u01C4-\u01CC" \
    "\u01F1-\u01F3" \
    "\u02B0-\u02B8" \
    "\u02D8-\u02DD" \
    "\u02E0-\u02E4" \
    "\u037A" \
    "\u0384\u0385" \
    "\u03D0-\u03D6" \
    "\u03F0-\u03F2" \
    "\u03F4\u03F5" \
    "\u03F9" \
    "\u0587" \
    "\u0675-\u0678" \
    "\u0E33" \
    "\u0EB3" \
    "\u0EDC\u0EDD" \
    "\u0F0C" \
    "\u0F77" \
    "\u0F79" \
    "\u10FC" \
    "\u1D2C-\u1D2E" \
    "\u1D30-\u1D3A" \
    "\u1D3C-\u1D4D" \
    "\u1D4F-\u1D6A" \
    "\u1D78" \
    "\u1D9B-\u1DBF" \
    "\u1E9A\u1E9B" \
    "\u1FBD" \
    "\u1FBF-\u1FC1" \
    "\u1FCD-\u1FCF" \
    "\u1FDD-\u1FDF" \
    "\u1FED\u1FEE" \
    "\u1FFD\u1FFE" \
    "\u2000-\u200A" \
    "\u2011" \
    "\u2017" \
    "\u2024-\u2026" \
    "\u202F" \
    "\u2033\u2034" \
    "\u2036\u2037" \
    "\u203C" \
    "\u203E" \
    "\u2047-\u2049" \
    "\u2057" \
    "\u205F" \
    "\u2070\u2071" \
    "\u2074-\u208E" \
    "\u2090-\u209C" \
    "\u20A8" \
    "\u2100-\u2103" \
    "\u2105-\u2107" \
    "\u2109-\u2113" \
    "\u2115\u2116" \
    "\u2119-\u211D" \
    "\u2120-\u2122" \
    "\u2124" \
    "\u2128" \
    "\u212C\u212D" \
    "\u212F-\u2131" \
    "\u2133-\u2139" \
    "\u213B-\u2140" \
    "\u2145-\u2149" \
    "\u2150-\u217F" \
    "\u2189" \
    "\u222C\u222D" \
    "\u222F\u2230" \
    "\u2460-\u24EA" \
    "\u2A0C" \
    "\u2A74-\u2A76" \
    "\u2C7C\u2C7D" \
    "\u2D6F" \
    "\u2E9F" \
    "\u2EF3" \
    "\u2F00-\u2FD5" \
    "\u3000" \
    "\u3036" \
    "\u3038-\u303A" \
    "\u309B\u309C" \
    "\u309F" \
    "\u30FF" \
    "\u3131-\u318E" \
    "\u3192-\u319F" \
    "\u3200-\u321E" \
    "\u3220-\u3247" \
    "\u3250-\u327E" \
    "\u3280-\u33FF" \
    "\uA69C\uA69D" \
    "\uA770" \
    "\uA7F2-\uA7F4" \
    "\uA7F8\uA7F9" \
    "\uAB5C-\uAB5F" \
    "\uAB69" \
    "\uFB00-\uFB06" \
    "\uFB13-\uFB17" \
    "\uFB20-\uFB29" \
    "\uFB4F-\uFBB1" \
    "\uFBD3-\uFD3D" \
    "\uFD50-\uFD8F" \
    "\uFD92-\uFDC7" \
    "\uFDF0-\uFDFC" \
    "\uFE10-\uFE19" \
    "\uFE30-\uFE44" \
    "\uFE47-\uFE52" \
    "\uFE54-\uFE66" \
    "\uFE68-\uFE6B" \
    "\uFE70-\uFE72" \
    "\uFE74" \
    "\uFE76-\uFEFC" \
    "\uFF01-\uFFBE" \
    "\uFFC2-\uFFC7" \
    "\uFFCA-\uFFCF" \
    "\uFFD2-\uFFD7" \
    "\uFFDA-\uFFDC" \
    "\uFFE0-\uFFE6" \
    "\uFFE8-\uFFEE" \
    "\u{10781}-\u{10785}" \
    "\u{10787}-\u{107B0}" \
    "\u{107B2}-\u{107BA}" \
    "\u{1D400}-\u{1D454}" \
    "\u{1D456}-\u{1D49C}" \
    "\u{1D49E}\u{1D49F}" \
    "\u{1D4A2}" \
    "\u{1D4A5}\u{1D4A6}" \
    "\u{1D4A9}-\u{1D4AC}" \
    "\u{1D4AE}-\u{1D4B9}" \
    "\u{1D4BB}" \
    "\u{1D4BD}-\u{1D4C3}" \
    "\u{1D4C5}-\u{1D505}" \
    "\u{1D507}-\u{1D50A}" \
    "\u{1D50D}-\u{1D514}" \
    "\u{1D516}-\u{1D51C}" \
    "\u{1D51E}-\u{1D539}" \
    "\u{1D53B}-\u{1D53E}" \
    "\u{1D540}-\u{1D544}" \
    "\u{1D546}" \
    "\u{1D54A}-\u{1D550}" \
    "\u{1D552}-\u{1D6A5}" \
    "\u{1D6A8}-\u{1D7CB}" \
    "\u{1D7CE}-\u{1D7FF}" \
    "\u{1E030}-\u{1E06D}" \
    "\u{1EE00}-\u{1EE03}" \
    "\u{1EE05}-\u{1EE1F}" \
    "\u{1EE21}\u{1EE22}" \
    "\u{1EE24}" \
    "\u{1EE27}" \
    "\u{1EE29}-\u{1EE32}" \
    "\u{1EE34}-\u{1EE37}" \
    "\u{1EE39}" \
    "\u{1EE3B}" \
    "\u{1EE42}" \
    "\u{1EE47}" \
    "\u{1EE49}" \
    "\u{1EE4B}" \
    "\u{1EE4D}-\u{1EE4F}" \
    "\u{1EE51}\u{1EE52}" \
    "\u{1EE54}" \
    "\u{1EE57}" \
    "\u{1EE59}" \
    "\u{1EE5B}" \
    "\u{1EE5D}" \
    "\u{1EE5F}" \
    "\u{1EE61}\u{1EE62}" \
    "\u{1EE64}" \
    "\u{1EE67}-\u{1EE6A}" \
    "\u{1EE6C}-\u{1EE72}" \
    "\u{1EE74}-\u{1EE77}" \
    "\u{1EE79}-\u{1EE7C}" \
    "\u{1EE7E}" \
    "\u{1EE80}-\u{1EE89}" \
    "\u{1EE8B}-\u{1EE9B}" \
    "\u{1EEA1}-\u{1EEA3}" \
    "\u{1EEA5}-\u{1EEA9}" \
    "\u{1EEAB}-\u{1EEBB}" \
    "\u{1F100}-\u{1F10A}" \
    "\u{1F110}-\u{1F12E}" \
    "\u{1F130}-\u{1F14F}" \
    "\u{1F16A}-\u{1F16C}" \
    "\u{1F190}" \
    "\u{1F200}-\u{1F202}" \
    "\u{1F210}-\u{1F23B}" \
    "\u{1F240}-\u{1F248}" \
    "\u{1F250}\u{1F251}" \
    "\u{1FBF0}-\u{1FBF9}" \
    "]"

  class_table = {
    "\u0300"=>230,
    "\u0301"=>230,
    "\u0302"=>230,
    "\u0303"=>230,
    "\u0304"=>230,
    "\u0305"=>230,
    "\u0306"=>230,
    "\u0307"=>230,
    "\u0308"=>230,
    "\u0309"=>230,
    "\u030A"=>230,
    "\u030B"=>230,
    "\u030C"=>230,
    "\u030D"=>230,
    "\u030E"=>230,
    "\u030F"=>230,
    "\u0310"=>230,
    "\u0311"=>230,
    "\u0312"=>230,
    "\u0313"=>230,
    "\u0314"=>230,
    "\u0315"=>232,
    "\u0316"=>220,
    "\u0317"=>220,
    "\u0318"=>220,
    "\u0319"=>220,
    "\u031A"=>232,
    "\u031B"=>216,
    "\u031C"=>220,
    "\u031D"=>220,
    "\u031E"=>220,
    "\u031F"=>220,
    "\u0320"=>220,
    "\u0321"=>202,
    "\u0322"=>202,
    "\u0323"=>220,
    "\u0324"=>220,
    "\u0325"=>220,
    "\u0326"=>220,
    "\u0327"=>202,
    "\u0328"=>202,
    "\u0329"=>220,
    "\u032A"=>220,
    "\u032B"=>220,
    "\u032C"=>220,
    "\u032D"=>220,
    "\u032E"=>220,
    "\u032F"=>220,
    "\u0330"=>220,
    "\u0331"=>220,
    "\u0332"=>220,
    "\u0333"=>220,
    "\u0334"=>1,
    "\u0335"=>1,
    "\u0336"=>1,
    "\u0337"=>1,
    "\u0338"=>1,
    "\u0339"=>220,
    "\u033A"=>220,
    "\u033B"=>220,
    "\u033C"=>220,
    "\u033D"=>230,
    "\u033E"=>230,
    "\u033F"=>230,
    "\u0340"=>230,
    "\u0341"=>230,
    "\u0342"=>230,
    "\u0343"=>230,
    "\u0344"=>230,
    "\u0345"=>240,
    "\u0346"=>230,
    "\u0347"=>220,
    "\u0348"=>220,
    "\u0349"=>220,
    "\u034A"=>230,
    "\u034B"=>230,
    "\u034C"=>230,
    "\u034D"=>220,
    "\u034E"=>220,
    "\u0350"=>230,
    "\u0351"=>230,
    "\u0352"=>230,
    "\u0353"=>220,
    "\u0354"=>220,
    "\u0355"=>220,
    "\u0356"=>220,
    "\u0357"=>230,
    "\u0358"=>232,
    "\u0359"=>220,
    "\u035A"=>220,
    "\u035B"=>230,
    "\u035C"=>233,
    "\u035D"=>234,
    "\u035E"=>234,
    "\u035F"=>233,
    "\u0360"=>234,
    "\u0361"=>234,
    "\u0362"=>233,
    "\u0363"=>230,
    "\u0364"=>230,
    "\u0365"=>230,
    "\u0366"=>230,
    "\u0367"=>230,
    "\u0368"=>230,
    "\u0369"=>230,
    "\u036A"=>230,
    "\u036B"=>230,
    "\u036C"=>230,
    "\u036D"=>230,
    "\u036E"=>230,
    "\u036F"=>230,
    "\u0483"=>230,
    "\u0484"=>230,
    "\u0485"=>230,
    "\u0486"=>230,
    "\u0487"=>230,
    "\u0591"=>220,
    "\u0592"=>230,
    "\u0593"=>230,
    "\u0594"=>230,
    "\u0595"=>230,
    "\u0596"=>220,
    "\u0597"=>230,
    "\u0598"=>230,
    "\u0599"=>230,
    "\u059A"=>222,
    "\u059B"=>220,
    "\u059C"=>230,
    "\u059D"=>230,
    "\u059E"=>230,
    "\u059F"=>230,
    "\u05A0"=>230,
    "\u05A1"=>230,
    "\u05A2"=>220,
    "\u05A3"=>220,
    "\u05A4"=>220,
    "\u05A5"=>220,
    "\u05A6"=>220,
    "\u05A7"=>220,
    "\u05A8"=>230,
    "\u05A9"=>230,
    "\u05AA"=>220,
    "\u05AB"=>230,
    "\u05AC"=>230,
    "\u05AD"=>222,
    "\u05AE"=>228,
    "\u05AF"=>230,
    "\u05B0"=>10,
    "\u05B1"=>11,
    "\u05B2"=>12,
    "\u05B3"=>13,
    "\u05B4"=>14,
    "\u05B5"=>15,
    "\u05B6"=>16,
    "\u05B7"=>17,
    "\u05B8"=>18,
    "\u05B9"=>19,
    "\u05BA"=>19,
    "\u05BB"=>20,
    "\u05BC"=>21,
    "\u05BD"=>22,
    "\u05BF"=>23,
    "\u05C1"=>24,
    "\u05C2"=>25,
    "\u05C4"=>230,
    "\u05C5"=>220,
    "\u05C7"=>18,
    "\u0610"=>230,
    "\u0611"=>230,
    "\u0612"=>230,
    "\u0613"=>230,
    "\u0614"=>230,
    "\u0615"=>230,
    "\u0616"=>230,
    "\u0617"=>230,
    "\u0618"=>30,
    "\u0619"=>31,
    "\u061A"=>32,
    "\u064B"=>27,
    "\u064C"=>28,
    "\u064D"=>29,
    "\u064E"=>30,
    "\u064F"=>31,
    "\u0650"=>32,
    "\u0651"=>33,
    "\u0652"=>34,
    "\u0653"=>230,
    "\u0654"=>230,
    "\u0655"=>220,
    "\u0656"=>220,
    "\u0657"=>230,
    "\u0658"=>230,
    "\u0659"=>230,
    "\u065A"=>230,
    "\u065B"=>230,
    "\u065C"=>220,
    "\u065D"=>230,
    "\u065E"=>230,
    "\u065F"=>220,
    "\u0670"=>35,
    "\u06D6"=>230,
    "\u06D7"=>230,
    "\u06D8"=>230,
    "\u06D9"=>230,
    "\u06DA"=>230,
    "\u06DB"=>230,
    "\u06DC"=>230,
    "\u06DF"=>230,
    "\u06E0"=>230,
    "\u06E1"=>230,
    "\u06E2"=>230,
    "\u06E3"=>220,
    "\u06E4"=>230,
    "\u06E7"=>230,
    "\u06E8"=>230,
    "\u06EA"=>220,
    "\u06EB"=>230,
    "\u06EC"=>230,
    "\u06ED"=>220,
    "\u0711"=>36,
    "\u0730"=>230,
    "\u0731"=>220,
    "\u0732"=>230,
    "\u0733"=>230,
    "\u0734"=>220,
    "\u0735"=>230,
    "\u0736"=>230,
    "\u0737"=>220,
    "\u0738"=>220,
    "\u0739"=>220,
    "\u073A"=>230,
    "\u073B"=>220,
    "\u073C"=>220,
    "\u073D"=>230,
    "\u073E"=>220,
    "\u073F"=>230,
    "\u0740"=>230,
    "\u0741"=>230,
    "\u0742"=>220,
    "\u0743"=>230,
    "\u0744"=>220,
    "\u0745"=>230,
    "\u0746"=>220,
    "\u0747"=>230,
    "\u0748"=>220,
    "\u0749"=>230,
    "\u074A"=>230,
    "\u07EB"=>230,
    "\u07EC"=>230,
    "\u07ED"=>230,
    "\u07EE"=>230,
    "\u07EF"=>230,
    "\u07F0"=>230,
    "\u07F1"=>230,
    "\u07F2"=>220,
    "\u07F3"=>230,
    "\u07FD"=>220,
    "\u0816"=>230,
    "\u0817"=>230,
    "\u0818"=>230,
    "\u0819"=>230,
    "\u081B"=>230,
    "\u081C"=>230,
    "\u081D"=>230,
    "\u081E"=>230,
    "\u081F"=>230,
    "\u0820"=>230,
    "\u0821"=>230,
    "\u0822"=>230,
    "\u0823"=>230,
    "\u0825"=>230,
    "\u0826"=>230,
    "\u0827"=>230,
    "\u0829"=>230,
    "\u082A"=>230,
    "\u082B"=>230,
    "\u082C"=>230,
    "\u082D"=>230,
    "\u0859"=>220,
    "\u085A"=>220,
    "\u085B"=>220,
    "\u0898"=>230,
    "\u0899"=>220,
    "\u089A"=>220,
    "\u089B"=>220,
    "\u089C"=>230,
    "\u089D"=>230,
    "\u089E"=>230,
    "\u089F"=>230,
    "\u08CA"=>230,
    "\u08CB"=>230,
    "\u08CC"=>230,
    "\u08CD"=>230,
    "\u08CE"=>230,
    "\u08CF"=>220,
    "\u08D0"=>220,
    "\u08D1"=>220,
    "\u08D2"=>220,
    "\u08D3"=>220,
    "\u08D4"=>230,
    "\u08D5"=>230,
    "\u08D6"=>230,
    "\u08D7"=>230,
    "\u08D8"=>230,
    "\u08D9"=>230,
    "\u08DA"=>230,
    "\u08DB"=>230,
    "\u08DC"=>230,
    "\u08DD"=>230,
    "\u08DE"=>230,
    "\u08DF"=>230,
    "\u08E0"=>230,
    "\u08E1"=>230,
    "\u08E3"=>220,
    "\u08E4"=>230,
    "\u08E5"=>230,
    "\u08E6"=>220,
    "\u08E7"=>230,
    "\u08E8"=>230,
    "\u08E9"=>220,
    "\u08EA"=>230,
    "\u08EB"=>230,
    "\u08EC"=>230,
    "\u08ED"=>220,
    "\u08EE"=>220,
    "\u08EF"=>220,
    "\u08F0"=>27,
    "\u08F1"=>28,
    "\u08F2"=>29,
    "\u08F3"=>230,
    "\u08F4"=>230,
    "\u08F5"=>230,
    "\u08F6"=>220,
    "\u08F7"=>230,
    "\u08F8"=>230,
    "\u08F9"=>220,
    "\u08FA"=>220,
    "\u08FB"=>230,
    "\u08FC"=>230,
    "\u08FD"=>230,
    "\u08FE"=>230,
    "\u08FF"=>230,
    "\u093C"=>7,
    "\u094D"=>9,
    "\u0951"=>230,
    "\u0952"=>220,
    "\u0953"=>230,
    "\u0954"=>230,
    "\u09BC"=>7,
    "\u09CD"=>9,
    "\u09FE"=>230,
    "\u0A3C"=>7,
    "\u0A4D"=>9,
    "\u0ABC"=>7,
    "\u0ACD"=>9,
    "\u0B3C"=>7,
    "\u0B4D"=>9,
    "\u0BCD"=>9,
    "\u0C3C"=>7,
    "\u0C4D"=>9,
    "\u0C55"=>84,
    "\u0C56"=>91,
    "\u0CBC"=>7,
    "\u0CCD"=>9,
    "\u0D3B"=>9,
    "\u0D3C"=>9,
    "\u0D4D"=>9,
    "\u0DCA"=>9,
    "\u0E38"=>103,
    "\u0E39"=>103,
    "\u0E3A"=>9,
    "\u0E48"=>107,
    "\u0E49"=>107,
    "\u0E4A"=>107,
    "\u0E4B"=>107,
    "\u0EB8"=>118,
    "\u0EB9"=>118,
    "\u0EBA"=>9,
    "\u0EC8"=>122,
    "\u0EC9"=>122,
    "\u0ECA"=>122,
    "\u0ECB"=>122,
    "\u0F18"=>220,
    "\u0F19"=>220,
    "\u0F35"=>220,
    "\u0F37"=>220,
    "\u0F39"=>216,
    "\u0F71"=>129,
    "\u0F72"=>130,
    "\u0F74"=>132,
    "\u0F7A"=>130,
    "\u0F7B"=>130,
    "\u0F7C"=>130,
    "\u0F7D"=>130,
    "\u0F80"=>130,
    "\u0F82"=>230,
    "\u0F83"=>230,
    "\u0F84"=>9,
    "\u0F86"=>230,
    "\u0F87"=>230,
    "\u0FC6"=>220,
    "\u1037"=>7,
    "\u1039"=>9,
    "\u103A"=>9,
    "\u108D"=>220,
    "\u135D"=>230,
    "\u135E"=>230,
    "\u135F"=>230,
    "\u1714"=>9,
    "\u1715"=>9,
    "\u1734"=>9,
    "\u17D2"=>9,
    "\u17DD"=>230,
    "\u18A9"=>228,
    "\u1939"=>222,
    "\u193A"=>230,
    "\u193B"=>220,
    "\u1A17"=>230,
    "\u1A18"=>220,
    "\u1A60"=>9,
    "\u1A75"=>230,
    "\u1A76"=>230,
    "\u1A77"=>230,
    "\u1A78"=>230,
    "\u1A79"=>230,
    "\u1A7A"=>230,
    "\u1A7B"=>230,
    "\u1A7C"=>230,
    "\u1A7F"=>220,
    "\u1AB0"=>230,
    "\u1AB1"=>230,
    "\u1AB2"=>230,
    "\u1AB3"=>230,
    "\u1AB4"=>230,
    "\u1AB5"=>220,
    "\u1AB6"=>220,
    "\u1AB7"=>220,
    "\u1AB8"=>220,
    "\u1AB9"=>220,
    "\u1ABA"=>220,
    "\u1ABB"=>230,
    "\u1ABC"=>230,
    "\u1ABD"=>220,
    "\u1ABF"=>220,
    "\u1AC0"=>220,
    "\u1AC1"=>230,
    "\u1AC2"=>230,
    "\u1AC3"=>220,
    "\u1AC4"=>220,
    "\u1AC5"=>230,
    "\u1AC6"=>230,
    "\u1AC7"=>230,
    "\u1AC8"=>230,
    "\u1AC9"=>230,
    "\u1ACA"=>220,
    "\u1ACB"=>230,
    "\u1ACC"=>230,
    "\u1ACD"=>230,
    "\u1ACE"=>230,
    "\u1B34"=>7,
    "\u1B44"=>9,
    "\u1B6B"=>230,
    "\u1B6C"=>220,
    "\u1B6D"=>230,
    "\u1B6E"=>230,
    "\u1B6F"=>230,
    "\u1B70"=>230,
    "\u1B71"=>230,
    "\u1B72"=>230,
    "\u1B73"=>230,
    "\u1BAA"=>9,
    "\u1BAB"=>9,
    "\u1BE6"=>7,
    "\u1BF2"=>9,
    "\u1BF3"=>9,
    "\u1C37"=>7,
    "\u1CD0"=>230,
    "\u1CD1"=>230,
    "\u1CD2"=>230,
    "\u1CD4"=>1,
    "\u1CD5"=>220,
    "\u1CD6"=>220,
    "\u1CD7"=>220,
    "\u1CD8"=>220,
    "\u1CD9"=>220,
    "\u1CDA"=>230,
    "\u1CDB"=>230,
    "\u1CDC"=>220,
    "\u1CDD"=>220,
    "\u1CDE"=>220,
    "\u1CDF"=>220,
    "\u1CE0"=>230,
    "\u1CE2"=>1,
    "\u1CE3"=>1,
    "\u1CE4"=>1,
    "\u1CE5"=>1,
    "\u1CE6"=>1,
    "\u1CE7"=>1,
    "\u1CE8"=>1,
    "\u1CED"=>220,
    "\u1CF4"=>230,
    "\u1CF8"=>230,
    "\u1CF9"=>230,
    "\u1DC0"=>230,
    "\u1DC1"=>230,
    "\u1DC2"=>220,
    "\u1DC3"=>230,
    "\u1DC4"=>230,
    "\u1DC5"=>230,
    "\u1DC6"=>230,
    "\u1DC7"=>230,
    "\u1DC8"=>230,
    "\u1DC9"=>230,
    "\u1DCA"=>220,
    "\u1DCB"=>230,
    "\u1DCC"=>230,
    "\u1DCD"=>234,
    "\u1DCE"=>214,
    "\u1DCF"=>220,
    "\u1DD0"=>202,
    "\u1DD1"=>230,
    "\u1DD2"=>230,
    "\u1DD3"=>230,
    "\u1DD4"=>230,
    "\u1DD5"=>230,
    "\u1DD6"=>230,
    "\u1DD7"=>230,
    "\u1DD8"=>230,
    "\u1DD9"=>230,
    "\u1DDA"=>230,
    "\u1DDB"=>230,
    "\u1DDC"=>230,
    "\u1DDD"=>230,
    "\u1DDE"=>230,
    "\u1DDF"=>230,
    "\u1DE0"=>230,
    "\u1DE1"=>230,
    "\u1DE2"=>230,
    "\u1DE3"=>230,
    "\u1DE4"=>230,
    "\u1DE5"=>230,
    "\u1DE6"=>230,
    "\u1DE7"=>230,
    "\u1DE8"=>230,
    "\u1DE9"=>230,
    "\u1DEA"=>230,
    "\u1DEB"=>230,
    "\u1DEC"=>230,
    "\u1DED"=>230,
    "\u1DEE"=>230,
    "\u1DEF"=>230,
    "\u1DF0"=>230,
    "\u1DF1"=>230,
    "\u1DF2"=>230,
    "\u1DF3"=>230,
    "\u1DF4"=>230,
    "\u1DF5"=>230,
    "\u1DF6"=>232,
    "\u1DF7"=>228,
    "\u1DF8"=>228,
    "\u1DF9"=>220,
    "\u1DFA"=>218,
    "\u1DFB"=>230,
    "\u1DFC"=>233,
    "\u1DFD"=>220,
    "\u1DFE"=>230,
    "\u1DFF"=>220,
    "\u20D0"=>230,
    "\u20D1"=>230,
    "\u20D2"=>1,
    "\u20D3"=>1,
    "\u20D4"=>230,
    "\u20D5"=>230,
    "\u20D6"=>230,
    "\u20D7"=>230,
    "\u20D8"=>1,
    "\u20D9"=>1,
    "\u20DA"=>1,
    "\u20DB"=>230,
    "\u20DC"=>230,
    "\u20E1"=>230,
    "\u20E5"=>1,
    "\u20E6"=>1,
    "\u20E7"=>230,
    "\u20E8"=>220,
    "\u20E9"=>230,
    "\u20EA"=>1,
    "\u20EB"=>1,
    "\u20EC"=>220,
    "\u20ED"=>220,
    "\u20EE"=>220,
    "\u20EF"=>220,
    "\u20F0"=>230,
    "\u2CEF"=>230,
    "\u2CF0"=>230,
    "\u2CF1"=>230,
    "\u2D7F"=>9,
    "\u2DE0"=>230,
    "\u2DE1"=>230,
    "\u2DE2"=>230,
    "\u2DE3"=>230,
    "\u2DE4"=>230,
    "\u2DE5"=>230,
    "\u2DE6"=>230,
    "\u2DE7"=>230,
    "\u2DE8"=>230,
    "\u2DE9"=>230,
    "\u2DEA"=>230,
    "\u2DEB"=>230,
    "\u2DEC"=>230,
    "\u2DED"=>230,
    "\u2DEE"=>230,
    "\u2DEF"=>230,
    "\u2DF0"=>230,
    "\u2DF1"=>230,
    "\u2DF2"=>230,
    "\u2DF3"=>230,
    "\u2DF4"=>230,
    "\u2DF5"=>230,
    "\u2DF6"=>230,
    "\u2DF7"=>230,
    "\u2DF8"=>230,
    "\u2DF9"=>230,
    "\u2DFA"=>230,
    "\u2DFB"=>230,
    "\u2DFC"=>230,
    "\u2DFD"=>230,
    "\u2DFE"=>230,
    "\u2DFF"=>230,
    "\u302A"=>218,
    "\u302B"=>228,
    "\u302C"=>232,
    "\u302D"=>222,
    "\u302E"=>224,
    "\u302F"=>224,
    "\u3099"=>8,
    "\u309A"=>8,
    "\uA66F"=>230,
    "\uA674"=>230,
    "\uA675"=>230,
    "\uA676"=>230,
    "\uA677"=>230,
    "\uA678"=>230,
    "\uA679"=>230,
    "\uA67A"=>230,
    "\uA67B"=>230,
    "\uA67C"=>230,
    "\uA67D"=>230,
    "\uA69E"=>230,
    "\uA69F"=>230,
    "\uA6F0"=>230,
    "\uA6F1"=>230,
    "\uA806"=>9,
    "\uA82C"=>9,
    "\uA8C4"=>9,
    "\uA8E0"=>230,
    "\uA8E1"=>230,
    "\uA8E2"=>230,
    "\uA8E3"=>230,
    "\uA8E4"=>230,
    "\uA8E5"=>230,
    "\uA8E6"=>230,
    "\uA8E7"=>230,
    "\uA8E8"=>230,
    "\uA8E9"=>230,
    "\uA8EA"=>230,
    "\uA8EB"=>230,
    "\uA8EC"=>230,
    "\uA8ED"=>230,
    "\uA8EE"=>230,
    "\uA8EF"=>230,
    "\uA8F0"=>230,
    "\uA8F1"=>230,
    "\uA92B"=>220,
    "\uA92C"=>220,
    "\uA92D"=>220,
    "\uA953"=>9,
    "\uA9B3"=>7,
    "\uA9C0"=>9,
    "\uAAB0"=>230,
    "\uAAB2"=>230,
    "\uAAB3"=>230,
    "\uAAB4"=>220,
    "\uAAB7"=>230,
    "\uAAB8"=>230,
    "\uAABE"=>230,
    "\uAABF"=>230,
    "\uAAC1"=>230,
    "\uAAF6"=>9,
    "\uABED"=>9,
    "\uFB1E"=>26,
    "\uFE20"=>230,
    "\uFE21"=>230,
    "\uFE22"=>230,
    "\uFE23"=>230,
    "\uFE24"=>230,
    "\uFE25"=>230,
    "\uFE26"=>230,
    "\uFE27"=>220,
    "\uFE28"=>220,
    "\uFE29"=>220,
    "\uFE2A"=>220,
    "\uFE2B"=>220,
    "\uFE2C"=>220,
    "\uFE2D"=>220,
    "\uFE2E"=>230,
    "\uFE2F"=>230,
    "\u{101FD}"=>220,
    "\u{102E0}"=>220,
    "\u{10376}"=>230,
    "\u{10377}"=>230,
    "\u{10378}"=>230,
    "\u{10379}"=>230,
    "\u{1037A}"=>230,
    "\u{10A0D}"=>220,
    "\u{10A0F}"=>230,
    "\u{10A38}"=>230,
    "\u{10A39}"=>1,
    "\u{10A3A}"=>220,
    "\u{10A3F}"=>9,
    "\u{10AE5}"=>230,
    "\u{10AE6}"=>220,
    "\u{10D24}"=>230,
    "\u{10D25}"=>230,
    "\u{10D26}"=>230,
    "\u{10D27}"=>230,
    "\u{10EAB}"=>230,
    "\u{10EAC}"=>230,
    "\u{10EFD}"=>220,
    "\u{10EFE}"=>220,
    "\u{10EFF}"=>220,
    "\u{10F46}"=>220,
    "\u{10F47}"=>220,
    "\u{10F48}"=>230,
    "\u{10F49}"=>230,
    "\u{10F4A}"=>230,
    "\u{10F4B}"=>220,
    "\u{10F4C}"=>230,
    "\u{10F4D}"=>220,
    "\u{10F4E}"=>220,
    "\u{10F4F}"=>220,
    "\u{10F50}"=>220,
    "\u{10F82}"=>230,
    "\u{10F83}"=>220,
    "\u{10F84}"=>230,
    "\u{10F85}"=>220,
    "\u{11046}"=>9,
    "\u{11070}"=>9,
    "\u{1107F}"=>9,
    "\u{110B9}"=>9,
    "\u{110BA}"=>7,
    "\u{11100}"=>230,
    "\u{11101}"=>230,
    "\u{11102}"=>230,
    "\u{11133}"=>9,
    "\u{11134}"=>9,
    "\u{11173}"=>7,
    "\u{111C0}"=>9,
    "\u{111CA}"=>7,
    "\u{11235}"=>9,
    "\u{11236}"=>7,
    "\u{112E9}"=>7,
    "\u{112EA}"=>9,
    "\u{1133B}"=>7,
    "\u{1133C}"=>7,
    "\u{1134D}"=>9,
    "\u{11366}"=>230,
    "\u{11367}"=>230,
    "\u{11368}"=>230,
    "\u{11369}"=>230,
    "\u{1136A}"=>230,
    "\u{1136B}"=>230,
    "\u{1136C}"=>230,
    "\u{11370}"=>230,
    "\u{11371}"=>230,
    "\u{11372}"=>230,
    "\u{11373}"=>230,
    "\u{11374}"=>230,
    "\u{11442}"=>9,
    "\u{11446}"=>7,
    "\u{1145E}"=>230,
    "\u{114C2}"=>9,
    "\u{114C3}"=>7,
    "\u{115BF}"=>9,
    "\u{115C0}"=>7,
    "\u{1163F}"=>9,
    "\u{116B6}"=>9,
    "\u{116B7}"=>7,
    "\u{1172B}"=>9,
    "\u{11839}"=>9,
    "\u{1183A}"=>7,
    "\u{1193D}"=>9,
    "\u{1193E}"=>9,
    "\u{11943}"=>7,
    "\u{119E0}"=>9,
    "\u{11A34}"=>9,
    "\u{11A47}"=>9,
    "\u{11A99}"=>9,
    "\u{11C3F}"=>9,
    "\u{11D42}"=>7,
    "\u{11D44}"=>9,
    "\u{11D45}"=>9,
    "\u{11D97}"=>9,
    "\u{11F41}"=>9,
    "\u{11F42}"=>9,
    "\u{16AF0}"=>1,
    "\u{16AF1}"=>1,
    "\u{16AF2}"=>1,
    "\u{16AF3}"=>1,
    "\u{16AF4}"=>1,
    "\u{16B30}"=>230,
    "\u{16B31}"=>230,
    "\u{16B32}"=>230,
    "\u{16B33}"=>230,
    "\u{16B34}"=>230,
    "\u{16B35}"=>230,
    "\u{16B36}"=>230,
    "\u{16FF0}"=>6,
    "\u{16FF1}"=>6,
    "\u{1BC9E}"=>1,
    "\u{1D165}"=>216,
    "\u{1D166}"=>216,
    "\u{1D167}"=>1,
    "\u{1D168}"=>1,
    "\u{1D169}"=>1,
    "\u{1D16D}"=>226,
    "\u{1D16E}"=>216,
    "\u{1D16F}"=>216,
    "\u{1D170}"=>216,
    "\u{1D171}"=>216,
    "\u{1D172}"=>216,
    "\u{1D17B}"=>220,
    "\u{1D17C}"=>220,
    "\u{1D17D}"=>220,
    "\u{1D17E}"=>220,
    "\u{1D17F}"=>220,
    "\u{1D180}"=>220,
    "\u{1D181}"=>220,
    "\u{1D182}"=>220,
    "\u{1D185}"=>230,
    "\u{1D186}"=>230,
    "\u{1D187}"=>230,
    "\u{1D188}"=>230,
    "\u{1D189}"=>230,
    "\u{1D18A}"=>220,
    "\u{1D18B}"=>220,
    "\u{1D1AA}"=>230,
    "\u{1D1AB}"=>230,
    "\u{1D1AC}"=>230,
    "\u{1D1AD}"=>230,
    "\u{1D242}"=>230,
    "\u{1D243}"=>230,
    "\u{1D244}"=>230,
    "\u{1E000}"=>230,
    "\u{1E001}"=>230,
    "\u{1E002}"=>230,
    "\u{1E003}"=>230,
    "\u{1E004}"=>230,
    "\u{1E005}"=>230,
    "\u{1E006}"=>230,
    "\u{1E008}"=>230,
    "\u{1E009}"=>230,
    "\u{1E00A}"=>230,
    "\u{1E00B}"=>230,
    "\u{1E00C}"=>230,
    "\u{1E00D}"=>230,
    "\u{1E00E}"=>230,
    "\u{1E00F}"=>230,
    "\u{1E010}"=>230,
    "\u{1E011}"=>230,
    "\u{1E012}"=>230,
    "\u{1E013}"=>230,
    "\u{1E014}"=>230,
    "\u{1E015}"=>230,
    "\u{1E016}"=>230,
    "\u{1E017}"=>230,
    "\u{1E018}"=>230,
    "\u{1E01B}"=>230,
    "\u{1E01C}"=>230,
    "\u{1E01D}"=>230,
    "\u{1E01E}"=>230,
    "\u{1E01F}"=>230,
    "\u{1E020}"=>230,
    "\u{1E021}"=>230,
    "\u{1E023}"=>230,
    "\u{1E024}"=>230,
    "\u{1E026}"=>230,
    "\u{1E027}"=>230,
    "\u{1E028}"=>230,
    "\u{1E029}"=>230,
    "\u{1E02A}"=>230,
    "\u{1E08F}"=>230,
    "\u{1E130}"=>230,
    "\u{1E131}"=>230,
    "\u{1E132}"=>230,
    "\u{1E133}"=>230,
    "\u{1E134}"=>230,
    "\u{1E135}"=>230,
    "\u{1E136}"=>230,
    "\u{1E2AE}"=>230,
    "\u{1E2EC}"=>230,
    "\u{1E2ED}"=>230,
    "\u{1E2EE}"=>230,
    "\u{1E2EF}"=>230,
    "\u{1E4EC}"=>232,
    "\u{1E4ED}"=>232,
    "\u{1E4EE}"=>220,
    "\u{1E4EF}"=>230,
    "\u{1E8D0}"=>220,
    "\u{1E8D1}"=>220,
    "\u{1E8D2}"=>220,
    "\u{1E8D3}"=>220,
    "\u{1E8D4}"=>220,
    "\u{1E8D5}"=>220,
    "\u{1E8D6}"=>220,
    "\u{1E944}"=>230,
    "\u{1E945}"=>230,
    "\u{1E946}"=>230,
    "\u{1E947}"=>230,
    "\u{1E948}"=>230,
    "\u{1E949}"=>230,
    "\u{1E94A}"=>7,
  }
  class_table.default = 0
  CLASS_TABLE = class_table.freeze

  DECOMPOSITION_TABLE = {
    "\u00C0"=>"A\u0300",
    "\u00C1"=>"A\u0301",
    "\u00C2"=>"A\u0302",
    "\u00C3"=>"A\u0303",
    "\u00C4"=>"A\u0308",
    "\u00C5"=>"A\u030A",
    "\u00C7"=>"C\u0327",
    "\u00C8"=>"E\u0300",
    "\u00C9"=>"E\u0301",
    "\u00CA"=>"E\u0302",
    "\u00CB"=>"E\u0308",
    "\u00CC"=>"I\u0300",
    "\u00CD"=>"I\u0301",
    "\u00CE"=>"I\u0302",
    "\u00CF"=>"I\u0308",
    "\u00D1"=>"N\u0303",
    "\u00D2"=>"O\u0300",
    "\u00D3"=>"O\u0301",
    "\u00D4"=>"O\u0302",
    "\u00D5"=>"O\u0303",
    "\u00D6"=>"O\u0308",
    "\u00D9"=>"U\u0300",
    "\u00DA"=>"U\u0301",
    "\u00DB"=>"U\u0302",
    "\u00DC"=>"U\u0308",
    "\u00DD"=>"Y\u0301",
    "\u00E0"=>"a\u0300",
    "\u00E1"=>"a\u0301",
    "\u00E2"=>"a\u0302",
    "\u00E3"=>"a\u0303",
    "\u00E4"=>"a\u0308",
    "\u00E5"=>"a\u030A",
    "\u00E7"=>"c\u0327",
    "\u00E8"=>"e\u0300",
    "\u00E9"=>"e\u0301",
    "\u00EA"=>"e\u0302",
    "\u00EB"=>"e\u0308",
    "\u00EC"=>"i\u0300",
    "\u00ED"=>"i\u0301",
    "\u00EE"=>"i\u0302",
    "\u00EF"=>"i\u0308",
    "\u00F1"=>"n\u0303",
    "\u00F2"=>"o\u0300",
    "\u00F3"=>"o\u0301",
    "\u00F4"=>"o\u0302",
    "\u00F5"=>"o\u0303",
    "\u00F6"=>"o\u0308",
    "\u00F9"=>"u\u0300",
    "\u00FA"=>"u\u0301",
    "\u00FB"=>"u\u0302",
    "\u00FC"=>"u\u0308",
    "\u00FD"=>"y\u0301",
    "\u00FF"=>"y\u0308",
    "\u0100"=>"A\u0304",
    "\u0101"=>"a\u0304",
    "\u0102"=>"A\u0306",
    "\u0103"=>"a\u0306",
    "\u0104"=>"A\u0328",
    "\u0105"=>"a\u0328",
    "\u0106"=>"C\u0301",
    "\u0107"=>"c\u0301",
    "\u0108"=>"C\u0302",
    "\u0109"=>"c\u0302",
    "\u010A"=>"C\u0307",
    "\u010B"=>"c\u0307",
    "\u010C"=>"C\u030C",
    "\u010D"=>"c\u030C",
    "\u010E"=>"D\u030C",
    "\u010F"=>"d\u030C",
    "\u0112"=>"E\u0304",
    "\u0113"=>"e\u0304",
    "\u0114"=>"E\u0306",
    "\u0115"=>"e\u0306",
    "\u0116"=>"E\u0307",
    "\u0117"=>"e\u0307",
    "\u0118"=>"E\u0328",
    "\u0119"=>"e\u0328",
    "\u011A"=>"E\u030C",
    "\u011B"=>"e\u030C",
    "\u011C"=>"G\u0302",
    "\u011D"=>"g\u0302",
    "\u011E"=>"G\u0306",
    "\u011F"=>"g\u0306",
    "\u0120"=>"G\u0307",
    "\u0121"=>"g\u0307",
    "\u0122"=>"G\u0327",
    "\u0123"=>"g\u0327",
    "\u0124"=>"H\u0302",
    "\u0125"=>"h\u0302",
    "\u0128"=>"I\u0303",
    "\u0129"=>"i\u0303",
    "\u012A"=>"I\u0304",
    "\u012B"=>"i\u0304",
    "\u012C"=>"I\u0306",
    "\u012D"=>"i\u0306",
    "\u012E"=>"I\u0328",
    "\u012F"=>"i\u0328",
    "\u0130"=>"I\u0307",
    "\u0134"=>"J\u0302",
    "\u0135"=>"j\u0302",
    "\u0136"=>"K\u0327",
    "\u0137"=>"k\u0327",
    "\u0139"=>"L\u0301",
    "\u013A"=>"l\u0301",
    "\u013B"=>"L\u0327",
    "\u013C"=>"l\u0327",
    "\u013D"=>"L\u030C",
    "\u013E"=>"l\u030C",
    "\u0143"=>"N\u0301",
    "\u0144"=>"n\u0301",
    "\u0145"=>"N\u0327",
    "\u0146"=>"n\u0327",
    "\u0147"=>"N\u030C",
    "\u0148"=>"n\u030C",
    "\u014C"=>"O\u0304",
    "\u014D"=>"o\u0304",
    "\u014E"=>"O\u0306",
    "\u014F"=>"o\u0306",
    "\u0150"=>"O\u030B",
    "\u0151"=>"o\u030B",
    "\u0154"=>"R\u0301",
    "\u0155"=>"r\u0301",
    "\u0156"=>"R\u0327",
    "\u0157"=>"r\u0327",
    "\u0158"=>"R\u030C",
    "\u0159"=>"r\u030C",
    "\u015A"=>"S\u0301",
    "\u015B"=>"s\u0301",
    "\u015C"=>"S\u0302",
    "\u015D"=>"s\u0302",
    "\u015E"=>"S\u0327",
    "\u015F"=>"s\u0327",
    "\u0160"=>"S\u030C",
    "\u0161"=>"s\u030C",
    "\u0162"=>"T\u0327",
    "\u0163"=>"t\u0327",
    "\u0164"=>"T\u030C",
    "\u0165"=>"t\u030C",
    "\u0168"=>"U\u0303",
    "\u0169"=>"u\u0303",
    "\u016A"=>"U\u0304",
    "\u016B"=>"u\u0304",
    "\u016C"=>"U\u0306",
    "\u016D"=>"u\u0306",
    "\u016E"=>"U\u030A",
    "\u016F"=>"u\u030A",
    "\u0170"=>"U\u030B",
    "\u0171"=>"u\u030B",
    "\u0172"=>"U\u0328",
    "\u0173"=>"u\u0328",
    "\u0174"=>"W\u0302",
    "\u0175"=>"w\u0302",
    "\u0176"=>"Y\u0302",
    "\u0177"=>"y\u0302",
    "\u0178"=>"Y\u0308",
    "\u0179"=>"Z\u0301",
    "\u017A"=>"z\u0301",
    "\u017B"=>"Z\u0307",
    "\u017C"=>"z\u0307",
    "\u017D"=>"Z\u030C",
    "\u017E"=>"z\u030C",
    "\u01A0"=>"O\u031B",
    "\u01A1"=>"o\u031B",
    "\u01AF"=>"U\u031B",
    "\u01B0"=>"u\u031B",
    "\u01CD"=>"A\u030C",
    "\u01CE"=>"a\u030C",
    "\u01CF"=>"I\u030C",
    "\u01D0"=>"i\u030C",
    "\u01D1"=>"O\u030C",
    "\u01D2"=>"o\u030C",
    "\u01D3"=>"U\u030C",
    "\u01D4"=>"u\u030C",
    "\u01D5"=>"U\u0308\u0304",
    "\u01D6"=>"u\u0308\u0304",
    "\u01D7"=>"U\u0308\u0301",
    "\u01D8"=>"u\u0308\u0301",
    "\u01D9"=>"U\u0308\u030C",
    "\u01DA"=>"u\u0308\u030C",
    "\u01DB"=>"U\u0308\u0300",
    "\u01DC"=>"u\u0308\u0300",
    "\u01DE"=>"A\u0308\u0304",
    "\u01DF"=>"a\u0308\u0304",
    "\u01E0"=>"A\u0307\u0304",
    "\u01E1"=>"a\u0307\u0304",
    "\u01E2"=>"\u00C6\u0304",
    "\u01E3"=>"\u00E6\u0304",
    "\u01E6"=>"G\u030C",
    "\u01E7"=>"g\u030C",
    "\u01E8"=>"K\u030C",
    "\u01E9"=>"k\u030C",
    "\u01EA"=>"O\u0328",
    "\u01EB"=>"o\u0328",
    "\u01EC"=>"O\u0328\u0304",
    "\u01ED"=>"o\u0328\u0304",
    "\u01EE"=>"\u01B7\u030C",
    "\u01EF"=>"\u0292\u030C",
    "\u01F0"=>"j\u030C",
    "\u01F4"=>"G\u0301",
    "\u01F5"=>"g\u0301",
    "\u01F8"=>"N\u0300",
    "\u01F9"=>"n\u0300",
    "\u01FA"=>"A\u030A\u0301",
    "\u01FB"=>"a\u030A\u0301",
    "\u01FC"=>"\u00C6\u0301",
    "\u01FD"=>"\u00E6\u0301",
    "\u01FE"=>"\u00D8\u0301",
    "\u01FF"=>"\u00F8\u0301",
    "\u0200"=>"A\u030F",
    "\u0201"=>"a\u030F",
    "\u0202"=>"A\u0311",
    "\u0203"=>"a\u0311",
    "\u0204"=>"E\u030F",
    "\u0205"=>"e\u030F",
    "\u0206"=>"E\u0311",
    "\u0207"=>"e\u0311",
    "\u0208"=>"I\u030F",
    "\u0209"=>"i\u030F",
    "\u020A"=>"I\u0311",
    "\u020B"=>"i\u0311",
    "\u020C"=>"O\u030F",
    "\u020D"=>"o\u030F",
    "\u020E"=>"O\u0311",
    "\u020F"=>"o\u0311",
    "\u0210"=>"R\u030F",
    "\u0211"=>"r\u030F",
    "\u0212"=>"R\u0311",
    "\u0213"=>"r\u0311",
    "\u0214"=>"U\u030F",
    "\u0215"=>"u\u030F",
    "\u0216"=>"U\u0311",
    "\u0217"=>"u\u0311",
    "\u0218"=>"S\u0326",
    "\u0219"=>"s\u0326",
    "\u021A"=>"T\u0326",
    "\u021B"=>"t\u0326",
    "\u021E"=>"H\u030C",
    "\u021F"=>"h\u030C",
    "\u0226"=>"A\u0307",
    "\u0227"=>"a\u0307",
    "\u0228"=>"E\u0327",
    "\u0229"=>"e\u0327",
    "\u022A"=>"O\u0308\u0304",
    "\u022B"=>"o\u0308\u0304",
    "\u022C"=>"O\u0303\u0304",
    "\u022D"=>"o\u0303\u0304",
    "\u022E"=>"O\u0307",
    "\u022F"=>"o\u0307",
    "\u0230"=>"O\u0307\u0304",
    "\u0231"=>"o\u0307\u0304",
    "\u0232"=>"Y\u0304",
    "\u0233"=>"y\u0304",
    "\u0340"=>"\u0300",
    "\u0341"=>"\u0301",
    "\u0343"=>"\u0313",
    "\u0344"=>"\u0308\u0301",
    "\u0374"=>"\u02B9",
    "\u037E"=>";",
    "\u0385"=>"\u00A8\u0301",
    "\u0386"=>"\u0391\u0301",
    "\u0387"=>"\u00B7",
    "\u0388"=>"\u0395\u0301",
    "\u0389"=>"\u0397\u0301",
    "\u038A"=>"\u0399\u0301",
    "\u038C"=>"\u039F\u0301",
    "\u038E"=>"\u03A5\u0301",
    "\u038F"=>"\u03A9\u0301",
    "\u0390"=>"\u03B9\u0308\u0301",
    "\u03AA"=>"\u0399\u0308",
    "\u03AB"=>"\u03A5\u0308",
    "\u03AC"=>"\u03B1\u0301",
    "\u03AD"=>"\u03B5\u0301",
    "\u03AE"=>"\u03B7\u0301",
    "\u03AF"=>"\u03B9\u0301",
    "\u03B0"=>"\u03C5\u0308\u0301",
    "\u03CA"=>"\u03B9\u0308",
    "\u03CB"=>"\u03C5\u0308",
    "\u03CC"=>"\u03BF\u0301",
    "\u03CD"=>"\u03C5\u0301",
    "\u03CE"=>"\u03C9\u0301",
    "\u03D3"=>"\u03D2\u0301",
    "\u03D4"=>"\u03D2\u0308",
    "\u0400"=>"\u0415\u0300",
    "\u0401"=>"\u0415\u0308",
    "\u0403"=>"\u0413\u0301",
    "\u0407"=>"\u0406\u0308",
    "\u040C"=>"\u041A\u0301",
    "\u040D"=>"\u0418\u0300",
    "\u040E"=>"\u0423\u0306",
    "\u0419"=>"\u0418\u0306",
    "\u0439"=>"\u0438\u0306",
    "\u0450"=>"\u0435\u0300",
    "\u0451"=>"\u0435\u0308",
    "\u0453"=>"\u0433\u0301",
    "\u0457"=>"\u0456\u0308",
    "\u045C"=>"\u043A\u0301",
    "\u045D"=>"\u0438\u0300",
    "\u045E"=>"\u0443\u0306",
    "\u0476"=>"\u0474\u030F",
    "\u0477"=>"\u0475\u030F",
    "\u04C1"=>"\u0416\u0306",
    "\u04C2"=>"\u0436\u0306",
    "\u04D0"=>"\u0410\u0306",
    "\u04D1"=>"\u0430\u0306",
    "\u04D2"=>"\u0410\u0308",
    "\u04D3"=>"\u0430\u0308",
    "\u04D6"=>"\u0415\u0306",
    "\u04D7"=>"\u0435\u0306",
    "\u04DA"=>"\u04D8\u0308",
    "\u04DB"=>"\u04D9\u0308",
    "\u04DC"=>"\u0416\u0308",
    "\u04DD"=>"\u0436\u0308",
    "\u04DE"=>"\u0417\u0308",
    "\u04DF"=>"\u0437\u0308",
    "\u04E2"=>"\u0418\u0304",
    "\u04E3"=>"\u0438\u0304",
    "\u04E4"=>"\u0418\u0308",
    "\u04E5"=>"\u0438\u0308",
    "\u04E6"=>"\u041E\u0308",
    "\u04E7"=>"\u043E\u0308",
    "\u04EA"=>"\u04E8\u0308",
    "\u04EB"=>"\u04E9\u0308",
    "\u04EC"=>"\u042D\u0308",
    "\u04ED"=>"\u044D\u0308",
    "\u04EE"=>"\u0423\u0304",
    "\u04EF"=>"\u0443\u0304",
    "\u04F0"=>"\u0423\u0308",
    "\u04F1"=>"\u0443\u0308",
    "\u04F2"=>"\u0423\u030B",
    "\u04F3"=>"\u0443\u030B",
    "\u04F4"=>"\u0427\u0308",
    "\u04F5"=>"\u0447\u0308",
    "\u04F8"=>"\u042B\u0308",
    "\u04F9"=>"\u044B\u0308",
    "\u0622"=>"\u0627\u0653",
    "\u0623"=>"\u0627\u0654",
    "\u0624"=>"\u0648\u0654",
    "\u0625"=>"\u0627\u0655",
    "\u0626"=>"\u064A\u0654",
    "\u06C0"=>"\u06D5\u0654",
    "\u06C2"=>"\u06C1\u0654",
    "\u06D3"=>"\u06D2\u0654",
    "\u0929"=>"\u0928\u093C",
    "\u0931"=>"\u0930\u093C",
    "\u0934"=>"\u0933\u093C",
    "\u0958"=>"\u0915\u093C",
    "\u0959"=>"\u0916\u093C",
    "\u095A"=>"\u0917\u093C",
    "\u095B"=>"\u091C\u093C",
    "\u095C"=>"\u0921\u093C",
    "\u095D"=>"\u0922\u093C",
    "\u095E"=>"\u092B\u093C",
    "\u095F"=>"\u092F\u093C",
    "\u09CB"=>"\u09C7\u09BE",
    "\u09CC"=>"\u09C7\u09D7",
    "\u09DC"=>"\u09A1\u09BC",
    "\u09DD"=>"\u09A2\u09BC",
    "\u09DF"=>"\u09AF\u09BC",
    "\u0A33"=>"\u0A32\u0A3C",
    "\u0A36"=>"\u0A38\u0A3C",
    "\u0A59"=>"\u0A16\u0A3C",
    "\u0A5A"=>"\u0A17\u0A3C",
    "\u0A5B"=>"\u0A1C\u0A3C",
    "\u0A5E"=>"\u0A2B\u0A3C",
    "\u0B48"=>"\u0B47\u0B56",
    "\u0B4B"=>"\u0B47\u0B3E",
    "\u0B4C"=>"\u0B47\u0B57",
    "\u0B5C"=>"\u0B21\u0B3C",
    "\u0B5D"=>"\u0B22\u0B3C",
    "\u0B94"=>"\u0B92\u0BD7",
    "\u0BCA"=>"\u0BC6\u0BBE",
    "\u0BCB"=>"\u0BC7\u0BBE",
    "\u0BCC"=>"\u0BC6\u0BD7",
    "\u0C48"=>"\u0C46\u0C56",
    "\u0CC0"=>"\u0CBF\u0CD5",
    "\u0CC7"=>"\u0CC6\u0CD5",
    "\u0CC8"=>"\u0CC6\u0CD6",
    "\u0CCA"=>"\u0CC6\u0CC2",
    "\u0CCB"=>"\u0CC6\u0CC2\u0CD5",
    "\u0D4A"=>"\u0D46\u0D3E",
    "\u0D4B"=>"\u0D47\u0D3E",
    "\u0D4C"=>"\u0D46\u0D57",
    "\u0DDA"=>"\u0DD9\u0DCA",
    "\u0DDC"=>"\u0DD9\u0DCF",
    "\u0DDD"=>"\u0DD9\u0DCF\u0DCA",
    "\u0DDE"=>"\u0DD9\u0DDF",
    "\u0F43"=>"\u0F42\u0FB7",
    "\u0F4D"=>"\u0F4C\u0FB7",
    "\u0F52"=>"\u0F51\u0FB7",
    "\u0F57"=>"\u0F56\u0FB7",
    "\u0F5C"=>"\u0F5B\u0FB7",
    "\u0F69"=>"\u0F40\u0FB5",
    "\u0F73"=>"\u0F71\u0F72",
    "\u0F75"=>"\u0F71\u0F74",
    "\u0F76"=>"\u0FB2\u0F80",
    "\u0F78"=>"\u0FB3\u0F80",
    "\u0F81"=>"\u0F71\u0F80",
    "\u0F93"=>"\u0F92\u0FB7",
    "\u0F9D"=>"\u0F9C\u0FB7",
    "\u0FA2"=>"\u0FA1\u0FB7",
    "\u0FA7"=>"\u0FA6\u0FB7",
    "\u0FAC"=>"\u0FAB\u0FB7",
    "\u0FB9"=>"\u0F90\u0FB5",
    "\u1026"=>"\u1025\u102E",
    "\u1B06"=>"\u1B05\u1B35",
    "\u1B08"=>"\u1B07\u1B35",
    "\u1B0A"=>"\u1B09\u1B35",
    "\u1B0C"=>"\u1B0B\u1B35",
    "\u1B0E"=>"\u1B0D\u1B35",
    "\u1B12"=>"\u1B11\u1B35",
    "\u1B3B"=>"\u1B3A\u1B35",
    "\u1B3D"=>"\u1B3C\u1B35",
    "\u1B40"=>"\u1B3E\u1B35",
    "\u1B41"=>"\u1B3F\u1B35",
    "\u1B43"=>"\u1B42\u1B35",
    "\u1E00"=>"A\u0325",
    "\u1E01"=>"a\u0325",
    "\u1E02"=>"B\u0307",
    "\u1E03"=>"b\u0307",
    "\u1E04"=>"B\u0323",
    "\u1E05"=>"b\u0323",
    "\u1E06"=>"B\u0331",
    "\u1E07"=>"b\u0331",
    "\u1E08"=>"C\u0327\u0301",
    "\u1E09"=>"c\u0327\u0301",
    "\u1E0A"=>"D\u0307",
    "\u1E0B"=>"d\u0307",
    "\u1E0C"=>"D\u0323",
    "\u1E0D"=>"d\u0323",
    "\u1E0E"=>"D\u0331",
    "\u1E0F"=>"d\u0331",
    "\u1E10"=>"D\u0327",
    "\u1E11"=>"d\u0327",
    "\u1E12"=>"D\u032D",
    "\u1E13"=>"d\u032D",
    "\u1E14"=>"E\u0304\u0300",
    "\u1E15"=>"e\u0304\u0300",
    "\u1E16"=>"E\u0304\u0301",
    "\u1E17"=>"e\u0304\u0301",
    "\u1E18"=>"E\u032D",
    "\u1E19"=>"e\u032D",
    "\u1E1A"=>"E\u0330",
    "\u1E1B"=>"e\u0330",
    "\u1E1C"=>"E\u0327\u0306",
    "\u1E1D"=>"e\u0327\u0306",
    "\u1E1E"=>"F\u0307",
    "\u1E1F"=>"f\u0307",
    "\u1E20"=>"G\u0304",
    "\u1E21"=>"g\u0304",
    "\u1E22"=>"H\u0307",
    "\u1E23"=>"h\u0307",
    "\u1E24"=>"H\u0323",
    "\u1E25"=>"h\u0323",
    "\u1E26"=>"H\u0308",
    "\u1E27"=>"h\u0308",
    "\u1E28"=>"H\u0327",
    "\u1E29"=>"h\u0327",
    "\u1E2A"=>"H\u032E",
    "\u1E2B"=>"h\u032E",
    "\u1E2C"=>"I\u0330",
    "\u1E2D"=>"i\u0330",
    "\u1E2E"=>"I\u0308\u0301",
    "\u1E2F"=>"i\u0308\u0301",
    "\u1E30"=>"K\u0301",
    "\u1E31"=>"k\u0301",
    "\u1E32"=>"K\u0323",
    "\u1E33"=>"k\u0323",
    "\u1E34"=>"K\u0331",
    "\u1E35"=>"k\u0331",
    "\u1E36"=>"L\u0323",
    "\u1E37"=>"l\u0323",
    "\u1E38"=>"L\u0323\u0304",
    "\u1E39"=>"l\u0323\u0304",
    "\u1E3A"=>"L\u0331",
    "\u1E3B"=>"l\u0331",
    "\u1E3C"=>"L\u032D",
    "\u1E3D"=>"l\u032D",
    "\u1E3E"=>"M\u0301",
    "\u1E3F"=>"m\u0301",
    "\u1E40"=>"M\u0307",
    "\u1E41"=>"m\u0307",
    "\u1E42"=>"M\u0323",
    "\u1E43"=>"m\u0323",
    "\u1E44"=>"N\u0307",
    "\u1E45"=>"n\u0307",
    "\u1E46"=>"N\u0323",
    "\u1E47"=>"n\u0323",
    "\u1E48"=>"N\u0331",
    "\u1E49"=>"n\u0331",
    "\u1E4A"=>"N\u032D",
    "\u1E4B"=>"n\u032D",
    "\u1E4C"=>"O\u0303\u0301",
    "\u1E4D"=>"o\u0303\u0301",
    "\u1E4E"=>"O\u0303\u0308",
    "\u1E4F"=>"o\u0303\u0308",
    "\u1E50"=>"O\u0304\u0300",
    "\u1E51"=>"o\u0304\u0300",
    "\u1E52"=>"O\u0304\u0301",
    "\u1E53"=>"o\u0304\u0301",
    "\u1E54"=>"P\u0301",
    "\u1E55"=>"p\u0301",
    "\u1E56"=>"P\u0307",
    "\u1E57"=>"p\u0307",
    "\u1E58"=>"R\u0307",
    "\u1E59"=>"r\u0307",
    "\u1E5A"=>"R\u0323",
    "\u1E5B"=>"r\u0323",
    "\u1E5C"=>"R\u0323\u0304",
    "\u1E5D"=>"r\u0323\u0304",
    "\u1E5E"=>"R\u0331",
    "\u1E5F"=>"r\u0331",
    "\u1E60"=>"S\u0307",
    "\u1E61"=>"s\u0307",
    "\u1E62"=>"S\u0323",
    "\u1E63"=>"s\u0323",
    "\u1E64"=>"S\u0301\u0307",
    "\u1E65"=>"s\u0301\u0307",
    "\u1E66"=>"S\u030C\u0307",
    "\u1E67"=>"s\u030C\u0307",
    "\u1E68"=>"S\u0323\u0307",
    "\u1E69"=>"s\u0323\u0307",
    "\u1E6A"=>"T\u0307",
    "\u1E6B"=>"t\u0307",
    "\u1E6C"=>"T\u0323",
    "\u1E6D"=>"t\u0323",
    "\u1E6E"=>"T\u0331",
    "\u1E6F"=>"t\u0331",
    "\u1E70"=>"T\u032D",
    "\u1E71"=>"t\u032D",
    "\u1E72"=>"U\u0324",
    "\u1E73"=>"u\u0324",
    "\u1E74"=>"U\u0330",
    "\u1E75"=>"u\u0330",
    "\u1E76"=>"U\u032D",
    "\u1E77"=>"u\u032D",
    "\u1E78"=>"U\u0303\u0301",
    "\u1E79"=>"u\u0303\u0301",
    "\u1E7A"=>"U\u0304\u0308",
    "\u1E7B"=>"u\u0304\u0308",
    "\u1E7C"=>"V\u0303",
    "\u1E7D"=>"v\u0303",
    "\u1E7E"=>"V\u0323",
    "\u1E7F"=>"v\u0323",
    "\u1E80"=>"W\u0300",
    "\u1E81"=>"w\u0300",
    "\u1E82"=>"W\u0301",
    "\u1E83"=>"w\u0301",
    "\u1E84"=>"W\u0308",
    "\u1E85"=>"w\u0308",
    "\u1E86"=>"W\u0307",
    "\u1E87"=>"w\u0307",
    "\u1E88"=>"W\u0323",
    "\u1E89"=>"w\u0323",
    "\u1E8A"=>"X\u0307",
    "\u1E8B"=>"x\u0307",
    "\u1E8C"=>"X\u0308",
    "\u1E8D"=>"x\u0308",
    "\u1E8E"=>"Y\u0307",
    "\u1E8F"=>"y\u0307",
    "\u1E90"=>"Z\u0302",
    "\u1E91"=>"z\u0302",
    "\u1E92"=>"Z\u0323",
    "\u1E93"=>"z\u0323",
    "\u1E94"=>"Z\u0331",
    "\u1E95"=>"z\u0331",
    "\u1E96"=>"h\u0331",
    "\u1E97"=>"t\u0308",
    "\u1E98"=>"w\u030A",
    "\u1E99"=>"y\u030A",
    "\u1E9B"=>"\u017F\u0307",
    "\u1EA0"=>"A\u0323",
    "\u1EA1"=>"a\u0323",
    "\u1EA2"=>"A\u0309",
    "\u1EA3"=>"a\u0309",
    "\u1EA4"=>"A\u0302\u0301",
    "\u1EA5"=>"a\u0302\u0301",
    "\u1EA6"=>"A\u0302\u0300",
    "\u1EA7"=>"a\u0302\u0300",
    "\u1EA8"=>"A\u0302\u0309",
    "\u1EA9"=>"a\u0302\u0309",
    "\u1EAA"=>"A\u0302\u0303",
    "\u1EAB"=>"a\u0302\u0303",
    "\u1EAC"=>"A\u0323\u0302",
    "\u1EAD"=>"a\u0323\u0302",
    "\u1EAE"=>"A\u0306\u0301",
    "\u1EAF"=>"a\u0306\u0301",
    "\u1EB0"=>"A\u0306\u0300",
    "\u1EB1"=>"a\u0306\u0300",
    "\u1EB2"=>"A\u0306\u0309",
    "\u1EB3"=>"a\u0306\u0309",
    "\u1EB4"=>"A\u0306\u0303",
    "\u1EB5"=>"a\u0306\u0303",
    "\u1EB6"=>"A\u0323\u0306",
    "\u1EB7"=>"a\u0323\u0306",
    "\u1EB8"=>"E\u0323",
    "\u1EB9"=>"e\u0323",
    "\u1EBA"=>"E\u0309",
    "\u1EBB"=>"e\u0309",
    "\u1EBC"=>"E\u0303",
    "\u1EBD"=>"e\u0303",
    "\u1EBE"=>"E\u0302\u0301",
    "\u1EBF"=>"e\u0302\u0301",
    "\u1EC0"=>"E\u0302\u0300",
    "\u1EC1"=>"e\u0302\u0300",
    "\u1EC2"=>"E\u0302\u0309",
    "\u1EC3"=>"e\u0302\u0309",
    "\u1EC4"=>"E\u0302\u0303",
    "\u1EC5"=>"e\u0302\u0303",
    "\u1EC6"=>"E\u0323\u0302",
    "\u1EC7"=>"e\u0323\u0302",
    "\u1EC8"=>"I\u0309",
    "\u1EC9"=>"i\u0309",
    "\u1ECA"=>"I\u0323",
    "\u1ECB"=>"i\u0323",
    "\u1ECC"=>"O\u0323",
    "\u1ECD"=>"o\u0323",
    "\u1ECE"=>"O\u0309",
    "\u1ECF"=>"o\u0309",
    "\u1ED0"=>"O\u0302\u0301",
    "\u1ED1"=>"o\u0302\u0301",
    "\u1ED2"=>"O\u0302\u0300",
    "\u1ED3"=>"o\u0302\u0300",
    "\u1ED4"=>"O\u0302\u0309",
    "\u1ED5"=>"o\u0302\u0309",
    "\u1ED6"=>"O\u0302\u0303",
    "\u1ED7"=>"o\u0302\u0303",
    "\u1ED8"=>"O\u0323\u0302",
    "\u1ED9"=>"o\u0323\u0302",
    "\u1EDA"=>"O\u031B\u0301",
    "\u1EDB"=>"o\u031B\u0301",
    "\u1EDC"=>"O\u031B\u0300",
    "\u1EDD"=>"o\u031B\u0300",
    "\u1EDE"=>"O\u031B\u0309",
    "\u1EDF"=>"o\u031B\u0309",
    "\u1EE0"=>"O\u031B\u0303",
    "\u1EE1"=>"o\u031B\u0303",
    "\u1EE2"=>"O\u031B\u0323",
    "\u1EE3"=>"o\u031B\u0323",
    "\u1EE4"=>"U\u0323",
    "\u1EE5"=>"u\u0323",
    "\u1EE6"=>"U\u0309",
    "\u1EE7"=>"u\u0309",
    "\u1EE8"=>"U\u031B\u0301",
    "\u1EE9"=>"u\u031B\u0301",
    "\u1EEA"=>"U\u031B\u0300",
    "\u1EEB"=>"u\u031B\u0300",
    "\u1EEC"=>"U\u031B\u0309",
    "\u1EED"=>"u\u031B\u0309",
    "\u1EEE"=>"U\u031B\u0303",
    "\u1EEF"=>"u\u031B\u0303",
    "\u1EF0"=>"U\u031B\u0323",
    "\u1EF1"=>"u\u031B\u0323",
    "\u1EF2"=>"Y\u0300",
    "\u1EF3"=>"y\u0300",
    "\u1EF4"=>"Y\u0323",
    "\u1EF5"=>"y\u0323",
    "\u1EF6"=>"Y\u0309",
    "\u1EF7"=>"y\u0309",
    "\u1EF8"=>"Y\u0303",
    "\u1EF9"=>"y\u0303",
    "\u1F00"=>"\u03B1\u0313",
    "\u1F01"=>"\u03B1\u0314",
    "\u1F02"=>"\u03B1\u0313\u0300",
    "\u1F03"=>"\u03B1\u0314\u0300",
    "\u1F04"=>"\u03B1\u0313\u0301",
    "\u1F05"=>"\u03B1\u0314\u0301",
    "\u1F06"=>"\u03B1\u0313\u0342",
    "\u1F07"=>"\u03B1\u0314\u0342",
    "\u1F08"=>"\u0391\u0313",
    "\u1F09"=>"\u0391\u0314",
    "\u1F0A"=>"\u0391\u0313\u0300",
    "\u1F0B"=>"\u0391\u0314\u0300",
    "\u1F0C"=>"\u0391\u0313\u0301",
    "\u1F0D"=>"\u0391\u0314\u0301",
    "\u1F0E"=>"\u0391\u0313\u0342",
    "\u1F0F"=>"\u0391\u0314\u0342",
    "\u1F10"=>"\u03B5\u0313",
    "\u1F11"=>"\u03B5\u0314",
    "\u1F12"=>"\u03B5\u0313\u0300",
    "\u1F13"=>"\u03B5\u0314\u0300",
    "\u1F14"=>"\u03B5\u0313\u0301",
    "\u1F15"=>"\u03B5\u0314\u0301",
    "\u1F18"=>"\u0395\u0313",
    "\u1F19"=>"\u0395\u0314",
    "\u1F1A"=>"\u0395\u0313\u0300",
    "\u1F1B"=>"\u0395\u0314\u0300",
    "\u1F1C"=>"\u0395\u0313\u0301",
    "\u1F1D"=>"\u0395\u0314\u0301",
    "\u1F20"=>"\u03B7\u0313",
    "\u1F21"=>"\u03B7\u0314",
    "\u1F22"=>"\u03B7\u0313\u0300",
    "\u1F23"=>"\u03B7\u0314\u0300",
    "\u1F24"=>"\u03B7\u0313\u0301",
    "\u1F25"=>"\u03B7\u0314\u0301",
    "\u1F26"=>"\u03B7\u0313\u0342",
    "\u1F27"=>"\u03B7\u0314\u0342",
    "\u1F28"=>"\u0397\u0313",
    "\u1F29"=>"\u0397\u0314",
    "\u1F2A"=>"\u0397\u0313\u0300",
    "\u1F2B"=>"\u0397\u0314\u0300",
    "\u1F2C"=>"\u0397\u0313\u0301",
    "\u1F2D"=>"\u0397\u0314\u0301",
    "\u1F2E"=>"\u0397\u0313\u0342",
    "\u1F2F"=>"\u0397\u0314\u0342",
    "\u1F30"=>"\u03B9\u0313",
    "\u1F31"=>"\u03B9\u0314",
    "\u1F32"=>"\u03B9\u0313\u0300",
    "\u1F33"=>"\u03B9\u0314\u0300",
    "\u1F34"=>"\u03B9\u0313\u0301",
    "\u1F35"=>"\u03B9\u0314\u0301",
    "\u1F36"=>"\u03B9\u0313\u0342",
    "\u1F37"=>"\u03B9\u0314\u0342",
    "\u1F38"=>"\u0399\u0313",
    "\u1F39"=>"\u0399\u0314",
    "\u1F3A"=>"\u0399\u0313\u0300",
    "\u1F3B"=>"\u0399\u0314\u0300",
    "\u1F3C"=>"\u0399\u0313\u0301",
    "\u1F3D"=>"\u0399\u0314\u0301",
    "\u1F3E"=>"\u0399\u0313\u0342",
    "\u1F3F"=>"\u0399\u0314\u0342",
    "\u1F40"=>"\u03BF\u0313",
    "\u1F41"=>"\u03BF\u0314",
    "\u1F42"=>"\u03BF\u0313\u0300",
    "\u1F43"=>"\u03BF\u0314\u0300",
    "\u1F44"=>"\u03BF\u0313\u0301",
    "\u1F45"=>"\u03BF\u0314\u0301",
    "\u1F48"=>"\u039F\u0313",
    "\u1F49"=>"\u039F\u0314",
    "\u1F4A"=>"\u039F\u0313\u0300",
    "\u1F4B"=>"\u039F\u0314\u0300",
    "\u1F4C"=>"\u039F\u0313\u0301",
    "\u1F4D"=>"\u039F\u0314\u0301",
    "\u1F50"=>"\u03C5\u0313",
    "\u1F51"=>"\u03C5\u0314",
    "\u1F52"=>"\u03C5\u0313\u0300",
    "\u1F53"=>"\u03C5\u0314\u0300",
    "\u1F54"=>"\u03C5\u0313\u0301",
    "\u1F55"=>"\u03C5\u0314\u0301",
    "\u1F56"=>"\u03C5\u0313\u0342",
    "\u1F57"=>"\u03C5\u0314\u0342",
    "\u1F59"=>"\u03A5\u0314",
    "\u1F5B"=>"\u03A5\u0314\u0300",
    "\u1F5D"=>"\u03A5\u0314\u0301",
    "\u1F5F"=>"\u03A5\u0314\u0342",
    "\u1F60"=>"\u03C9\u0313",
    "\u1F61"=>"\u03C9\u0314",
    "\u1F62"=>"\u03C9\u0313\u0300",
    "\u1F63"=>"\u03C9\u0314\u0300",
    "\u1F64"=>"\u03C9\u0313\u0301",
    "\u1F65"=>"\u03C9\u0314\u0301",
    "\u1F66"=>"\u03C9\u0313\u0342",
    "\u1F67"=>"\u03C9\u0314\u0342",
    "\u1F68"=>"\u03A9\u0313",
    "\u1F69"=>"\u03A9\u0314",
    "\u1F6A"=>"\u03A9\u0313\u0300",
    "\u1F6B"=>"\u03A9\u0314\u0300",
    "\u1F6C"=>"\u03A9\u0313\u0301",
    "\u1F6D"=>"\u03A9\u0314\u0301",
    "\u1F6E"=>"\u03A9\u0313\u0342",
    "\u1F6F"=>"\u03A9\u0314\u0342",
    "\u1F70"=>"\u03B1\u0300",
    "\u1F71"=>"\u03B1\u0301",
    "\u1F72"=>"\u03B5\u0300",
    "\u1F73"=>"\u03B5\u0301",
    "\u1F74"=>"\u03B7\u0300",
    "\u1F75"=>"\u03B7\u0301",
    "\u1F76"=>"\u03B9\u0300",
    "\u1F77"=>"\u03B9\u0301",
    "\u1F78"=>"\u03BF\u0300",
    "\u1F79"=>"\u03BF\u0301",
    "\u1F7A"=>"\u03C5\u0300",
    "\u1F7B"=>"\u03C5\u0301",
    "\u1F7C"=>"\u03C9\u0300",
    "\u1F7D"=>"\u03C9\u0301",
    "\u1F80"=>"\u03B1\u0313\u0345",
    "\u1F81"=>"\u03B1\u0314\u0345",
    "\u1F82"=>"\u03B1\u0313\u0300\u0345",
    "\u1F83"=>"\u03B1\u0314\u0300\u0345",
    "\u1F84"=>"\u03B1\u0313\u0301\u0345",
    "\u1F85"=>"\u03B1\u0314\u0301\u0345",
    "\u1F86"=>"\u03B1\u0313\u0342\u0345",
    "\u1F87"=>"\u03B1\u0314\u0342\u0345",
    "\u1F88"=>"\u0391\u0313\u0345",
    "\u1F89"=>"\u0391\u0314\u0345",
    "\u1F8A"=>"\u0391\u0313\u0300\u0345",
    "\u1F8B"=>"\u0391\u0314\u0300\u0345",
    "\u1F8C"=>"\u0391\u0313\u0301\u0345",
    "\u1F8D"=>"\u0391\u0314\u0301\u0345",
    "\u1F8E"=>"\u0391\u0313\u0342\u0345",
    "\u1F8F"=>"\u0391\u0314\u0342\u0345",
    "\u1F90"=>"\u03B7\u0313\u0345",
    "\u1F91"=>"\u03B7\u0314\u0345",
    "\u1F92"=>"\u03B7\u0313\u0300\u0345",
    "\u1F93"=>"\u03B7\u0314\u0300\u0345",
    "\u1F94"=>"\u03B7\u0313\u0301\u0345",
    "\u1F95"=>"\u03B7\u0314\u0301\u0345",
    "\u1F96"=>"\u03B7\u0313\u0342\u0345",
    "\u1F97"=>"\u03B7\u0314\u0342\u0345",
    "\u1F98"=>"\u0397\u0313\u0345",
    "\u1F99"=>"\u0397\u0314\u0345",
    "\u1F9A"=>"\u0397\u0313\u0300\u0345",
    "\u1F9B"=>"\u0397\u0314\u0300\u0345",
    "\u1F9C"=>"\u0397\u0313\u0301\u0345",
    "\u1F9D"=>"\u0397\u0314\u0301\u0345",
    "\u1F9E"=>"\u0397\u0313\u0342\u0345",
    "\u1F9F"=>"\u0397\u0314\u0342\u0345",
    "\u1FA0"=>"\u03C9\u0313\u0345",
    "\u1FA1"=>"\u03C9\u0314\u0345",
    "\u1FA2"=>"\u03C9\u0313\u0300\u0345",
    "\u1FA3"=>"\u03C9\u0314\u0300\u0345",
    "\u1FA4"=>"\u03C9\u0313\u0301\u0345",
    "\u1FA5"=>"\u03C9\u0314\u0301\u0345",
    "\u1FA6"=>"\u03C9\u0313\u0342\u0345",
    "\u1FA7"=>"\u03C9\u0314\u0342\u0345",
    "\u1FA8"=>"\u03A9\u0313\u0345",
    "\u1FA9"=>"\u03A9\u0314\u0345",
    "\u1FAA"=>"\u03A9\u0313\u0300\u0345",
    "\u1FAB"=>"\u03A9\u0314\u0300\u0345",
    "\u1FAC"=>"\u03A9\u0313\u0301\u0345",
    "\u1FAD"=>"\u03A9\u0314\u0301\u0345",
    "\u1FAE"=>"\u03A9\u0313\u0342\u0345",
    "\u1FAF"=>"\u03A9\u0314\u0342\u0345",
    "\u1FB0"=>"\u03B1\u0306",
    "\u1FB1"=>"\u03B1\u0304",
    "\u1FB2"=>"\u03B1\u0300\u0345",
    "\u1FB3"=>"\u03B1\u0345",
    "\u1FB4"=>"\u03B1\u0301\u0345",
    "\u1FB6"=>"\u03B1\u0342",
    "\u1FB7"=>"\u03B1\u0342\u0345",
    "\u1FB8"=>"\u0391\u0306",
    "\u1FB9"=>"\u0391\u0304",
    "\u1FBA"=>"\u0391\u0300",
    "\u1FBB"=>"\u0391\u0301",
    "\u1FBC"=>"\u0391\u0345",
    "\u1FBE"=>"\u03B9",
    "\u1FC1"=>"\u00A8\u0342",
    "\u1FC2"=>"\u03B7\u0300\u0345",
    "\u1FC3"=>"\u03B7\u0345",
    "\u1FC4"=>"\u03B7\u0301\u0345",
    "\u1FC6"=>"\u03B7\u0342",
    "\u1FC7"=>"\u03B7\u0342\u0345",
    "\u1FC8"=>"\u0395\u0300",
    "\u1FC9"=>"\u0395\u0301",
    "\u1FCA"=>"\u0397\u0300",
    "\u1FCB"=>"\u0397\u0301",
    "\u1FCC"=>"\u0397\u0345",
    "\u1FCD"=>"\u1FBF\u0300",
    "\u1FCE"=>"\u1FBF\u0301",
    "\u1FCF"=>"\u1FBF\u0342",
    "\u1FD0"=>"\u03B9\u0306",
    "\u1FD1"=>"\u03B9\u0304",
    "\u1FD2"=>"\u03B9\u0308\u0300",
    "\u1FD3"=>"\u03B9\u0308\u0301",
    "\u1FD6"=>"\u03B9\u0342",
    "\u1FD7"=>"\u03B9\u0308\u0342",
    "\u1FD8"=>"\u0399\u0306",
    "\u1FD9"=>"\u0399\u0304",
    "\u1FDA"=>"\u0399\u0300",
    "\u1FDB"=>"\u0399\u0301",
    "\u1FDD"=>"\u1FFE\u0300",
    "\u1FDE"=>"\u1FFE\u0301",
    "\u1FDF"=>"\u1FFE\u0342",
    "\u1FE0"=>"\u03C5\u0306",
    "\u1FE1"=>"\u03C5\u0304",
    "\u1FE2"=>"\u03C5\u0308\u0300",
    "\u1FE3"=>"\u03C5\u0308\u0301",
    "\u1FE4"=>"\u03C1\u0313",
    "\u1FE5"=>"\u03C1\u0314",
    "\u1FE6"=>"\u03C5\u0342",
    "\u1FE7"=>"\u03C5\u0308\u0342",
    "\u1FE8"=>"\u03A5\u0306",
    "\u1FE9"=>"\u03A5\u0304",
    "\u1FEA"=>"\u03A5\u0300",
    "\u1FEB"=>"\u03A5\u0301",
    "\u1FEC"=>"\u03A1\u0314",
    "\u1FED"=>"\u00A8\u0300",
    "\u1FEE"=>"\u00A8\u0301",
    "\u1FEF"=>"`",
    "\u1FF2"=>"\u03C9\u0300\u0345",
    "\u1FF3"=>"\u03C9\u0345",
    "\u1FF4"=>"\u03C9\u0301\u0345",
    "\u1FF6"=>"\u03C9\u0342",
    "\u1FF7"=>"\u03C9\u0342\u0345",
    "\u1FF8"=>"\u039F\u0300",
    "\u1FF9"=>"\u039F\u0301",
    "\u1FFA"=>"\u03A9\u0300",
    "\u1FFB"=>"\u03A9\u0301",
    "\u1FFC"=>"\u03A9\u0345",
    "\u1FFD"=>"\u00B4",
    "\u2000"=>"\u2002",
    "\u2001"=>"\u2003",
    "\u2126"=>"\u03A9",
    "\u212A"=>"K",
    "\u212B"=>"A\u030A",
    "\u219A"=>"\u2190\u0338",
    "\u219B"=>"\u2192\u0338",
    "\u21AE"=>"\u2194\u0338",
    "\u21CD"=>"\u21D0\u0338",
    "\u21CE"=>"\u21D4\u0338",
    "\u21CF"=>"\u21D2\u0338",
    "\u2204"=>"\u2203\u0338",
    "\u2209"=>"\u2208\u0338",
    "\u220C"=>"\u220B\u0338",
    "\u2224"=>"\u2223\u0338",
    "\u2226"=>"\u2225\u0338",
    "\u2241"=>"\u223C\u0338",
    "\u2244"=>"\u2243\u0338",
    "\u2247"=>"\u2245\u0338",
    "\u2249"=>"\u2248\u0338",
    "\u2260"=>"=\u0338",
    "\u2262"=>"\u2261\u0338",
    "\u226D"=>"\u224D\u0338",
    "\u226E"=>"<\u0338",
    "\u226F"=>">\u0338",
    "\u2270"=>"\u2264\u0338",
    "\u2271"=>"\u2265\u0338",
    "\u2274"=>"\u2272\u0338",
    "\u2275"=>"\u2273\u0338",
    "\u2278"=>"\u2276\u0338",
    "\u2279"=>"\u2277\u0338",
    "\u2280"=>"\u227A\u0338",
    "\u2281"=>"\u227B\u0338",
    "\u2284"=>"\u2282\u0338",
    "\u2285"=>"\u2283\u0338",
    "\u2288"=>"\u2286\u0338",
    "\u2289"=>"\u2287\u0338",
    "\u22AC"=>"\u22A2\u0338",
    "\u22AD"=>"\u22A8\u0338",
    "\u22AE"=>"\u22A9\u0338",
    "\u22AF"=>"\u22AB\u0338",
    "\u22E0"=>"\u227C\u0338",
    "\u22E1"=>"\u227D\u0338",
    "\u22E2"=>"\u2291\u0338",
    "\u22E3"=>"\u2292\u0338",
    "\u22EA"=>"\u22B2\u0338",
    "\u22EB"=>"\u22B3\u0338",
    "\u22EC"=>"\u22B4\u0338",
    "\u22ED"=>"\u22B5\u0338",
    "\u2329"=>"\u3008",
    "\u232A"=>"\u3009",
    "\u2ADC"=>"\u2ADD\u0338",
    "\u304C"=>"\u304B\u3099",
    "\u304E"=>"\u304D\u3099",
    "\u3050"=>"\u304F\u3099",
    "\u3052"=>"\u3051\u3099",
    "\u3054"=>"\u3053\u3099",
    "\u3056"=>"\u3055\u3099",
    "\u3058"=>"\u3057\u3099",
    "\u305A"=>"\u3059\u3099",
    "\u305C"=>"\u305B\u3099",
    "\u305E"=>"\u305D\u3099",
    "\u3060"=>"\u305F\u3099",
    "\u3062"=>"\u3061\u3099",
    "\u3065"=>"\u3064\u3099",
    "\u3067"=>"\u3066\u3099",
    "\u3069"=>"\u3068\u3099",
    "\u3070"=>"\u306F\u3099",
    "\u3071"=>"\u306F\u309A",
    "\u3073"=>"\u3072\u3099",
    "\u3074"=>"\u3072\u309A",
    "\u3076"=>"\u3075\u3099",
    "\u3077"=>"\u3075\u309A",
    "\u3079"=>"\u3078\u3099",
    "\u307A"=>"\u3078\u309A",
    "\u307C"=>"\u307B\u3099",
    "\u307D"=>"\u307B\u309A",
    "\u3094"=>"\u3046\u3099",
    "\u309E"=>"\u309D\u3099",
    "\u30AC"=>"\u30AB\u3099",
    "\u30AE"=>"\u30AD\u3099",
    "\u30B0"=>"\u30AF\u3099",
    "\u30B2"=>"\u30B1\u3099",
    "\u30B4"=>"\u30B3\u3099",
    "\u30B6"=>"\u30B5\u3099",
    "\u30B8"=>"\u30B7\u3099",
    "\u30BA"=>"\u30B9\u3099",
    "\u30BC"=>"\u30BB\u3099",
    "\u30BE"=>"\u30BD\u3099",
    "\u30C0"=>"\u30BF\u3099",
    "\u30C2"=>"\u30C1\u3099",
    "\u30C5"=>"\u30C4\u3099",
    "\u30C7"=>"\u30C6\u3099",
    "\u30C9"=>"\u30C8\u3099",
    "\u30D0"=>"\u30CF\u3099",
    "\u30D1"=>"\u30CF\u309A",
    "\u30D3"=>"\u30D2\u3099",
    "\u30D4"=>"\u30D2\u309A",
    "\u30D6"=>"\u30D5\u3099",
    "\u30D7"=>"\u30D5\u309A",
    "\u30D9"=>"\u30D8\u3099",
    "\u30DA"=>"\u30D8\u309A",
    "\u30DC"=>"\u30DB\u3099",
    "\u30DD"=>"\u30DB\u309A",
    "\u30F4"=>"\u30A6\u3099",
    "\u30F7"=>"\u30EF\u3099",
    "\u30F8"=>"\u30F0\u3099",
    "\u30F9"=>"\u30F1\u3099",
    "\u30FA"=>"\u30F2\u3099",
    "\u30FE"=>"\u30FD\u3099",
    "\uF900"=>"\u8C48",
    "\uF901"=>"\u66F4",
    "\uF902"=>"\u8ECA",
    "\uF903"=>"\u8CC8",
    "\uF904"=>"\u6ED1",
    "\uF905"=>"\u4E32",
    "\uF906"=>"\u53E5",
    "\uF907"=>"\u9F9C",
    "\uF908"=>"\u9F9C",
    "\uF909"=>"\u5951",
    "\uF90A"=>"\u91D1",
    "\uF90B"=>"\u5587",
    "\uF90C"=>"\u5948",
    "\uF90D"=>"\u61F6",
    "\uF90E"=>"\u7669",
    "\uF90F"=>"\u7F85",
    "\uF910"=>"\u863F",
    "\uF911"=>"\u87BA",
    "\uF912"=>"\u88F8",
    "\uF913"=>"\u908F",
    "\uF914"=>"\u6A02",
    "\uF915"=>"\u6D1B",
    "\uF916"=>"\u70D9",
    "\uF917"=>"\u73DE",
    "\uF918"=>"\u843D",
    "\uF919"=>"\u916A",
    "\uF91A"=>"\u99F1",
    "\uF91B"=>"\u4E82",
    "\uF91C"=>"\u5375",
    "\uF91D"=>"\u6B04",
    "\uF91E"=>"\u721B",
    "\uF91F"=>"\u862D",
    "\uF920"=>"\u9E1E",
    "\uF921"=>"\u5D50",
    "\uF922"=>"\u6FEB",
    "\uF923"=>"\u85CD",
    "\uF924"=>"\u8964",
    "\uF925"=>"\u62C9",
    "\uF926"=>"\u81D8",
    "\uF927"=>"\u881F",
    "\uF928"=>"\u5ECA",
    "\uF929"=>"\u6717",
    "\uF92A"=>"\u6D6A",
    "\uF92B"=>"\u72FC",
    "\uF92C"=>"\u90CE",
    "\uF92D"=>"\u4F86",
    "\uF92E"=>"\u51B7",
    "\uF92F"=>"\u52DE",
    "\uF930"=>"\u64C4",
    "\uF931"=>"\u6AD3",
    "\uF932"=>"\u7210",
    "\uF933"=>"\u76E7",
    "\uF934"=>"\u8001",
    "\uF935"=>"\u8606",
    "\uF936"=>"\u865C",
    "\uF937"=>"\u8DEF",
    "\uF938"=>"\u9732",
    "\uF939"=>"\u9B6F",
    "\uF93A"=>"\u9DFA",
    "\uF93B"=>"\u788C",
    "\uF93C"=>"\u797F",
    "\uF93D"=>"\u7DA0",
    "\uF93E"=>"\u83C9",
    "\uF93F"=>"\u9304",
    "\uF940"=>"\u9E7F",
    "\uF941"=>"\u8AD6",
    "\uF942"=>"\u58DF",
    "\uF943"=>"\u5F04",
    "\uF944"=>"\u7C60",
    "\uF945"=>"\u807E",
    "\uF946"=>"\u7262",
    "\uF947"=>"\u78CA",
    "\uF948"=>"\u8CC2",
    "\uF949"=>"\u96F7",
    "\uF94A"=>"\u58D8",
    "\uF94B"=>"\u5C62",
    "\uF94C"=>"\u6A13",
    "\uF94D"=>"\u6DDA",
    "\uF94E"=>"\u6F0F",
    "\uF94F"=>"\u7D2F",
    "\uF950"=>"\u7E37",
    "\uF951"=>"\u964B",
    "\uF952"=>"\u52D2",
    "\uF953"=>"\u808B",
    "\uF954"=>"\u51DC",
    "\uF955"=>"\u51CC",
    "\uF956"=>"\u7A1C",
    "\uF957"=>"\u7DBE",
    "\uF958"=>"\u83F1",
    "\uF959"=>"\u9675",
    "\uF95A"=>"\u8B80",
    "\uF95B"=>"\u62CF",
    "\uF95C"=>"\u6A02",
    "\uF95D"=>"\u8AFE",
    "\uF95E"=>"\u4E39",
    "\uF95F"=>"\u5BE7",
    "\uF960"=>"\u6012",
    "\uF961"=>"\u7387",
    "\uF962"=>"\u7570",
    "\uF963"=>"\u5317",
    "\uF964"=>"\u78FB",
    "\uF965"=>"\u4FBF",
    "\uF966"=>"\u5FA9",
    "\uF967"=>"\u4E0D",
    "\uF968"=>"\u6CCC",
    "\uF969"=>"\u6578",
    "\uF96A"=>"\u7D22",
    "\uF96B"=>"\u53C3",
    "\uF96C"=>"\u585E",
    "\uF96D"=>"\u7701",
    "\uF96E"=>"\u8449",
    "\uF96F"=>"\u8AAA",
    "\uF970"=>"\u6BBA",
    "\uF971"=>"\u8FB0",
    "\uF972"=>"\u6C88",
    "\uF973"=>"\u62FE",
    "\uF974"=>"\u82E5",
    "\uF975"=>"\u63A0",
    "\uF976"=>"\u7565",
    "\uF977"=>"\u4EAE",
    "\uF978"=>"\u5169",
    "\uF979"=>"\u51C9",
    "\uF97A"=>"\u6881",
    "\uF97B"=>"\u7CE7",
    "\uF97C"=>"\u826F",
    "\uF97D"=>"\u8AD2",
    "\uF97E"=>"\u91CF",
    "\uF97F"=>"\u52F5",
    "\uF980"=>"\u5442",
    "\uF981"=>"\u5973",
    "\uF982"=>"\u5EEC",
    "\uF983"=>"\u65C5",
    "\uF984"=>"\u6FFE",
    "\uF985"=>"\u792A",
    "\uF986"=>"\u95AD",
    "\uF987"=>"\u9A6A",
    "\uF988"=>"\u9E97",
    "\uF989"=>"\u9ECE",
    "\uF98A"=>"\u529B",
    "\uF98B"=>"\u66C6",
    "\uF98C"=>"\u6B77",
    "\uF98D"=>"\u8F62",
    "\uF98E"=>"\u5E74",
    "\uF98F"=>"\u6190",
    "\uF990"=>"\u6200",
    "\uF991"=>"\u649A",
    "\uF992"=>"\u6F23",
    "\uF993"=>"\u7149",
    "\uF994"=>"\u7489",
    "\uF995"=>"\u79CA",
    "\uF996"=>"\u7DF4",
    "\uF997"=>"\u806F",
    "\uF998"=>"\u8F26",
    "\uF999"=>"\u84EE",
    "\uF99A"=>"\u9023",
    "\uF99B"=>"\u934A",
    "\uF99C"=>"\u5217",
    "\uF99D"=>"\u52A3",
    "\uF99E"=>"\u54BD",
    "\uF99F"=>"\u70C8",
    "\uF9A0"=>"\u88C2",
    "\uF9A1"=>"\u8AAA",
    "\uF9A2"=>"\u5EC9",
    "\uF9A3"=>"\u5FF5",
    "\uF9A4"=>"\u637B",
    "\uF9A5"=>"\u6BAE",
    "\uF9A6"=>"\u7C3E",
    "\uF9A7"=>"\u7375",
    "\uF9A8"=>"\u4EE4",
    "\uF9A9"=>"\u56F9",
    "\uF9AA"=>"\u5BE7",
    "\uF9AB"=>"\u5DBA",
    "\uF9AC"=>"\u601C",
    "\uF9AD"=>"\u73B2",
    "\uF9AE"=>"\u7469",
    "\uF9AF"=>"\u7F9A",
    "\uF9B0"=>"\u8046",
    "\uF9B1"=>"\u9234",
    "\uF9B2"=>"\u96F6",
    "\uF9B3"=>"\u9748",
    "\uF9B4"=>"\u9818",
    "\uF9B5"=>"\u4F8B",
    "\uF9B6"=>"\u79AE",
    "\uF9B7"=>"\u91B4",
    "\uF9B8"=>"\u96B8",
    "\uF9B9"=>"\u60E1",
    "\uF9BA"=>"\u4E86",
    "\uF9BB"=>"\u50DA",
    "\uF9BC"=>"\u5BEE",
    "\uF9BD"=>"\u5C3F",
    "\uF9BE"=>"\u6599",
    "\uF9BF"=>"\u6A02",
    "\uF9C0"=>"\u71CE",
    "\uF9C1"=>"\u7642",
    "\uF9C2"=>"\u84FC",
    "\uF9C3"=>"\u907C",
    "\uF9C4"=>"\u9F8D",
    "\uF9C5"=>"\u6688",
    "\uF9C6"=>"\u962E",
    "\uF9C7"=>"\u5289",
    "\uF9C8"=>"\u677B",
    "\uF9C9"=>"\u67F3",
    "\uF9CA"=>"\u6D41",
    "\uF9CB"=>"\u6E9C",
    "\uF9CC"=>"\u7409",
    "\uF9CD"=>"\u7559",
    "\uF9CE"=>"\u786B",
    "\uF9CF"=>"\u7D10",
    "\uF9D0"=>"\u985E",
    "\uF9D1"=>"\u516D",
    "\uF9D2"=>"\u622E",
    "\uF9D3"=>"\u9678",
    "\uF9D4"=>"\u502B",
    "\uF9D5"=>"\u5D19",
    "\uF9D6"=>"\u6DEA",
    "\uF9D7"=>"\u8F2A",
    "\uF9D8"=>"\u5F8B",
    "\uF9D9"=>"\u6144",
    "\uF9DA"=>"\u6817",
    "\uF9DB"=>"\u7387",
    "\uF9DC"=>"\u9686",
    "\uF9DD"=>"\u5229",
    "\uF9DE"=>"\u540F",
    "\uF9DF"=>"\u5C65",
    "\uF9E0"=>"\u6613",
    "\uF9E1"=>"\u674E",
    "\uF9E2"=>"\u68A8",
    "\uF9E3"=>"\u6CE5",
    "\uF9E4"=>"\u7406",
    "\uF9E5"=>"\u75E2",
    "\uF9E6"=>"\u7F79",
    "\uF9E7"=>"\u88CF",
    "\uF9E8"=>"\u88E1",
    "\uF9E9"=>"\u91CC",
    "\uF9EA"=>"\u96E2",
    "\uF9EB"=>"\u533F",
    "\uF9EC"=>"\u6EBA",
    "\uF9ED"=>"\u541D",
    "\uF9EE"=>"\u71D0",
    "\uF9EF"=>"\u7498",
    "\uF9F0"=>"\u85FA",
    "\uF9F1"=>"\u96A3",
    "\uF9F2"=>"\u9C57",
    "\uF9F3"=>"\u9E9F",
    "\uF9F4"=>"\u6797",
    "\uF9F5"=>"\u6DCB",
    "\uF9F6"=>"\u81E8",
    "\uF9F7"=>"\u7ACB",
    "\uF9F8"=>"\u7B20",
    "\uF9F9"=>"\u7C92",
    "\uF9FA"=>"\u72C0",
    "\uF9FB"=>"\u7099",
    "\uF9FC"=>"\u8B58",
    "\uF9FD"=>"\u4EC0",
    "\uF9FE"=>"\u8336",
    "\uF9FF"=>"\u523A",
    "\uFA00"=>"\u5207",
    "\uFA01"=>"\u5EA6",
    "\uFA02"=>"\u62D3",
    "\uFA03"=>"\u7CD6",
    "\uFA04"=>"\u5B85",
    "\uFA05"=>"\u6D1E",
    "\uFA06"=>"\u66B4",
    "\uFA07"=>"\u8F3B",
    "\uFA08"=>"\u884C",
    "\uFA09"=>"\u964D",
    "\uFA0A"=>"\u898B",
    "\uFA0B"=>"\u5ED3",
    "\uFA0C"=>"\u5140",
    "\uFA0D"=>"\u55C0",
    "\uFA10"=>"\u585A",
    "\uFA12"=>"\u6674",
    "\uFA15"=>"\u51DE",
    "\uFA16"=>"\u732A",
    "\uFA17"=>"\u76CA",
    "\uFA18"=>"\u793C",
    "\uFA19"=>"\u795E",
    "\uFA1A"=>"\u7965",
    "\uFA1B"=>"\u798F",
    "\uFA1C"=>"\u9756",
    "\uFA1D"=>"\u7CBE",
    "\uFA1E"=>"\u7FBD",
    "\uFA20"=>"\u8612",
    "\uFA22"=>"\u8AF8",
    "\uFA25"=>"\u9038",
    "\uFA26"=>"\u90FD",
    "\uFA2A"=>"\u98EF",
    "\uFA2B"=>"\u98FC",
    "\uFA2C"=>"\u9928",
    "\uFA2D"=>"\u9DB4",
    "\uFA2E"=>"\u90DE",
    "\uFA2F"=>"\u96B7",
    "\uFA30"=>"\u4FAE",
    "\uFA31"=>"\u50E7",
    "\uFA32"=>"\u514D",
    "\uFA33"=>"\u52C9",
    "\uFA34"=>"\u52E4",
    "\uFA35"=>"\u5351",
    "\uFA36"=>"\u559D",
    "\uFA37"=>"\u5606",
    "\uFA38"=>"\u5668",
    "\uFA39"=>"\u5840",
    "\uFA3A"=>"\u58A8",
    "\uFA3B"=>"\u5C64",
    "\uFA3C"=>"\u5C6E",
    "\uFA3D"=>"\u6094",
    "\uFA3E"=>"\u6168",
    "\uFA3F"=>"\u618E",
    "\uFA40"=>"\u61F2",
    "\uFA41"=>"\u654F",
    "\uFA42"=>"\u65E2",
    "\uFA43"=>"\u6691",
    "\uFA44"=>"\u6885",
    "\uFA45"=>"\u6D77",
    "\uFA46"=>"\u6E1A",
    "\uFA47"=>"\u6F22",
    "\uFA48"=>"\u716E",
    "\uFA49"=>"\u722B",
    "\uFA4A"=>"\u7422",
    "\uFA4B"=>"\u7891",
    "\uFA4C"=>"\u793E",
    "\uFA4D"=>"\u7949",
    "\uFA4E"=>"\u7948",
    "\uFA4F"=>"\u7950",
    "\uFA50"=>"\u7956",
    "\uFA51"=>"\u795D",
    "\uFA52"=>"\u798D",
    "\uFA53"=>"\u798E",
    "\uFA54"=>"\u7A40",
    "\uFA55"=>"\u7A81",
    "\uFA56"=>"\u7BC0",
    "\uFA57"=>"\u7DF4",
    "\uFA58"=>"\u7E09",
    "\uFA59"=>"\u7E41",
    "\uFA5A"=>"\u7F72",
    "\uFA5B"=>"\u8005",
    "\uFA5C"=>"\u81ED",
    "\uFA5D"=>"\u8279",
    "\uFA5E"=>"\u8279",
    "\uFA5F"=>"\u8457",
    "\uFA60"=>"\u8910",
    "\uFA61"=>"\u8996",
    "\uFA62"=>"\u8B01",
    "\uFA63"=>"\u8B39",
    "\uFA64"=>"\u8CD3",
    "\uFA65"=>"\u8D08",
    "\uFA66"=>"\u8FB6",
    "\uFA67"=>"\u9038",
    "\uFA68"=>"\u96E3",
    "\uFA69"=>"\u97FF",
    "\uFA6A"=>"\u983B",
    "\uFA6B"=>"\u6075",
    "\uFA6C"=>"\u{242EE}",
    "\uFA6D"=>"\u8218",
    "\uFA70"=>"\u4E26",
    "\uFA71"=>"\u51B5",
    "\uFA72"=>"\u5168",
    "\uFA73"=>"\u4F80",
    "\uFA74"=>"\u5145",
    "\uFA75"=>"\u5180",
    "\uFA76"=>"\u52C7",
    "\uFA77"=>"\u52FA",
    "\uFA78"=>"\u559D",
    "\uFA79"=>"\u5555",
    "\uFA7A"=>"\u5599",
    "\uFA7B"=>"\u55E2",
    "\uFA7C"=>"\u585A",
    "\uFA7D"=>"\u58B3",
    "\uFA7E"=>"\u5944",
    "\uFA7F"=>"\u5954",
    "\uFA80"=>"\u5A62",
    "\uFA81"=>"\u5B28",
    "\uFA82"=>"\u5ED2",
    "\uFA83"=>"\u5ED9",
    "\uFA84"=>"\u5F69",
    "\uFA85"=>"\u5FAD",
    "\uFA86"=>"\u60D8",
    "\uFA87"=>"\u614E",
    "\uFA88"=>"\u6108",
    "\uFA89"=>"\u618E",
    "\uFA8A"=>"\u6160",
    "\uFA8B"=>"\u61F2",
    "\uFA8C"=>"\u6234",
    "\uFA8D"=>"\u63C4",
    "\uFA8E"=>"\u641C",
    "\uFA8F"=>"\u6452",
    "\uFA90"=>"\u6556",
    "\uFA91"=>"\u6674",
    "\uFA92"=>"\u6717",
    "\uFA93"=>"\u671B",
    "\uFA94"=>"\u6756",
    "\uFA95"=>"\u6B79",
    "\uFA96"=>"\u6BBA",
    "\uFA97"=>"\u6D41",
    "\uFA98"=>"\u6EDB",
    "\uFA99"=>"\u6ECB",
    "\uFA9A"=>"\u6F22",
    "\uFA9B"=>"\u701E",
    "\uFA9C"=>"\u716E",
    "\uFA9D"=>"\u77A7",
    "\uFA9E"=>"\u7235",
    "\uFA9F"=>"\u72AF",
    "\uFAA0"=>"\u732A",
    "\uFAA1"=>"\u7471",
    "\uFAA2"=>"\u7506",
    "\uFAA3"=>"\u753B",
    "\uFAA4"=>"\u761D",
    "\uFAA5"=>"\u761F",
    "\uFAA6"=>"\u76CA",
    "\uFAA7"=>"\u76DB",
    "\uFAA8"=>"\u76F4",
    "\uFAA9"=>"\u774A",
    "\uFAAA"=>"\u7740",
    "\uFAAB"=>"\u78CC",
    "\uFAAC"=>"\u7AB1",
    "\uFAAD"=>"\u7BC0",
    "\uFAAE"=>"\u7C7B",
    "\uFAAF"=>"\u7D5B",
    "\uFAB0"=>"\u7DF4",
    "\uFAB1"=>"\u7F3E",
    "\uFAB2"=>"\u8005",
    "\uFAB3"=>"\u8352",
    "\uFAB4"=>"\u83EF",
    "\uFAB5"=>"\u8779",
    "\uFAB6"=>"\u8941",
    "\uFAB7"=>"\u8986",
    "\uFAB8"=>"\u8996",
    "\uFAB9"=>"\u8ABF",
    "\uFABA"=>"\u8AF8",
    "\uFABB"=>"\u8ACB",
    "\uFABC"=>"\u8B01",
    "\uFABD"=>"\u8AFE",
    "\uFABE"=>"\u8AED",
    "\uFABF"=>"\u8B39",
    "\uFAC0"=>"\u8B8A",
    "\uFAC1"=>"\u8D08",
    "\uFAC2"=>"\u8F38",
    "\uFAC3"=>"\u9072",
    "\uFAC4"=>"\u9199",
    "\uFAC5"=>"\u9276",
    "\uFAC6"=>"\u967C",
    "\uFAC7"=>"\u96E3",
    "\uFAC8"=>"\u9756",
    "\uFAC9"=>"\u97DB",
    "\uFACA"=>"\u97FF",
    "\uFACB"=>"\u980B",
    "\uFACC"=>"\u983B",
    "\uFACD"=>"\u9B12",
    "\uFACE"=>"\u9F9C",
    "\uFACF"=>"\u{2284A}",
    "\uFAD0"=>"\u{22844}",
    "\uFAD1"=>"\u{233D5}",
    "\uFAD2"=>"\u3B9D",
    "\uFAD3"=>"\u4018",
    "\uFAD4"=>"\u4039",
    "\uFAD5"=>"\u{25249}",
    "\uFAD6"=>"\u{25CD0}",
    "\uFAD7"=>"\u{27ED3}",
    "\uFAD8"=>"\u9F43",
    "\uFAD9"=>"\u9F8E",
    "\uFB1D"=>"\u05D9\u05B4",
    "\uFB1F"=>"\u05F2\u05B7",
    "\uFB2A"=>"\u05E9\u05C1",
    "\uFB2B"=>"\u05E9\u05C2",
    "\uFB2C"=>"\u05E9\u05BC\u05C1",
    "\uFB2D"=>"\u05E9\u05BC\u05C2",
    "\uFB2E"=>"\u05D0\u05B7",
    "\uFB2F"=>"\u05D0\u05B8",
    "\uFB30"=>"\u05D0\u05BC",
    "\uFB31"=>"\u05D1\u05BC",
    "\uFB32"=>"\u05D2\u05BC",
    "\uFB33"=>"\u05D3\u05BC",
    "\uFB34"=>"\u05D4\u05BC",
    "\uFB35"=>"\u05D5\u05BC",
    "\uFB36"=>"\u05D6\u05BC",
    "\uFB38"=>"\u05D8\u05BC",
    "\uFB39"=>"\u05D9\u05BC",
    "\uFB3A"=>"\u05DA\u05BC",
    "\uFB3B"=>"\u05DB\u05BC",
    "\uFB3C"=>"\u05DC\u05BC",
    "\uFB3E"=>"\u05DE\u05BC",
    "\uFB40"=>"\u05E0\u05BC",
    "\uFB41"=>"\u05E1\u05BC",
    "\uFB43"=>"\u05E3\u05BC",
    "\uFB44"=>"\u05E4\u05BC",
    "\uFB46"=>"\u05E6\u05BC",
    "\uFB47"=>"\u05E7\u05BC",
    "\uFB48"=>"\u05E8\u05BC",
    "\uFB49"=>"\u05E9\u05BC",
    "\uFB4A"=>"\u05EA\u05BC",
    "\uFB4B"=>"\u05D5\u05B9",
    "\uFB4C"=>"\u05D1\u05BF",
    "\uFB4D"=>"\u05DB\u05BF",
    "\uFB4E"=>"\u05E4\u05BF",
    "\u{1109A}"=>"\u{11099}\u{110BA}",
    "\u{1109C}"=>"\u{1109B}\u{110BA}",
    "\u{110AB}"=>"\u{110A5}\u{110BA}",
    "\u{1112E}"=>"\u{11131}\u{11127}",
    "\u{1112F}"=>"\u{11132}\u{11127}",
    "\u{1134B}"=>"\u{11347}\u{1133E}",
    "\u{1134C}"=>"\u{11347}\u{11357}",
    "\u{114BB}"=>"\u{114B9}\u{114BA}",
    "\u{114BC}"=>"\u{114B9}\u{114B0}",
    "\u{114BE}"=>"\u{114B9}\u{114BD}",
    "\u{115BA}"=>"\u{115B8}\u{115AF}",
    "\u{115BB}"=>"\u{115B9}\u{115AF}",
    "\u{11938}"=>"\u{11935}\u{11930}",
    "\u{1D15E}"=>"\u{1D157}\u{1D165}",
    "\u{1D15F}"=>"\u{1D158}\u{1D165}",
    "\u{1D160}"=>"\u{1D158}\u{1D165}\u{1D16E}",
    "\u{1D161}"=>"\u{1D158}\u{1D165}\u{1D16F}",
    "\u{1D162}"=>"\u{1D158}\u{1D165}\u{1D170}",
    "\u{1D163}"=>"\u{1D158}\u{1D165}\u{1D171}",
    "\u{1D164}"=>"\u{1D158}\u{1D165}\u{1D172}",
    "\u{1D1BB}"=>"\u{1D1B9}\u{1D165}",
    "\u{1D1BC}"=>"\u{1D1BA}\u{1D165}",
    "\u{1D1BD}"=>"\u{1D1B9}\u{1D165}\u{1D16E}",
    "\u{1D1BE}"=>"\u{1D1BA}\u{1D165}\u{1D16E}",
    "\u{1D1BF}"=>"\u{1D1B9}\u{1D165}\u{1D16F}",
    "\u{1D1C0}"=>"\u{1D1BA}\u{1D165}\u{1D16F}",
    "\u{2F800}"=>"\u4E3D",
    "\u{2F801}"=>"\u4E38",
    "\u{2F802}"=>"\u4E41",
    "\u{2F803}"=>"\u{20122}",
    "\u{2F804}"=>"\u4F60",
    "\u{2F805}"=>"\u4FAE",
    "\u{2F806}"=>"\u4FBB",
    "\u{2F807}"=>"\u5002",
    "\u{2F808}"=>"\u507A",
    "\u{2F809}"=>"\u5099",
    "\u{2F80A}"=>"\u50E7",
    "\u{2F80B}"=>"\u50CF",
    "\u{2F80C}"=>"\u349E",
    "\u{2F80D}"=>"\u{2063A}",
    "\u{2F80E}"=>"\u514D",
    "\u{2F80F}"=>"\u5154",
    "\u{2F810}"=>"\u5164",
    "\u{2F811}"=>"\u5177",
    "\u{2F812}"=>"\u{2051C}",
    "\u{2F813}"=>"\u34B9",
    "\u{2F814}"=>"\u5167",
    "\u{2F815}"=>"\u518D",
    "\u{2F816}"=>"\u{2054B}",
    "\u{2F817}"=>"\u5197",
    "\u{2F818}"=>"\u51A4",
    "\u{2F819}"=>"\u4ECC",
    "\u{2F81A}"=>"\u51AC",
    "\u{2F81B}"=>"\u51B5",
    "\u{2F81C}"=>"\u{291DF}",
    "\u{2F81D}"=>"\u51F5",
    "\u{2F81E}"=>"\u5203",
    "\u{2F81F}"=>"\u34DF",
    "\u{2F820}"=>"\u523B",
    "\u{2F821}"=>"\u5246",
    "\u{2F822}"=>"\u5272",
    "\u{2F823}"=>"\u5277",
    "\u{2F824}"=>"\u3515",
    "\u{2F825}"=>"\u52C7",
    "\u{2F826}"=>"\u52C9",
    "\u{2F827}"=>"\u52E4",
    "\u{2F828}"=>"\u52FA",
    "\u{2F829}"=>"\u5305",
    "\u{2F82A}"=>"\u5306",
    "\u{2F82B}"=>"\u5317",
    "\u{2F82C}"=>"\u5349",
    "\u{2F82D}"=>"\u5351",
    "\u{2F82E}"=>"\u535A",
    "\u{2F82F}"=>"\u5373",
    "\u{2F830}"=>"\u537D",
    "\u{2F831}"=>"\u537F",
    "\u{2F832}"=>"\u537F",
    "\u{2F833}"=>"\u537F",
    "\u{2F834}"=>"\u{20A2C}",
    "\u{2F835}"=>"\u7070",
    "\u{2F836}"=>"\u53CA",
    "\u{2F837}"=>"\u53DF",
    "\u{2F838}"=>"\u{20B63}",
    "\u{2F839}"=>"\u53EB",
    "\u{2F83A}"=>"\u53F1",
    "\u{2F83B}"=>"\u5406",
    "\u{2F83C}"=>"\u549E",
    "\u{2F83D}"=>"\u5438",
    "\u{2F83E}"=>"\u5448",
    "\u{2F83F}"=>"\u5468",
    "\u{2F840}"=>"\u54A2",
    "\u{2F841}"=>"\u54F6",
    "\u{2F842}"=>"\u5510",
    "\u{2F843}"=>"\u5553",
    "\u{2F844}"=>"\u5563",
    "\u{2F845}"=>"\u5584",
    "\u{2F846}"=>"\u5584",
    "\u{2F847}"=>"\u5599",
    "\u{2F848}"=>"\u55AB",
    "\u{2F849}"=>"\u55B3",
    "\u{2F84A}"=>"\u55C2",
    "\u{2F84B}"=>"\u5716",
    "\u{2F84C}"=>"\u5606",
    "\u{2F84D}"=>"\u5717",
    "\u{2F84E}"=>"\u5651",
    "\u{2F84F}"=>"\u5674",
    "\u{2F850}"=>"\u5207",
    "\u{2F851}"=>"\u58EE",
    "\u{2F852}"=>"\u57CE",
    "\u{2F853}"=>"\u57F4",
    "\u{2F854}"=>"\u580D",
    "\u{2F855}"=>"\u578B",
    "\u{2F856}"=>"\u5832",
    "\u{2F857}"=>"\u5831",
    "\u{2F858}"=>"\u58AC",
    "\u{2F859}"=>"\u{214E4}",
    "\u{2F85A}"=>"\u58F2",
    "\u{2F85B}"=>"\u58F7",
    "\u{2F85C}"=>"\u5906",
    "\u{2F85D}"=>"\u591A",
    "\u{2F85E}"=>"\u5922",
    "\u{2F85F}"=>"\u5962",
    "\u{2F860}"=>"\u{216A8}",
    "\u{2F861}"=>"\u{216EA}",
    "\u{2F862}"=>"\u59EC",
    "\u{2F863}"=>"\u5A1B",
    "\u{2F864}"=>"\u5A27",
    "\u{2F865}"=>"\u59D8",
    "\u{2F866}"=>"\u5A66",
    "\u{2F867}"=>"\u36EE",
    "\u{2F868}"=>"\u36FC",
    "\u{2F869}"=>"\u5B08",
    "\u{2F86A}"=>"\u5B3E",
    "\u{2F86B}"=>"\u5B3E",
    "\u{2F86C}"=>"\u{219C8}",
    "\u{2F86D}"=>"\u5BC3",
    "\u{2F86E}"=>"\u5BD8",
    "\u{2F86F}"=>"\u5BE7",
    "\u{2F870}"=>"\u5BF3",
    "\u{2F871}"=>"\u{21B18}",
    "\u{2F872}"=>"\u5BFF",
    "\u{2F873}"=>"\u5C06",
    "\u{2F874}"=>"\u5F53",
    "\u{2F875}"=>"\u5C22",
    "\u{2F876}"=>"\u3781",
    "\u{2F877}"=>"\u5C60",
    "\u{2F878}"=>"\u5C6E",
    "\u{2F879}"=>"\u5CC0",
    "\u{2F87A}"=>"\u5C8D",
    "\u{2F87B}"=>"\u{21DE4}",
    "\u{2F87C}"=>"\u5D43",
    "\u{2F87D}"=>"\u{21DE6}",
    "\u{2F87E}"=>"\u5D6E",
    "\u{2F87F}"=>"\u5D6B",
    "\u{2F880}"=>"\u5D7C",
    "\u{2F881}"=>"\u5DE1",
    "\u{2F882}"=>"\u5DE2",
    "\u{2F883}"=>"\u382F",
    "\u{2F884}"=>"\u5DFD",
    "\u{2F885}"=>"\u5E28",
    "\u{2F886}"=>"\u5E3D",
    "\u{2F887}"=>"\u5E69",
    "\u{2F888}"=>"\u3862",
    "\u{2F889}"=>"\u{22183}",
    "\u{2F88A}"=>"\u387C",
    "\u{2F88B}"=>"\u5EB0",
    "\u{2F88C}"=>"\u5EB3",
    "\u{2F88D}"=>"\u5EB6",
    "\u{2F88E}"=>"\u5ECA",
    "\u{2F88F}"=>"\u{2A392}",
    "\u{2F890}"=>"\u5EFE",
    "\u{2F891}"=>"\u{22331}",
    "\u{2F892}"=>"\u{22331}",
    "\u{2F893}"=>"\u8201",
    "\u{2F894}"=>"\u5F22",
    "\u{2F895}"=>"\u5F22",
    "\u{2F896}"=>"\u38C7",
    "\u{2F897}"=>"\u{232B8}",
    "\u{2F898}"=>"\u{261DA}",
    "\u{2F899}"=>"\u5F62",
    "\u{2F89A}"=>"\u5F6B",
    "\u{2F89B}"=>"\u38E3",
    "\u{2F89C}"=>"\u5F9A",
    "\u{2F89D}"=>"\u5FCD",
    "\u{2F89E}"=>"\u5FD7",
    "\u{2F89F}"=>"\u5FF9",
    "\u{2F8A0}"=>"\u6081",
    "\u{2F8A1}"=>"\u393A",
    "\u{2F8A2}"=>"\u391C",
    "\u{2F8A3}"=>"\u6094",
    "\u{2F8A4}"=>"\u{226D4}",
    "\u{2F8A5}"=>"\u60C7",
    "\u{2F8A6}"=>"\u6148",
    "\u{2F8A7}"=>"\u614C",
    "\u{2F8A8}"=>"\u614E",
    "\u{2F8A9}"=>"\u614C",
    "\u{2F8AA}"=>"\u617A",
    "\u{2F8AB}"=>"\u618E",
    "\u{2F8AC}"=>"\u61B2",
    "\u{2F8AD}"=>"\u61A4",
    "\u{2F8AE}"=>"\u61AF",
    "\u{2F8AF}"=>"\u61DE",
    "\u{2F8B0}"=>"\u61F2",
    "\u{2F8B1}"=>"\u61F6",
    "\u{2F8B2}"=>"\u6210",
    "\u{2F8B3}"=>"\u621B",
    "\u{2F8B4}"=>"\u625D",
    "\u{2F8B5}"=>"\u62B1",
    "\u{2F8B6}"=>"\u62D4",
    "\u{2F8B7}"=>"\u6350",
    "\u{2F8B8}"=>"\u{22B0C}",
    "\u{2F8B9}"=>"\u633D",
    "\u{2F8BA}"=>"\u62FC",
    "\u{2F8BB}"=>"\u6368",
    "\u{2F8BC}"=>"\u6383",
    "\u{2F8BD}"=>"\u63E4",
    "\u{2F8BE}"=>"\u{22BF1}",
    "\u{2F8BF}"=>"\u6422",
    "\u{2F8C0}"=>"\u63C5",
    "\u{2F8C1}"=>"\u63A9",
    "\u{2F8C2}"=>"\u3A2E",
    "\u{2F8C3}"=>"\u6469",
    "\u{2F8C4}"=>"\u647E",
    "\u{2F8C5}"=>"\u649D",
    "\u{2F8C6}"=>"\u6477",
    "\u{2F8C7}"=>"\u3A6C",
    "\u{2F8C8}"=>"\u654F",
    "\u{2F8C9}"=>"\u656C",
    "\u{2F8CA}"=>"\u{2300A}",
    "\u{2F8CB}"=>"\u65E3",
    "\u{2F8CC}"=>"\u66F8",
    "\u{2F8CD}"=>"\u6649",
    "\u{2F8CE}"=>"\u3B19",
    "\u{2F8CF}"=>"\u6691",
    "\u{2F8D0}"=>"\u3B08",
    "\u{2F8D1}"=>"\u3AE4",
    "\u{2F8D2}"=>"\u5192",
    "\u{2F8D3}"=>"\u5195",
    "\u{2F8D4}"=>"\u6700",
    "\u{2F8D5}"=>"\u669C",
    "\u{2F8D6}"=>"\u80AD",
    "\u{2F8D7}"=>"\u43D9",
    "\u{2F8D8}"=>"\u6717",
    "\u{2F8D9}"=>"\u671B",
    "\u{2F8DA}"=>"\u6721",
    "\u{2F8DB}"=>"\u675E",
    "\u{2F8DC}"=>"\u6753",
    "\u{2F8DD}"=>"\u{233C3}",
    "\u{2F8DE}"=>"\u3B49",
    "\u{2F8DF}"=>"\u67FA",
    "\u{2F8E0}"=>"\u6785",
    "\u{2F8E1}"=>"\u6852",
    "\u{2F8E2}"=>"\u6885",
    "\u{2F8E3}"=>"\u{2346D}",
    "\u{2F8E4}"=>"\u688E",
    "\u{2F8E5}"=>"\u681F",
    "\u{2F8E6}"=>"\u6914",
    "\u{2F8E7}"=>"\u3B9D",
    "\u{2F8E8}"=>"\u6942",
    "\u{2F8E9}"=>"\u69A3",
    "\u{2F8EA}"=>"\u69EA",
    "\u{2F8EB}"=>"\u6AA8",
    "\u{2F8EC}"=>"\u{236A3}",
    "\u{2F8ED}"=>"\u6ADB",
    "\u{2F8EE}"=>"\u3C18",
    "\u{2F8EF}"=>"\u6B21",
    "\u{2F8F0}"=>"\u{238A7}",
    "\u{2F8F1}"=>"\u6B54",
    "\u{2F8F2}"=>"\u3C4E",
    "\u{2F8F3}"=>"\u6B72",
    "\u{2F8F4}"=>"\u6B9F",
    "\u{2F8F5}"=>"\u6BBA",
    "\u{2F8F6}"=>"\u6BBB",
    "\u{2F8F7}"=>"\u{23A8D}",
    "\u{2F8F8}"=>"\u{21D0B}",
    "\u{2F8F9}"=>"\u{23AFA}",
    "\u{2F8FA}"=>"\u6C4E",
    "\u{2F8FB}"=>"\u{23CBC}",
    "\u{2F8FC}"=>"\u6CBF",
    "\u{2F8FD}"=>"\u6CCD",
    "\u{2F8FE}"=>"\u6C67",
    "\u{2F8FF}"=>"\u6D16",
    "\u{2F900}"=>"\u6D3E",
    "\u{2F901}"=>"\u6D77",
    "\u{2F902}"=>"\u6D41",
    "\u{2F903}"=>"\u6D69",
    "\u{2F904}"=>"\u6D78",
    "\u{2F905}"=>"\u6D85",
    "\u{2F906}"=>"\u{23D1E}",
    "\u{2F907}"=>"\u6D34",
    "\u{2F908}"=>"\u6E2F",
    "\u{2F909}"=>"\u6E6E",
    "\u{2F90A}"=>"\u3D33",
    "\u{2F90B}"=>"\u6ECB",
    "\u{2F90C}"=>"\u6EC7",
    "\u{2F90D}"=>"\u{23ED1}",
    "\u{2F90E}"=>"\u6DF9",
    "\u{2F90F}"=>"\u6F6E",
    "\u{2F910}"=>"\u{23F5E}",
    "\u{2F911}"=>"\u{23F8E}",
    "\u{2F912}"=>"\u6FC6",
    "\u{2F913}"=>"\u7039",
    "\u{2F914}"=>"\u701E",
    "\u{2F915}"=>"\u701B",
    "\u{2F916}"=>"\u3D96",
    "\u{2F917}"=>"\u704A",
    "\u{2F918}"=>"\u707D",
    "\u{2F919}"=>"\u7077",
    "\u{2F91A}"=>"\u70AD",
    "\u{2F91B}"=>"\u{20525}",
    "\u{2F91C}"=>"\u7145",
    "\u{2F91D}"=>"\u{24263}",
    "\u{2F91E}"=>"\u719C",
    "\u{2F91F}"=>"\u{243AB}",
    "\u{2F920}"=>"\u7228",
    "\u{2F921}"=>"\u7235",
    "\u{2F922}"=>"\u7250",
    "\u{2F923}"=>"\u{24608}",
    "\u{2F924}"=>"\u7280",
    "\u{2F925}"=>"\u7295",
    "\u{2F926}"=>"\u{24735}",
    "\u{2F927}"=>"\u{24814}",
    "\u{2F928}"=>"\u737A",
    "\u{2F929}"=>"\u738B",
    "\u{2F92A}"=>"\u3EAC",
    "\u{2F92B}"=>"\u73A5",
    "\u{2F92C}"=>"\u3EB8",
    "\u{2F92D}"=>"\u3EB8",
    "\u{2F92E}"=>"\u7447",
    "\u{2F92F}"=>"\u745C",
    "\u{2F930}"=>"\u7471",
    "\u{2F931}"=>"\u7485",
    "\u{2F932}"=>"\u74CA",
    "\u{2F933}"=>"\u3F1B",
    "\u{2F934}"=>"\u7524",
    "\u{2F935}"=>"\u{24C36}",
    "\u{2F936}"=>"\u753E",
    "\u{2F937}"=>"\u{24C92}",
    "\u{2F938}"=>"\u7570",
    "\u{2F939}"=>"\u{2219F}",
    "\u{2F93A}"=>"\u7610",
    "\u{2F93B}"=>"\u{24FA1}",
    "\u{2F93C}"=>"\u{24FB8}",
    "\u{2F93D}"=>"\u{25044}",
    "\u{2F93E}"=>"\u3FFC",
    "\u{2F93F}"=>"\u4008",
    "\u{2F940}"=>"\u76F4",
    "\u{2F941}"=>"\u{250F3}",
    "\u{2F942}"=>"\u{250F2}",
    "\u{2F943}"=>"\u{25119}",
    "\u{2F944}"=>"\u{25133}",
    "\u{2F945}"=>"\u771E",
    "\u{2F946}"=>"\u771F",
    "\u{2F947}"=>"\u771F",
    "\u{2F948}"=>"\u774A",
    "\u{2F949}"=>"\u4039",
    "\u{2F94A}"=>"\u778B",
    "\u{2F94B}"=>"\u4046",
    "\u{2F94C}"=>"\u4096",
    "\u{2F94D}"=>"\u{2541D}",
    "\u{2F94E}"=>"\u784E",
    "\u{2F94F}"=>"\u788C",
    "\u{2F950}"=>"\u78CC",
    "\u{2F951}"=>"\u40E3",
    "\u{2F952}"=>"\u{25626}",
    "\u{2F953}"=>"\u7956",
    "\u{2F954}"=>"\u{2569A}",
    "\u{2F955}"=>"\u{256C5}",
    "\u{2F956}"=>"\u798F",
    "\u{2F957}"=>"\u79EB",
    "\u{2F958}"=>"\u412F",
    "\u{2F959}"=>"\u7A40",
    "\u{2F95A}"=>"\u7A4A",
    "\u{2F95B}"=>"\u7A4F",
    "\u{2F95C}"=>"\u{2597C}",
    "\u{2F95D}"=>"\u{25AA7}",
    "\u{2F95E}"=>"\u{25AA7}",
    "\u{2F95F}"=>"\u7AEE",
    "\u{2F960}"=>"\u4202",
    "\u{2F961}"=>"\u{25BAB}",
    "\u{2F962}"=>"\u7BC6",
    "\u{2F963}"=>"\u7BC9",
    "\u{2F964}"=>"\u4227",
    "\u{2F965}"=>"\u{25C80}",
    "\u{2F966}"=>"\u7CD2",
    "\u{2F967}"=>"\u42A0",
    "\u{2F968}"=>"\u7CE8",
    "\u{2F969}"=>"\u7CE3",
    "\u{2F96A}"=>"\u7D00",
    "\u{2F96B}"=>"\u{25F86}",
    "\u{2F96C}"=>"\u7D63",
    "\u{2F96D}"=>"\u4301",
    "\u{2F96E}"=>"\u7DC7",
    "\u{2F96F}"=>"\u7E02",
    "\u{2F970}"=>"\u7E45",
    "\u{2F971}"=>"\u4334",
    "\u{2F972}"=>"\u{26228}",
    "\u{2F973}"=>"\u{26247}",
    "\u{2F974}"=>"\u4359",
    "\u{2F975}"=>"\u{262D9}",
    "\u{2F976}"=>"\u7F7A",
    "\u{2F977}"=>"\u{2633E}",
    "\u{2F978}"=>"\u7F95",
    "\u{2F979}"=>"\u7FFA",
    "\u{2F97A}"=>"\u8005",
    "\u{2F97B}"=>"\u{264DA}",
    "\u{2F97C}"=>"\u{26523}",
    "\u{2F97D}"=>"\u8060",
    "\u{2F97E}"=>"\u{265A8}",
    "\u{2F97F}"=>"\u8070",
    "\u{2F980}"=>"\u{2335F}",
    "\u{2F981}"=>"\u43D5",
    "\u{2F982}"=>"\u80B2",
    "\u{2F983}"=>"\u8103",
    "\u{2F984}"=>"\u440B",
    "\u{2F985}"=>"\u813E",
    "\u{2F986}"=>"\u5AB5",
    "\u{2F987}"=>"\u{267A7}",
    "\u{2F988}"=>"\u{267B5}",
    "\u{2F989}"=>"\u{23393}",
    "\u{2F98A}"=>"\u{2339C}",
    "\u{2F98B}"=>"\u8201",
    "\u{2F98C}"=>"\u8204",
    "\u{2F98D}"=>"\u8F9E",
    "\u{2F98E}"=>"\u446B",
    "\u{2F98F}"=>"\u8291",
    "\u{2F990}"=>"\u828B",
    "\u{2F991}"=>"\u829D",
    "\u{2F992}"=>"\u52B3",
    "\u{2F993}"=>"\u82B1",
    "\u{2F994}"=>"\u82B3",
    "\u{2F995}"=>"\u82BD",
    "\u{2F996}"=>"\u82E6",
    "\u{2F997}"=>"\u{26B3C}",
    "\u{2F998}"=>"\u82E5",
    "\u{2F999}"=>"\u831D",
    "\u{2F99A}"=>"\u8363",
    "\u{2F99B}"=>"\u83AD",
    "\u{2F99C}"=>"\u8323",
    "\u{2F99D}"=>"\u83BD",
    "\u{2F99E}"=>"\u83E7",
    "\u{2F99F}"=>"\u8457",
    "\u{2F9A0}"=>"\u8353",
    "\u{2F9A1}"=>"\u83CA",
    "\u{2F9A2}"=>"\u83CC",
    "\u{2F9A3}"=>"\u83DC",
    "\u{2F9A4}"=>"\u{26C36}",
    "\u{2F9A5}"=>"\u{26D6B}",
    "\u{2F9A6}"=>"\u{26CD5}",
    "\u{2F9A7}"=>"\u452B",
    "\u{2F9A8}"=>"\u84F1",
    "\u{2F9A9}"=>"\u84F3",
    "\u{2F9AA}"=>"\u8516",
    "\u{2F9AB}"=>"\u{273CA}",
    "\u{2F9AC}"=>"\u8564",
    "\u{2F9AD}"=>"\u{26F2C}",
    "\u{2F9AE}"=>"\u455D",
    "\u{2F9AF}"=>"\u4561",
    "\u{2F9B0}"=>"\u{26FB1}",
    "\u{2F9B1}"=>"\u{270D2}",
    "\u{2F9B2}"=>"\u456B",
    "\u{2F9B3}"=>"\u8650",
    "\u{2F9B4}"=>"\u865C",
    "\u{2F9B5}"=>"\u8667",
    "\u{2F9B6}"=>"\u8669",
    "\u{2F9B7}"=>"\u86A9",
    "\u{2F9B8}"=>"\u8688",
    "\u{2F9B9}"=>"\u870E",
    "\u{2F9BA}"=>"\u86E2",
    "\u{2F9BB}"=>"\u8779",
    "\u{2F9BC}"=>"\u8728",
    "\u{2F9BD}"=>"\u876B",
    "\u{2F9BE}"=>"\u8786",
    "\u{2F9BF}"=>"\u45D7",
    "\u{2F9C0}"=>"\u87E1",
    "\u{2F9C1}"=>"\u8801",
    "\u{2F9C2}"=>"\u45F9",
    "\u{2F9C3}"=>"\u8860",
    "\u{2F9C4}"=>"\u8863",
    "\u{2F9C5}"=>"\u{27667}",
    "\u{2F9C6}"=>"\u88D7",
    "\u{2F9C7}"=>"\u88DE",
    "\u{2F9C8}"=>"\u4635",
    "\u{2F9C9}"=>"\u88FA",
    "\u{2F9CA}"=>"\u34BB",
    "\u{2F9CB}"=>"\u{278AE}",
    "\u{2F9CC}"=>"\u{27966}",
    "\u{2F9CD}"=>"\u46BE",
    "\u{2F9CE}"=>"\u46C7",
    "\u{2F9CF}"=>"\u8AA0",
    "\u{2F9D0}"=>"\u8AED",
    "\u{2F9D1}"=>"\u8B8A",
    "\u{2F9D2}"=>"\u8C55",
    "\u{2F9D3}"=>"\u{27CA8}",
    "\u{2F9D4}"=>"\u8CAB",
    "\u{2F9D5}"=>"\u8CC1",
    "\u{2F9D6}"=>"\u8D1B",
    "\u{2F9D7}"=>"\u8D77",
    "\u{2F9D8}"=>"\u{27F2F}",
    "\u{2F9D9}"=>"\u{20804}",
    "\u{2F9DA}"=>"\u8DCB",
    "\u{2F9DB}"=>"\u8DBC",
    "\u{2F9DC}"=>"\u8DF0",
    "\u{2F9DD}"=>"\u{208DE}",
    "\u{2F9DE}"=>"\u8ED4",
    "\u{2F9DF}"=>"\u8F38",
    "\u{2F9E0}"=>"\u{285D2}",
    "\u{2F9E1}"=>"\u{285ED}",
    "\u{2F9E2}"=>"\u9094",
    "\u{2F9E3}"=>"\u90F1",
    "\u{2F9E4}"=>"\u9111",
    "\u{2F9E5}"=>"\u{2872E}",
    "\u{2F9E6}"=>"\u911B",
    "\u{2F9E7}"=>"\u9238",
    "\u{2F9E8}"=>"\u92D7",
    "\u{2F9E9}"=>"\u92D8",
    "\u{2F9EA}"=>"\u927C",
    "\u{2F9EB}"=>"\u93F9",
    "\u{2F9EC}"=>"\u9415",
    "\u{2F9ED}"=>"\u{28BFA}",
    "\u{2F9EE}"=>"\u958B",
    "\u{2F9EF}"=>"\u4995",
    "\u{2F9F0}"=>"\u95B7",
    "\u{2F9F1}"=>"\u{28D77}",
    "\u{2F9F2}"=>"\u49E6",
    "\u{2F9F3}"=>"\u96C3",
    "\u{2F9F4}"=>"\u5DB2",
    "\u{2F9F5}"=>"\u9723",
    "\u{2F9F6}"=>"\u{29145}",
    "\u{2F9F7}"=>"\u{2921A}",
    "\u{2F9F8}"=>"\u4A6E",
    "\u{2F9F9}"=>"\u4A76",
    "\u{2F9FA}"=>"\u97E0",
    "\u{2F9FB}"=>"\u{2940A}",
    "\u{2F9FC}"=>"\u4AB2",
    "\u{2F9FD}"=>"\u{29496}",
    "\u{2F9FE}"=>"\u980B",
    "\u{2F9FF}"=>"\u980B",
    "\u{2FA00}"=>"\u9829",
    "\u{2FA01}"=>"\u{295B6}",
    "\u{2FA02}"=>"\u98E2",
    "\u{2FA03}"=>"\u4B33",
    "\u{2FA04}"=>"\u9929",
    "\u{2FA05}"=>"\u99A7",
    "\u{2FA06}"=>"\u99C2",
    "\u{2FA07}"=>"\u99FE",
    "\u{2FA08}"=>"\u4BCE",
    "\u{2FA09}"=>"\u{29B30}",
    "\u{2FA0A}"=>"\u9B12",
    "\u{2FA0B}"=>"\u9C40",
    "\u{2FA0C}"=>"\u9CFD",
    "\u{2FA0D}"=>"\u4CCE",
    "\u{2FA0E}"=>"\u4CED",
    "\u{2FA0F}"=>"\u9D67",
    "\u{2FA10}"=>"\u{2A0CE}",
    "\u{2FA11}"=>"\u4CF8",
    "\u{2FA12}"=>"\u{2A105}",
    "\u{2FA13}"=>"\u{2A20E}",
    "\u{2FA14}"=>"\u{2A291}",
    "\u{2FA15}"=>"\u9EBB",
    "\u{2FA16}"=>"\u4D56",
    "\u{2FA17}"=>"\u9EF9",
    "\u{2FA18}"=>"\u9EFE",
    "\u{2FA19}"=>"\u9F05",
    "\u{2FA1A}"=>"\u9F0F",
    "\u{2FA1B}"=>"\u9F16",
    "\u{2FA1C}"=>"\u9F3B",
    "\u{2FA1D}"=>"\u{2A600}",
  }.freeze

  KOMPATIBLE_TABLE = {
    "\u00A0"=>" ",
    "\u00A8"=>" \u0308",
    "\u00AA"=>"a",
    "\u00AF"=>" \u0304",
    "\u00B2"=>"2",
    "\u00B3"=>"3",
    "\u00B4"=>" \u0301",
    "\u00B5"=>"\u03BC",
    "\u00B8"=>" \u0327",
    "\u00B9"=>"1",
    "\u00BA"=>"o",
    "\u00BC"=>"1\u20444",
    "\u00BD"=>"1\u20442",
    "\u00BE"=>"3\u20444",
    "\u0132"=>"IJ",
    "\u0133"=>"ij",
    "\u013F"=>"L\u00B7",
    "\u0140"=>"l\u00B7",
    "\u0149"=>"\u02BCn",
    "\u017F"=>"s",
    "\u01C4"=>"D\u017D",
    "\u01C5"=>"D\u017E",
    "\u01C6"=>"d\u017E",
    "\u01C7"=>"LJ",
    "\u01C8"=>"Lj",
    "\u01C9"=>"lj",
    "\u01CA"=>"NJ",
    "\u01CB"=>"Nj",
    "\u01CC"=>"nj",
    "\u01F1"=>"DZ",
    "\u01F2"=>"Dz",
    "\u01F3"=>"dz",
    "\u02B0"=>"h",
    "\u02B1"=>"\u0266",
    "\u02B2"=>"j",
    "\u02B3"=>"r",
    "\u02B4"=>"\u0279",
    "\u02B5"=>"\u027B",
    "\u02B6"=>"\u0281",
    "\u02B7"=>"w",
    "\u02B8"=>"y",
    "\u02D8"=>" \u0306",
    "\u02D9"=>" \u0307",
    "\u02DA"=>" \u030A",
    "\u02DB"=>" \u0328",
    "\u02DC"=>" \u0303",
    "\u02DD"=>" \u030B",
    "\u02E0"=>"\u0263",
    "\u02E1"=>"l",
    "\u02E2"=>"s",
    "\u02E3"=>"x",
    "\u02E4"=>"\u0295",
    "\u037A"=>" \u0345",
    "\u0384"=>" \u0301",
    "\u03D0"=>"\u03B2",
    "\u03D1"=>"\u03B8",
    "\u03D2"=>"\u03A5",
    "\u03D5"=>"\u03C6",
    "\u03D6"=>"\u03C0",
    "\u03F0"=>"\u03BA",
    "\u03F1"=>"\u03C1",
    "\u03F2"=>"\u03C2",
    "\u03F4"=>"\u0398",
    "\u03F5"=>"\u03B5",
    "\u03F9"=>"\u03A3",
    "\u0587"=>"\u0565\u0582",
    "\u0675"=>"\u0627\u0674",
    "\u0676"=>"\u0648\u0674",
    "\u0677"=>"\u06C7\u0674",
    "\u0678"=>"\u064A\u0674",
    "\u0E33"=>"\u0E4D\u0E32",
    "\u0EB3"=>"\u0ECD\u0EB2",
    "\u0EDC"=>"\u0EAB\u0E99",
    "\u0EDD"=>"\u0EAB\u0EA1",
    "\u0F0C"=>"\u0F0B",
    "\u0F77"=>"\u0FB2\u0F81",
    "\u0F79"=>"\u0FB3\u0F81",
    "\u10FC"=>"\u10DC",
    "\u1D2C"=>"A",
    "\u1D2D"=>"\u00C6",
    "\u1D2E"=>"B",
    "\u1D30"=>"D",
    "\u1D31"=>"E",
    "\u1D32"=>"\u018E",
    "\u1D33"=>"G",
    "\u1D34"=>"H",
    "\u1D35"=>"I",
    "\u1D36"=>"J",
    "\u1D37"=>"K",
    "\u1D38"=>"L",
    "\u1D39"=>"M",
    "\u1D3A"=>"N",
    "\u1D3C"=>"O",
    "\u1D3D"=>"\u0222",
    "\u1D3E"=>"P",
    "\u1D3F"=>"R",
    "\u1D40"=>"T",
    "\u1D41"=>"U",
    "\u1D42"=>"W",
    "\u1D43"=>"a",
    "\u1D44"=>"\u0250",
    "\u1D45"=>"\u0251",
    "\u1D46"=>"\u1D02",
    "\u1D47"=>"b",
    "\u1D48"=>"d",
    "\u1D49"=>"e",
    "\u1D4A"=>"\u0259",
    "\u1D4B"=>"\u025B",
    "\u1D4C"=>"\u025C",
    "\u1D4D"=>"g",
    "\u1D4F"=>"k",
    "\u1D50"=>"m",
    "\u1D51"=>"\u014B",
    "\u1D52"=>"o",
    "\u1D53"=>"\u0254",
    "\u1D54"=>"\u1D16