_option("--all",
               "Restore all installed gems to pristine",
               "condition") do |value, options|
      options[:all] = value
    end

    add_option("--skip=gem_name",
               "used on --all, skip if name == gem_name") do |value, options|
      options[:skip] ||= []
      options[:skip] << value
    end

    add_option("--[no-]extensions",
               "Restore gems with extensions",
               "in addition to regular gems") do |value, options|
      options[:extensions_set] = true
      options[:extensions]     = value
    end

    add_option("--only-missing-extensions",
               "Only restore gems with missing extensions") do |value, options|
      options[:only_missing_extensions] = value
    end

    add_option("--only-executables",
               "Only restore executables") do |value, options|
      options[:only_executables] = value
    end

    add_option("--only-plugins",
               "Only restore plugins") do |value, options|
      options[:only_plugins] = value
    end

    add_option("-E", "--[no-]env-shebang",
               "Rewrite executables with a shebang",
               "of /usr/bin/env") do |value, options|
      options[:env_shebang] = value
    end

    add_option("-i", "--install-dir DIR",
               "Gem repository to get gems restored") do |value, options|
      options[:install_dir] = File.expand_path(value)
    end

    add_option("-n", "--bindir DIR",
               "Directory where executables are",
               "located") do |value, options|
      options[:bin_dir] = File.expand_path(value)
    end

    add_version_option("restore to", "pristine condition")
  end

  def arguments # :nodoc:
    "GEMNAME       gem to restore to pristine condition (unless --all)"
  end

  def defaults_str # :nodoc:
    "--extensions"
  end

  def description # :nodoc:
    <<-EOF
The pristine command compares an installed gem with the contents of its
cached .gem file and restores any files that don't match the cached .gem's
copy.

If you have made modifications to an installed gem, the pristine command
will revert them.  All extensions are rebuilt and all bin stubs for the gem
are regenerated after checking for modifications.

If the cached gem cannot be found it will be downloaded.

If --no-extensions is provided pristine will not attempt to restore a gem
with an extension.

If --extensions is given (but not --all or gem names) only gems with
extensions will be restored.
    EOF
  end

  def usage # :nodoc:
    "#{program_name} [GEMNAME ...]"
  end

  def execute
    install_dir = options[:install_dir]

    specification_record = install_dir ? Gem::SpecificationRecord.from_path(install_dir) : Gem::Specification.specification_record

    specs = if options[:all]
      specification_record.map

    # `--extensions` must be explicitly given to pristine only gems
    # with extensions.
    elsif options[:extensions_set] &&
          options[:extensions] && options[:args].empty?
      specification_record.select do |spec|
        spec.extensions && !spec.extensions.empty?
      end
    elsif options[:only_missing_extensions]
      specification_record.select(&:missing_extensions?)
    else
      get_all_gem_names.sort.flat_map do |gem_name|
        specification_record.find_all_by_name(gem_name, options[:version]).reverse
      end
    end

    specs = specs.select {|spec| spec.platform == RUBY_ENGINE || Gem::Platform.local === spec.platform || spec.platform == Gem