 Gem::Specification.find_all_by_name "rubygems-update", requirement
    installed_gems = update_gem("rubygems-update", requirement) if installed_gems.empty? || installed_gems.first.version != version
    return if installed_gems.empty?

    install_rubygems installed_gems.first
  end

  def update_rubygems_arguments # :nodoc:
    args = []
    args << "--silent" if options[:silent]
    args << "--prefix" << Gem.prefix if Gem.prefix
    args << "--no-document" unless options[:document].include?("rdoc") || options[:document].include?("ri")
    args << "--no-format-executable" if options[:no_format_executable]
    args << "--previous-version" << Gem::VERSION
    args
  end

  def which_to_update(highest_installed_gems, gem_names)
    result = []

    highest_installed_gems.each do |_l_name, l_spec|
      next if !gem_names.empty? &&
              gem_names.none? {|name| name == l_spec.name }

      highest_remote_tup = highest_remote_name_tuple l_spec
      next unless highest_remote_tup

      result << highest_remote_tup
    end

    result
  end

  private

  #
  # Oldest version we support downgrading to. This is the version that
  # originally ships with the oldest supported patch version of ruby.
  #
  def oldest_supported_version
    @oldest_supported_version ||=
      Gem::Version.new("3.3.3")
  end
end
                                                                                                                                                                                                              rubygems/rubygems/commands/push_command.rb                                                          0000644                 00000006346 15040313420 0015042 0                                                                                                    ustar 00                                                                                                                                                                                                                                                       # frozen_string_literal: true

require_relative "../command"
require_relative "../local_remote_options"
require_relative "../gemcutter_utilities"
require_relative "../package"

class Gem::Commands::PushCommand < Gem::Command
  include Gem::LocalRemoteOptions
  include Gem::GemcutterUtilities

  def description # :nodoc:
    <<-EOF
The push command uploads a gem to the push server (the default is
https://rubygems.org) and adds it to the index.

The gem can be removed from the index and deleted from the server using the yank
command.  For further discussion see the help for the yank command.

The push command will use ~/.gem/credentials to authenticate to a server, but you can use the RubyGems environment variable GEM_HOST_API_KEY to set the api key to authenticate.
    EOF
  end

  def arguments # :nodoc:
    "GEM       built gem to push up"
  end

  def usage # :nodoc:
    "#{program_name} GEM"
  end

  def initialize
    super "push", "Push a gem up to the gem server", host: host, attestations: []

    @user_defined_host = false

    add_proxy_option
    add_key_option
    add_otp_option

    add_option("--host HOST",
               "Push to another gemcutter-compatible host",
               "  (e.g. https://rubygems.org)") do |value, options|
      options[:host] = value
      @user_defined_host = true
    end

    add_option("--attestation FILE",
                "Push with sigstore attestations") do |value, options|
      options[:attestations] << value
    end

    @host = nil
  end

  def execute
    gem_name = get_one_gem_name
    default_gem_server, push_host = get_hosts_for(gem_name)

    @host = if @user_defined_host
      options[:host]
    elsif default_gem_server
      default_gem_server
    elsif push_host
      push_host
    else
      options[:host]
    end

    sign_in @host, scope: get_push_scope

    send_gem(gem_name)
  end

  def send_gem(name)
    args = [:post, "api/v1/gems"]

    _, push_host = get_hosts_for(name)

    @host ||= push_host

    # Always include @host, even if it's nil
    args += [@host, push_host]

    say "Pushing gem to #{@host || Gem.host}..."

    response = send_push_request(name, args)

    with_response response
  end

  private

  def send_push_request(name, args)
    rubygems_api_request(*args, scope: get_push_scope) do |request|
      body = Gem.read_binary name
      if options[:attestations].any?
        request.set_form([
          ["gem", body, { filename: name, content_type: "application/octet-stream" }],
          get_attestations_part,
        ], "multipart/form-data")
      else
        request.body = body
        request.add_field "Content-Type",   "application/octet-stream"
        request.add_field "Content-Length", request.body.size
      end
      request.add_field "Authorization", api_key
    end
  end

  def get_hosts_for(name)
    gem_metadata = Gem::Package.new(name).spec.metadata

    [
      gem_metadata["default_gem_server"],
      gem_metadata["allowed_push_host"],
    ]
  end

  def get_push_scope
    :push_rubygem
  end

  def get_attestations_part
    bundles = "[" + options[:attestations].map do |attestation|
      Gem.read_binary(attestation)
    end.join(",") + "]"
    [
      "attestations",
      bundles,
      { content_type: "application/json" },
    ]
  end
end
                                                                                                                                                                                                                                                                                          rubygems/rubygems/commands/open_command.rb                                                          0000644                 00000003635 15040313420 0015022 0                                                                                                    ustar 00                                                                                                                                                                                                                                                       # frozen_string_literal: true

require_relative "../command"
require_relative "../version_option"

class Gem::Commands::OpenCommand < Gem::Command
  include Gem::VersionOption

  def initialize
    super "open", "Open gem sources in editor"

    add_option("-e", "--editor COMMAND", String,
               "Prepends COMMAND to gem path. Could be used to specify editor.") do |command, options|
      options[:editor] = command || get_env_editor
    end
    add_option("-v", "--version VERSION", String,
               "Opens specific gem version") do |version|
      options[:version] = version
    end
  end

  def arguments # :nodoc:
    "GEMNAME     name of gem to open in editor"
  end

  def defaults_str # :nodoc:
    "-e #{get_env_editor}"
  end

  def description # :nodoc:
    <<-EOF
        The open command opens gem in editor and changes current path
        to gem's source directory.
        Editor command can be specified with -e option, otherwise rubygems
        will look for editor in $EDITOR, $VISUAL and $GEM_EDITOR variables.
    EOF
  end

  def usage # :nodoc:
    "#{program_name} [-e COMMAND] GEMNAME"
  end

  def get_env_editor
    ENV["GEM_EDITOR"] ||
      ENV["VISUAL"] ||
      ENV["EDITOR"] ||
      "vi"
  end

  def execute
    @version = options[:version] || Gem::Requirement.default
    @editor  = options[:editor] || get_env_editor

    found = open_gem(get_one_gem_name)

    terminate_interaction 1 unless found
  end

  def open_gem(name)
    spec = spec_for name

    return false unless spec

    if spec.default_gem?
      say "'#{name}' is a default gem and can't be opened."
      return false
    end

    open_editor(spec.full_gem_path)
  end

  def open_editor(path)
    system(*@editor.split(/\s+/) + [path], { chdir: path })
  end

  def spec_for(name)
    spec = Gem::Specification.find_all_by_name(name, @version).first

    return spec if spec

    say "Unable to find gem '#{name}'"
  end
end
                                                                                                   rubygems/rubygems/commands/sources_command.rb                                                       0000644                 00000013375 15040313420 0015546 0                                                                                                    ustar 00                                                                                                                                                                                                                                                       # frozen_string_literal: true

require_relative "../command"
require_relative "../remote_fetcher"
require_relative "../spec_fetcher"
require_relative "../local_remote_options"

class Gem::Commands::SourcesCommand < Gem::Command
  include Gem::LocalRemoteOptions

  def initialize
    require "fileutils"

    super "sources",
          "Manage the sources and cache file RubyGems uses to search for gems"

    add_option "-a", "--add SOURCE_URI", "Add source" do |value, options|
      options[:add] = value
    end

    add_option "-l", "--list", "List sources" do |value, options|
      options[:list] = value
    end

    add_option "-r", "--remove SOURCE_URI", "Remove source" do |value, options|
      options[:remove] = value
    end

    add_option "-c", "--clear-all",
               "Remove all sources (clear the cache)" do |value, options|
      options[:clear_all] = value
    end

    add_option "-u", "--update", "Update source cache" do |value, options|
      options[:update] = value
    end

    add_option "-f", "--[no-]force", "Do not show any confirmation prompts and behave as if 'yes' was always answered" do |value, options|
      options[:force] = value
    end

    add_proxy_option
  end

  def add_source(source_uri) # :nodoc:
    check_rubygems_https source_uri

    source = Gem::Source.new source_uri

    check_typo_squatting(source)

    begin
      if Gem.sources.include? source
        say "source #{source_uri} already present in the cache"
      else
        source.load_specs :released
        Gem.sources << source
        Gem.configuration.write

        say "#{source_uri} added to sources"
      end
    rescue Gem::URI::Error, ArgumentError
      say "#{source_uri} is not a URI"
      terminate_interaction 1
    rescue Gem::RemoteFetcher::FetchError => e
      say "Error fetching #{Gem::Uri.redact(source.uri)}:\n\t#{e.message}"
      terminate_interaction 1
    end
  end

  def check_typo_squatting(source)
    if source.typo_squatting?("rubygems.org")
      question = <<-QUESTION.chomp
#{source.uri} is too similar to https://rubygems.org

Do you want to add this source?
      QUESTION

      terminate_interaction 1 unless options[:force] || ask_yes_no(question)
    end
  end

  def check_rubygems_https(source_uri) # :nodoc:
    uri = Gem::URI source_uri

    if uri.scheme && uri.scheme.casecmp("http").zero? &&
       uri.host.casecmp("rubygems.org").zero?
      question = <<-QUESTION.chomp
https://rubygems.