# frozen_string_literal: true

##
# The SourceSet chooses the best available method to query a remote index.
#
# Kind off like BestSet but filters the sources for gems

class Gem::Resolver::SourceSet < Gem::Resolver::Set
  ##
  # Creates a SourceSet for the given +sources+ or Gem::sources if none are
  # specified.  +sources+ must be a Gem::SourceList.

  def initialize
    super()

    @links = {}
    @sets  = {}
  end

  def find_all(req) # :nodoc:
    if set = get_set(req.dependency.name)
      set.find_all req
    else
      []
    end
  end

  # potentially no-op
  def prefetch(reqs) # :nodoc:
    reqs.each do |req|
      if set = get_set(req.dependency.name)
        set.prefetch reqs
      end
    end
  end

  def add_source_gem(name, source)
    @links[name] = source
  end

  private

  def get_set(name)
    link = @links[name]
    @sets[link] ||= Gem::Source.new(link).dependency_resolver_set if link
  end
end
                                                                                             rubygems/rubygems/resolver/vendor_specification.rb                                                  0000644                 00000001101 15040313417 0016610 0                                                                                                    ustar 00                                                                                                                                                                                                                                                       # frozen_string_literal: true

##
# A VendorSpecification represents a gem that has been unpacked into a project
# and is being loaded through a gem dependencies file through the +path:+
# option.

class Gem::Resolver::VendorSpecification < Gem::Resolver::SpecSpecification
  def ==(other) # :nodoc:
    self.class === other &&
      @set  == other.set &&
      @spec == other.spec &&
      @source == other.source
  end

  ##
  # This is a null install as this gem was unpacked into a directory.
  # +options+ are ignored.

  def install(options = {})
    yield nil
  end
end
                                                                                                                                                                                                                                                                                                                                                                                                                                                               rubygems/rubygems/resolver/requirement_list.rb                                                      0000644                 00000002530 15040313417 0016015 0                                                                                                    ustar 00                                                                                                                                                                                                                                                       # frozen_string_literal: true

##
# The RequirementList is used to hold the requirements being considered
# while resolving a set of gems.
#
# The RequirementList acts like a queue where the oldest items are removed
# first.

class Gem::Resolver::RequirementList
  include Enumerable

  ##
  # Creates a new RequirementList.

  def initialize
    @exact = []
    @list = []
  end

  def initialize_copy(other) # :nodoc:
    @exact = @exact.dup
    @list = @list.dup
  end

  ##
  # Adds Resolver::DependencyRequest +req+ to this requirements list.

  def add(req)
    if req.requirement.exact?
      @exact.push req
    else
      @list.push req
    end
    req
  end

  ##
  # Enumerates requirements in the list

  def each # :nodoc:
    return enum_for __method__ unless block_given?

    @exact.each do |requirement|
      yield requirement
    end

    @list.each do |requirement|
      yield requirement
    end
  end

  ##
  # How many elements are in the list

  def size
    @exact.size + @list.size
  end

  ##
  # Is the list empty?

  def empty?
    @exact.empty? && @list.empty?
  end

  ##
  # Remove the oldest DependencyRequest from the list.

  def remove
    return @exact.shift unless @exact.empty?
    @list.shift
  end

  ##
  # Returns the oldest five entries from the list.

  def next5
    x = @exact[0,5]
    x + @list[0,5 - x.size]
  end
end
                                                                                                                                                                        rubygems/rubygems/resolver/vendor_set.rb                                                            0000644                 00000003647 15040313417 0014604 0                                                                                                    ustar 00                                                                                                                                                                                                                                                       # frozen_string_literal: true

##
# A VendorSet represents gems that have been unpacked into a specific
# directory that contains a gemspec.
#
# This is used for gem dependency file support.
#
# Example:
#
#   set = Gem::Resolver::VendorSet.new
#
#   set.add_vendor_gem 'rake', 'vendor/rake'
#
# The directory vendor/rake must contain an unpacked rake gem along with a
# rake.gemspec (watching the given name).

class Gem::Resolver::VendorSet < Gem::Resolver::Set
  ##
  # The specifications for this set.

  attr_reader :specs # :nodoc:

  def initialize # :nodoc:
    super()

    @directories = {}
    @specs       = {}
  end

  ##
  # Adds a specification to the set with the given +name+ which has been
  # unpacked into the given +directory+.

  def add_vendor_gem(name, directory) # :nodoc:
    gemspec = File.join directory, "#{name}.gemspec"

    spec = Gem::Specification.load gemspec

    raise Gem::GemNotFoundException,
          "unable to find #{gemspec} for gem #{name}" unless spec

    spec.full_gem_path = File.expand_path directory

    @specs[spec.name]  = spec
    @directories[spec] = directory

    spec
  end

  ##
  # Returns an Array of VendorSpecification objects matching the
  # DependencyRequest +req+.

  def find_all(req)
    @specs.values.select do |spec|
      req.match? spec
    end.map do |spec|
      source = Gem::Source::Vendor.new @di