      end
#
#       extend Gem::Deprecate
#       deprecate :some_instance_method, "X.z", 2011, 4
#       rubygems_deprecate :some_old_method, "Modern#some_new_method"
#
#       class << self
#         extend Gem::Deprecate
#         deprecate :some_class_method, :none, 2011, 4
#       end
#     end
#
#
# Example usage of +rubygems_deprecate_command+:
#
#     class Gem::Commands::QueryCommand < Gem::Command
#       extend Gem::Deprecate
#       rubygems_deprecate_command
#
#       # ...
#     end
#
#
# Example usage of +skip_during+:
#
#     class TestSomething < Gem::Testcase
#       def test_some_thing_with_deprecations
#         Gem::Deprecate.skip_during do
#           actual_stdout, actual_stderr = capture_output do
#             Gem.something_deprecated
#           end
#           assert_empty actual_stdout
#           assert_equal(expected, actual_stderr)
#         end
#       end
#     end

module Gem
  module Deprecate
    def self.skip # :nodoc:
      @skip ||= false
    end

    def self.skip=(v) # :nodoc:
      @skip = v
    end

    ##
    # Temporarily turn off warnings. Intended for tests only.

    def skip_during
      original = Gem::Deprecate.skip
      Gem::Deprecate.skip = true
      yield
    ensure
      Gem::Deprecate.skip = original
    end

    def self.next_rubygems_major_version # :nodoc:
      Gem::Version.new(Gem.rubygems_version.segments.first).bump
    end

    ##
    # Simple deprecation method that deprecates +name+ by wrapping it up
    # in a dummy method. It warns on each call to the dummy method
    # telling the user of +repl+ (unless +repl+ is :none) and the
    # year/month that it is planned to go away.

    def deprecate(name, repl, year, month)
      class_eval do
        old = "_deprecated_#{name}"
        alias_method old, name
        define_method name do |*args, &block|
          klass = is_a? Module
          target = klass ? "#{self}." : "#{self.class}#"
          msg = [
            "NOTE: #{target}#{name} is deprecated",
            repl == :none ? " with no replacement" : "; use #{repl} instead",
            format(". It will be removed on or after %4d-%02d.", year, month),
            "\n#{target}#{name} called from #{Gem.location_of_caller.join(":")}",
          ]
          warn "#{msg.join}." unless Gem::Deprecate.skip
          send old, *args, &block
        end
        ruby2_keywords name if respond_to?(:ruby2_keywords, true)
      end
    end

    ##
    # Simple deprecation method that deprecates +name+ by wrapping it up
    # in a dummy method. It warns on each call to the dummy method
    # telling the user of +repl+ (unless +repl+ is :none) and the
    # Rubygems version that it is planned to go away.

    def rubygems_deprecate(name, replacement=:none)
      class_eval do
        old = "_deprecated_#{name}"
        alias_method old, name
        define_method name do |*args, &block|
          klass = is_a? Module
          target = klass ? "#{self}." : "#{self.class}#"
          msg = [
            "NOTE: #{target}#{name} is deprecated",
            replacement == :none ? " with no replacement" : "; use #{replacement} instead",
            ". It will be removed in Rubygems #{Gem::Deprecate.next_rubygems_major_version}",
            "\n#{target}#{name} called from #{Gem.location_of_caller.join(":")}",
          ]
          warn "#{msg.join}." unless Gem::Deprecate.skip
          send old, *args, &block
        end
        ruby2_keywords name if respond_to?(:ruby2_keywords, true)
      end
    end

    # Deprecation method to deprecate Rubygems commands
    def rubygems_deprecate_command(version = Gem::Deprecate.next_rubygems_major_version)
      class_eval do
        define_method "deprecated?" do
          true
        end

        define_method "deprecation_warning" do
          msg = [
            "#{command} command is deprecated",
            ". It will be removed in Rubygems #{version}.\n",
          ]

          alert_warning msg.join.to_s unless Gem::Deprecate.skip
        end
      end
    end

    module_function :rubygems_deprecate, :rubygems_deprecate_command, :skip_during
  end
end
                                                                                                                                                                                                                                                                                                                                                                                                                                                                                         rubygems/rubygems/security/signer.rb                                                                0000644                 00000013702 15040313417 0013722 0                                                                                                    ustar 00                                                                                                                                                                                                                                                       # frozen_string_literal: true

##
# Basic OpenSSL-based package signing class.

require_relative "../user_interaction"

class Gem::Security::Signer
  include Gem::UserInteraction

  ##
  # The chain of certificates for signing including the signing certificate

  attr_accessor :cert_chain

  ##
  # The private key for the signing certificate

  attr_accessor :key

  ##
  # The digest algorithm used to create the signature

  attr_reader :digest_algorithm

  ##
  # The name of the digest algorithm, used to pull digests out of the hash by
  # name.

  attr_reader :digest_name # :nodoc:

  ##
  # Gem::Security::Signer options

  attr_reader :options

  DEFAULT_OPTIONS = {
    expiration_length_days: 365,
  }.freeze

  ##
  # Attempts to re-sign an expired cert with a given private key
  def self.re_sign_cert(expired_cert, expired_cert_path, private_key)
    return unless expired_cert.not_after < Time.now

    expiry = expired_cert.not_after.strftime("%Y%m%d%H%M%S")
    expired_cert_file = "#{File.basename(expired_cert_path)}.expired.#{expiry}"
    new_expired_cert_path = File.join(Gem.user_home, ".gem", expired_cert_file)

    Gem::Security.write(expired_cert, new_expired_cert_path)

    re_signed_cert = Gem::Security.re_sign(
      expired_cert,
      private_key,
      (Gem::Security::ONE_DAY * Gem.configuration.cert_expiration_length_days)
    )

    Gem::Security.write(re_signed_cert, expired_cert_path)

    yield(expired_cert_path, new_expired_cert_path) if block_given?
  end

  ##
  # Creates a new signer with an RSA +key+ or path to a key, and a certificate
  # +chain+ containing X509 certificates, encoding certificates or paths to
  # certificates.

  def initialize(key, cert_chain, passphrase = nil, options = {})
    @cert_chain = cert_chain
    @key        = key
    @passphrase = passphrase
    @options = DEFAULT_OPTIONS.merge(options)

    unless @key
      default_key = File.join Gem.default_key_path
      @key = default_key if File.exist? default_key
    end

    unless @cert_chain
      default_cert = File.join Gem.default_cert_path
      @cert_chain = [default_cert] if File.exist? default_cert
    end

    @digest_name      = Gem::Security::DIGEST_NAME
    @digest_algorithm = Gem::Security.create_digest(@digest_name)

    if @key && !@key.is_a?(OpenSSL::PKey::PKey)
      @key = OpenSSL::PKey.read(File.read(@key), @passphrase)
    end

    if @cert_chain
      @cert_chain = @cert_chain.compact.map do |cert|
        next cert if OpenSSL::X509::Certificate === cert

        cert = File.read cert if File.exist? cert

        OpenSSL::X509::Certificate.new cert
      end

      load_cert_chain
    end
  end

  ##
  # Extracts the full name of +cert+.  If the certificate has a subjectAltName
  # this value is preferred, otherwise the subject is used.

  def extract_name(cert) # :nodoc:
    subject_alt_name = cert.extensions.fin