   = serial

    cert.not_before = Time.now
    cert.not_after  = Time.now + age

    cert.subject    = subject

    ef = OpenSSL::X509::ExtensionFactory.new nil, cert

    cert.extensions = extensions.map do |ext_name, value|
      ef.create_extension ext_name, value
    end

    cert
  end

  ##
  # Gets the right public key from a PKey instance

  def self.get_public_key(key)
    # Ruby 3.0 (Ruby/OpenSSL 2.2) or later
    return OpenSSL::PKey.read(key.public_to_der) if key.respond_to?(:public_to_der)
    return key.public_key unless key.is_a?(OpenSSL::PKey::EC)

    ec_key = OpenSSL::PKey::EC.new(key.group.curve_name)
    ec_key.public_key = key.public_key
    ec_key
  end

  ##
  # Creates a self-signed certificate with an issuer and subject from +email+,
  # a subject alternative name of +email+ and the given +extensions+ for the
  # +key+.

  def self.create_cert_email(email, key, age = ONE_YEAR, extensions = EXTENSIONS)
    subject = email_to_name email

    extensions = extensions.merge "subjectAltName" => "email:#{email}"

    create_cert_self_signed subject, key, age, extensions
  end

  ##
  # Creates a self-signed certificate with an issuer and subject of +subject+
  # and the given +extensions+ for the +key+.

  def self.create_cert_self_signed(subject, key, age = ONE_YEAR, extensions = EXTENSIONS, serial = 1)
    certificate = create_cert subject, key, age, extensions

    sign certificate, key, certificate, age, extensions, serial
  end

  ##
  # Creates a new digest instance using the specified +algorithm+. The default
  # is SHA256.

  def self.create_digest(algorithm = DIGEST_NAME)
    OpenSSL::Digest.new(algorithm)
  end

  ##
  # Creates a new key pair of the specified +algorithm+. RSA, DSA, and EC
  # are supported.

  def self.create_key(algorithm)
    if defined?(OpenSSL::PKey)
      case algorithm.downcase
      when "dsa"
        OpenSSL::PKey::DSA.new(RSA_DSA_KEY_LENGTH)
      when "rsa"
        OpenSSL::PKey::RSA.new(RSA_DSA_KEY_LENGTH)
      when "ec"
        OpenSSL::PKey::EC.generate(EC_NAME)
      else
        raise Gem::Security::Exception,
        "#{algorithm} algorithm not found. RSA, DSA, and EC algorithms are supported."
      end
    end
  end

  ##
  # Turns +email_address+ into an OpenSSL::X509::Name

  def self.email_to_name(email_address)
    email_address = email_address.gsub(/[^\w@.-]+/i, "_")

    cn, dcs = email_address.split "@"

    dcs = dcs.split "."

    OpenSSL::X509::Name.new([
      ["CN", cn],
      *dcs.map {|dc| ["DC", dc] },
    ])
  end

  ##
  # Signs +expired_certificate+ with +private_key+ if the keys match and the
  # expired certificate was self-signed.
  #--
  # TODO increment serial

  def self.re_sign(expired_certificate, private_key, age = ONE_YEAR, extensions = EXTENSIONS)
    raise Gem::Security::Exception,
          "incorrect signing key for re-signing " +
          expired_certificate.subject.to_s unless
      expired_certificate.check_private_key(private_key)

    unless expired_certificate.subject.to_s ==
           expired_certificate.issuer.to_s
      subject = alt_name_or_x509_entry expired_certificate, :subject
      issuer  = alt_name_or_x509_entry expired_certificate, :issuer

      raise Gem::Security::Exception,
            "#{subject} is not self-signed, contact #{issuer} " \
            "to obtain a valid certificate"
    end

    serial = expired_certificate.serial + 1

    create_cert_self_signed(expired_certificate.subject, private_key, age,
                            extensions, serial)
  end

  ##
  # Resets the trust directory for verifying gems.

  def self.reset
    @trust_dir = nil
  end

  ##
  # Sign the public key from +certificate+ with the +signing_key+ and
  # +signing_cert+, using the Gem::Security::DIGEST_NAME.  Uses the
  # default certificate validity range and extensions.
  #
  # Returns the newly signed certificate.

  def self.sign(certificate, signing_key, signing_cert, age = ONE_YEAR, extensions = EXTENSIONS, serial = 1)
    signee_subject = certificate.subject
    signee_key     = certificate.public_key

    alt_name = certificate.extensions.find do |extension|
      extension.oid == "subjectAltName"
    end

    extensions = extensions.merge "subjectAltName" => alt_name.value if
      alt_name

    issuer_alt_name = signing_cert.extensions.find do |extension|
      extension.oid == "subjectAltName"
    end

    extensions = extensions.merge "issuerAltName" => issuer_alt_name.value if
      issuer_alt_name

    signed = create_cert signee_subject, signee_key, age, extensions, serial
    signed.issuer = signing_cert.subject

    signed.sign signing_key, Gem::Security::DIGEST_NAME
  end

  ##
  # Returns a Gem::Security::TrustDir which wraps the directory where trusted
  # certificates live.

  def self.trust_dir
    return @trust_dir if @trust_dir

    dir = File.join Gem.user_home, ".gem", "trust"

    @trust_dir ||= Gem::Security::TrustDir.new dir
  end

  ##
  # Enumerates the trusted certificates via Gem::Security::TrustDir.

  def self.trusted_certificates(&block)
    trust_dir.each_certificate(&block)
  end

  ##
  # Writes +pemmable+, which must respond to +to_pem+ to +path+ with the given
  # +permissions+. If passed +cipher+ and +passphrase+ those arguments will be
  # passed to +to_pem+.

  def self.write(pemmable, path, permissions = 0o600, passphrase = nil, cipher = KEY_CIPHER)
    path = File.expand_path path

    File.open path, "wb", permissions do |io|
      if passphrase && cipher
        io.write pemmable.to_pem cipher, passphrase
      else
        io.write pemmable.to_pem
      end
    end

    path
  end

  reset
end

if Gem::HAVE_OPENSSL
  require_relative "security/policy"
  require_relative "security/policies"
  require_relative "security/trust_dir"
end

require_relative "security/signer"
                                                                                                                                                                                                                                                                                                                          rubygems/rubygems/tsort/lib/tsort.rb                                                                0000644                 00000034751 15040313423 0013664 0                                                                                                    ustar 00                                                                                                                                                                                                                                                       aPods/Molinillo
[PubGrub]:            https://github.com/jhawthorn/pub_grub
[GH-net-protocol-14]: https://github.com/ruby/net-protocol/pull/14
[GH-pathname-20]:     https://github.com/ruby/pathname/pull/20
[GH-6791]:            https://github.com/ruby/ruby/pull/6791
[GH-6868]:            https://github.com/ruby/ruby/pull/6868
[GH-rubygems-4475]:   https://github.com/rubygems/rubygems/pull/4475
[GH-rubygems-6149]:   https://github.com/rubygems/rubygems/pull/6149
[GH-rubygems-6167]:   https://github.com/rubygems/rubygems/pull/6167
[sec-156615]:         https://hackerone.com/reports/156615
[CVE-2021-33621]:     https://www.ruby-lang.org/en/news/2022/11/22/http-response-splitting-in-cgi-cve-2021-33621/
[wasm/README.md]:     https://github.com/ruby/ruby/blob/master/wasm/README.md
[ruby.wasm]:          https://github.com/ruby/ruby.wasm
                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                             ruby/expect.rb                                                                                      0000644                 00000004304 15040313423 0007336 0                                                                                                    ustar 00                                                                                                                                                                                                                                                       # frozen_string_literal: true
$expect_verbose = false

class IO
  # call-seq:
  #   IO#expect(pattern,timeout=9999999)                  ->  Array
  #   IO#expect(pattern,timeout=9999999) { |result                                                                                                                                                                                                                                                                                                                            ruby/rinda/rinda.rb                                                                                 0000644                 00000015123 15040313423 0010241 0                                                                                                    ustar 00                                                                                                                                                                                                                                                       pattern matches, followed by elements indicating which the
  # pattern which matched to the anchor in the regular expression.
  #
  # The optional timeout parameter defines, in seconds, the total time to wait
  # for the pattern.  If the timeout expires or eof is found, nil is returned
  # or yielded.  However, the buffer in a timeout session is kept for the next
  # expect call.  The default timeout is 9999999 seconds.
  def expect(pat,timeout=9999999)
    buf = ''.dup
    case pat
    when String
      e_pat = Regexp.new(Regexp.quote(pat))
    when Regexp
      e_pat = pat
    else
      raise TypeError, "unsupported pattern class: #{pat.class}"
    end
    @unusedBuf ||= ''
    while true
      if not @unusedBuf.empty?
        c = @unusedBuf.slice!(0)
      elsif !IO.select([self],nil,nil,timeout) or eof? then
        result = nil
        @unusedBuf = buf
        break
      else
        c = getc
      end
      buf << c
      if $expect_verbose
        STDOUT.print c
        STDOUT.flush
      end
      if mat=e_pat.match(buf) then
        result = [buf,*mat.captures]
        break
      end
    end
    if block_given? then
      yield result
    else
      return result
    end
    nil
  end
end
                                                                                                                                                                                                                                                                                                                            ruby/rinda/rinda.rb                                                                                 0000644                 00000015123 15040313423 0010241 0                                                                                                    ustar 00                                                                                                                                                                                                                                                       # frozen_string_literal: false
require 'drb/drb'

##
# A module to implement the Linda distributed computing paradigm in Ruby.
#
# Rinda is part of DRb (dRuby).
#
# == Example(s)
#
# See the sample/drb/ directory in the Ruby distribution, from 1.8.2 onwards.
#
#--
# TODO
# == Introduction to Linda/rinda?
#
# == Why is this library separate from DRb?

module Rinda

  ##
  # Rinda error base class

  class RindaError < RuntimeError; end

  ##
  # Raised when a hash-based tuple has an invalid key.

  class InvalidHashTupleKey < RindaError; end

  ##
  # Raised when trying to use a canceled tuple.

  class RequestCanceledError < ThreadError; end

  ##
  # Raised when trying to use an expired tuple.

  class RequestExpiredError < ThreadError; end

  ##
  # A tuple is the elementary object in Rinda programming.
  # Tuples may be matched against templates if the tuple and
  # the template are the same size.

  class Tuple

    ##
    # Creates a new Tuple from +ary_or_hash+ which must be an Array or Hash.

    def initialize(ary_or_hash)
      if hash?(ary_or_hash)
        init_with_hash(ary_or_hash)
      else
        init_with_ary(ary_or_hash)
      end
    end

    ##
    # The number of elements in the tuple.

    def size
      @tuple.size
    end

    ##
    # Accessor method for elements of the tuple.

    def [](k)
      @tuple[k]
    end

    ##
    # Fetches item +k+ from the tuple.

    def fetch(k)
      @tuple.fetch(k)
    end

    ##
    # Iterate through the tuple, yielding the index or key, and the
    # value, thus ensuring arrays are iterated similarly to hashes.

    def each # FIXME
      if Hash === @tuple
        @tuple.each { |k, v| yield(k, v) }
      else
        @tuple.each_with_index { |v, k| yield(k, v) }
      end
    end

    ##
    # Return the tuple itself
    def value
      @tuple
    end

    private

    def hash?(ary_or_hash)
      ary_or_hash.respond_to?(:keys)
    end

    ##
    # Munges +ary+ into a valid Tuple.

    def init_with_ary(ary)
      @tuple = Array.new(ary.size)
      @tuple.size.times do |i|
        @tuple[i] = ary[i]
      end
    end

    ##
    # Ensures +hash+ is a valid Tuple.

    def init_with_hash(hash)
      @tuple = Hash.new
      hash.each do |k, v|
        raise InvalidHashTupleKey unless String === k
        @tuple[k] = v
      end
    end

  end

  ##
  # Templates are used to match tuples in Rinda.

  class Template < Tuple

    ##
    # Matches this template against +tuple+.  The +tuple+ must be the same
    # size as the template.  An element with a +nil+ value in a template acts
    # as a wildcard, matching any value in the corresponding position in the
    # tuple.  Elements of the template match the +tuple+ if the are #== or
    # #===.
    #
    #   Template.new([:foo, 5]).match   Tuple.new([:foo, 5]) # => true
    #   Template.new([:foo, nil]).match Tuple.new([:foo, 5]) # => true
    #   Template.new([String]).match    Tuple.new(['hello']) # => true
    #
    #   Template.new([:foo]).match      Tuple.new([:foo, 5]) # => false
    #   Template.new([:foo, 6]).match   Tuple.new([:foo, 5]) # => false
    #   Template.new([:foo, nil]).match Tuple.new([:foo])    # => false
    #   Template.new([:foo, 6]).match   Tuple.new([:foo])    # => false

    def match(tuple)
      return false unless tuple.respond_to?(:size)
      return false unless tuple.respond_to?(:fetch)
      return false unless self.size == tuple.size
      each do |k, v|
        begin
          it = tuple.fetch(k)
        rescue
          return false
        end
        next if v.nil?
        next if v == it
        next if v === it
        return false
      end
      return true
    end

    ##
    # Alias for #match.

    def ===(tuple)
      match(tuple)
    end

  end

  ##
  # <i>Documentation?</i>

  class DRbObjectTemplate

    ##
    # Creates a new DRbObjectTemplate that will match against +uri+ and +ref+.

    def initialize(uri=nil, ref=nil)
      @drb_uri = uri
      @drb_ref = ref
    end

    ##
    # This DRbObjectTemplate matches +ro+ if the remote object's drburi and
    # drbref are the same.  +nil+ is used as a wildcard.

    def ===(ro)
      return true if super(ro)
      unless @drb_uri.nil?
        return false unless (@drb_uri === ro.__drburi rescue false)
      end
      unless @drb_ref.nil?
        return false unless (@drb_ref === ro.__drbref rescue false)
      end
      true
    end

  end

  ##
  # TupleSpaceProxy allows a remote Tuplespace to appear as local.

  class TupleSpaceProxy
    ##
    # A Port ensures that a moved tuple arrives properly at its destination
    # and does not get lost.
    #
    # See https://bugs.ruby-lang.org/issues/8                                                                                                                                                                                                                                                                                                                                                                                                                                             ruby/rinda/ring.rb                                                                                  0000644                 00000031061 15040313423 0010102 0                                                                                                    ustar 00                                                                                                                                                                                                                                                       See TupleSpace#take.

    def take(tuple, sec=nil, &block)
      Port.deliver do |port|
        @ts.move(DRbObject.new(port), tuple, sec, &block)
      end
    end

    ##
    # Reads +tuple+ from the proxied TupleSpace.  See TupleSpace#read.

    def read(tuple, sec=nil, &block)
      @ts.read(tuple, sec, &block)
    end

    ##
    # Reads all tuples matching +tuple+ from the proxied TupleSpace.  See
    # TupleSpace#read_all.

    def read_all(tuple)
      @ts.read_all(tuple)
    end

    ##
    # Registers for notifications of event +ev+ on the proxied TupleSpace.
    # See TupleSpace#notify

    def notify(ev, tuple, sec=nil)
      @ts.notify(ev, tuple, sec)
    end

  end

  ##
  # An SimpleRenewer allows a TupleSpace to check if a TupleEntry is still
  # alive.

  class SimpleRenewer

    include DRbUndumped

    ##
    # Creates a new SimpleRenewer that keeps an object alive for another +sec+
    # seconds.

    def initialize(sec=180)
      @sec = sec
    end

    ##
    # Called by the TupleSpace to check if the object is still alive.

    def renew
      @sec
    end
  end

end

                                                                                                                                                                                                                                                                                                                                                                                                                                             ruby/rinda/ring.rb                                                                                  0000644                 00000031061 15040313423 0010102 0                                                                                                    ustar 00                                                                                                                                                                                                                                                       # frozen_string_literal: false
#
# Note: Rinda::Ring API is unstable.
#
require 'drb/drb'
require_relative 'rinda'
require 'ipaddr'

module Rinda

  ##
  # The default port Ring discovery will use.

  Ring_PORT = 7647

  ##
  # A RingServer allows a Rinda::TupleSpace to be located via UDP broadcasts.
  # Default service location uses the following steps:
  #
  # 1. A RingServer begins listening on the network broadcast UDP address.
  # 2. A RingFinger sends a UDP packet containing the DRb URI where it will
  #    listen for a reply.
  # 3. The RingServer receives the UDP packet and connects back to the
  #    provided DRb URI with the DRb service.
  #
  # A RingServer requires a TupleSpace:
  #
  #   ts = Rinda::TupleSpace.new
  #   rs = Rinda::RingServer.new
  #
  # RingServer can also listen on multicast addresses for announcements.  This
  # allows multiple RingServers to run on the same host.  To use network
  # broadcast and multicast:
  #
  #   ts = Rinda::TupleSpace.new
  #   rs = Rinda::RingServer.new ts, %w[Socket::INADDR_ANY, 239.0.0.1 ff02::1]

  class RingServer

    include DRbUndumped

    ##
    # Special renewer for the RingServer to allow shutdown

    class Renewer # :nodoc:
      include DRbUndumped

      ##
      # Set to false to shutdown future requests using this Renewer

      attr_writer :renew

      def initialize # :nodoc:
        @renew = true
      end

      def renew # :nodoc:
        @renew ? 1 : true
      end
    end

    ##
    # Advertises +ts+ on the given +addresses+ at +port+.
    #
    # If +addresses+ is omitted only the UDP broadcast address is used.
    #
    # +addresses+ can contain multiple addresses.  If a multicast address is
    # given in +addresses+ then the RingServer will listen for multicast
    # queries.
    #
    # If you use IPv4 multicast you may need to set an address of the inbound
    # interface which joins a multicast group.
    #
    #   ts = Rinda::TupleSpace.new
    #   rs = Rinda::RingServer.new(ts, [['239.0.0.1', '9.5.1.1']])
    #
    # You can set addresses as an Array Object.  The first element of the
    # Array is a multicast address and the second is an inbound interface
    # address.  If the second is omitted                        rubygems/rubygems/stub_specification.rb                                                             0000644                 00000011726 15040313423 0014442 0                                                                                                    ustar 00                                                                                                                                                                                                                                                       # frozen_string_literal: true

##
# Gem::StubSpecification reads the stub: line from the gemspec.  This prevents
# us having to eval the entire gemspec in order to find out certain
# information.

class Gem::StubSpecification < Gem::BasicSpecification
  # :nodoc:
  PREFIX = "# stub: "

  # :nodoc:
  OPEN_MODE = "r:UTF-8:-"

  class StubLine # :nodoc: all
    attr_reader :name, :version, :platform, :require_paths, :extensions,
                :full_name

    NO_EXTENSIONS = [].freeze

    # These are common require paths.
    REQUIRE_PATHS = { # :nodoc:
      "lib" => "lib",
      "test" => "test",
      "ext" => "ext",
    }.freeze

    # These are common require path lists.  This hash is used to optimize
    # and consolidate require_path objects.  Most specs just specify "lib"
    # in their require paths, so lets take advantage of that by pre-allocating
    # a require path list for that case.
    REQUIRE_PATH_LIST = { # :nodoc:
      "lib" => ["lib"].freeze,
    }.freeze

    def initialize(data, extensions)
      parts          = data[PREFIX.length..-1].split(" ", 4)
      @name          = -parts[0]
      @version       = if Gem::Version.correct?(parts[1])
        Gem::Version.new(parts[1])
      else
        Gem::Version.new(0)
      end

      @platform      = Gem::Platform.new parts[2]
      @extensions    = extensions
      @full_name     = if platform == Gem::Platform::RUBY
        "#{name}-#{version}"
      else
        "#{name}-#{version}-#{platform}"
      end

      path_list = parts.last
      @require_paths = REQUIRE_PATH_LIST[path_list] || path_list.split("\0").map! do |x|
        REQUIRE_PATHS[x] || x
      end
    end
  end

  def self.default_gemspec_stub(filename, base_dir, gems_dir)
    new filename, base_dir, gems_dir, true
  end

  def self.gemspec_stub(filename, base_dir, gems_dir)
    new filename, base_dir, gems_dir, false
  end

  attr_reader :base_dir, :gems_dir

  def initialize(filename, base_dir, gems_dir, default_gem)
    super()

    self.loaded_from = filename
    @data            = nil
    @name            = nil
    @spec            = nil
    @base_dir        = base_dir
    @gems_dir        = gems_dir
    @default_gem     = default_gem
  end

  ##
  # True when this gem has been activated

  def activated?
    @activated ||= !loaded_spec.nil?
  end

  def default_gem?
    @default_gem
  end

  def build_extensions # :nodoc:
    return if default_gem?
    return if extensions.empty?

    to_spec.build_extensions
  end

  ##
  # If the gemspec contains a stubline, returns a StubLine instance. Otherwise
  # returns the full Gem::Specification.

  def data
    unless @data
      begin
        saved_lineno = $.

        Gem.open_file loaded_from, OPEN_MODE do |file|
          file.readline # discard encoding line
          stubline = file.readline
          if stubline.start_with?(PREFIX)
            extline = file.readline

            extensions =
              if extline.delete_prefix!(PREFIX)
                extline.chomp!
                extline.split "\0"
              else
                StubLine::NO_EXTENSIONS
              end

            stubline.chomp! # readline(chomp: true) allocates 3x as much as .readline.chomp!
            @data = StubLine.new stubline, extensions
          end
        rescue EOFError
        end
      ensure
        $. = saved_lineno
      end
    end

    @data ||= to_spec
  end

  private :data

  def raw_require_paths # :nodoc:
    data.require_paths
  end

  def missing_extensions?
    return false if default_gem?
    return false if extensions.empty?
    return false if File.exist? gem_build_complete_path

    to_spec.missing_extensions?
  end

  ##
  # Name of the gem

  def name
    data.name
  end

  ##
  # Platform of the gem

  def platform
    data.platform
  end

  ##
  # Extensions for this gem

  def extensions
    data.extensions
  end

  ##
  # Version of the gem

  def version
    data.version
  end

  def full_name
    data.full_name
  end

  ##
  # The full Gem::Specification for this gem, loaded from evalling its gemspec

  def spec
    @spec ||= loaded_spec if @data
    @spec ||= Gem::Specification.load(loaded_from)
  end
  alias_method :to_spec, :spec

  ##
  # Is this StubSpecification valid? i.e. have we found a stub line, OR does
  # the filename contain a valid gemspec?

  def valid?
    data
  end

  ##
  # Is there a stub line present for this StubSpecification?

  def stubbed?
    data.is_a? StubLine
  end

  def ==(other) # :nodoc:
    self.class === other &&
      name == other.name &&
      version == other.version &&
      platform == other.platform
  end

  alias_method :eql?, :== # :nodoc:

  def hash # :nodoc:
    name.hash ^ version.hash ^ platform.hash
  end

  def <=>(other) # :nodoc:
    sort_obj <=> other.sort_obj
  end

  def sort_obj # :nodoc:
    [name, version, Gem::Platform.sort_priority(platform)]
  end

  private

  def loaded_spec
    spec = Gem.loaded_specs[name]
    return unless spec && spec.version == version && spec.default_gem? == default_gem?

    spec
  end
end
                                          rubygems/rubygems/specification.rb                                                                  0000644                 00000212040 15040313423 0013375 0                                                                                                    ustar 00                                                                                                                                                                                                                                                       # frozen_string_literal: true

#
#--
# Copyright 2006 by Chad Fowler, Rich Kilmer, Jim Weirich and others.
# All rights reserved.
# See LICENSE.txt for permissions.
#++

require_relative "deprecate"
require_relative "basic_specification"
require_relative "stub_specification"
require_relative "platform"
require_relative "specification_record"
require_relative "util/list"

require "rbconfig"

##
# The Specification class contains the information for a gem.  Typically
# defined in a .gemspec file or a Rakefile, and looks like this:
#
#   Gem::Specification.new do |s|
#     s.name        = 'example'
#     s.version     = '0.1.0'
#     s.licenses    = ['MIT']
#     s.summary     = "This is an example!"
#     s.description = "Much longer explanation of the example!"
#     s.authors     = ["Ruby Coder"]
#     s.email       = 'rubycoder@example.com'
#     s.files       = ["lib/example.rb"]
#     s.homepage    = 'https://rubygems.org/gems/example'
#     s.metadata    = { "source_code_uri" => "https://github.com/example/example" }
#   end
#
# Starting in RubyGems 2.0, a Specification can hold arbitrary
# metadata.  See #metadata for restrictions on the format and size of metadata
# items you may add to a specification.

class Gem::Specification < Gem::BasicSpecification
  extend Gem::Deprecate

  # REFACTOR: Consider breaking out this version stuff into a separate
  # module. There's enough special stuff around it that it may justify
  # a separate class.

  ##
  # The version number of a specification that does not specify one
  # (i.e. RubyGems 0.7 or earlier).

  NONEXISTENT_SPECIFICATION_VERSION = -1

  ##
  # The specification version applied to any new Specification instances
  # created.  This should be bumped whenever something in the spec format
  # changes.
  #
  # Specification Version History:
  #
  #   spec   ruby
  #    ver    ver yyyy-mm-dd description
  #     -1 <0.8.0            pre-spec-version-history
  #      1  0.8.0 2004-08-01 Deprecated "test_suite_file" for "test_files"
  #                          "test_file=x" is a shortcut for "test_files=[x]"
  #      2  0.9.5 2007-10-01 Added "required_rubygems_version"
  #                          Now forward-compatible with future versions
  #      3  1.3.2 2009-01-03 Added Fixnum validation to specification_version
  #      4  1.9.0 2011-06-07 Added metadata
  #--
  # When updating this number, be sure to also update #to_ruby.
  #
  # NOTE RubyGems < 1.2 cannot load specification versions > 2.

  CURRENT_SPECIFICATION_VERSION = 4 # :nodoc:

  ##
  # An informal list of changes to the specification.  The highest-valued
  # key should be equal to the CURRENT_SPECIFICATION_VERSION.

  SPECIFICATION_VERSION_HISTORY = { # :nodoc:
    -1 => ["(RubyGems versions up to and including 0.7 did not have versioned specifications)"],
    1 => [
      'Deprecated "test_suite_file" in favor of the new, but equivalent, "test_files"',
      '"test_file=x" is a shortcut for "test_files=[x]"',
    ],
    2 => [
      'Added "required_rubygems_version"',
      "Now forward-compatible with future versions",
    ],
    3 => [
      "Added Fixnum validation to the specification_version",
    ],
    4 => [
      "Added sandboxed freeform metadata to the specification version.",
    ],
  }.freeze

  MARSHAL_FIELDS = { # :nodoc:
    -1 => 16,
    1 => 16,
    2 => 16,
    3 => 17,
    4 => 18,
  }.freeze

  today = Time.now.utc
  TODAY = Time.utc(today.year, today.month, today.day) # :nodoc:

  @load_cache = {} # :nodoc:
  @load_cache_mutex = Thread::Mutex.new

  VALID_NAME_PATTERN = /\A[a-zA-Z0-9\.\-\_]+\z/ # :nodoc:

  # :startdoc:

  ##
  # List of attribute names: [:name, :version, ...]

  @@required_attributes = [:rubygems_version,
                           :specification_version,
                           :name,
                           :version,
                           :date,
                           :summary,
                           :require_paths]

  ##
  # Map of attribute names to default values.

  @@default_value = {
    authors: [],
    autorequire: nil,
    bindir: "bin",
    cert_chain: [],
    date: nil,
    dependencies: [],
    description: nil,
    email: nil,
    executables: [],
    extensions: [],
    extra_rdoc_files: [],
    files: [],
    homepage: nil,
    licenses: [],
    metadata: {},
    name: nil,
    platform: Gem::Platform::RUBY,
    post_install_message: nil,
    rdoc_options: [],
    require_paths: ["lib"],
    required_ruby_version: Gem::Requirement.default,
    required_rubygems_version: Gem::Requirement.default,
    requirements: [],
    rubygems_version: Gem::VERSION,
    signing_key: nil,
    specification_version: CURRENT_SPECIFICATION_VERSION,
    summary: nil,
    test_files: [],
    version: nil,
  }.freeze

  # rubocop:disable Style/MutableConstant
  INITIALIZE_CODE_FOR_DEFAULTS = {} # :nodoc:
  # rubocop:enable Style/MutableConstant

  @@default_value.each do |k,v|
    INITIALIZE_CODE_FOR_DEFAULTS[k] = case v
                                      when [], {}, true, false, nil, Numeric, Symbol
                                        v.inspect
                                      when String
                                        v.dump
                                      else
                                        "default_value(:#{k}).dup"
    end
  end

  @@attributes = @@default_value.keys.sort_by(&:to_s)
  @@array_attributes = @@default_value.select {|_k,v| v.is_a?(Array) }.keys
  @@nil_attributes, @@non_nil_attributes = @@default_value.keys.partition do |k|
    @@default_value[k].nil?
  end

  # Sentinel object to represent "not found" stubs
  NOT_FOUND = Struct.new(:to_spec, :this).new # :nodoc:
  deprecate_constant :NOT_FOUND

  # Tracking removed method calls to warn users during build time.
  REMOVED_METHODS = [:rubyforge_project=, :mark_version].freeze # :nodoc:
  def removed_method_calls
    @removed_method_calls ||= []
  end

  ######################################################################
  # :section: Required gemspec attributes

  ##
  # This gem's name.
  #
  # Usage:
  #
  #   spec.name = 'rake'

  attr_accessor :name

  ##
  # This gem's version.
  #
  # The version string can contain numbers and periods, such as +1.0.0+.
  # A gem is a 'prerelease' gem if the version has a letter in it, such as
  # +1.0.0.pre+.
  #
  # Usage:
  #
  #   spec.version = '0.4.1'

  attr_reader :version

  ##
  # A short summary of this gem's description.  Displayed in <tt>gem list -d</tt>.
  #
  # The #description should be more detailed than the summary.
  #
  # Usage:
  #
  #   spec.summary = "This is a small summary of my gem"

  attr_reader :summary

  ##
  # Files included in this gem.  You cannot append to this accessor, you must
  # assign to it.
  #
  # Only add files you can require to this list, not directories, etc.
  #
  # Directories are automatically stripped from this list when building a gem,
  # other non-files cause an error.
  #
  # Usage:
  #
  #   require 'rake'
  #   spec.files = FileList['lib/**/*.rb',
  #                         'bin/*',
  #                         '[A-Z]*'].to_a
  #
  #   # or without Rake...
  #   spec.files = Dir['lib/**/*.rb'] + Dir['bin/*']
  #   spec.files += Dir['[A-Z]*']
  #   spec.files.reject! { |fn| fn.include? "CVS" }

  def files
    # DO NOT CHANGE TO ||= ! This is not a normal accessor. (yes, it sucks)
    # DOC: Why isn't it normal? Why does it suck? How can we fix this?
    @files = [@files,
              @test_files,
              add_bindir(@executables),
              @extra_rdoc_files,
              @extensions].flatten.compact.uniq.sort
  end

  ##
  # A list of authors for this gem.
  #
  # Alternatively, a single author can be specified by assigning a string to
  # +spec.author+
  #
  # Usage:
  #
  #   spec.authors = ['John Jones', 'Mary Smith']

  def authors=(value)
    @authors = Array(value).flatten.grep(String)
  end

  ######################################################################
  # :section: Recommended gemspec attributes

  ##
  # The version of Ruby required by this gem
  #
  # Usage:
  #
  #   spec.required_ruby_version = '>= 2.7.0'

  attr_reader :required_ruby_version

  ##
  # A long description of this gem
  #
  # The description should be more detailed than the summary but not
  # excessively long.  A few paragraphs is a recommended length with no
  # examples or formatting.
  #
  # Usage:
  #
  #   spec.description = <<~EOF
  #     Rake is a Make-like program implemented in Ruby. Tasks and
  #     dependencies are specified in standard Ruby syntax.
  #   EOF

  attr_reader :description

  ##
  # A contact email address (or addresses) for this gem
  #
  # Usage:
  #
  #   spec.email = 'john.jones@example.com'
  #   spec.email = ['jack@example.com', 'jill@example.com']

  attr_accessor :email

  ##
  # The URL of this gem's home page
  #
  # Usage:
  #
  #   spec.homepage = 'https://github.com/ruby/rake'

  attr_accessor :homepage

  ##
  # The license for this gem.
  #
  # The license must be no more than 64 characters.
  #
  # This should just be the name of your license. The full text of the license
  # should be inside of the gem (at the top level) when you build it.
  #
  # The simplest way is to specify the standard SPDX ID
  # https://spdx.org/licenses/ for the license.
  # Ideally, you should pick one that is OSI (Open Source Initiative)
  # https://opensource.org/licenses/ approved.
  #
  # The most commonly used OSI-approved licenses are MIT and Apache-2.0.
  # GitHub also provides a license picker at https://choosealicense.com/.
  #
  # You can also use a custom license file along with your gemspec and specify
  # a LicenseRef-<idstring>, where idstring is the name of the file containing
  # the license text.
  #
  # You should specify a license for your gem so that people know how they are
  # permitted to use it and any restrictions you're placing on it.  Not
  # specifying a license means all rights are reserved; others have no right
  # to use the code for any purpose.
  #
  # You can set multiple licenses with #licenses=
  #
  # Usage:
  #   spec.license = 'MIT'

  def license=(o)
    self.licenses = [o]
  end

  ##
  # The license(s) for the library.
  #
  # Each license must be a short name, no more than 64 characters.
  #
  # This should just be the name of your license. The full
  # text of the license should be inside of the gem when you build it.
  #
  # See #license= for more discussion
  #
  # Usage:
  #   spec.licenses = ['MIT', 'GPL-2.0']

  def licenses=(licenses)
    @licenses = Array licenses
  end

  ##
  # The metadata holds extra data for this gem that may be useful to other
  # consumers and is settable by gem authors.
  #
  # Metadata items have the following restrictions:
  #
  # * The metadata must be a Hash object
  # * All keys and values must be Strings
  # * Keys can be a maximum of 128 bytes and values can be a maximum of 1024
  #   bytes
  # * All strings must be UTF-8, no binary data is allowed
  #
  # You can use metadata to specify links to your gem's homepage, codebase,
  # documentation, wiki, mailing list, issue tracker and changelog.
  #
  #   s.metadata = {
  #     "bug_tracker_uri"   => "https://example.com/user/bestgemever/issues",
  #     "changelog_uri"     => "https://example.com/user/bestgemever/CHANGELOG.md",
  #     "documentation_uri" => "https://www.example.info/gems/bestgemever/0.0.1",
  #     "homepage_uri"      => "https://bestgemever.example.io",
  #     "mailing_list_uri"  => "https://groups.example.com/bestgemever",
  #     "source_code_uri"   => "https://example.com/user/bestgemever",
  #     "wiki_uri"          => "https://example.com/user/bestgemever/wiki",
  #     "funding_uri"       => "https://example.com/donate"
  #   }
  #
  # These links will be used on your gem's page on rubygems.org and must pass
  # validation against following regex.
  #
  #   %r{\Ahttps?:\/\/([^\s:@]+:[^\s:@]*@)?[A-Za-z\d\-]+(\.[A-Za-z\d\-]+)+\.?(:\d{1,5})?([\/?]\S*)?\z}

  attr_accessor :metadata

  ######################################################################
  # :section: Optional gemspec attributes

  ##
  # Singular (alternative) writer for #authors
  #
  # Usage:
  #
  #   spec.author = 'John Jones'

  def author=(o)
    self.authors = [o]
  end

  ##
  # The path in the gem for executable scripts.  Usually 'exe'
  #
  # Usage:
  #
  #   spec.bindir = 'exe'

  attr_accessor :bindir

  ##
  # The certificate chain used to sign this gem.  See Gem::Security for
  # details.

  attr_accessor :cert_chain

  ##
  # A message that gets displayed after the gem is installed.
  #
  # Usage:
  #
  #   spec.post_install_message = "Thanks for installing!"

  attr_accessor :post_install_message

  ##
  # The platform this gem runs on.
  #
  # This is usually Gem::Platform::RUBY or Gem::Platform::CURRENT.
  #
  # Most gems contain pure Ruby code; they should simply leave the default
  # value in place.  Some gems contain C (or other) code to be compiled into a
  # Ruby "extension".  The gem should leave the default value in place unless
  # the code will only compile on a certain type of system.  Some gems consist
  # of pre-compiled code ("binary gems").  It's especially important that they
  # set the platform attribute appropriately.  A shortcut is to set the
  # platform to Gem::Platform::CURRENT, which will cause the gem builder to set
  # the platform to the appropriate value for the system on which the build is
  # being performed.
  #
  # If this attribute is set to a non-default value, it will be included in
  # the filename of the gem when it is built such as:
  # nokogiri-1.6.0-x86-mingw32.gem
  #
  # Usage:
  #
  #   spec.platform = Gem::Platform.local

  def platform=(platform)
    @original_platform = platform

    case platform
    when Gem::Platform::CURRENT then
      @new_platform = Gem::Platform.local
      @original_platform = @new_platform.to_s

    when Gem::Platform then
      @new_platform = platform

    # legacy constants
    when nil, Gem::Platform::RUBY then
      @new_platform = Gem::Platform::RUBY
    when "mswin32" then # was Gem::Platform::WIN32
      @new_platform = Gem::Platform.new "x86-mswin32"
    when "i586-linux" then # was Gem::Platform::LINUX_586
      @new_platform = Gem::Platform.new "x86-linux"
    when "powerpc-darwin" then # was Gem::Platform::DARWIN
      @new_platform = Gem::Platform.new "ppc-darwin"
    else
      @new_platform = Gem::Platform.new platform
    end

    @platform = @new_platform.to_s

    invalidate_memoized_attributes
  end

  ##
  # Paths in the gem to add to <code>$LOAD_PATH</code> when this gem is
  # activated.
  #--
  # See also #require_paths
  #++
  # If you have an extension you do not need to add <code>"ext"</code> to the
  # require path, the extension build process will copy the extension files
  # into "lib" for you.
  #
  # The default value is <code>"lib"</code>
  #
  # Usage:
  #
  #   # If all library files are in the root directory...
  #   spec.require_paths = ['.']

  def require_paths=(val)
    @require_paths = Array(val)
  end

  ##
  # The RubyGems version required by this gem

  attr_reader :required_rubygems_version

  ##
  # The key used to sign this gem.  See Gem::Security for details.

  attr_accessor :signing_key

  ##
  # Adds a development dependency named +gem+ with +requirements+ to this
  # gem.
  #
  # Usage:
  #
  #   spec.add_development_dependency 'example', '~> 1.1', '>= 1.1.4'
  #
  # Development dependencies aren't installed by default and aren't
  # activated when a gem is required.

  def add_development_dependency(gem, *requirements)
    add_dependency_with_type(gem, :development, requirements)
  end

  ##
  # Adds a runtime dependency named +gem+ with +requirements+ to this gem.
  #
  # Usage:
  #
  #   spec.add_dependency 'example', '~> 1.1', '>= 1.1.4'

  def add_dependency(gem, *requirements)
    if requirements.uniq.size != requirements.size
      warn "WARNING: duplicated #{gem} dependency #{requirements}"
    end

    add_dependency_with_type(gem, :runtime, requirements)
  end

  ##
  # Executables included in the gem.
  #
  # For example, the rake gem has rake as an executable. You don't specify the
  # full path (as in bin/rake); all application-style files are expected to be
  # found in bindir.  These files must be executable Ruby files.  Files that
  # use bash or other interpreters will not work.
  #
  # Executables included may only be ruby scripts, not scripts for other
  # languages or compiled binaries.
  #
  # Usage:
  #
  #   spec.executables << 'rake'

  def executables
    @executables ||= []
  end

  ##
  # Extensions to build when installing the gem, specifically the paths to
  # extconf.rb-style files used to compile extensions.
  #
  # These files will be run when the gem is installed, causing the C (or
  # whatever) code to be compiled on the user's machine.
  #
  # Usage:
  #
  #  spec.extensions << 'ext/rmagic/extconf.rb'
  #
  # See Gem::Ext::Builder for information about writing extensions for gems.

  def extensions
    @extensions ||= []
  end

  ##
  # Extra files to add to RDoc such as README or doc/examples.txt
  #
  # When the user elects to generate the RDoc documentation for a gem (typically
  # at install time), all the library files are sent to RDoc for processing.
  # This option allows you to have some non-code files included for a more
  # complete set of documentation.
  #
  # Usage:
  #
  #  spec.extra_rdoc_files = ['README', 'doc/user-guide.txt']

  def extra_rdoc_files
    @extra_rdoc_files ||= []
  end

  ##
  # The version of RubyGems that installed this gem.  Returns
  # <code>Gem::Version.new(0)</code> for gems installed by versions earlier
  # than RubyGems 2.2.0.

  def installed_by_version # :nodoc:
    @installed_by_version ||= Gem::Version.new(0)
  end

  ##
  # Sets the version of RubyGems that installed this gem.  See also
  # #installed_by_version.

  def installed_by_version=(version) # :nodoc:
    @installed_by_version = Gem::Version.new version
  end

  ##
  # Specifies the rdoc options to be used when generating API documentation.
  #
  # Usage:
  #
  #   spec.rdoc_options << '--title' << 'Rake -- Ruby Make' <<
  #     '--main' << 'README' <<
  #     '--line-numbers'

  def rdoc_options
    @rdoc_options ||= []
  end

  LATEST_RUBY_WITHOUT_PATCH_VERSIONS = Gem::Version.new("2.1")

  ##
  # The version of Ruby required by this gem.  The ruby version can be
  # specified to the patch-level:
  #
  #   $ ruby -v -e 'p Gem.ruby_version'
  #   ruby 2.0.0p247 (2013-06-27 revision 41674) [x86_64-darwin12.4.0]
  #   #<Gem::Version "2.0.0.247">
  #
  # Prereleases can also be specified.
  #
  # Usage:
  #
  #  # This gem will work with 1.8.6 or greater...
  #  spec.required_ruby_version = '>= 1.8.6'
  #
  #  # Only with final releases of major version 2 where minor version is at least 3
  #  spec.required_ruby_version = '~> 2.3'
  #
  #  # Only prereleases or final releases after 2.6.0.preview2
  #  spec.required_ruby_version = '> 2.6.0.preview2'
  #
  #  # This gem will work with 2.3.0 or greater, including major version 3, but lesser than 4.0.0
  #  spec.required_ruby_version = '>= 2.3', '< 4'

  def required_ruby_version=(req)
    @required_ruby_version = Gem::Requirement.create req

    @required_ruby_version.requirements.map! do |op, v|
      if v >= LATEST_RUBY_WITHOUT_PATCH_VERSIONS && v.release.segments.size == 4
        [op == "~>" ? "=" : op, Gem::Version.new(v.segments.tap {|s| s.delete_at(3) }.join("."))]
      else
        [op, v]
      end
    end
  end

  ##
  # The RubyGems version required by this gem

  def required_rubygems_version=(req)
    @required_rubygems_version = Gem::Requirement.create req
  end

  ##
  # Lists the external (to RubyGems) requirements that must be met for this gem
  # to work.  It's simply information for the user.
  #
  # Usage:
  #
  #   spec.requirements << 'libmagick, v6.0'
  #   spec.requirements << 'A good graphics card'

  def requirements
    @requirements ||= []
  end

  ##
  # A collection of unit test files.  They will be loaded as unit tests when
  # the user requests a gem to be unit tested.
  #
  # Usage:
  #   spec.test_files = Dir.glob('test/tc_*.rb')
  #   spec.test_files = ['tests/test-suite.rb']

  def test_files=(files) # :nodoc:
    @test_files = Array files
  end

  ######################################################################
  # :section: Read-only attributes

  ##
  # The version of RubyGems used to create this gem.

  attr_accessor :rubygems_version

  ##
  # The path where this gem installs its extensions.

  def extensions_dir
    @extensions_dir ||= super
  end

  ######################################################################
  # :section: Specification internals

  ##
  # True when this gemspec has been activated. This attribute is not persisted.

  attr_accessor :activated

  alias_method :activated?, :activated

  ##
  # Autorequire was used by old RubyGems to automatically require a file.
  #
  # Deprecated: It is neither supported nor functional.

  attr_accessor :autorequire # :nodoc:

  ##
  # Sets the default executable for this gem.
  #
  # Deprecated: You must now specify the executable name to  Gem.bin_path.

  attr_writer :default_executable
  rubygems_deprecate :default_executable=

  ##
  # Allows deinstallation of gems with legacy platforms.

  attr_writer :original_platform # :nodoc:

  ##
  # The Gem::Specification version of this gemspec.
  #
  # Do not set this, it is set automatically when the gem is packaged.

  attr_accessor :specification_version

  def self._all # :nodoc:
    specification_record.all
  end

  def self.clear_load_cache # :nodoc:
    @load_cache_mutex.synchronize do
      @load_cache.clear
    end
  end
  private_class_method :clear_load_cache

  def self.gem_path # :nodoc:
    Gem.path
  end
  private_class_method :gem_path

  def self.each_gemspec(dirs) # :nodoc:
    dirs.each do |dir|
      Gem::Util.glob_files_in_dir("*.gemspec", dir).each do |path|
        yield path
      end
    end
  end

  def self.gemspec_stubs_in(dir, pattern) # :nodoc:
    Gem::Util.glob_files_in_dir(pattern, dir).map {|path| yield path }.select(&:valid?)
  end

  def self.each_spec(dirs) # :nodoc:
    each_gemspec(dirs) do |path|
      spec = self.load path
      yield spec if spec
    end
  end

  ##
  # Returns a Gem::StubSpecification for every installed gem

  def self.stubs
    specification_record.stubs
  end

  ##
  # Returns a Gem::StubSpecification for default gems

  def self.default_stubs(pattern = "*.gemspec")
    base_dir = Gem.default_dir
    gems_dir = File.join base_dir, "gems"
    gemspec_stubs_in(Gem.default_specifications_dir, pattern) do |path|
      Gem::StubSpecification.default_gemspec_stub(path, base_dir, gems_dir)
    end
  end

  ##
  # Returns a Gem::StubSpecification for installed gem named +name+
  # only returns stubs that match Gem.platforms

  def self.stubs_for(name)
    specification_record.stubs_for(name)
  end

  ##
  # Finds stub specifications matching a pattern from the standard locations,
  # optionally filtering out specs not matching the current platform
  #
  def self.stubs_for_pattern(pattern, match_platform = true) # :nodoc:
    specification_record.stubs_for_pattern(pattern, match_platform)
  end

  def self._resort!(specs) # :nodoc:
    specs.sort! do |a, b|
      names = a.name <=> b.name
      next names if names.nonzero?
      versions = b.version <=> a.version
      next versions if versions.nonzero?
      platforms = Gem::Platform.sort_priority(b.platform) <=> Gem::Platform.sort_priority(a.platform)
      next platforms if platforms.nonzero?
      default_gem = a.default_gem_priority <=> b.default_gem_priority
      next default_gem if default_gem.nonzero?
      a.base_dir_priority(gem_path) <=> b.base_dir_priority(gem_path)
    end
  end

  ##
  # Loads the default specifications. It should be called only once.

  def self.load_defaults
    each_spec([Gem.default_specifications_dir]) do |spec|
      # #load returns nil if the spec is bad, so we just ignore
      # it at this stage
      Gem.register_default_spec(spec)
    end
  end

  ##
  # Adds +spec+ to the known specifications, keeping the collection
  # properly sorted.

  def self.add_spec(spec)
    specification_record.add_spec(spec)
  end

  ##
  # Removes +spec+ from the known specs.

  def self.remove_spec(spec)
    specification_record.remove_spec(spec)
  end

  ##
  # Returns all specifications. This method is discouraged from use.
  # You probably want to use one of the Enumerable methods instead.

  def self.all
    warn "NOTE: Specification.all called from #{caller(1, 1).first}" unless
      Gem::Deprecate.skip
    _all
  end

  ##
  # Sets the known specs to +specs+.

  def self.all=(specs)
    specification_record.all = specs
  end

  ##
  # Return full names of all specs in sorted order.

  def self.all_names
    specification_record.all_names
  end

  ##
  # Return the list of all array-oriented instance variables.
  #--
  # Not sure why we need to use so much stupid reflection in here...

  def self.array_attributes
    @@array_attributes.dup
  end

  ##
  # Return the list of all instance variables.
  #--
  # Not sure why we need to use so much stupid reflection in here...

  def self.attribute_names
    @@attributes.dup
  end

  ##
  # Return the directories that Specification uses to find specs.

  def self.dirs
    @@dirs ||= Gem::SpecificationRecord.dirs_from(gem_path)
  end

  ##
  # Set the directories that Specification uses to find specs. Setting
  # this resets the list of known specs.

  def self.dirs=(dirs)
    reset

    @@dirs = Gem::SpecificationRecord.dirs_from(Array(dirs))
  end

  extend Enumerable

  ##
  # Enumerate every known spec.  See ::dirs= and ::add_spec to set the list of
  # specs.

  def self.each(&block)
    specification_record.each(&block)
  end

  ##
  # Returns every spec that matches +name+ and optional +requirements+.

  def self.find_all_by_name(name, *requirements)
    specification_record.find_all_by_name(name, *requirements)
  end

  ##
  # Returns every spec that has the given +full_name+

  def self.find_all_by_full_name(full_name)
    stubs.select {|s| s.full_name == full_name }.map(&:to_spec)
  end

  ##
  # Find the best specification matching a +name+ and +requirements+. Raises
  # if the dependency doesn't resolve to a valid specification.

  def self.find_by_name(name, *requirements)
    requirements = Gem::Requirement.default if requirements.empty?

    Gem::Dependency.new(name, *requirements).to_spec
  end

  ##
  # Find the best specification matching a +full_name+.
  def self.find_by_full_name(full_name)
    stubs.find {|s| s.full_name == full_name }&.to_spec
  end

  ##
  # Return the best specification that contains the file matching +path+.

  def self.find_by_path(path)
    specification_record.find_by_path(path)
  end

  ##
  # Return the best specification that contains the file matching +path+
  # amongst the specs that are not activated.

  def self.find_inactive_by_path(path)
    specification_record.find_inactive_by_path(path)
  end

  ##
  # Return the best specification that contains the file matching +path+, among
  # those already activated.

  def self.find_active_stub_by_path(path)
    specification_record.find_active_stub_by_path(path)
  end

  ##
  # Return currently unresolved specs that contain the file matching +path+.

  def self.find_in_unresolved(path)
    unresolved_specs.find_all {|spec| spec.contains_requirable_file? path }
  end

  ##
  # Search through all unresolved deps and sub-dependencies and return
  # specs that contain the file matching +path+.

  def self.find_in_unresolved_tree(path)
    unresolved_specs.each do |spec|
      spec.traverse do |_from_spec, _dep, to_spec, trail|
        if to_spec.has_conflicts? || to_spec.conficts_when_loaded_with?(trail)
          :next
        else
          return trail.reverse if to_spec.contains_requirable_file? path
        end
      end
    end

    []
  end

  def self.unresolved_specs
    unresolved_deps.values.flat_map(&:to_specs)
  end
  private_class_method :unresolved_specs

  ##
  # Special loader for YAML files.  When a Specification object is loaded
  # from a YAML file, it bypasses the normal Ruby object initialization
  # routine (#initialize).  This method makes up for that and deals with
  # gems of different ages.
  #
  # +input+ can be anything that YAML.load() accepts: String or IO.

  def self.from_yaml(input)
    Gem.load_yaml

    input = normalize_yaml_input input
    spec = Gem::SafeYAML.safe_load input

    if spec && spec.class == FalseClass
      raise Gem::EndOfYAMLException
    end

    unless Gem::Specification === spec
      raise Gem::Exception, "YAML data doesn't evaluate to gem specification"
    end

    spec.specification_version ||= NONEXISTENT_SPECIFICATION_VERSION
    spec.reset_nil_attributes_to_default
    spec.flatten_require_paths

    spec
  end

  ##
  # Return the latest specs, optionally including prerelease specs if
  # +prerelease+ is true.

  def self.latest_specs(prerelease = false)
    specification_record.latest_specs(prerelease)
  end

  ##
  # Return the latest installed spec for gem +name+.

  def self.latest_spec_for(name)
    specification_record.latest_spec_for(name)
  end

  def self._latest_specs(specs, prerelease = false) # :nodoc:
    result = {}

    specs.reverse_each do |spec|
      unless prerelease
        next if spec.version.prerelease?
      end

      result[spec.name] = spec
    end

    result.flat_map(&:last).sort_by(&:name)
  end

  ##
  # Loads Ruby format gemspec from +file+.

  def self.load(file)
    return unless file

    spec = @load_cache_mutex.synchronize { @load_cache[file] }
    return spec if spec

    return unless File.file?(file)

    code = Gem.open_file(file, "r:UTF-8:-", &:read)

    begin
      spec = eval code, binding, file

      if Gem::Specification === spec
        spec.loaded_from = File.expand_path file.to_s
        @load_cache_mutex.synchronize do
          prev = @load_cache[file]
          if prev
            spec = prev
          else
            @load_cache[file] = spec
          end
        end
        return spec
      end

      warn "[#{file}] isn't a Gem::Specification (#{spec.class} instead)."
    rescue SignalException, SystemExit
      raise
    rescue SyntaxError, StandardError => e
      warn "Invalid gemspec in [#{file}]: #{e}"
    end

    nil
  end

  ##
  # Specification attributes that must be non-nil

  def self.non_nil_attributes
    @@non_nil_attributes.dup
  end

  ##
  # Make sure the YAML specification is properly formatted with dashes

  def self.normalize_yaml_input(input)
    result = input.respond_to?(:read) ? input.read : input
    result = "--- " + result unless result.start_with?("--- ")
    result = result.dup
    result.gsub!(/ !!null \n/, " \n")
    # date: 2011-04-26 00:00:00.000000000Z
    # date: 2011-04-26 00:00:00.000000000 Z
    result.gsub!(/^(date: \d{4}-\d{2}-\d{2} \d{2}:\d{2}:\d{2}\.\d+?)Z/, '\1 Z')
    result
  end

  ##
  # Return a list of all outdated local gem names.  This method is HEAVY
  # as it must go fetch specifications from the server.
  #
  # Use outdated_and_latest_version if you wish to retrieve the latest remote
  # version as well.

  def self.outdated
    outdated_and_latest_version.map {|local, _| local.name }
  end

  ##
  # Enumerates the outdated local gems yielding the local specification and
  # the latest remote version.
  #
  # This method may take some time to return as it must check each local gem
  # against the server's index.

  def self.outdated_and_latest_version
    return enum_for __method__ unless block_given?

    # TODO: maybe we should switch to rubygems' version service?
    fetcher = Gem::SpecFetcher.fetcher

    latest_specs(true).each do |local_spec|
      dependency =
        Gem::Dependency.new local_spec.name, ">= #{local_spec.version}"

      remotes, = fetcher.search_for_dependency dependency
      remotes  = remotes.map {|n, _| n.version }

      latest_remote = remotes.sort.last

      yield [local_spec, latest_remote] if
        latest_remote && local_spec.version < latest_remote
    end

    nil
  end

  ##
  # Is +name+ a required attribute?

  def self.required_attribute?(name)
    @@required_attributes.include? name.to_sym
  end

  ##
  # Required specification attributes

  def self.required_attributes
    @@required_attributes.dup
  end

  ##
  # Reset the list of known specs, running pre and post reset hooks
  # registered in Gem.

  def self.reset
    @@dirs = nil
    Gem.pre_reset_hooks.each(&:call)
    @specification_record = nil
    clear_load_cache

    unless unresolved_deps.empty?
      unresolved = unresolved_deps.filter_map do |name, dep|
        matching_versions = find_all_by_name(name)
        next if dep.latest_version? && matching_versions.any?(&:default_gem?)

        [dep, matching_versions.uniq(&:full_name)]
      end.to_h

      unless unresolved.empty?
        warn "WARN: Unresolved or ambiguous specs during Gem::Specification.reset:"
        unresolved.each do |dep, versions|
          warn "      #{dep}"

          unless versions.empty?
            warn "      Available/installed versions of this gem:"
            versions.each {|s| warn "      - #{s.version}" }
          end
        end
        warn "WARN: Clearing out unresolved specs. Try 'gem cleanup <gem>'"
        warn "Please report a bug if this causes problems."
      end

      unresolved_deps.clear
    end
    Gem.post_reset_hooks.each(&:call)
  end

  ##
  # Keeps track of all currently known specifications

  def self.specification_record
    @specification_record ||= Gem::SpecificationRecord.new(dirs)
  end

  # DOC: This method needs documented or nodoc'd
  def self.unresolved_deps
    @unresolved_deps ||= Hash.new {|h, n| h[n] = Gem::Dependency.new n }
  end

  ##
  # Load custom marshal format, re-initializing defaults as needed

  def self._load(str)
    Gem.load_yaml
    Gem.load_safe_marshal

    yaml_set = false
    retry_count = 0

    array = begin
      Gem::SafeMarshal.safe_load str
    rescue ArgumentError => e
      # Avoid an infinite retry loop when the argument error has nothing to do
      # with the classes not being defined.
      # 1 retry each allowed in case all 3 of
      # - YAML
      # - YAML::Syck::DefaultKey
      # - YAML::PrivateType
      # need to be defined
      raise if retry_count >= 3

      #
      # Some very old marshaled specs included references to `YAML::PrivateType`
      # and `YAML::Syck::DefaultKey` constants due to bugs in the old emitter
      # that generated them. Workaround the issue by defining the necessary
      # constants and retrying.
      #
      message = e.message
      raise unless message.include?("YAML::")

      unless Object.const_defined?(:YAML)
        Object.const_set "YAML", Psych
        yaml_set = true
      end

      if message.include?("YAML::Syck::")
        YAML.const_set "Syck", YAML unless YAML.const_defined?(:Syck)

        YAML::Syck.const_set "DefaultKey", Class.new if message.include?("YAML::Syck::DefaultKey") && !YAML::Syck.const_defined?(:DefaultKey)
      elsif message.include?("YAML::PrivateType") && !YAML.const_defined?(:PrivateType)
        YAML.const_set "PrivateType", Class.new
      end

      retry_count += 1
      retry
    ensure
      Object.__send__(:remove_const, "YAML") if yaml_set
    end

    spec = Gem::Specification.new
    spec.instance_variable_set :@specification_version, array[1]

    current_version = CURRENT_SPECIFICATION_VERSION

    field_count = if spec.specification_version > current_version
      spec.instance_variable_set :@specification_version,
                                 current_version
      MARSHAL_FIELDS[current_version]
    else
      MARSHAL_FIELDS[spec.specification_version]
    end

    if array.size < field_count
      raise TypeError, "invalid Gem::Specification format #{array.inspect}"
    end

    spec.instance_variable_set :@rubygems_version,          array[0]
    # spec version
    spec.instance_variable_set :@name,                      array[2]
    spec.instance_variable_set :@version,                   array[3]
    spec.date =                                             array[4]
    spec.instance_variable_set :@summary,                   array[5]
    spec.instance_variable_set :@required_ruby_version,     array[6]
    spec.instance_variable_set :@required_rubygems_version, array[7]
    spec.platform =                                         array[8]
    spec.instance_variable_set :@dependencies,              array[9]
    # offset due to rubyforge_project removal
    spec.instance_variable_set :@email,                     array[11]
    spec.instance_variable_set :@authors,                   array[12]
    spec.instance_variable_set :@description,               array[13]
    spec.instance_variable_set :@homepage,                  array[14]
    spec.instance_variable_set :@has_rdoc,                  array[15]
    spec.instance_variable_set :@licenses,                  array[17]
    spec.instance_variable_set :@metadata,                  array[18]
    spec.instance_variable_set :@loaded,                    false
    spec.instance_variable_set :@activated,                 false

    spec
  end

  def <=>(other) # :nodoc:
    sort_obj <=> other.sort_obj
  end

  def ==(other) # :nodoc:
    self.class === other &&
      name == other.name &&
      version == other.version &&
      platform == other.platform
  end

  ##
  # Dump only crucial instance variables.
  #--
  # MAINTAIN ORDER!
  # (down with the man)

  def _dump(limit)
    Marshal.dump [
      @rubygems_version,
      @specification_version,
      @name,
      @version,
      date,
      @summary,
      @required_ruby_version,
      @required_rubygems_version,
      @original_platform,
      @dependencies,
      "", # rubyforge_project
      @email,
      @authors,
      @description,
      @homepage,
      true, # has_rdoc
      @new_platform,
      @licenses,
      @metadata,
    ]
  end

  ##
  # Activate this spec, registering it as a loaded spec and adding
  # it's lib paths to $LOAD_PATH. Returns true if the spec was
  # activated, false if it was previously activated. Freaks out if
  # there are conflicts upon activation.

  def activate
    other = Gem.loaded_specs[name]
    if other
      check_version_conflict other
      return false
    end

    raise_if_conflicts

    activate_dependencies
    add_self_to_load_path

    Gem.loaded_specs[name] = self
    @activated = true
    @loaded = true

    true
  end

  ##
  # Activate all unambiguously resolved runtime dependencies of this
  # spec. Add any ambiguous dependencies to the unresolved list to be
  # resolved later, as needed.

  def activate_dependencies
    unresolved = Gem::Specification.unresolved_deps

    runtime_dependencies.each do |spec_dep|
      if loaded = Gem.loaded_specs[spec_dep.name]
        next if spec_dep.matches_spec? loaded

        msg = "can't satisfy '#{spec_dep}', already activated '#{loaded.full_name}'"
        e = Gem::LoadError.new msg
        e.name = spec_dep.name

        raise e
      end

      specs = spec_dep.matching_specs(true).uniq(&:full_name)

      if specs.size == 0
        raise Gem::MissingSpecError.new(spec_dep.name, spec_dep.requirement, "at: #{spec_file}")
      elsif specs.size == 1
        specs.first.activate
      else
        name = spec_dep.name
        unresolved[name] = unresolved[name].merge spec_dep
      end
    end

    unresolved.delete self.name
  end

  ##
  # Abbreviate the spec for downloading.  Abbreviated specs are only used for
  # searching, downloading and related activities and do not need deployment
  # specific information (e.g. list of files).  So we abbreviate the spec,
  # making it much smaller for quicker downloads.

  def abbreviate
    self.files = []
    self.test_files = []
    self.rdoc_options = []
    self.extra_rdoc_files = []
    self.cert_chain = []
  end

  ##
  # Sanitize the descriptive fields in the spec.  Sometimes non-ASCII
  # characters will garble the site index.  Non-ASCII characters will
  # be replaced by their XML entity equivalent.

  def sanitize
    self.summary              = sanitize_string(summary)
    self.description          = sanitize_string(description)
    self.post_install_message = sanitize_string(post_install_message)
    self.authors              = authors.collect {|a| sanitize_string(a) }
  end

  ##
  # Sanitize a single string.

  def sanitize_string(string)
    return string unless string

    # HACK: the #to_s is in here because RSpec has an Array of Arrays of
    # Strings for authors.  Need a way to disallow bad values on gemspec
    # generation.  (Probably won't happen.)
    string.to_s
  end

  ##
  # Returns an array with bindir attached to each executable in the
  # +executables+ list

  def add_bindir(executables)
    return nil if executables.nil?

    if @bindir
      Array(executables).map {|e| File.join(@bindir, e) }
    else
      executables
    end
  rescue StandardError
    nil
  end

  ##
  # Adds a dependency on gem +dependency+ with type +type+ that requires
  # +requirements+.  Valid types are currently <tt>:runtime</tt> and
  # <tt>:development</tt>.

  def add_dependency_with_type(dependency, type, requirements)
    requirements = if requirements.empty?
      Gem::Requirement.default
    else
      requirements.flatten
    end

    unless dependency.respond_to?(:name) &&
           dependency.respond_to?(:requirement)
      dependency = Gem::Dependency.new(dependency.to_s, requirements, type)
    end

    dependencies << dependency
  end

  private :add_dependency_with_type

  alias_method :add_runtime_dependency, :add_dependency

  ##
  # Adds this spec's require paths to LOAD_PATH, in the proper location.

  def add_self_to_load_path
    return if default_gem?

    paths = full_require_paths

    Gem.add_to_load_path(*paths)
  end

  ##
  # Singular reader for #authors.  Returns the first author in the list

  def author
    (val = authors) && val.first
  end

  ##
  # The list of author names who wrote this gem.
  #
  #   spec.authors = ['Chad Fowler', 'Jim Weirich', 'Rich Kilmer']

  def authors
    @authors ||= []
  end

  ##
  # Returns the full path to installed gem's bin directory.
  #
  # NOTE: do not confuse this with +bindir+, which is just 'bin', not
  # a full path.

  def bin_dir
    @bin_dir ||= File.join gem_dir, bindir
  end

  ##
  # Returns the full path to an executable named +name+ in this gem.

  def bin_file(name)
    File.join bin_dir, name
  end

  ##
  # Returns the build_args used to install the gem

  def build_args
    if File.exist? build_info_file
      build_info = File.readlines build_info_file
      build_info = build_info.map(&:strip)
      build_info.delete ""
      build_info
    else
      []
    end
  end

  ##
  # Builds extensions for this platform if the gem has extensions listed and
  # the gem.build_complete file is missing.

  def build_extensions # :nodoc:
    return if extensions.empty?
    return if default_gem?
    # we need to fresh build when same name and version of default gems
    return if self.class.find_by_full_name(full_name)&.default_gem?
    return if File.exist? gem_build_complete_path
    return unless File.writable?(base_dir)
    return unless File.exist?(File.join(base_dir, "extensions"))

    begin
      # We need to require things in $LOAD_PATH without looking for the
      # extension we are about to build.
      unresolved_deps = Gem::Specification.unresolved_deps.dup
      Gem::Specification.unresolved_deps.clear

      require_relative "config_file"
      require_relative "ext"
      require_relative "user_interaction"

      ui = Gem::SilentUI.new
      Gem::DefaultUserInteraction.use_ui ui do
        builder = Gem::Ext::Builder.new self
        builder.build_extensions
      end
    ensure
      ui&.close
      Gem::Specification.unresolved_deps.replace unresolved_deps
    end
  end

  ##
  # Returns the full path to the build info directory

  def build_info_dir
    File.join base_dir, "build_info"
  end

  ##
  # Returns the full path to the file containing the build
  # information generated when the gem was installed

  def build_inf