>~/.gem/gem-public_cert.pem</code>
#
# Obviously this RubyGems trust infrastructure doesn't exist yet.  Also, in
# the "real world", issuers actually generate the child certificate from a
# certificate request, rather than sign an existing certificate.  And our
# hypothetical infrastructure is missing a certificate revocation system.
# These are that can be fixed in the future...
#
# At this point you should know how to do all of these new and interesting
# things:
#
# * build a gem signing key and certificate
# * adjust your security policy
# * modify your trusted certificate list
# * sign a certificate
#
# == Manually verifying signatures
#
# In case you don't trust RubyGems you can verify gem signatures manually:
#
# 1. Fetch and unpack the gem
#
#      gem fetch some_signed_gem
#      tar -xf some_signed_gem-1.0.gem
#
# 2. Grab the public key from the gemspec
#
#      gem spec some_signed_gem-1.0.gem cert_chain | \
#        ruby -rpsych -e 'puts Psych.load($stdin)' > public_key.crt
#
# 3. Generate a SHA1 hash of the data.tar.gz
#
#      openssl dgst -sha1 < data.tar.gz > my.hash
#
# 4. Verify the signature
#
#      openssl rsautl -verify -inkey public_key.crt -certin \
#        -in data.tar.gz.sig > verified.hash
#
# 5. Compare your hash to the verified hash
#
#      diff -s verified.hash my.hash
#
# 6. Repeat 5 and 6 with metadata.gz
#
# == OpenSSL Reference
#
# The .pem files generated by --build and --sign are PEM files.  Here's a
# couple of useful OpenSSL commands for manipulating them:
#
#   # convert a PEM format X509 certificate into DER format:
#   # (note: Windows .cer files are X509 certificates in DER format)
#   $ openssl x509 -in input.pem -outform der -out output.der
#
#   # print out the certificate in a human-readable format:
#   $ openssl x509 -in input.pem -noout -text
#
# And you can do the same thing with the private key file as well:
#
#   # convert a PEM format RSA key into DER format:
#   $ openssl rsa -in input_key.pem -outform der -out output_key.der
#
#   # print out the key in a human readable format:
#   $ openssl rsa -in input_key.pem -noout -text
#
# == Bugs/TODO
#
# * There's no way to define a system-wide trust list.
# * custom security policies (from a YAML file, etc)
# * Simple method to generate a signed certificate request
# * Support for OCSP, SCVP, CRLs, or some other form of cert status check
#   (list is in order of preference)
# * Support for encrypted private keys
# * Some sort of semi-formal trust hierarchy (see long-winded explanation
#   above)
# * Path discovery (for gem certificate chains that don't have a self-signed
#   root) -- by the way, since we don't have this, THE ROOT OF THE CERTIFICATE
#   CHAIN MUST BE SELF SIGNED if Policy#verify_root is true (and it is for the
#   MediumSecurity and HighSecurity policies)
# * Better explanation of X509 naming (ie, we don't have to use email
#   addresses)
# * Honor AIA field (see note about OCSP above)
# * Honor extension restrictions
# * Might be better to store the certificate chain as a PKCS#7 or PKCS#12
#   file, instead of an array embedded in the metadata.
#
# == Original author
#
# Paul Duncan <pabs@pablotron.org>
# https://pablotron.org/

module Gem::Security
  ##
  # Gem::Security default exception type

  class Exception < Gem::Exception; end

  ##
  # Used internally to select the signing digest from all computed digests

  DIGEST_NAME = "SHA256" # :nodoc:

  ##
  # Length of keys created by RSA and DSA keys

  RSA_DSA_KEY_LENGTH = 3072

  ##
  # Default algorithm to use when building a key pair

  DEFAULT_KEY_ALGORITHM = "RSA"

  ##
  # Named curve used for Elliptic Curve

  EC_NAME = "secp384r1"

  ##
  # Cipher used to encrypt the key pair used to sign gems.
  # Must be in the list returned by OpenSSL::Cipher.ciphers

  KEY_CIPHER = OpenSSL::Cipher.new("AES-256-CBC") if defined?(OpenSSL::Cipher)

  ##
  # One day in seconds

  ONE_DAY = 86_400

  ##
  # One year in seconds

  ONE_YEAR = ONE_DAY * 365

  ##
  # The default set of extensions are:
  #
  # * The certificate is not a certificate authority
  # * The key for the certificate may be used for key and data encipherment
  #   and digital signatures
  # * The certificate contains a subject key identifier

  EXTENSIONS = {
    "basicConstraints" => "CA:FALSE",
    "keyUsage" =>
      "keyEncipherment,dataEncipherment,digitalSignature",
    "subjectKeyIdentifier" => "hash",
  }.freeze

  def self.alt_name_or_x509_entry(certificate, x509_entry)
    alt_name = certificate.extensions.find do |extension|
      extension.oid =