Merge a set of command options with the set of default options (without
  # modifying the default option hash).

  def merge_options(new_options)
    @options = @defaults.clone
    new_options.each {|k,v| @options[k] = v }
  end

  ##
  # True if the command handles the given argument list.

  def handles?(args)
    parser.parse!(args.dup)
    true
  rescue StandardError
    false
  end

  ##
  # Handle the given list of arguments by parsing them and recording the
  # results.

  def handle_options(args)
    args = add_extra_args(args)
    check_deprecated_options(args)
    @options = Marshal.load Marshal.dump @defaults # deep copy
    parser.parse!(args)
    @options[:args] = args
  end

  ##
  # Adds extra args from ~/.gemrc

  def add_extra_args(args)
    result = []

    s_extra = Gem::Command.specific_extra_args(@command)
    extra = Gem::Command.extra_args + s_extra

    until extra.empty? do
      ex = []
      ex << extra.shift
      ex << extra.shift if /^[^-]/.match?(extra.first.to_s)
      result << ex if handles?(ex)
    end

    result.flatten!
    result.concat(args)
    result
  end

  def deprecated?
    false
  end

  private

  def option_is_deprecated?(option)
    @deprecated_options[command].key?(option)
  end

  def add_parser_description # :nodoc:
    return unless description

    formatted = description.split("\n\n").map do |chunk|
      wrap chunk, 80 - 4
    end.join "\n"

    @parser.separator nil
    @parser.separator "  Description:"
    formatted.each_line do |line|
      @parser.separator "    #{line.rstrip}"
    end
  end

  def add_parser_options # :nodoc:
    @parser.separator nil

    regular_options = @option_groups.delete :options

    configure_options "", regular_options

    @option_groups.sort_by {|n,_| n.to_s }.each do |group_name, option_list|
      @parser.separator nil
      configure_options group_name, option_list
    end
  end

  ##
  # Adds a section with +title+ and +content+ to the parser help view.  Used
  # for adding command arguments and default arguments.

  def add_parser_run_info(title, content)
    return if content.empty?

    @parser.separator nil
    @parser.separator "  #{title}:"
    content.each_line do |line|
      @parser.separator "    #{line.rstrip}"
    end
  end

  def add_parser_summary # :nodoc:
    return unless @summary

    @parser.separator nil
    @parser.separator "  Summary:"
    wrap(@summary, 80 - 4).each_line do |line|
      @parser.separator "    #{line.strip}"
    end
  end

  ##
  # Create on demand parser.

  def parser
    create_option_parser if @parser.nil?
    @parser
  end

  ##
  # Creates an option parser and fills it in with the help info for the
  # command.

  def create_option_parser
    @parser = Gem::OptionParser.new

    add_parser_options

    @parser.separator nil
    configure_options "Common", Gem::Command.common_options

    add_parser_run_info "Arguments", arguments
    add_parser_summary
    add_parser_description
    add_parser_run_info "Defaults", defaults_str
  end

  def configure_options(header, option_list)
    return if option_list.nil? || option_list.empty?

    header = header.to_s.empty? ? "" : "#{header} "
    @parser.separator "  #{header}Options:"

    option_list.each do |args, handler|
      @parser.on(*args) do |value|
        handler.call(value, @options)
      end
    end

    @parser.separator ""
  end

  ##
  # Wraps +text+ to +width+

  def wrap(text, width) # :doc:
    text.gsub(/(.{1,#{width}})( +|$\n?)|(.{1,#{width}})/, "\\1\\3\n")
  end

  # ----------------------------------------------------------------
  # Add the options common to all commands.

  add_common_option("-h", "--help",
                    "Get help on this command") do |_value, options|
    options[:help] = true
  end

  add_common_option("-V", "--[no-]verbose",
                    "Set the verbose level of output") do |value, _options|
    # Set us to "really verbose" so the progress meter works
    if Gem.configuration.verbose && value
      Gem.configuration.verbose = 1
    else
      Gem.configuration.verbose = value
    end
  end

  add_common_option("-q", "--quiet", "Silence command progress meter") do |_value, _options|
    Gem.configuration.verbose = false
  end

  add_common_option("--silent",
                    "Silence RubyGems output") do |_value, options|
    options[:silent] = true
  end

  # Backtrace and config-file are added so they show up in the help
  # commands.  Both options are actually handled before the other
  # options get parsed.

  add_common_option("--config-file FILE",
                    "Use this config file instead of default") do
  end

  add_common_option("--backtrace",
                    "Show stack backtrace on errors") do
  end

  add_common_option("--debug",
                    "Turn on Ruby debugging") do
  end

  add_common_option("--norc",
                    "Avoid loading any .gemrc file") do
  end

  # :stopdoc:

  HELP = <<-HELP
RubyGems is a package manager for Ruby.

  Usage:
    gem -h/--help
    gem -v/--version
    gem [global options...] command [arguments...] [options...]

  Global options:
    -C PATH                      run as if gem was started in <PATH>
                                 instead of the current working directory

  Examples:
    gem install rake
    gem list --local
    gem build package.gemspec
    gem push package-0.0.1.gem
    gem help install

  Further help:
    gem help commands            list all 'gem' commands
    gem help examples            show some examples of usage
    gem help gem_dependencies    gem dependencies file guide
    gem help platforms           gem platforms guide
    gem help <COMMAND>           show help on COMMAND
                                   (e.g. 'gem help install')
  Further information:
    https://guides.rubygems.org
  HELP

  # :startdoc:
end

##
# \Commands will be placed in this namespace

module Gem::Commands
end
                                                                                                                                                                                                             rubygems/rubygems/s3_uri_signer.rb                                                                  0000644                 00000013732 15040313423 0013337 0                                                                                                    ustar 00                                                                                                                                                                                                                                                       # frozen_string_literal: true

require_relative "openssl"

##
# S3URISigner implements AWS SigV4 for S3 Source to avoid a dependency on the aws-sdk-* gems
# More on AWS SigV4: https://docs.aws.amazon.com/AmazonS3/latest/API/sig-v4-authenticating-requests.html
class Gem::S3URISigner
  class ConfigurationError < Gem::Exception
    def initialize(message)
      super message
    end

    def to_s # :nodoc:
      super.to_s
    end
  end

  class InstanceProfileError < Gem::Exception
    def initialize(message)
      super message
    end

    def to_s # :nodoc:
      super.to_s
    end
  end

  attr_accessor :uri

  def initialize(uri)
    @uri = uri
  end

  ##
  # Signs S3 URI using query-params according to the reference: https://docs.aws.amazon.com/AmazonS3/latest/API/sigv4-query-string-auth.html
  def sign(expiration = 86_400)
    s3_config = fetch_s3_config

    current_time = Time.now.utc
    date_time = current_time.strftime("%Y%m%dT%H%m%SZ")
    date = date_time[0,8]

    credential_info = "#{date}/#{s3_config.region}/s3/aws4_request"
    canonical_host = "#{uri.host}.s3.#{s3_config.region}.amazonaws.com"

    query_params = generate_canonical_query_params(s3_config, date_time, credential_info, expiration)
    canonical_request = generate_canonical_request(canonical_host, query_params)
    string_to_sign = generate_string_to_sign(date_time, credential_info, canonical_request)
    signature = generate_signature(s3_config, date, string_to_sign)

    Gem::URI.parse("https://#{canonical_host}#{uri.path}?#{query_params}&X-Amz-Signature=#{signature}")
  end

  private

  S3Config = Struct.new :access_key_id, :secret_access_key, :security_token, :region

  def generate_canonical_query_params(s3_config, date_time, credential_info, expiration)
    canonical_params = {}
    canonical_params["X-Amz-Algorithm"] = "AWS4-HMAC-SHA256"
    canonical_params["X-Amz-Credential"] = "#{s3_config.access_key_id}/#{credential_info}"
    canonical_params["X-Amz-Date"] = date_time
    canonical_params["X-Amz-Expires"] = expiration.to_s
    canonical_params["X-Amz-SignedHeaders"] = "host"
    canonical_params["X-Amz-Security-Token"] = s3_config.security_token if s3_config.security_token

    # Sorting is required to generate proper signature
    canonical_params.sort.to_h.map do |key, value|
      "#{base64_uri_escape(key)}=#{base64_uri_escape(value)}"
    end.join("&")
  end

  def generate_canonical_request(canonical_host, query_params)
    [
      "GET",
      uri.path,
      query_params,
      "host:#{canonical_host}",
      "", # empty params
      "host",
      "UNSIGNED-PAYLOAD",
    ].join("\n")
  end

  def generate_string_to_sign(date_time, credential_info, canonical_request)
    [
      "AWS4-HMAC-SHA256",
      date_time,
      credential_info,
      OpenSSL::Digest::SHA256.hexdigest(canonical_request),
    ].join("\n")
  end

  def generate_signature(s3_config, date, string_to_sign)
    date_key = OpenSSL::HMAC.digest("sha256", "AWS4" + s3_config.secret_access_key, date)
    date_region_key = OpenSSL::HMAC.digest("sha256", date_key, s3_config.region)
    date_region_service_key = OpenSSL::HMAC.digest("sha256", date_region_key, "s3")
    signing_key = OpenSSL::HMAC.digest("sha256", date_region_service_key, "aws4_request")
    OpenSSL::HMAC.hexdigest("sha256", signing_key, string_to_sign)
  end

  ##
  # Extracts S3 configuration for S3 bucket
  def fetch_s3_config
    return S3Config.new(uri.user, uri.password, nil, "us-east-1") if uri.user && uri.password

    s3_source = Gem.configuration[:s3_source] || Gem.configuration["s3_source"]
    host = uri.host
    raise ConfigurationError.new("no s3_source key exists in .gemrc") unless s3_source

    auth = s3_source[host] || s3_source[host.to_sym]
    raise ConfigurationError.new("no key for host #{host} in s3_source in .gemrc") unless auth

    provider = auth[:provider] || auth["provider"]
    case provider
    when "env"
      id = ENV["AWS_ACCESS_KEY_ID"]
      secret = ENV["AWS_SECRET_ACCESS_KEY"]
      security_token = ENV["AWS_SESSION_TOKEN"]
    when "instance_profile"
      credentials = ec2_metadata_credentials_json
      id = credentials["AccessKeyId"]
      secret = credentials["SecretAccessKey"]
      security_token = credentials["Token"]
    else
      id = auth[:id] || auth["id"]
      secret = auth[:secret] || auth["secret"]
      security_token = auth[:security_token] || auth["security_token"]
    end

    raise ConfigurationError.new("s3_source for #{host} missing id or secret") unless id && secret

    region = auth[:region] || auth["region"] || "us-east-1"
    S3Config.new(id, secret, security_token, region)
  end

  def base64_uri_escape(str)
    str.gsub(%r{[\+/=\n]}, BASE64_URI_TRANSLATE)
  end

  def ec2_metadata_credentials_json
    require_relative "vendored_net_http"
    require_relative "request"
    require_relative "request/connection_pools"
    require "json"

    iam_info = ec2_metadata_request(EC2_IAM_INFO)
    # Expected format: arn:aws:iam::<id>:instance-profile/<role_name>
    role_name = iam_info["InstanceProfileArn"].split("/").last
    ec2_metadata_request(EC2_IAM_SECURITY_CREDENTIALS + role_name)
  end

  def ec2_metadata_request(url)
    uri = Gem::URI(url)
    @request_pool ||= create_request_pool(uri)
    request = Gem::Request.new(uri, Gem::Net::HTTP::Get, nil, @request_pool)
    response = request.fetch

    case response
    when Gem::Net::HTTPOK then
      JSON.parse(response.body)
    else
      raise InstanceProfileError.new("Unable to fetch AWS metadata from #{uri}: #{response.message} #{response.code}")
    end
  end

  def create_request_pool(uri)
    proxy_uri = Gem::Request.proxy_uri(Gem::Request.get_proxy_from_env(uri.scheme))
    certs = Gem::Request.get_cert_files
    Gem::Request::ConnectionPools.new(proxy_uri, certs).pool_for(uri)
  end

  BASE64_URI_TRANSLATE = { "+" => "%2B", "/" => "%2F", "=" => "%3D", "\n" => "" }.freeze
  EC2_IAM_INFO = "http://169.254.169.254/latest/meta-data/iam/info"
  EC2_IAM_SECURITY_CREDENTIALS = "http://169.254.169.254/latest/meta-data/iam/security-credentials/"
end
                                      rubygems/rubygems/security_option.rb                                                                0000644                 00000002074 15040313423 0014020 0                                                                                                    ustar 00                                                                                                                                                                                                                                                       # frozen_string_literal: true

#--
# Copyr