<?php
/**
 * Handle frontend scripts
 *
 * @package Kkart\Classes
 * @version 2.3.0
 */

use Automattic\Jetpack\Constants;

if ( ! defined( 'ABSPATH' ) ) {
	exit;
}

/**
 * Frontend scripts class.
 */
class KKART_Frontend_Scripts {

	/**
	 * Contains an array of script handles registered by KKART.
	 *
	 * @var array
	 */
	private static $scripts = array();

	/**
	 * Contains an array of script handles registered by KKART.
	 *
	 * @var array
	 */
	private static $styles = array();

	/**
	 * Contains an array of script handles localized by KKART.
	 *
	 * @var array
	 */
	private static $wp_localize_scripts = array();

	/**
	 * Hook in methods.
	 */
	public static function init() {
		add_action( 'wp_enqueue_scripts', array( __CLASS__, 'load_scripts' ) );
		add_action( 'wp_print_scripts', array( __CLASS__, 'localize_printed_scripts' ), 5 );
		add_action( 'wp_print_footer_scripts', array( __CLASS__, 'localize_printed_scripts' ), 5 );
	}

	/**
	 * Get styles for the frontend.
	 *
	 * @return array
	 */
	public static function get_styles() {
		$version = Constants::get_constant( 'KKART_VERSION' );

		return apply_filters(
			'kkart_enqueue_styles',
			array(
				'kkart-layout'      => array(
					'src'     => self::get_asset_url( 'assets/css/kkart-layout.css' ),
					'deps'    => '',
					'version' => $version,
					'media'   => 'all',
					'has_rtl' => true,
				),
				'kkart-smallscreen' => array(
					'src'     => self::get_asset_url( 'assets/css/kkart-smallscreen.css' ),
					'deps'    => 'kkart-layout',
					'version' => $version,
					'media'   => 'only screen and (max-width: ' . apply_filters( 'kkart_style_smallscreen_breakpoint', '768px' ) . ')',
					'has_rtl' => true,
				),
				'kkart-general'     => array(
					'src'     => self::get_asset_url( 'assets/css/kkart.css' ),
					'deps'    => '',
					'version' => $version,
					'media'   => 'all',
					'has_rtl' => true,
				),
			)
		);
	}

	/**
	 * Return asset URL.
	 *
	 * @param string $path Assets path.
	 * @return string
	 */
	private static function get_asset_url( $path ) {
		return apply_filters( 'kkart_get_asset_url', plugins_url( $path, KKART_PLUGIN_FILE ), $path );
	}

	/**
	 * Register a script for use.
	 *
	 * @uses   wp_register_script()
	 * @param  string   $handle    Name of the script. Should be unique.
	 * @param  string   $path      Full URL of the script, or path of the script relative to the WordPress root directory.
	 * @param  string[] $deps      An array of registered script handles this script depends on.
	 * @param  string   $version   String specifying script version number, if it has one, which is added to the URL as a query string for cache busting purposes. If version is set to false, a version number is automatically added equal to current installed WordPress version. If set to null, no version is added.
	 * @param  boolean  $in_footer Whether to enqueue the script before </body> instead of in the <head>. Default 'false'.
	 */
	private static function register_script( $handle, $path, $deps = array( 'jquery' ), $version = KKART_VERSION, $in_footer = true ) {
		self::$scripts[] = $handle;
		wp_register_script( $handle, $path, $deps, $version, $in_footer );
	}

	/**
	 * Register and enqueue a script for use.
	 *
	 * @uses   wp_enqueue_script()
	 * @param  string   $handle    Name of the script. Should be unique.
	 * @param  string   $path      Full URL of the script, or path of the script relative to the WordPress root directory.
	 * @param  string[] $deps      An array of registered script handles this script depends on.
	 * @param  string   $version   String specifying script version number, if it has one, which is added to the URL as a query string for cache busting purposes. If version is set to false, a version number is automatically added equal to current installed WordPress version. If set to null, no version is added.
	 * @param  boolean  $in_footer Whether to enqueue the script before </body> instead of in the <head>. Default 'false'.
	 */
	private static function enqueue_script( $handle, $path = '', $deps = array( 'jquery' ), $version = KKART_VERSION, $in_footer = true ) {
		if ( ! in_array( $handle, self::$scripts, true ) && $path ) {
			self::register_script( $handle, $path, $deps, $version, $in_footer );
		}
		wp_enqueue_script( $handle );
	}

	/**
	 * Register a style for use.
	 *
	 * @uses   wp_register_style()
	 * @param  string   $handle  Name of the stylesheet. Should be unique.
	 * @param  string   $path    Full URL of the stylesheet, or path of the stylesheet relative to the WordPress root directory.
	 * @param  string[] $deps    An array of registered stylesheet handles this stylesheet depends on.
	 * @param  string   $version String specifying stylesheet version number, if it has one, which is added to the URL as a query string for cache busting purposes. If version is set to false, a version number is automatically added equal to current installed WordPress version. If set to null, no version is added.
	 * @param  string   $media   The media for which this stylesheet has been defined. Accepts media types like 'all', 'print' and 'screen', or media queries like '(orientation: portrait)' and '(max-width: 640px)'.
	 * @param  boolean  $has_rtl If has RTL version to load too.
	 */
	private static function register_style( $handle, $path, $deps = array(), $version = KKART_VERSION, $media = 'all', $has_rtl = false ) {
		self::$styles[] = $handle;
		wp_register_style( $handle, $path, $deps, $version, $media );

		if ( $has_rtl ) {
			wp_style_add_data( $handle, 'rtl', 'replace' );
		}
	}

	/**
	 * Register and enqueue a styles for use.
	 *
	 * @uses   wp_enqueue_style()
	 * @param  string   $handle  Name of the stylesheet. Should be unique.
	 * @param  string   $path    Full URL of the stylesheet, or path of the stylesheet relative to the WordPress root directory.
	 * @param  string[] $deps    An array of registered stylesheet handles this stylesheet depends on.
	 * @param  string   $version String specifying stylesheet version number, if it has one, which is added to the URL as a query string for cache busting purposes. If version is set to false, a version number is automatically added equal to current installed WordPress version. If set to null, no version is added.
	 * @param  string   $media   The media for which this stylesheet has been defined. Accepts media types like 'all', 'print' and 'screen', or media queries like '(orientation: portrait)' and '(max-width: 640px)'.
	 * @param  boolean  $has_rtl If has RTL version to load too.
	 */
	private static function enqueue_style( $handle, $path = '', $deps = array(), $version = KKART_VERSION, $media = 'all', $has_rtl = false ) {
		if ( ! in_array( $handle, self::$styles, true ) && $path ) {
			self::register_style( $handle, $path, $deps, $version, $media, $has_rtl );
		}
		wp_enqueue_style( $handle );
	}

	/**
	 * Register all KKART scripts.
	 */
	private static function register_scripts() {
		$suffix  = Constants::is_true( 'SCRIPT_DEBUG' ) ? '' : '.min';
		$version = Constants::get_constant( 'KKART_VERSION' );

		$register_scripts = array(
			'flexslider'                 => array(
				'src'     => self::get_asset_url( 'assets/js/flexslider/jquery.flexslider' . $suffix . '.js' ),
				'deps'    => array( 'jquery' ),
				'version' => '2.7.2',
			),
			'js-cookie'                  => array(
				'src'     => self::get_asset_url( 'assets/js/js-cookie/js.cookie' . $suffix . '.js' ),
				'deps'    => array(),
				'version' => '2.1.4',
			),
			'jquery-blockui'             => array(
				'src'     => self::get_asset_url( 'assets/js/jquery-blockui/jquery.blockUI' . $suffix . '.js' ),
				'deps'    => array( 'jquery' ),
				'version' => '2.70',
			),
			'jquery-cookie'              => array( // deprecated.
				'src'     => self::get_asset_url( 'assets/js/jquery-cookie/jquery.cookie' . $suffix . '.js' ),
				'deps'    => array( 'jquery' ),
				'version' => '1.4.1',
			),
			'jquery-payment'             => array(
				'src'     => self::get_asset_url( 'assets/js/jquery-payment/jquery.payment' . $suffix . '.js' ),
				'deps'    => array( 'jquery' ),
				'version' => '3.0.0',
			),
			'photoswipe'                 => array(
				'src'     => self::get_asset_url( 'assets/js/photoswipe/photoswipe' . $suffix . '.js' ),
				'deps'    => array(),
				'version' => '4.1.1',
			),
			'photoswipe-ui-default'      => array(
				'src'     => self::get_asset_url( 'assets/js/photoswipe/photoswipe-ui-default' . $suffix . '.js' ),
				'deps'    => array( 'photoswipe' ),
				'version' => '4.1.1',
			),
			'prettyPhoto'                => array( // deprecated.
				'src'     => self::get_asset_url( 'assets/js/prettyPhoto/jquery.prettyPhoto' . $suffix . '.js' ),
				'deps'    => array( 'jquery' ),
				'version' => '3.1.6',
			),
			'prettyPhoto-init'           => array( // deprecated.
				'src'     => self::get_asset_url( 'assets/js/prettyPhoto/jquery.prettyPhoto.init' . $suffix . '.js' ),
				'deps'    => array( 'jquery', 'prettyPhoto' ),
				'version' => $version,
			),
			'select2'                    => array(
				'src'     => self::get_asset_url( 'assets/js/select2/select2.full' . $suffix . '.js' ),
				'deps'    => array( 'jquery' ),
				'version' => '4.0.3',
			),
			'selectWoo'                  => array(
				'src'     => self::get_asset_url( 'assets/js/selectWoo/selectWoo.full' . $suffix . '.js' ),
				'deps'    => array( 'jquery' ),
				'version' => '1.0.6',
			),
			'kkart-address-i18n'            => array(
				'src'     => self::get_asset_url( 'assets/js/frontend/address-i18n' . $suffix . '.js' ),
				'deps'    => array( 'jquery', 'kkart-country-select' ),
				'version' => $version,
			),
			'kkart-add-payment-method'      => array(
				'src'     => self::get_asset_url( 'assets/js/frontend/add-payment-method' . $suffix . '.js' ),
				'deps'    => array( 'jquery', 'kkart' ),
				'version' => $version,
			),
			'kkart-cart'                    => array(
				'src'     => self::get_asset_url( 'assets/js/frontend/cart' . $suffix . '.js' ),
				'deps'    => array( 'jquery', 'kkart', 'kkart-country-select', 'kkart-address-i18n' ),
				'version' => $version,
			),
			'kkart-cart-fragments'          => array(
				'src'     => self::get_asset_url( 'assets/js/frontend/cart-fragments' . $suffix . '.js' ),
				'deps'    => array( 'jquery', 'js-cookie' ),
				'version' => $version,
			),
			'kkart-checkout'                => array(
				'src'     => self::get_asset_url( 'assets/js/frontend/checkout' . $suffix . '.js' ),
				'deps'    => array( 'jquery', 'kkart', 'kkart-country-select', 'kkart-address-i18n' ),
				'version' => $version,
			),
			'kkart-country-select'          => array(
				'src'     => self::get_asset_url( 'assets/js/frontend/country-select' . $suffix . '.js' ),
				'deps'    => array( 'jquery' ),
				'version' => $version,
			),
			'kkart-credit-card-form'        => array(
				'src'     => self::get_asset_url( 'assets/js/frontend/credit-card-form' . $suffix . '.js' ),
				'deps'    => array( 'jquery', 'jquery-payment' ),
				'version' => $version,
			),
			'kkart-add-to-cart'             => array(
				'src'     => self::get_asset_url( 'assets/js/frontend/add-to-cart' . $suffix . '.js' ),
				'deps'    => array( 'jquery', 'jquery-blockui' ),
				'version' => $version,
			),
			'kkart-add-to-cart-variation'   => array(
				'src'     => self::get_asset_url( 'assets/js/frontend/add-to-cart-variation' . $suffix . '.js' ),
				'deps'    => array( 'jquery', 'wp-util', 'jquery-blockui' ),
				'version' => $version,
			),
			'kkart-geolocation'             => array(
				'src'     => self::get_asset_url( 'assets/js/frontend/geolocation' . $suffix . '.js' ),
				'deps'    => array( 'jquery' ),
				'version' => $version,
			),
			'kkart-lost-password'           => array(
				'src'     => self::get_asset_url( 'assets/js/frontend/lost-password' . $suffix . '.js' ),
				'deps'    => array( 'jquery', 'kkart' ),
				'version' => $version,
			),
			'kkart-password-strength-meter' => array(
				'src'     => self::get_asset_url( 'assets/js/frontend/password-strength-meter' . $suffix . '.js' ),
				'deps'    => array( 'jquery', 'password-strength-meter' ),
				'version' => $version,
			),
			'kkart-single-product'          => array(
				'src'     => self::get_asset_url( 'assets/js/frontend/single-product' . $suffix . '.js' ),
				'deps'    => array( 'jquery' ),
				'version' => $version,
			),
			'kkart'                => array(
				'src'     => self::get_asset_url( 'assets/js/frontend/kkart' . $suffix . '.js' ),
				'deps'    => array( 'jquery', 'jquery-blockui', 'js-cookie' ),
				'version' => $version,
			),
			'zoom'                       => array(
				'src'     => self::get_asset_url( 'assets/js/zoom/jquery.zoom' . $suffix . '.js' ),
				'deps'    => array( 'jquery' ),
				'version' => '1.7.21',
			),
		);
		foreach ( $register_scripts as $name => $props ) {
			self::register_script( $name, $props['src'], $props['deps'], $props['version'] );
		}
	}

	/**
	 * Register all KKART styles.
	 */
	private static function register_styles() {
		$version = Constants::get_constant( 'KKART_VERSION' );

		$register_styles = array(
			'photoswipe'                  => array(
				'src'     => self::get_asset_url( 'assets/css/photoswipe/photoswipe.min.css' ),
				'deps'    => array(),
				'version' => $version,
				'has_rtl' => false,
			),
			'photoswipe-default-skin'     => array(
				'src'     => self::get_asset_url( 'assets/css/photoswipe/default-skin/default-skin.min.css' ),
				'deps'    => array( 'photoswipe' ),
				'version' => $version,
				'has_rtl' => false,
			),
			'select2'                     => array(
				'src'     => self::get_asset_url( 'assets/css/select2.css' ),
				'deps'    => array(),
				'version' => $version,
				'has_rtl' => false,
			),
			'kkart_prettyPhoto_css' => array( // deprecated.
				'src'     => self::get_asset_url( 'assets/css/prettyPhoto.css' ),
				'deps'    => array(),
				'version' => $version,
				'has_rtl' => true,
			),
		);
		foreach ( $register_styles as $name => $props ) {
			self::register_style( $name, $props['src'], $props['deps'], $props['version'], 'all', $props['has_rtl'] );
		}
	}

	/**
	 * Register/queue frontend scripts.
	 */
	public static function load_scripts() {
		global $post;

		if ( ! did_action( 'before_kkart_init' ) ) {
			return;
		}

		self::register_scripts();
		self::register_styles();

		if ( 'yes' === get_option( 'kkart_enable_ajax_add_to_cart' ) ) {
			self::enqueue_script( 'kkart-add-to-cart' );
		}
		// if ( is_cart() ) {
			// self::enqueue_script( 'kkart-cart' );
		// }
		if ( is_cart() || is_checkout() || is_account_page() ) {
			self::enqueue_script( 'selectWoo' );
			self::enqueue_style( 'select2' );

			// Password strength meter. Load in checkout, account login and edit account page.
			if ( ( 'no' === get_option( 'kkart_registration_generate_password' ) && ! is_user_logged_in() ) || is_edit_account_page() || is_lost_password_page() ) {
				self::enqueue_script( 'kkart-password-strength-meter' );
			}
		}
		// if ( is_checkout() ) {
			// self::enqueue_script( 'kkart-checkout' );
		// }
		if ( is_add_payment_method_page() ) {
			self::enqueue_script( 'kkart-add-payment-method' );
		}
		if ( is_lost_password_page() ) {
			self::enqueue_script( 'kkart-lost-password' );
		}

		// Load gallery scripts on product pages only if supported.
		if ( is_product() || ( ! empty( $post->post_content ) && strstr( $post->post_content, '[product_page' ) ) ) {
			if ( current_theme_supports( 'kkart-product-gallery-zoom' ) ) {
				self::enqueue_script( 'zoom' );
			}
			if ( current_theme_supports( 'kkart-product-gallery-slider' ) ) {
				self::enqueue_script( 'flexslider' );
			}
			if ( current_theme_supports( 'kkart-product-gallery-lightbox' ) ) {
				self::enqueue_script( 'photoswipe-ui-default' );
				self::enqueue_style( 'photoswipe-default-skin' );
				add_action( 'wp_footer', 'kkart_photoswipe' );
			}
			self::enqueue_script( 'kkart-single-product' );
		}

		if ( 'geolocation_ajax' === get_option( 'kkart_default_customer_address' ) ) {
			$ua = strtolower( kkart_get_user_agent() ); // Exclude common bots from geolocation by user agent.

			if ( ! strstr( $ua, 'bot' ) && ! strstr( $ua, 'spider' ) && ! strstr( $ua, 'crawl' ) ) {
				self::enqueue_script( 'kkart-geolocation' );
			}
		}

		// Global frontend scripts.
		self::enqueue_script( 'kkart' );
		self::enqueue_script( 'kkart-cart-fragments' );

		// CSS Styles.
		$enqueue_styles = self::get_styles();
		if ( $enqueue_styles ) {
			foreach ( $enqueue_styles as $handle => $args ) {
				if ( ! isset( $args['has_rtl'] ) ) {
					$args['has_rtl'] = false;
				}

				self::enqueue_style( $handle, $args['src'], $args['deps'], $args['version'], $args['media'], $args['has_rtl'] );
			}
		}

		// Placeholder style.
		wp_register_style( 'kkart-inline', false ); // phpcs:ignore
		wp_enqueue_style( 'kkart-inline' );

		if ( true === kkart_string_to_bool( get_option( 'kkart_checkout_highlight_required_fields', 'yes' ) ) ) {
			wp_add_inline_style( 'kkart-inline', '.kkart form .form-row .required { visibility: visible; }' );
		} else {
			wp_add_inline_style( 'kkart-inline', '.kkart form .form-row .required { visibility: hidden; }' );
		}
	}

	/**
	 * Localize a KKART script once.
	 *
	 * @since 2.3.0 this needs less wp_script_is() calls due to https://core.trac.wordpress.org/ticket/28404 being added in WP 4.0.
	 * @param string $handle Script handle the data will be attached to.
	 */
	private static function localize_script( $handle ) {
		if ( ! in_array( $handle, self::$wp_localize_scripts, true ) && wp_script_is( $handle ) ) {
			$data = self::get_script_data( $handle );

			if ( ! $data ) {
				return;
			}

			$name                        = str_replace( '-', '_', $handle ) . '_params';
			self::$wp_localize_scripts[] = $handle;
			wp_localize_script( $handle, $name, apply_filters( $name, $data ) );
		}
	}

	/**
	 * Return data for script handles.
	 *
	 * @param  string $handle Script handle the data will be attached to.
	 * @return array|bool
	 */
	private static function get_script_data( $handle ) {
		global $wp;

		switch ( $handle ) {
			case 'kkart':
				$params = array(
					'ajax_url'    => KKART()->ajax_url(),
					'kkart_ajax_url' => KKART_AJAX::get_endpoint( '%%endpoint%%' ),
				);
				break;
			case 'kkart-geolocation':
				$params = array(
					'kkart_ajax_url'  => KKART_AJAX::get_endpoint( '%%endpoint%%' ),
					'home_url'     => remove_query_arg( 'lang', home_url() ), // FIX for WPML compatibility.
					'is_available' => ! ( is_cart() || is_account_page() || is_checkout() || is_customize_preview() ) ? '1' : '0',
					'hash'         => isset( $_GET['v'] ) ? kkart_clean( wp_unslash( $_GET['v'] ) ) : '', // WPCS: input var ok, CSRF ok.
				);
				break;
			case 'kkart-single-product':
				$params = array(
					'i18n_required_rating_text' => esc_attr__( 'Please select a rating', 'kkart' ),
					'review_rating_required'    => kkart_review_ratings_required() ? 'yes' : 'no',
					'flexslider'                => apply_filters(
						'kkart_single_product_carousel_options',
						array(
							'rtl'            => is_rtl(),
							'animation'      => 'slide',
							'smoothHeight'   => true,
							'directionNav'   => false,
							'controlNav'     => 'thumbnails',
							'slideshow'      => false,
							'animationSpeed' => 500,
							'animationLoop'  => false, // Breaks photoswipe pagination if true.
							'allowOneSlide'  => false,
						)
					),
					'zoom_enabled'              => apply_filters( 'kkart_single_product_zoom_enabled', get_theme_support( 'kkart-product-gallery-zoom' ) ),
					'zoom_options'              => apply_filters( 'kkart_single_product_zoom_options', array() ),
					'photoswipe_enabled'        => apply_filters( 'kkart_single_product_photoswipe_enabled', get_theme_support( 'kkart-product-gallery-lightbox' ) ),
					'photoswipe_options'        => apply_filters(
						'kkart_single_product_photoswipe_options',
						array(
							'shareEl'               => false,
							'closeOnScroll'         => false,
							'history'               => false,
							'hideAnimationDuration' => 0,
							'showAnimationDuration' => 0,
						)
					),
					'flexslider_enabled'        => apply_filters( 'kkart_single_product_flexslider_enabled', get_theme_support( 'kkart-product-gallery-slider' ) ),
				);
				break;
			case 'kkart-checkout':
				$params = array(
					'ajax_url'                  => KKART()->ajax_url(),
					'kkart_ajax_url'               => KKART_AJAX::get_endpoint( '%%endpoint%%' ),
					'update_order_review_nonce' => wp_create_nonce( 'update-order-review' ),
					'apply_coupon_nonce'        => wp_create_nonce( 'apply-coupon' ),
					'remove_coupon_nonce'       => wp_create_nonce( 'remove-coupon' ),
					'option_guest_checkout'     => get_option( 'kkart_enable_guest_checkout' ),
					'checkout_url'              => KKART_AJAX::get_endpoint( 'checkout' ),
					'is_checkout'               => is_checkout() ? 1 : 0,
					'debug_mode'                => Constants::is_true( 'WP_DEBUG' ),
					'i18n_checkout_error'       => esc_attr__( 'Error processing checkout. Please try again.', 'kkart' ),
				);
				break;
			case 'kkart-address-i18n':
				$params = array(
					'locale'             => wp_json_encode( KKART()->countries->get_country_locale() ),
					'locale_fields'      => wp_json_encode( KKART()->countries->get_country_locale_field_selectors() ),
					'i18n_required_text' => esc_attr__( 'required', 'kkart' ),
					'i18n_optional_text' => esc_html__( 'optional', 'kkart' ),
				);
				break;
			case 'kkart-cart':
				$params = array(
					'ajax_url'                     => KKART()->ajax_url(),
					'kkart_ajax_url'                  => KKART_AJAX::get_endpoint( '%%endpoint%%' ),
					'update_shipping_method_nonce' => wp_create_nonce( 'update-shipping-method' ),
					'apply_coupon_nonce'           => wp_create_nonce( 'apply-coupon' ),
					'remove_coupon_nonce'          => wp_create_nonce( 'remove-coupon' ),
				);
				break;
			case 'kkart-cart-fragments':
				$params = array(
					'ajax_url'        => KKART()->ajax_url(),
					'kkart_ajax_url'     => KKART_AJAX::get_endpoint( '%%endpoint%%' ),
					'cart_hash_key'   => apply_filters( 'kkart_cart_hash_key', 'kkart_cart_hash_' . md5( get_current_blog_id() . '_' . get_site_url( get_current_blog_id(), '/' ) . get_template() ) ),
					'fragment_name'   => apply_filters( 'kkart_cart_fragment_name', 'kkart_fragments_' . md5( get_current_blog_id() . '_' . get_site_url( get_current_blog_id(), '/' ) . get_template() ) ),
					'request_timeout' => 5000,
				);
				break;
			case 'kkart-add-to-cart':
				$params = array(
					'ajax_url'                => KKART()->ajax_url(),
					'kkart_ajax_url'             => KKART_AJAX::get_endpoint( '%%endpoint%%' ),
					'i18n_view_cart'          => esc_attr__( 'View cart', 'kkart' ),
					'cart_url'                => apply_filters( 'kkart_add_to_cart_redirect', kkart_get_cart_url(), null ),
					'is_cart'                 => is_cart(),
					'cart_redirect_after_add' => get_option( 'kkart_cart_redirect_after_add' ),
				);
				break;
			case 'kkart-add-to-cart-variation':
				// We also need the wp.template for this script :).
				?>
				<script type="text/template" id="tmpl-variation-template">
					<div class="kkart-variation-description">{{{ data.variation.variation_description }}}</div>
					<div class="kkart-variation-price">{{{ data.variation.price_html }}}</div>
					<div class="kkart-variation-availability">{{{ data.variation.availability_html }}}</div>
				</script>
				<script type="text/template" id="tmpl-unavailable-variation-template">
					<p><?php esc_html_e( 'Sorry, this product is unavailable. Please choose a different combination.', 'kkart' ); ?></p>
				</script>
				<?php

				$params = array(
					'kkart_ajax_url'                      => KKART_AJAX::get_endpoint( '%%endpoint%%' ),
					'i18n_no_matching_variations_text' => esc_attr__( 'Sorry, no products matched your selection. Please choose a different combination.', 'kkart' ),
					'i18n_make_a_selection_text'       => esc_attr__( 'Please select some product options before adding this product to your cart.', 'kkart' ),
					'i18n_unavailable_text'            => esc_attr__( 'Sorry, this product is unavailable. Please choose a different combination.', 'kkart' ),
				);
				break;
			case 'kkart-country-select':
				$params = array(
					'countries'                 => wp_json_encode( array_merge( KKART()->countries->get_allowed_country_states(), KKART()->countries->get_shipping_country_states() ) ),
					'i18n_select_state_text'    => esc_attr__( 'Select an option&hellip;', 'kkart' ),
					'i18n_no_matches'           => _x( 'No matches found', 'enhanced select', 'kkart' ),
					'i18n_ajax_error'           => _x( 'Loading failed', 'enhanced select', 'kkart' ),
					'i18n_input_too_short_1'    => _x( 'Please enter 1 or more characters', 'enhanced select', 'kkart' ),
					'i18n_input_too_short_n'    => _x( 'Please enter %qty% or more characters', 'enhanced select', 'kkart' ),
					'i18n_input_too_long_1'     => _x( 'Please delete 1 character', 'enhanced select', 'kkart' ),
					'i18n_input_too_long_n'     => _x( 'Please delete %qty% characters', 'enhanced select', 'kkart' ),
					'i18n_selection_too_long_1' => _x( 'You can only select 1 item', 'enhanced select', 'kkart' ),
					'i18n_selection_too_long_n' => _x( 'You can only select %qty% items', 'enhanced select', 'kkart' ),
					'i18n_load_more'            => _x( 'Loading more results&hellip;', 'enhanced select', 'kkart' ),
					'i18n_searching'            => _x( 'Searching&hellip;', 'enhanced select', 'kkart' ),
				);
				break;
			case 'kkart-password-strength-meter':
				$params = array(
					'min_password_strength' => apply_filters( 'kkart_min_password_strength', 3 ),
					'stop_checkout'         => apply_filters( 'kkart_enforce_password_strength_meter_on_checkout', false ),
					'i18n_password_error'   => esc_attr__( 'Please enter a stronger password.', 'kkart' ),
					'i18n_password_hint'    => esc_attr( wp_get_password_hint() ),
				);
				break;
			default:
				$params = false;
		}

		$params = apply_filters_deprecated( $handle . '_params', array( $params ), '3.0.0', 'kkart_get_script_data' );

		return apply_filters( 'kkart_get_script_data', $params, $handle );
	}

	/**
	 * Localize scripts only when enqueued.
	 */
	public static function localize_printed_scripts() {
		foreach ( self::$scripts as $handle ) {
			self::localize_script( $handle );
		}
	}
}

KKART_Frontend_Scripts::init();
