<?php
/**
 * Privacy/GDPR related functionality which ties into WordPress functionality.
 *
 * @since 3.4.0
 * @package Kkart\Classes
 */

defined( 'ABSPATH' ) || exit;

if ( ! class_exists( 'KKART_Privacy_Background_Process', false ) ) {
	include_once __DIR__ . '/class-kkart-privacy-background-process.php';
}

/**
 * KKART_Privacy Class.
 */
class KKART_Privacy extends KKART_Abstract_Privacy {

	/**
	 * Background process to clean up orders.
	 *
	 * @var KKART_Privacy_Background_Process
	 */
	protected static $background_process;

	/**
	 * Init - hook into events.
	 */
	public function __construct() {
		parent::__construct( __( 'Kkart', 'kkart' ) );

		if ( ! self::$background_process ) {
			self::$background_process = new KKART_Privacy_Background_Process();
		}

		// Include supporting classes.
		include_once __DIR__ . '/class-kkart-privacy-erasers.php';
		include_once __DIR__ . '/class-kkart-privacy-exporters.php';

		// This hook registers Kkart data exporters.
		$this->add_exporter( 'kkart-customer-data', __( 'Kkart Customer Data', 'kkart' ), array( 'KKART_Privacy_Exporters', 'customer_data_exporter' ) );
		$this->add_exporter( 'kkart-customer-orders', __( 'Kkart Customer Orders', 'kkart' ), array( 'KKART_Privacy_Exporters', 'order_data_exporter' ) );
		$this->add_exporter( 'kkart-customer-downloads', __( 'Kkart Customer Downloads', 'kkart' ), array( 'KKART_Privacy_Exporters', 'download_data_exporter' ) );
		$this->add_exporter( 'kkart-customer-tokens', __( 'Kkart Customer Payment Tokens', 'kkart' ), array( 'KKART_Privacy_Exporters', 'customer_tokens_exporter' ) );

		// This hook registers Kkart data erasers.
		$this->add_eraser( 'kkart-customer-data', __( 'Kkart Customer Data', 'kkart' ), array( 'KKART_Privacy_Erasers', 'customer_data_eraser' ) );
		$this->add_eraser( 'kkart-customer-orders', __( 'Kkart Customer Orders', 'kkart' ), array( 'KKART_Privacy_Erasers', 'order_data_eraser' ) );
		$this->add_eraser( 'kkart-customer-downloads', __( 'Kkart Customer Downloads', 'kkart' ), array( 'KKART_Privacy_Erasers', 'download_data_eraser' ) );
		$this->add_eraser( 'kkart-customer-tokens', __( 'Kkart Customer Payment Tokens', 'kkart' ), array( 'KKART_Privacy_Erasers', 'customer_tokens_eraser' ) );

		// Cleanup orders daily - this is a callback on a daily cron event.
		add_action( 'kkart_cleanup_personal_data', array( $this, 'queue_cleanup_personal_data' ) );

		// Handles custom anonomization types not included in core.
		add_filter( 'wp_privacy_anonymize_data', array( $this, 'anonymize_custom_data_types' ), 10, 3 );

		// When this is fired, data is removed in a given order. Called from bulk actions.
		add_action( 'kkart_remove_order_personal_data', array( 'KKART_Privacy_Erasers', 'remove_order_personal_data' ) );
	}

	/**
	 * Add privacy policy content for the privacy policy page.
	 *
	 * @since 3.4.0
	 */
	public function get_privacy_message() {
		$content = '<div class="wp-suggested-text">' .
			'<p class="privacy-policy-tutorial">' .
				__( 'This sample language includes the basics around what personal data your store may be collecting, storing and sharing, as well as who may have access to that data. Depending on what settings are enabled and which additional plugins are used, the specific information shared by your store will vary. We recommend consulting with a lawyer when deciding what information to disclose on your privacy policy.', 'kkart' ) .
			'</p>' .
			'<p>' . __( 'We collect information about you during the checkout process on our store.', 'kkart' ) . '</p>' .
			'<h2>' . __( 'What we collect and store', 'kkart' ) . '</h2>' .
			'<p>' . __( 'While you visit our site, we’ll track:', 'kkart' ) . '</p>' .
			'<ul>' .
				'<li>' . __( 'Products you’ve viewed:  we’ll use this to, for example, show you products you’ve recently viewed', 'kkart' ) . '</li>' .
				'<li>' . __( 'Location, IP address and browser type: we’ll use this for purposes like estimating taxes and shipping', 'kkart' ) . '</li>' .
				'<li>' . __( 'Shipping address: we’ll ask you to enter this so we can, for instance, estimate shipping before you place an order, and send you the order!', 'kkart' ) . '</li>' .
			'</ul>' .
			'<p>' . __( 'We’ll also use cookies to keep track of cart contents while you’re browsing our site.', 'kkart' ) . '</p>' .
			'<p class="privacy-policy-tutorial">' .
				__( 'Note: you may want to further detail your cookie policy, and link to that section from here.', 'kkart' ) .
			'</p>' .
			'<p>' . __( 'When you purchase from us, we’ll ask you to provide information including your name, billing address, shipping address, email address, phone number, credit card/payment details and optional account information like username and password. We’ll use this information for purposes, such as, to:', 'kkart' ) . '</p>' .
			'<ul>' .
				'<li>' . __( 'Send you information about your account and order', 'kkart' ) . '</li>' .
				'<li>' . __( 'Respond to your requests, including refunds and complaints', 'kkart' ) . '</li>' .
				'<li>' . __( 'Process payments and prevent fraud', 'kkart' ) . '</li>' .
				'<li>' . __( 'Set up your account for our store', 'kkart' ) . '</li>' .
				'<li>' . __( 'Comply with any legal obligations we have, such as calculating taxes', 'kkart' ) . '</li>' .
				'<li>' . __( 'Improve our store offerings', 'kkart' ) . '</li>' .
				'<li>' . __( 'Send you marketing messages, if you choose to receive them', 'kkart' ) . '</li>' .
			'</ul>' .
			'<p>' . __( 'If you create an account, we will store your name, address, email and phone number, which will be used to populate the checkout for future orders.', 'kkart' ) . '</p>' .
			'<p>' . __( 'We generally store information about you for as long as we need the information for the purposes for which we collect and use it, and we are not legally required to continue to keep it. For example, we will store order information for XXX years for tax and accounting purposes. This includes your name, email address and billing and shipping addresses.', 'kkart' ) . '</p>' .
			'<p>' . __( 'We will also store comments or reviews, if you choose to leave them.', 'kkart' ) . '</p>' .
			'<h2>' . __( 'Who on our team has access', 'kkart' ) . '</h2>' .
			'<p>' . __( 'Members of our team have access to the information you provide us. For example, both Administrators and Shop Managers can access:', 'kkart' ) . '</p>' .
			'<ul>' .
				'<li>' . __( 'Order information like what was purchased, when it was purchased and where it should be sent, and', 'kkart' ) . '</li>' .
				'<li>' . __( 'Customer information like your name, email address, and billing and shipping information.', 'kkart' ) . '</li>' .
			'</ul>' .
			'<p>' . __( 'Our team members have access to this information to help fulfill orders, process refunds and support you.', 'kkart' ) . '</p>' .
			'<h2>' . __( 'What we share with others', 'kkart' ) . '</h2>' .
			'<p class="privacy-policy-tutorial">' .
				__( 'In this section you should list who you’re sharing data with, and for what purpose. This could include, but may not be limited to, analytics, marketing, payment gateways, shipping providers, and third party embeds.', 'kkart' ) .
			'</p>' .
			'<p>' . __( 'We share information with third parties who help us provide our orders and store services to you; for example --', 'kkart' ) . '</p>' .
			'<h3>' . __( 'Payments', 'kkart' ) . '</h3>' .
			'<p class="privacy-policy-tutorial">' .
				__( 'In this subsection you should list which third party payment processors you’re using to take payments on your store since these may handle customer data. We’ve included PayPal as an example, but you should remove this if you’re not using PayPal.', 'kkart' ) .
			'</p>' .
			'<p>' . __( 'We accept payments through PayPal. When processing payments, some of your data will be passed to PayPal, including information required to process or support the payment, such as the purchase total and billing information.', 'kkart' ) . '</p>' .
			'<p>' . __( 'Please see the <a href="https://www.paypal.com/us/webapps/mpp/ua/privacy-full">PayPal Privacy Policy</a> for more details.', 'kkart' ) . '</p>' .
			'</div>';

		return apply_filters( 'kkart_privacy_policy_content', $content );
	}

	/**
	 * Spawn events for order cleanup.
	 */
	public function queue_cleanup_personal_data() {
		self::$background_process->push_to_queue( array( 'task' => 'trash_pending_orders' ) );
		self::$background_process->push_to_queue( array( 'task' => 'trash_failed_orders' ) );
		self::$background_process->push_to_queue( array( 'task' => 'trash_cancelled_orders' ) );
		self::$background_process->push_to_queue( array( 'task' => 'anonymize_completed_orders' ) );
		self::$background_process->push_to_queue( array( 'task' => 'delete_inactive_accounts' ) );
		self::$background_process->save()->dispatch();
	}

	/**
	 * Handle some custom types of data and anonymize them.
	 *
	 * @param string $anonymous Anonymized string.
	 * @param string $type Type of data.
	 * @param string $data The data being anonymized.
	 * @return string Anonymized string.
	 */
	public function anonymize_custom_data_types( $anonymous, $type, $data ) {
		switch ( $type ) {
			case 'address_state':
			case 'address_country':
				$anonymous = ''; // Empty string - we don't want to store anything after removal.
				break;
			case 'phone':
				$anonymous = preg_replace( '/\d/u', '0', $data );
				break;
			case 'numeric_id':
				$anonymous = 0;
				break;
		}
		return $anonymous;
	}

	/**
	 * Find and trash old orders.
	 *
	 * @since 3.4.0
	 * @param  int $limit Limit orders to process per batch.
	 * @return int Number of orders processed.
	 */
	public static function trash_pending_orders( $limit = 20 ) {
		$option = kkart_parse_relative_date_option( get_option( 'kkart_trash_pending_orders' ) );

		if ( empty( $option['number'] ) ) {
			return 0;
		}

		return self::trash_orders_query(
			apply_filters(
				'kkart_trash_pending_orders_query_args',
				array(
					'date_created' => '<' . strtotime( '-' . $option['number'] . ' ' . $option['unit'] ),
					'limit'        => $limit, // Batches of 20.
					'status'       => 'kkart-pending',
					'type'         => 'shop_order',
				)
			)
		);
	}

	/**
	 * Find and trash old orders.
	 *
	 * @since 3.4.0
	 * @param  int $limit Limit orders to process per batch.
	 * @return int Number of orders processed.
	 */
	public static function trash_failed_orders( $limit = 20 ) {
		$option = kkart_parse_relative_date_option( get_option( 'kkart_trash_failed_orders' ) );

		if ( empty( $option['number'] ) ) {
			return 0;
		}

		return self::trash_orders_query(
			apply_filters(
				'kkart_trash_failed_orders_query_args',
				array(
					'date_created' => '<' . strtotime( '-' . $option['number'] . ' ' . $option['unit'] ),
					'limit'        => $limit, // Batches of 20.
					'status'       => 'kkart-failed',
					'type'         => 'shop_order',
				)
			)
		);
	}

	/**
	 * Find and trash old orders.
	 *
	 * @since 3.4.0
	 * @param  int $limit Limit orders to process per batch.
	 * @return int Number of orders processed.
	 */
	public static function trash_cancelled_orders( $limit = 20 ) {
		$option = kkart_parse_relative_date_option( get_option( 'kkart_trash_cancelled_orders' ) );

		if ( empty( $option['number'] ) ) {
			return 0;
		}

		return self::trash_orders_query(
			apply_filters(
				'kkart_trash_cancelled_orders_query_args',
				array(
					'date_created' => '<' . strtotime( '-' . $option['number'] . ' ' . $option['unit'] ),
					'limit'        => $limit, // Batches of 20.
					'status'       => 'kkart-cancelled',
					'type'         => 'shop_order',
				)
			)
		);
	}

	/**
	 * For a given query trash all matches.
	 *
	 * @since 3.4.0
	 * @param array $query Query array to pass to kkart_get_orders().
	 * @return int Count of orders that were trashed.
	 */
	protected static function trash_orders_query( $query ) {
		$orders = kkart_get_orders( $query );
		$count  = 0;

		if ( $orders ) {
			foreach ( $orders as $order ) {
				$order->delete( false );
				$count ++;
			}
		}

		return $count;
	}

	/**
	 * Anonymize old completed orders.
	 *
	 * @since 3.4.0
	 * @param  int $limit Limit orders to process per batch.
	 * @return int Number of orders processed.
	 */
	public static function anonymize_completed_orders( $limit = 20 ) {
		$option = kkart_parse_relative_date_option( get_option( 'kkart_anonymize_completed_orders' ) );

		if ( empty( $option['number'] ) ) {
			return 0;
		}

		return self::anonymize_orders_query(
			apply_filters(
				'kkart_anonymize_completed_orders_query_args',
				array(
					'date_created' => '<' . strtotime( '-' . $option['number'] . ' ' . $option['unit'] ),
					'limit'        => $limit, // Batches of 20.
					'status'       => 'kkart-completed',
					'anonymized'   => false,
					'type'         => 'shop_order',
				)
			)
		);
	}

	/**
	 * For a given query, anonymize all matches.
	 *
	 * @since 3.4.0
	 * @param array $query Query array to pass to kkart_get_orders().
	 * @return int Count of orders that were anonymized.
	 */
	protected static function anonymize_orders_query( $query ) {
		$orders = kkart_get_orders( $query );
		$count  = 0;

		if ( $orders ) {
			foreach ( $orders as $order ) {
				KKART_Privacy_Erasers::remove_order_personal_data( $order );
				$count ++;
			}
		}

		return $count;
	}

	/**
	 * Delete inactive accounts.
	 *
	 * @since 3.4.0
	 * @param  int $limit Limit users to process per batch.
	 * @return int Number of users processed.
	 */
	public static function delete_inactive_accounts( $limit = 20 ) {
		$option = kkart_parse_relative_date_option( get_option( 'kkart_delete_inactive_accounts' ) );

		if ( empty( $option['number'] ) ) {
			return 0;
		}

		return self::delete_inactive_accounts_query( strtotime( '-' . $option['number'] . ' ' . $option['unit'] ), $limit );
	}

	/**
	 * Delete inactive accounts.
	 *
	 * @since 3.4.0
	 * @param int $timestamp Timestamp to delete customers before.
	 * @param int $limit     Limit number of users to delete per run.
	 * @return int Count of customers that were deleted.
	 */
	protected static function delete_inactive_accounts_query( $timestamp, $limit = 20 ) {
		$count      = 0;
		$user_query = new WP_User_Query(
			array(
				'fields'     => 'ID',
				'number'     => $limit,
				'role__in'   => apply_filters(
					'kkart_delete_inactive_account_roles',
					array(
						'Customer',
						'Subscriber',
					)
				),
				'meta_query' => array( // phpcs:ignore WordPress.DB.SlowDBQuery.slow_db_query_meta_query
					'relation' => 'AND',
					array(
						'key'     => 'kkart_last_active',
						'value'   => (string) $timestamp,
						'compare' => '<',
						'type'    => 'NUMERIC',
					),
					array(
						'key'     => 'kkart_last_active',
						'value'   => '0',
						'compare' => '>',
						'type'    => 'NUMERIC',
					),
				),
			)
		);

		$user_ids = $user_query->get_results();

		if ( $user_ids ) {
			if ( ! function_exists( 'wp_delete_user' ) ) {
				require_once KKART_ADMIN_DIR . 'includes/user.php';
			}

			foreach ( $user_ids as $user_id ) {
				wp_delete_user( $user_id );
				$count ++;
			}
		}

		return $count;
	}
}

new KKART_Privacy();
