<?php
/**
 * Initialize this version of the REST API.
 *
 * @package Kkart\RestApi
 */

namespace Automattic\Kkart\RestApi;

defined( 'ABSPATH' ) || exit;

use Automattic\Kkart\RestApi\Utilities\SingletonTrait;

/**
 * Class responsible for loading the REST API and all REST API namespaces.
 */
class Server {
	use SingletonTrait;

	/**
	 * REST API namespaces and endpoints.
	 *
	 * @var array
	 */
	protected $controllers = array();

	/**
	 * Hook into WordPress ready to init the REST API as needed.
	 */
	public function init() {
		add_action( 'rest_api_init', array( $this, 'register_rest_routes' ), 10 );
	}

	/**
	 * Register REST API routes.
	 */
	public function register_rest_routes() {
		foreach ( $this->get_rest_namespaces() as $namespace => $controllers ) {
			foreach ( $controllers as $controller_name => $controller_class ) {
				$this->controllers[ $namespace ][ $controller_name ] = new $controller_class();
				$this->controllers[ $namespace ][ $controller_name ]->register_routes();
			}
		}
	}

	/**
	 * Get API namespaces - new namespaces should be registered here.
	 *
	 * @return array List of Namespaces and Main controller classes.
	 */
	protected function get_rest_namespaces() {
		return apply_filters(
			'kkart_rest_api_get_rest_namespaces',
			array(
				'kkart/v1' => $this->get_v1_controllers(),
				'kkart/v2' => $this->get_v2_controllers(),
				'kkart/v3' => $this->get_v3_controllers(),
			)
		);
	}

	/**
	 * List of controllers in the kkart/v1 namespace.
	 *
	 * @return array
	 */
	protected function get_v1_controllers() {
		return array(
			'coupons'                  => 'KKART_REST_Coupons_V1_Controller',
			'customer-downloads'       => 'KKART_REST_Customer_Downloads_V1_Controller',
			'customers'                => 'KKART_REST_Customers_V1_Controller',
			'order-notes'              => 'KKART_REST_Order_Notes_V1_Controller',
			'order-refunds'            => 'KKART_REST_Order_Refunds_V1_Controller',
			'orders'                   => 'KKART_REST_Orders_V1_Controller',
			'product-attribute-terms'  => 'KKART_REST_Product_Attribute_Terms_V1_Controller',
			'product-attributes'       => 'KKART_REST_Product_Attributes_V1_Controller',
			'product-categories'       => 'KKART_REST_Product_Categories_V1_Controller',
			'product-reviews'          => 'KKART_REST_Product_Reviews_V1_Controller',
			'product-shipping-classes' => 'KKART_REST_Product_Shipping_Classes_V1_Controller',
			'product-tags'             => 'KKART_REST_Product_Tags_V1_Controller',
			'products'                 => 'KKART_REST_Products_V1_Controller',
			'reports-sales'            => 'KKART_REST_Report_Sales_V1_Controller',
			'reports-top-sellers'      => 'KKART_REST_Report_Top_Sellers_V1_Controller',
			'reports'                  => 'KKART_REST_Reports_V1_Controller',
			'tax-classes'              => 'KKART_REST_Tax_Classes_V1_Controller',
			'taxes'                    => 'KKART_REST_Taxes_V1_Controller',
			'webhooks'                 => 'KKART_REST_Webhooks_V1_Controller',
			'webhook-deliveries'       => 'KKART_REST_Webhook_Deliveries_V1_Controller',
		);
	}

	/**
	 * List of controllers in the kkart/v2 namespace.
	 *
	 * @return array
	 */
	protected function get_v2_controllers() {
		return array(
			'coupons'                  => 'KKART_REST_Coupons_V2_Controller',
			'customer-downloads'       => 'KKART_REST_Customer_Downloads_V2_Controller',
			'customers'                => 'KKART_REST_Customers_V2_Controller',
			'network-orders'           => 'KKART_REST_Network_Orders_V2_Controller',
			'order-notes'              => 'KKART_REST_Order_Notes_V2_Controller',
			'order-refunds'            => 'KKART_REST_Order_Refunds_V2_Controller',
			'orders'                   => 'KKART_REST_Orders_V2_Controller',
			'product-attribute-terms'  => 'KKART_REST_Product_Attribute_Terms_V2_Controller',
			'product-attributes'       => 'KKART_REST_Product_Attributes_V2_Controller',
			'product-categories'       => 'KKART_REST_Product_Categories_V2_Controller',
			'product-reviews'          => 'KKART_REST_Product_Reviews_V2_Controller',
			'product-shipping-classes' => 'KKART_REST_Product_Shipping_Classes_V2_Controller',
			'product-tags'             => 'KKART_REST_Product_Tags_V2_Controller',
			'products'                 => 'KKART_REST_Products_V2_Controller',
			'product-variations'       => 'KKART_REST_Product_Variations_V2_Controller',
			'reports-sales'            => 'KKART_REST_Report_Sales_V2_Controller',
			'reports-top-sellers'      => 'KKART_REST_Report_Top_Sellers_V2_Controller',
			'reports'                  => 'KKART_REST_Reports_V2_Controller',
			'settings'                 => 'KKART_REST_Settings_V2_Controller',
			'settings-options'         => 'KKART_REST_Setting_Options_V2_Controller',
			'shipping-zones'           => 'KKART_REST_Shipping_Zones_V2_Controller',
			'shipping-zone-locations'  => 'KKART_REST_Shipping_Zone_Locations_V2_Controller',
			'shipping-zone-methods'    => 'KKART_REST_Shipping_Zone_Methods_V2_Controller',
			'tax-classes'              => 'KKART_REST_Tax_Classes_V2_Controller',
			'taxes'                    => 'KKART_REST_Taxes_V2_Controller',
			'webhooks'                 => 'KKART_REST_Webhooks_V2_Controller',
			'webhook-deliveries'       => 'KKART_REST_Webhook_Deliveries_V2_Controller',
			'system-status'            => 'KKART_REST_System_Status_V2_Controller',
			'system-status-tools'      => 'KKART_REST_System_Status_Tools_V2_Controller',
			'shipping-methods'         => 'KKART_REST_Shipping_Methods_V2_Controller',
			'payment-gateways'         => 'KKART_REST_Payment_Gateways_V2_Controller',
		);
	}

	/**
	 * List of controllers in the kkart/v3 namespace.
	 *
	 * @return array
	 */
	protected function get_v3_controllers() {
		return array(
			'coupons'                  => 'KKART_REST_Coupons_Controller',
			'customer-downloads'       => 'KKART_REST_Customer_Downloads_Controller',
			'customers'                => 'KKART_REST_Customers_Controller',
			'network-orders'           => 'KKART_REST_Network_Orders_Controller',
			'order-notes'              => 'KKART_REST_Order_Notes_Controller',
			'order-refunds'            => 'KKART_REST_Order_Refunds_Controller',
			'orders'                   => 'KKART_REST_Orders_Controller',
			'product-attribute-terms'  => 'KKART_REST_Product_Attribute_Terms_Controller',
			'product-attributes'       => 'KKART_REST_Product_Attributes_Controller',
			'product-categories'       => 'KKART_REST_Product_Categories_Controller',
			'product-reviews'          => 'KKART_REST_Product_Reviews_Controller',
			'product-shipping-classes' => 'KKART_REST_Product_Shipping_Classes_Controller',
			'product-tags'             => 'KKART_REST_Product_Tags_Controller',
			'products'                 => 'KKART_REST_Products_Controller',
			'product-variations'       => 'KKART_REST_Product_Variations_Controller',
			'reports-sales'            => 'KKART_REST_Report_Sales_Controller',
			'reports-top-sellers'      => 'KKART_REST_Report_Top_Sellers_Controller',
			'reports-orders-totals'    => 'KKART_REST_Report_Orders_Totals_Controller',
			'reports-products-totals'  => 'KKART_REST_Report_Products_Totals_Controller',
			'reports-customers-totals' => 'KKART_REST_Report_Customers_Totals_Controller',
			'reports-coupons-totals'   => 'KKART_REST_Report_Coupons_Totals_Controller',
			'reports-reviews-totals'   => 'KKART_REST_Report_Reviews_Totals_Controller',
			'reports'                  => 'KKART_REST_Reports_Controller',
			'settings'                 => 'KKART_REST_Settings_Controller',
			'settings-options'         => 'KKART_REST_Setting_Options_Controller',
			'shipping-zones'           => 'KKART_REST_Shipping_Zones_Controller',
			'shipping-zone-locations'  => 'KKART_REST_Shipping_Zone_Locations_Controller',
			'shipping-zone-methods'    => 'KKART_REST_Shipping_Zone_Methods_Controller',
			'tax-classes'              => 'KKART_REST_Tax_Classes_Controller',
			'taxes'                    => 'KKART_REST_Taxes_Controller',
			'webhooks'                 => 'KKART_REST_Webhooks_Controller',
			'system-status'            => 'KKART_REST_System_Status_Controller',
			'system-status-tools'      => 'KKART_REST_System_Status_Tools_Controller',
			'shipping-methods'         => 'KKART_REST_Shipping_Methods_Controller',
			'payment-gateways'         => 'KKART_REST_Payment_Gateways_Controller',
			'data'                     => 'KKART_REST_Data_Controller',
			'data-continents'          => 'KKART_REST_Data_Continents_Controller',
			'data-countries'           => 'KKART_REST_Data_Countries_Controller',
			'data-currencies'          => 'KKART_REST_Data_Currencies_Controller',
			'yoco-webhook'             => 'KKART_REST_Yoco_Webhook_Controller',
		);
	}

	/**
	 * Return the path to the package.
	 *
	 * @return string
	 */
	public static function get_path() {
		return dirname( __DIR__ );
	}
}
