<?php
/**
 * Kkart Meta Boxes
 *
 * Sets up the write panels used by products and orders (custom post types).
 *
 * @package Kkart\Admin\Meta Boxes
 */

use Automattic\Jetpack\Constants;

defined( 'ABSPATH' ) || exit;

/**
 * KKART_Admin_Meta_Boxes.
 */
class KKART_Admin_Meta_Boxes {

	/**
	 * Is meta boxes saved once?
	 *
	 * @var boolean
	 */
	private static $saved_meta_boxes = false;

	/**
	 * Meta box error messages.
	 *
	 * @var array
	 */
	public static $meta_box_errors = array();

	/**
	 * Constructor.
	 */
	public function __construct() {
		add_action( 'add_meta_boxes', array( $this, 'remove_meta_boxes' ), 10 );
		add_action( 'add_meta_boxes', array( $this, 'rename_meta_boxes' ), 20 );
		add_action( 'add_meta_boxes', array( $this, 'add_meta_boxes' ), 30 );
		add_action( 'save_post', array( $this, 'save_meta_boxes' ), 1, 2 );

		/**
		 * Save Order Meta Boxes.
		 *
		 * In order:
		 *      Save the order items.
		 *      Save the order totals.
		 *      Save the order downloads.
		 *      Save order data - also updates status and sends out admin emails if needed. Last to show latest data.
		 *      Save actions - sends out other emails. Last to show latest data.
		 */
		add_action( 'kkart_process_shop_order_meta', 'KKART_Meta_Box_Order_Items::save', 10 );
		add_action( 'kkart_process_shop_order_meta', 'KKART_Meta_Box_Order_Downloads::save', 30, 2 );
		add_action( 'kkart_process_shop_order_meta', 'KKART_Meta_Box_Order_Data::save', 40 );
		add_action( 'kkart_process_shop_order_meta', 'KKART_Meta_Box_Order_Actions::save', 50, 2 );

		// Save Product Meta Boxes.
		add_action( 'kkart_process_product_meta', 'KKART_Meta_Box_Product_Data::save', 10, 2 );
		add_action( 'kkart_process_product_meta', 'KKART_Meta_Box_Product_Images::save', 20, 2 );

		// Save Coupon Meta Boxes.
		add_action( 'kkart_process_shop_coupon_meta', 'KKART_Meta_Box_Coupon_Data::save', 10, 2 );

		// Save Rating Meta Boxes.
		add_filter( 'wp_update_comment_data', 'KKART_Meta_Box_Product_Reviews::save', 1 );

		// Error handling (for showing errors from meta boxes on next page load).
		add_action( 'admin_notices', array( $this, 'output_errors' ) );
		add_action( 'shutdown', array( $this, 'save_errors' ) );
	}

	/**
	 * Add an error message.
	 *
	 * @param string $text Error to add.
	 */
	public static function add_error( $text ) {
		self::$meta_box_errors[] = $text;
	}

	/**
	 * Save errors to an option.
	 */
	public function save_errors() {
		update_option( 'kkart_meta_box_errors', self::$meta_box_errors );
	}

	/**
	 * Show any stored error messages.
	 */
	public function output_errors() {
		$errors = array_filter( (array) get_option( 'kkart_meta_box_errors' ) );

		if ( ! empty( $errors ) ) {

			echo '<div id="kkart_errors" class="error notice is-dismissible">';

			foreach ( $errors as $error ) {
				echo '<p>' . wp_kses_post( $error ) . '</p>';
			}

			echo '</div>';

			// Clear.
			delete_option( 'kkart_meta_box_errors' );
		}
	}

	/**
	 * Add KKART Meta boxes.
	 */
	public function add_meta_boxes() {
		$screen    = get_current_screen();
		$screen_id = $screen ? $screen->id : '';

		// Products.
		add_meta_box( 'postexcerpt', __( 'Product short description', 'kkart' ), 'KKART_Meta_Box_Product_Short_Description::output', 'product', 'normal' );
		
		add_meta_box( 'kkart-product-images', __( 'Product gallery', 'kkart' ), 'KKART_Meta_Box_Product_Images::output', 'product', 'normal', 'high' );
		add_meta_box( 'kkart-product-data', __( 'Product data', 'kkart' ), 'KKART_Meta_Box_Product_Data::output', 'product', 'normal', 'high' );
		
		// Orders.
		foreach ( kkart_get_order_types( 'order-meta-boxes' ) as $type ) {
			$order_type_object = get_post_type_object( $type );
			/* Translators: %s order type name. */
			add_meta_box( 'kkart-order-data', sprintf( __( '%s data', 'kkart' ), $order_type_object->labels->singular_name ), 'KKART_Meta_Box_Order_Data::output', $type, 'normal', 'high' );
			add_meta_box( 'kkart-order-items', __( 'Items', 'kkart' ), 'KKART_Meta_Box_Order_Items::output', $type, 'normal', 'high' );
			/* Translators: %s order type name. */
			add_meta_box( 'kkart-order-notes', sprintf( __( '%s notes', 'kkart' ), $order_type_object->labels->singular_name ), 'KKART_Meta_Box_Order_Notes::output', $type, 'side', 'default' );
			add_meta_box( 'kkart-order-downloads', __( 'Downloadable product permissions', 'kkart' ) . kkart_help_tip( __( 'Note: Permissions for order items will automatically be granted when the order status changes to processing/completed.', 'kkart' ) ), 'KKART_Meta_Box_Order_Downloads::output', $type, 'normal', 'default' );
			/* Translators: %s order type name. */
			add_meta_box( 'kkart-order-actions', sprintf( __( '%s actions', 'kkart' ), $order_type_object->labels->singular_name ), 'KKART_Meta_Box_Order_Actions::output', $type, 'side', 'high' );
		}

		// Coupons.
		add_meta_box( 'kkart-coupon-data', __( 'Coupon data', 'kkart' ), 'KKART_Meta_Box_Coupon_Data::output', 'shop_coupon', 'normal', 'high' );

		// Comment rating.
		if ( 'comment' === $screen_id && isset( $_GET['c'] ) && metadata_exists( 'comment', kkart_clean( wp_unslash( $_GET['c'] ) ), 'rating' ) ) { // phpcs:ignore WordPress.Security.NonceVerification.Recommended
			add_meta_box( 'kkart-rating', __( 'Rating', 'kkart' ), 'KKART_Meta_Box_Product_Reviews::output', 'comment', 'normal', 'high' );
		}
	}

	/**
	 * Remove bloat.
	 */
	public function remove_meta_boxes() {
		remove_meta_box( 'postexcerpt', 'product', 'normal' );
		remove_meta_box( 'product_shipping_classdiv', 'product', 'side' );
		remove_meta_box( 'commentsdiv', 'product', 'normal' );
		remove_meta_box( 'commentstatusdiv', 'product', 'side' );
		remove_meta_box( 'commentstatusdiv', 'product', 'normal' );
		remove_meta_box( 'woothemes-settings', 'shop_coupon', 'normal' );
		remove_meta_box( 'commentstatusdiv', 'shop_coupon', 'normal' );
		remove_meta_box( 'slugdiv', 'shop_coupon', 'normal' );

		foreach ( kkart_get_order_types( 'order-meta-boxes' ) as $type ) {
			remove_meta_box( 'commentsdiv', $type, 'normal' );
			remove_meta_box( 'woothemes-settings', $type, 'normal' );
			remove_meta_box( 'commentstatusdiv', $type, 'normal' );
			remove_meta_box( 'slugdiv', $type, 'normal' );
			remove_meta_box( 'submitdiv', $type, 'side' );
		}
	}

	/**
	 * Rename core meta boxes.
	 */
	public function rename_meta_boxes() {
		global $post;

		// Comments/Reviews.
		if ( isset( $post ) && ( 'publish' === $post->post_status || 'private' === $post->post_status ) && post_type_supports( 'product', 'comments' ) ) {
			remove_meta_box( 'commentsdiv', 'product', 'normal' );
			add_meta_box( 'commentsdiv', __( 'Reviews', 'kkart' ), 'post_comment_meta_box', 'product', 'normal' );
		}
	}

	/**
	 * Check if we're saving, the trigger an action based on the post type.
	 *
	 * @param  int    $post_id Post ID.
	 * @param  object $post Post object.
	 */
	public function save_meta_boxes( $post_id, $post ) {
		$post_id = absint( $post_id );

		// $post_id and $post are required
		if ( empty( $post_id ) || empty( $post ) || self::$saved_meta_boxes ) {
			return;
		}

		// Dont' save meta boxes for revisions or autosaves.
		if ( Constants::is_true( 'DOING_AUTOSAVE' ) || is_int( wp_is_post_revision( $post ) ) || is_int( wp_is_post_autosave( $post ) ) ) {
			return;
		}

		// Check the nonce.
		if ( empty( $_POST['kkart_meta_nonce'] ) || ! wp_verify_nonce( wp_unslash( $_POST['kkart_meta_nonce'] ), 'kkart_save_data' ) ) { // phpcs:ignore WordPress.Security.ValidatedSanitizedInput.InputNotSanitized
			return;
		}

		// Check the post being saved == the $post_id to prevent triggering this call for other save_post events.
		if ( empty( $_POST['post_ID'] ) || absint( $_POST['post_ID'] ) !== $post_id ) {
			return;
		}

		// Check user has permission to edit.
		if ( ! current_user_can( 'edit_post', $post_id ) ) {
			return;
		}

		// We need this save event to run once to avoid potential endless loops. This would have been perfect:
		// remove_action( current_filter(), __METHOD__ );
		// But cannot be used due to https://github.com/kkart/kkart/issues/6485
		// When that is patched in core we can use the above.
		self::$saved_meta_boxes = true;

		// Check the post type.
		if ( in_array( $post->post_type, kkart_get_order_types( 'order-meta-boxes' ), true ) ) {
			do_action( 'kkart_process_shop_order_meta', $post_id, $post );
		} elseif ( in_array( $post->post_type, array( 'product', 'shop_coupon' ), true ) ) {
			do_action( 'kkart_process_' . $post->post_type . '_meta', $post_id, $post );
		}
	}
}

new KKART_Admin_Meta_Boxes();
