<?php
/**
 * Onboarding - set up automated tax calculation.
 *
 * @package kkart Admin
 */

namespace Automattic\Kkart\Admin\Features;

use Automattic\Kkart\Admin\Features\OnboardingTasks;
use Automattic\Kkart\Admin\Notes\ConfirmTaxSettings;

/**
 * This contains logic for setting up shipping when the profiler completes.
 */
class OnboardingAutomateTaxes {
	/**
	 * Constructor.
	 */
	public function __construct() {
		add_action(
			'kkart_onboarding_profile_completed',
			array(
				__CLASS__,
				'on_onboarding_profile_completed',
			)
		);

		add_action(
			'jetpack_authorize_ending_authorized',
			array(
				__CLASS__,
				'on_onboarding_profile_completed',
			)
		);
	}

	/**
	 * Set up automated taxes.
	 */
	public static function on_onboarding_profile_completed() {
		$jetpack_connected = null;
		$wcs_version       = null;
		$wcs_tos_accepted  = null;

		if ( class_exists( '\Jetpack_Data' ) ) {
			$user_token        = \Jetpack_Data::get_access_token( JETPACK_MASTER_USER );
			$jetpack_connected = isset( $user_token->external_user_id );
		}

		if ( class_exists( '\KKART_Connect_Loader' ) ) {
			$wcs_version = \KKART_Connect_Loader::get_wcs_version();
		}

		if ( class_exists( '\KKART_Connect_Options' ) ) {
			$wcs_tos_accepted = \KKART_Connect_Options::get_option( 'tos_accepted' );
		}

		if ( $jetpack_connected && $wcs_version && $wcs_tos_accepted && self::automated_tax_is_supported() ) {
			update_option( 'kkart_connect_taxes_enabled', 'yes' );
			update_option( 'kkart_calc_taxes', 'yes' );
			self::track_tax_automation();
			ConfirmTaxSettings::possibly_add_note();
		}
	}

	/**
	 * Check if automated taxes are supported.
	 */
	private static function automated_tax_is_supported() {
		return in_array( KKART()->countries->get_base_country(), OnboardingTasks::get_automated_tax_supported_countries(), true );
	}

	/**
	 * Track when a user has tax automation enabled.
	 */
	private static function track_tax_automation() {
		kkart_admin_record_tracks_event( 'tasklist_task_completed', array( 'task_name' => 'tax_automated' ) );
	}
}
