# Writers

A **_writer_** is an object that inherits from `Laminas\Log\Writer\AbstractWriter`.
A writer's responsibility is to record log data to a storage backend.

## Writing to Streams

`Laminas\Log\Writer\Stream` sends log data to a [PHP stream](http://www.php.net/stream).

To write log data to the PHP output buffer, use the `php://output` stream.
Alternately, you can send log data directly to a stream like `STDERR`
(`php://stderr`).

```php
$writer = new Laminas\Log\Writer\Stream('php://output');
$logger = new Laminas\Log\Logger();
$logger->addWriter($writer);

$logger->info('Informational message');
```

To write data to a file, use one of the [filesystem
streams](http://www.php.net/manual/en/wrappers.php#wrappers.file):

```php
$writer = new Laminas\Log\Writer\Stream('/path/to/logfile');
$logger = new Laminas\Log\Logger();
$logger->addWriter($writer);

$logger->info('Informational message');
```

By default, the stream opens in the append access mode ("a"). To open it with a
different access mode, the `Laminas\Log\Writer\Stream` constructor accepts an
optional second parameter for the stream mode.

The constructor of `Laminas\Log\Writer\Stream` also accepts an existing stream resource:

```php
$stream = @fopen('/path/to/logfile', 'a', false);
if (! $stream) {
    throw new Exception('Failed to open stream');
}

$writer = new Laminas\Log\Writer\Stream($stream);
$logger = new Laminas\Log\Logger();
$logger->addWriter($writer);

$logger->info('Informational message');
```

You cannot specify the stream access mode for existing stream resources. Doing
so causes a `Laminas\Log\Exception` to be thrown.

You can use an array of options in place of the stream argument when
constructing a `Stream` instance; when doing so, the `stream` key is required:

```php
$writer = new Laminas\Log\Writer\Stream([
    'stream' => 'php://output',
]);
$logger = new Laminas\Log\Logger();
$logger->addWriter($writer);

$logger->info('Informational message');
```

The following table details all allowed constructor arguments and their
corresponding configuration options.

Constructor Argument | Option Name | Default | Description
-------------------- | ----------- | ------- | -----------
`$streamOrUrl` | stream | None; required | Stream resource or URL to open and log to
`$mode` | mode | "a" | Stream access mode to use when opening a stream URL
`$logSeparator` | log\_separator | `PHP_EOL` | Separator string to use between entries
`$filePermissions` | chmod | `null` | Permissions mode to use for stream resource; defaults to existing file/stream permissions

## Writing to Databases

`Laminas\Log\Writer\Db` writes log information to a database table using
`Laminas\Db\Adapter\Adapter`. The constructor of `Laminas\Log\Writer\Db` receives a
`Laminas\Db\Adapter\Adapter` instance, a table name, an optional mapping of event
data to database columns, and an optional string contains the character
separator for the log array:

```php
$dbconfig = [
    // Sqlite Configuration
    'driver' => 'Pdo',
    'dsn' => 'sqlite:' . __DIR__ . '/tmp/sqlite.db',
];
$db = new Laminas\Db\Adapter\Adapter($dbconfig);

$writer = new Laminas\Log\Writer\Db($db, 'log_table_name');
$logger = new Laminas\Log\Logger();
$logger->addWriter($writer);

$logger->info('Informational message');
```

The example above writes a single row of log data to the database table named
`log_table_name` table. The database columns will be created according to the
event array generated by the `Laminas\Log\Logger` instance.

If we specify the mapping of the events with the database columns, the log will
store only to the selected fields in the database:

```php
$dbconfig = [
    // Sqlite Configuration
    'driver' => 'Pdo',
    'dsn' => 'sqlite:' . __DIR__ . '/tmp/sqlite.db',
];
$db = new Laminas\Db\Adapter\Adapter($dbconfig);

$mapping = [
    'timestamp' => 'date',
    'priority'  => 'type',
    'message'   => 'event',
];
$writer = new Laminas\Log\Writer\Db($db, 'log_table_name', $mapping);
$logger = new Laminas\Log\Logger();
$logger->addWriter($writer);

$logger->info('Informational message');
```

In this example, the writer stores only the log timestamp, priority, and
message, in the database fields date, type, and event, respectively.

`Laminas\Log\Writer\Db` has a fourth optional parameter in the constructor. This
parameter is the character separator for array fields in the log event. For
instance, if we have a log event that contains the field `extra`, and that field
is an array, its elements will be translated as `extra_field`, where '_' is the
character separator (default), and 'field' is the subname of the specific field
found in the `extra` array.

## Writing to FirePHP

`Laminas\Log\Writer\FirePHP` writes log information to the
[FirePHP](http://www.firephp.org/) Firefox extension. In order to use it, you
must install the FirePHPCore server library and the FirePHP browser extension.

To install the FirePHPCore library you can use
[Composer](https://getcomposer.org):

```bash
$ composer require firephp/firephp-core
```

## Writing to ChromePHP

`Laminas\Log\Writer\ChromePHP` sends log data to the
[ChromePHP](https://chrome.google.com/webstore/detail/chrome-logger/noaneddfkdjfnfdakjjmocngnfkfehhd)
Chrome extension.

To use the ChromePHP writer, you will also need to include the [ChromePHP
Library](http://craig.is/writing/chrome-logger) library in your application. Use
[Composer](https://getcomposer.org) to do this:

```bash
$ composer require ccampbell/chromephp
```

## Writing to Mail

`Laminas\Log\Writer\Mail` takes a configuration array or `Laminas\Mail\Message`. Basic usage looks like

```php
$message = new \Laminas\Mail\Message();
$message->setTo('email@example.com');

$writer = new \Laminas\Log\Writer\Mail($message);
$logger = new \Laminas\Log\Logger();
$logger->addWriter($writer);

// goes to mail message
$logger->info('Informational message');
```

An email of the logged information will be sent via `sendmail` by default. You may also provide a `Laminas\Mail\Transport`
during construction. For configuration options checkout the `Laminas\Mail\Transport` documentation.

```php
$writer = new Laminas\Log\Writer\Mail($mail, $transport);
```

`Laminas\Log\Writer\mail` may also be constructed with a configuration array. The configuration array accepts the following
keys:

```php
[
    'subject_prepend_text' => '',
    'transport' => $transport
    'mail' => $mail,
    'filters' => [],
    'formatter' => []
]
```

And expects the following data:

Array Index | Accepted Values | Description
----------- | --------------- | -----------
`subject_prepend_text` | string | Mail message
`transport` | `Transport\TransportInterface` | Transport method
`mail` | `Laminas\Mail\Message` | mail message
`mail` | `array` | `Laminas\Mail\Message` factory array
`filters` | array, int, string, Laminas\Log\Filter\FilterInterface | Log filter(s)
`formatter` | array, string, Laminas\Log\Formatter\FormatterInterface | Log formatter(s)

Basic usage of the configuration array looks like:

```php
$transport = new \Laminas\Mail\Transport\Smtp();
// TODO configure the SMTP transport

$message = new \Laminas\Mail\Message();
// TODO configure the Mail message

$writer = new \Laminas\Log\Writer\Mail([
    'subject_prepend_text' => 'Start of the subject',
    'transport' => $transport,
    'mail' => $mail,
    'filters' => [],
    'formatter' => []
]);

$logger = new \Laminas\Log\Logger();
$logger->addWriter($writer);

// goes to mail message
$logger->info('Informational message');
```

To use the `Laminas\Mail\Message` factory array construction will look like:

```php
$transport = new \Laminas\Mail\Transport\Smtp();
// TODO configure the SMTP transport

$writer = new \Laminas\Log\Writer\Mail([
    'subject_prepend_text' => 'Start of the subject',
    'transport' => $transport,
    'mail' => [
        'to' => 'email@example.com'
    ],
    'filters' => [],
    'formatter' => []
]);

$logger = new \Laminas\Log\Logger();
$logger->addWriter($writer);

// goes to mail message
$logger->info('Informational message');
```

## Writing to MongoDB

Using the legacy mongo php driver (may be used up to up to php 5.6)

In this example `Laminas\Log\Writer\MongoDB` uses an array for construction. Available keys include:

```php
[
    'save_options' => [],
    'collection'   => '',
    'database'     => '',
    'mongo'        => $mongo,
    'filters'      => [],
    'formatter'    => []
]
```

`collection`, `database`, and `mongo` are required. Each key accepts:

Array Index    | Accepted Values                                        | Description
-------------- | ------------------------------------------------------ | -----------
`save_options` | array                                                  | MongoDB driver options
`collection`   | string                                                 | collection name
`database`     | string                                                 | database name
`mongo`        | `Mongo` or `MongoClient`                               | MongoDB connection object
`filters`      | array, int, string, `Laminas\Log\Filter\FilterInterface`  | Log filter(s)
`formatter`    | array, string, `Laminas\Log\Formatter\FormatterInterface` | Log formatter(s)

And `Laminas\Log\Writer\MongoDB` is used like this:

```php
$mongo = new MongoClient();

$writer = new \Laminas\Log\Writer\MongoDB([
    'save_options' => [], // MongoDB Driver Options
    'collection'   => 'collectionName',
    'database'     => 'databaseName',
    'mongo'        => $mongo,
    'filters'      => [],
    'formatter'    => []
]);

$logger = new \Laminas\Log\Logger();
$logger->addWriter($writer);

// goes to Mongo DB
$logger->info('Informational message');
```

It may also be constructed by passing the arguments directly

```php
$mongo       = new MongoClient();
$database    = 'databaseName';
$collection  = 'collectionName';
$saveOptions = [];

$writer = new \Laminas\Log\Writer\MongoDB($mongo, $database, $collection, $saveOptions);
$logger = new \Laminas\Log\Logger();
$logger->addWriter($writer);

// goes to Mongo DB
$logger->info('Informational message');
```

Using the mongodb driver (php 5.4 and later)

In this example `Laminas\Log\Writer\MongoDB` uses an array for construction. Available keys include:

```php
[
    'save_options' => [],
    'collection'   => '',
    'database'     => '',
    'manager'      => $mongo,
    'filters'      => [],
    'formatter'    => []
]
```

`collection`, `database`, and `manager` are required. Each key accepts:

Array Index    | Accepted Values                                        | Description
---------------| ------------------------------------------------------ | -----------
`save_options` | array                                                  | MongoDB driver options
`collection`   | string                                                 | collection name
`database`     | string                                                 | database name
`manager`      | `MongoDB\Driver\Manager`                               | MongoDB connection object
`filters`      | array, int, string, `Laminas\Log\Filter\FilterInterface`  | Log filter(s)
`formatter`    | array, string, `Laminas\Log\Formatter\FormatterInterface` | Log formatter(s)

And `Laminas\Log\Writer\MongoDB` is used like this:

```php
$mongo = new MongoDB\Driver\Manager();

$writer = new \Laminas\Log\Writer\MongoDB([
    'save_options' => [], // MongoDB Driver Options
    'collection'   => 'collectionName',
    'database'     => 'databaseName',
    'manager'      => $mongo,
    'filters'      => [],
    'formatter'    => []
]);

$logger = new \Laminas\Log\Logger();
$logger->addWriter($writer);

// goes to Mongo DB
$logger->info('Informational message');
```

It may also be constructed by passing the arguments directly

```php
$mongo       = new MongoDB\Driver\Manager();
$database    = 'databaseName';
$collection  = 'collectionName';
$saveOptions = [];

$writer = new \Laminas\Log\Writer\MongoDB($manager, $database, $collection, $saveOptions);
$logger = new \Laminas\Log\Logger();
$logger->addWriter($writer);

// goes to Mongo DB
$logger->info('Informational message');
```

## Writing to Syslog

`Laminas\Log\Writer\Syslog` is a writer generates system log messages from the data it receives.

```php
$writer = new \Laminas\Log\Writer\Syslog();
$logger = new \Laminas\Log\Logger();
$logger->addWriter($writer);

// goes to system log
$logger->info('Informational message');
```

The application name and syslog facility may be set

Array Index | Accepted Values | Description
----------- | --------------- | -----------
`application` | string | application name
`facility` | string | syslog facility [list of facilities](http://php.net/openlog)
`filters` | array, int, string, Laminas\Log\Filter\FilterInterface | Log filter(s)
`formatter` | array, string, Laminas\Log\Formatter\FormatterInterface | Log formatter(s)

```php
$writer = new \Laminas\Log\Writer\Syslog([
    'application' => '',
    'facility' => '',
    'filters' => [],
    'formatter' => []
]);

$logger = new \Laminas\Log\Logger();
$logger->addWriter($writer);

$logger->info('Informational message');
```

## Writing to Zend Monitor

`Laminas\Log\Writer\ZendMonitor` writes log data to the Zend Monitor on a Zend Server. If the web server is not a Laminas
Server or Zend Monitor is not enabled it will fail silently.

```php
$writer = new \Laminas\Log\Writer\ZendMonitor();
$logger = new \Laminas\Log\Logger();
$logger->addWriter($writer);

// goes to Zend Monitor
$logger->info('Informational message');
```

`Laminas\Log\Writer\AbstractWriter` options are available.

Array Index | Accepted Values | Description
----------- | --------------- | -----------
`filters` | array, int, string, Laminas\Log\Filter\FilterInterface | Log filter(s)
`formatter` | array, string, Laminas\Log\Formatter\FormatterInterface | Log formatter(s)

```php
$writer = new \Laminas\Log\Writer\ZendMonitor([
    'filters' => [],
    'formatter' => []
]);

$logger = new \Laminas\Log\Logger();
$logger->addWriter($writer);

// goes to Zend Monitor
$logger->info('Informational message');
```

## Stubbing Out the Writer

`Laminas\Log\Writer\Noop` is a stub that does not write log data to anything; it is
useful for disabling logging or stubbing out logging during tests:

```php
$writer = new Laminas\Log\Writer\Noop;
$logger = new Laminas\Log\Logger();
$logger->addWriter($writer);

// goes nowhere
$logger->info('Informational message');
```

### Migration from 2.0-2.3 to 2.4+

Version 2.4 adds support for PHP 7. In PHP 7, `null` is a reserved keyword, which required renaming
the `Null` log writer. If you were using the `Null` writer directly previously, you will now receive
an `E_USER_DEPRECATED` notice on instantiation. Please update your code to refer to the `Noop` class
instead.

Users pulling their `Null` writer instance from the writer plugin manager receive a `Noop` instance
instead starting in 2.4.0.

## Testing with the Mock Writer

`Laminas\Log\Writer\Mock` is a simple writer that records the raw data it receives
in an array that it exposes as a public property.

```php
$mock = new Laminas\Log\Writer\Mock;
$logger = new Laminas\Log\Logger();
$logger->addWriter($mock);

$logger->info('Informational message');

var_dump($mock->events[0]);

// Array
// (
//    [timestamp] => 2007-04-06T07:16:37-07:00
//    [message] => Informational message
//    [priority] => 6
//    [priorityName] => INFO
// )
```

To clear the events logged by the mock, reset the `$events` property:

```php
$mock->events = [];
```

## Compositing Writers

There is no composite writer object. However, a `Logger` instance can write to
any number of writers, effectively making it a composite logger.

To utilize this functionality, add writers via the `Logger::addWriter()` method:

```php
$writer1 = new Laminas\Log\Writer\Stream('/path/to/first/logfile');
$writer2 = new Laminas\Log\Writer\Stream('/path/to/second/logfile');

$logger = new Laminas\Log\Logger();
$logger->addWriter($writer1);
$logger->addWriter($writer2);

// goes to both writers
$logger->info('Informational message');
```

You can also specify the priority number for each writer to change the order of writing. The
priority number is an integer number passed as second parameter in the
`addWriter()` method. Internally, `SplPriorityQueue` is used to manage writers,
which means that:

- higher integer values indicate higher priority (triggered earliest);
- lower integer values (including negative values) have lower priority
(triggered last).

## Distinction between priority in the queue of writers and the filter `Priority`

When you add a writer to the logger, you can set its priority as a second
argument. This priority is the priority in the queue of all writers of the
logger. It can be any integer, and the default is `1` (`\Laminas\Log\Logger::ALERT`)
for the writers; the bigger the number, the higher the priority. A writer with a
lower priority will be triggered later, so `ALERT` is triggered after `DEBUG`.
For details on the list of the priorities, see the section entitled [Using Built-in Priorities](intro.md#using-built-in-priorities).

This priority should not be confused with the [`Priority` filter](filters.md#available-filters),
which determines if a message meets a _severity_ threshold. When a `Priority`
filter is attached, the writer will output the message only when the filter is
lower or equal (by default) to the priority (the biggest severity is 0).
