<?php
/**
 * Simple class to integrate with Cisco Webex XML API.
 * Built on version 5.9 of Reference Guide.
 * Based on Webex class by Joshua McGinnis (http://joshuamcginnis.com/2010/07/12/webex-api-php-sdk).
 *
 * @author Joubert Guimarães de Assis "RedRat" <joubert@redrat.com.br>
 * @copyright Copyright (c) 2013, RedRat Consultoria
 * @license GPL version 2
 * @see Github, animes and mangás, cute girls and PHP, much PHP
 * @link http://developer.cisco.com/documents/4733862/4736722/xml_api_5+9.pdf
 * @link https://github.com/joubertredrat/phpebex-php-webex
 */

class Webex {
	
	private $username;
	private $password;
	private $siteID;
	private $partnerID;

	private $url_prefix;
	private $url_host;

	private $send_mode;

	private $data;

	private $action;

	const SEND_CURL 			= 'curl';
	const SEND_FSOCKS 			= 'fsocks';
	const PREFIX_HTTP 			= 'http';
	const PREFIX_HTTPS 			= 'https';
	const SUFIX_XML_API 		= 'WBXService/XMLService';
	const WEBEX_DOMAIN 			= 'webex.com';
	const XML_VERSION			= '1.0';
	const XML_ENCODING			= 'UTF-8';
	const USER_AGENT			= '';

	const API_SCHEMA_MEETING	= 'http://www.webex.com/schemas/2002/06/service/meeting';
	const API_SCHEMA_SERVICE	= 'http://www.webex.com/schemas/2002/06/service';

	const DATA_SENDER				= 'sender';
	const DATA_SENDER_POST_HEADER 	= 'post_header';
	const DATA_SENDER_POST_BODY 	= 'post_body';
	const DATA_SENDER_XML 			= 'xml';
	const DATA_RESPONSE				= 'response';
	const DATA_RESPONSE_XML			= 'xml';
	const DATA_RESPONSE_DATA		= 'data';

	/**
	 * Constructor of class.
	 *
	 * @return void
	 */
	public function __construct() {
		$this->action = 0;
		$this->response = array();
		$this->send_mode = in_array(self::SEND_CURL, get_loaded_extensions()) ? self::SEND_CURL : self::SEND_FSOCKS;
	}

	/**
	 * Get a possible modes to send a POST data.
	 *
	 * @return array Returns a list of send modes.
	 */
	public static function get_sendmode() {
		return array(self::SEND_CURL, self::SEND_FSOCKS);
	}

	/**
	 * Validates a customer webex domain.
	 *
	 * @param string $url Url to validate.
	 * @return bool Return true if a valid url or false if not.
	 */
	public static function validate_url($url) {
		$regex = "/^(http|https):\/\/(([A-Z0-9][A-Z0-9_-]*)+.(" . self::WEBEX_DOMAIN . ")$)/i";
		return (bool) preg_match($regex, $url);
	}

	/**
	 * Get port used by API.
	 *
	 * @param string $prefix Prefix to get a port.
	 * @return int Return a port.
	 */
	public static function get_port($prefix) {
		switch($prefix) {
			case self::PREFIX_HTTP:
				return 80;
			break;
			case self::PREFIX_HTTPS:
				return 443;
			break;
			default:
				exit(__CLASS__ . ' error report: Wrong prefix');
			break;
		}
	}

	/**
	 * Mode to send data.
	 *
	 * @param string mode to send.
	 * @return void
	 */
	public function set_sendmode($mode) {
		if(!in_array($mode, self::get_sendmode()))
			exit(__CLASS__ . ' error report: Wrong send mode');
		$this->send_mode = $mode;
	}

	/**
	 * Auth data to integrate with API.
	 *
	 * @param string $username Username to auth.
	 * @param string $password Password to auth.
	 * @param string $siteID Customer site id.
	 * @param string $partnerID Customer partnerID id.
	 * @return void
	 */
	public function set_auth($username, $password, $siteID, $partnerID, $url) {
		$this->username 	= $username;
		$this->password 	= $password;
		$this->siteID 		= $siteID;
		$this->partnerID 	= $partnerID;
		if(!self::validate_url($url))
			exit(__CLASS__ . ' error report: Wrong webex url');
		list($this->url_prefix, $this->url_host) = preg_split("$://$", $url);
	}

	/**
	 * Generates a XML to send a data to API.
	 *
	 * @param array $data Data to insert in XML in format:
	 *		$data['service']	= 'meeting';
	 *		$data['xml_header'] = '<item><subitem>data</subitem></item>';
	 *		$data['xml_body']	= '<item><subitem>data</subitem></item>';
	 * @return string Returns a XML generated.
	 */
	private function get_xml($data) {
		$xml 	= array();
		$xml[]	= '<?xml version="' . self::XML_VERSION . '" encoding="' . self::XML_ENCODING . '"?>';
		$xml[]	= 	'<serv:message xmlns:xsi="http://www.w3.org/2001/XMLSchema-instance">';
		$xml[]	= 		'<header>';
		$xml[]	= 			'<securityContext>';
		$xml[]	= 				'<webExID>' . $this->username . '</webExID>';
		$xml[]	= 				'<password>' . $this->password . '</password>';
		$xml[]	= 				'<siteName>' . $this->siteID . '</siteName>';
		$xml[]	= 				'<partnerID>' . $this->partnerID . '</partnerID>';
		if(isset($data['xml_header']))
			$xml[]	= 			$data['xml_header'];
		$xml[]	= 			'</securityContext>';
		$xml[]	= 		'</header>';
		$xml[]	= 		'<body>';
		$xml[]	= 			'<bodyContent xsi:type="java:com.webex.service.binding.' . $data['service'] . '">';
		$xml[]	= 				$data['xml_body'];
		$xml[]	= 			'</bodyContent>';				
		$xml[]	= 		'</body>';
		$xml[]	= 	'</serv:message>'; 
		return implode('', $xml);
	}

	/**
	 * Test if have a auth data to use a API.
	 *
	 * @return bool Returns true if have data and false if not.
	 */
	public function has_auth() {
		return (bool) $this->username && $this->password && $this->siteID && $this->partnerID;
	}

	/**
	 * Generates a header and a body to send data to API.
	 *
	 * @return string Returns a response from API.
	 */
	private function send($xml) {
    	$post_data['UID'] = $this->username;
    	$post_data['PWD'] = $this->password;
    	$post_data['SID'] = $this->siteID;
		$post_data['PID'] = $this->partnerID;
		$post_data['XML'] = $xml;

		// Really I dont know why xml api give a error on http_build_query :(
		$post_string = '';
		foreach ($post_data as $variable => $value) {
			$post_string .= '' . $variable . '=' . urlencode($value) . '&';
		}

		$post_header 		= array();
		$post_header[] 		= 'POST /' . self::SUFIX_XML_API . ' HTTP/1.0';
		$post_header[] 		= 'Host: ' . $this->url_host;
		$post_header[] 		= 'User-Agent: ' . self::USER_AGENT;
		if($this->send_mode == self::SEND_FSOCKS) {
			$post_header[] 	= 'Content-Type: application/xml';
			$post_header[] 	= 'Content-Length: ' . strlen($xml);
		}
		$data = array();
		$data['post_header'] = $post_header;
	    $data['post_string'] = $post_string; 
		$this->data[$this->action][self::DATA_SENDER][self::DATA_SENDER_POST_HEADER] 	= $post_header;
		$this->data[$this->action][self::DATA_SENDER][self::DATA_SENDER_POST_BODY] 		= $post_header;
		$this->data[$this->action][self::DATA_SENDER][self::DATA_SENDER_XML] 			= $xml; 
		return $this->{'send_' . $this->send_mode}($data);
	}

	/**
	 * Send a data to Webex API using PHP curl.
	 *
	 * @param array $data Data to send to API in format:
	 * 		$data['post_header'] = "blablabla";
	 * 		$data['post_header'] = "post_string";
	 * @return string Returns a response from API.
	 */
	private function send_curl($data) { 
		extract($data);  
		$ch = curl_init();
		curl_setopt($ch, CURLOPT_URL, $this->url_prefix . '://' . $this->url_host . '/' . self::SUFIX_XML_API);
		curl_setopt($ch, CURLOPT_PORT, self::get_port($this->url_prefix));
		curl_setopt($ch, CURLOPT_SSL_VERIFYPEER, false);
		curl_setopt($ch, CURLOPT_RETURNTRANSFER, true);
		curl_setopt($ch, CURLOPT_HTTPHEADER, $post_header);
		curl_setopt($ch, CURLOPT_POSTFIELDS, $post_string);
		curl_setopt($ch, CURLOPT_RETURNTRANSFER, 1);
		$response = curl_exec($ch); 
		if($response === false)
			exit(__CLASS__ . ' error report: Curl error - ' . curl_error($ch));
		curl_close($ch);
		return $response;
	}

	/**
	 * Send a data to Webex API using PHP file functions.
	 *
	 * @param array $data Data to send to API in format:
	 * 		$data['post_header'] = "blablabla";
	 * 		$data['post_header'] = "post_string";
	 * @return string Returns a response from API.
	 * @todo haha, I need to test this :)
	 */
	private function send_fsocks($data) {
		extract($data);
		$post_data = implode("\n", $post_header) . "\n\n" . $post_string . "\n";
		$fp = fsockopen($this->url_host, self::get_port($this->url_prefix), $errno, $error);
		if($fp) {
			fwrite($fp, $post_data);
			$response = '';
			while(!feof($fp)) {
				$response .= fgets($fp, 1024);
			}
			return $response;
		}
		else
			exit(__CLASS__ . ' error report: Fsocks error - (' . $errno . ') ' . $error);
	}

	/**
	 * Get response from a API.
	 *
	 * @param string $type Type of data to be requested.
	 * @param int $number number of sender to be requested.
	 * @return string|object Return a response.
	 */
	public function get_response($type = self::DATA_RESPONSE_DATA, $number = null) {
		if(isset($number) && is_int($number)) {
			if($number < 1 || $number > ($this->action - 1))
				exit(__CLASS__ . ' error report: Invalid response number');
			$number--;
		}
		else
			$number = ($this->action - 1);
		var_dump($number);
		switch($type) {
			case self::DATA_RESPONSE_XML:
			case self::DATA_RESPONSE_DATA:
				return $this->data[$number][self::DATA_RESPONSE][$type];
			break;
			default:
				exit(__CLASS__ . ' error report: I don\'t undestood that data you needs');
			break;
		}
	}

	/**
	 * @todo documentate in future.
	 */
	public function meeting_LstsummaryMeeting($maximumNum = 100) {
		if(!$this->has_auth())
			exit(__CLASS__ . ' error report: Auth data not found');
		$xml_body 	= array();
		$xml_body[] = "<listControl>"; 
		$xml_body[] = 		'<maximumNum>' . $maximumNum . '</maximumNum>';
		$xml_body[] = '</listControl>';
		$xml_body[] = '<order>';
		$xml_body[] = 	'<orderBy>STARTTIME</orderBy>';
		$xml_body[] = 	'<orderAD>DESC</orderAD>';
		$xml_body[] = '</order>';
		$xml_body[] = '<dateScope>';
		$xml_body[] = '</dateScope>'; 
		$data['xml_body'] 	= implode('', $xml_body); 
		$data['service'] 	= str_replace("_", ".", __FUNCTION__);
		$xml 				= $this->get_xml($data);
		$response 			= $this->send($xml); 
	
		$resposeXMLString = simplexml_load_string($response);

    // $xmlNode = simplexml_load_file('https://vision-1f7f-50c9.my.webex.com/WBXService/XMLService');
    //echo "<pre>"; print_r($this->xmlToArray($resposeXMLString));die();
    //echo "<pre>"; print_r( $this->xmlToArray($resposeXMLString)['message']['serv:body']['serv:bodyContent']['meet:meeting']);
		 
	//	$xml 						= simplexml_load_string($response);
	//	$Data 						= new stdClass;
	//	$Data->header 				= new stdClass;
	//	$Data->header->response 	= new stdClass;
	//	$Data->bodyContent 			= array(); 
	//	$node 					    = $xml->children(self::API_SCHEMA_SERVICE);
		//$asd = $xml->bodyContent;
	 
	//	echo "<pre>"; print_r($node);
	//	$Data->header->response->result 	= (string) $node[0]->response->result;
	//	$Data->header->response->gsbStatus 	= (string) $node[0]->response->gsbStatus;
        //echo "<pre>";print_r($response);"</pre>";
       // echo "<pre>";print_r($node);"</pre>";
       
       #####
      $newmetting = array();
      $newmetting = $this->xmlToArray($resposeXMLString)['message']['serv:body']['serv:bodyContent']['meet:meeting'];
       
       ##### 
		if(is_array($newmetting) && !empty($newmetting)){
			foreach($newmetting as $meeting) { 
			//	echo "<pre>";print_r($meeting);"</pre>"; echo "*--------------------------------*";
				if($meeting['meet:meetingKey']) {
					$MeetingData 						= new stdClass;
					$MeetingData->meetingKey 			= (string) $meeting['meet:meetingKey'];
					$MeetingData->confName 				= (string) $meeting['meet:confName'];
					$MeetingData->meetingType 			= (string) $meeting['meet:meetingType'];
					$MeetingData->hostWebExID 			= (string) $meeting['meet:hostWebExID'];
					$MeetingData->otherHostWebExID 		= (string) $meeting['meet:otherHostWebExID'];
					$MeetingData->timeZoneID 			= (string) $meeting['meet:timeZoneID'];
					$MeetingData->timeZone 				= (string) $meeting['meet:timeZone'];
					$MeetingData->status 				= (string) $meeting['meet:status'];
					$MeetingData->startDate 			= (string) $meeting['meet:startDate'];
					$MeetingData->duration 				= (string) $meeting['meet:duration'];
					$MeetingData->listStatus 			= (string) $meeting['meet:listStatus'];
					$MeetingData->hostJoined 			= (string) $meeting['meet:hostJoined'];
					$MeetingData->participantsJoined 	= (string) $meeting['meet:participantsJoined'];
					$MeetingData->telePresence 			= (string) $meeting['meet:telePresence'];
					$Data->bodyContent[] 				= $MeetingData;
				}
			}
		}
		
		$this->data[$this->action][self::DATA_RESPONSE][self::DATA_RESPONSE_DATA] 	= $Data;
		$this->data[$this->action][self::DATA_RESPONSE][self::DATA_RESPONSE_XML] 	= $response;
		$this->action++;
		//print_r($Data); die;
		return $Data;
	} 
	
	function create_meeting($meeting_password='', $title='', $duration='20', $start_date=''){
		$xml = '';
		$xml = '<serv:message xmlns:xsi="http://www.w3.org/2001/XMLSchema-instance">';
		$xml .= '<header>';
		$xml .= '<securityContext>';
		$xml .= 	'<webExID>' . $this->username . '</webExID>';
		$xml .= 	'<password>' . $this->password . '</password>';
		$xml .= 	'<siteName>' . $this->siteID . '</siteName>';
		$xml .= 	'<partnerID>' . $this->partnerID . '</partnerID>';
		$xml .= '</securityContext>';
		$xml .= '</header>';
		$xml .= '<body>';
		$xml .= '<bodyContent xsi:type="java:com.webex.service.binding.meeting.CreateMeeting">';
        
        if($meeting_password!=''){
			$xml .= '<accessControl>';
				$xml .= '<meetingPassword>'.$meeting_password.'</meetingPassword>';
			$xml .= '</accessControl>';
		}
		
        $xml .= '<metaData>';
            $xml .= '<confName>'.$title.'</confName>';
            //$xml .= '<meetingType>105</meetingType>';
        $xml .= '</metaData>';

        $xml .= '<enableOptions>';
            $xml .= '<multiVideo>True</multiVideo>';
            $xml .= '<audioVideo>True</audioVideo>';
            $xml .= '<supportUCFRichMedia>true</supportUCFRichMedia>';
            $xml .= '<chatAllAttendees>true</chatAllAttendees>';
            $xml .= '<chatHost>true</chatHost>';
            $xml .= '<chatPresenter>true</chatPresenter>';
            $xml .= '<attendeeList>true</attendeeList>';
            $xml .= '<voip>true</voip>';
        $xml .= '</enableOptions>';
       $xml .='<telephony>';
           $xml .='<telephonySupport>OTHER</telephonySupport>';
           $xml .='<extTelephonyDescription>!#TollFreeCallInCountryCode:1_1
					!#TollFreeCallInTelephonyCountryCode:+1
					!#TollFreeCallInNumber:8003477829
					!#ParticipantCode:194187</extTelephonyDescription>';
        $xml .='</telephony>';
        /*$xml .='<participants>';
            $xml .='<maxUserNumber>6</maxUserNumber>';
            //$xml .='<maxUserNumber>'.$userNumbers.'</maxUserNumber>';
            $xml .='<attendees>';
                $xml .='<attendee>';   
                    $xml .='<person>';
                        $xml .='<name>'.$host_name.'</name>';
                        $xml .='<email>'.$host_email.'</email>';
                    $xml .='</person>';
                    $xml .='<role>HOST</role>';
                $xml .='</attendee>';
                $xml .='<attendee>';   
                    $xml .='<person>';
                        $xml .='<name>'.$speaker_name.'</name>';
                        $xml .='<email>'.$speaker_email.'</email>';
                    $xml .='</person>';
                    $xml .='<role>ATTENDEE</role>';
                $xml .='</attendee>';
            $xml .='</attendees>';
        $xml .='</participants>';*/

        $xml .= '<schedule>';
            $xml .= '<startDate>'.$start_date.'</startDate>';
            $xml .= '<duration>'.$duration.'</duration>';
           // if($time_zome!='')$xml .= '<timeZoneID>20</timeZoneID>';
        $xml .= '</schedule>';
		$xml .= '</bodyContent>';
		$xml .= '</body>';
		$xml .= '</serv:message>';
		$response 	= $this->send($xml);
		return $response;
	}
	
	function delete_meeting($meetingID){
		$xml = '';
		$xml = '<serv:message xmlns:xsi="http://www.w3.org/2001/XMLSchema-instance">';
		$xml .= '<header>';
		$xml .= '<securityContext>';
		$xml .= 	'<webExID>' . $this->username . '</webExID>';
		$xml .= 	'<password>' . $this->password . '</password>';
		$xml .= 	'<siteName>' . $this->siteID . '</siteName>';
		$xml .= 	'<partnerID>' . $this->partnerID . '</partnerID>';
		$xml .= '</securityContext>';
		$xml .= '</header>';
		
		$xml .= '<body>';
		$xml .= '<bodyContent xsi:type="java:com.webex.service.binding.meeting.DelMeeting">';
		$xml .= '<meetingKey>'.$meetingID.'</meetingKey>';
		$xml .= '</bodyContent>';
		$xml .= '</body>';
		$xml .= '</serv:message>';
		$response 	= $this->send($xml);
		return $response;
	}
	
	function get_meeting($meetingID){
		$xml = '';
		$xml = '<serv:message xmlns:xsi="http://www.w3.org/2001/XMLSchema-instance">';
		$xml .= '<header>';
		$xml .= '<securityContext>';
		$xml .= 	'<webExID>' . $this->username . '</webExID>';
		$xml .= 	'<password>' . $this->password . '</password>';
		$xml .= 	'<siteName>' . $this->siteID . '</siteName>';
		$xml .= 	'<partnerID>' . $this->partnerID . '</partnerID>';
		$xml .= '</securityContext>';
		$xml .= '</header>';
		
		$xml .= '<body>';
		$xml .= '<bodyContent  xsi:type="java:com.webex.service.binding.meeting.GetMeeting">';
		$xml .= '<meetingKey>'.$meetingID.'</meetingKey>';
		$xml .= '</bodyContent>';
		$xml .= '</body>';
		$xml .= '</serv:message>';
		$response 	= $this->send($xml);
		//echo "<pre>"; print_r($response);
		$resposeXMLString = simplexml_load_string($response);
        $newmetting = array();
        $newmetting = $this->xmlToArray($resposeXMLString)['message']['serv:body']['serv:bodyContent'];
       // echo "<pre>"; print_r($newmetting);
        if(is_array($newmetting) && !empty($newmetting)){
		//	foreach($newmetting as $meeting) {
				//echo "<pre>";print_r($meeting);echo "*--------------------------------*";
				if($newmetting['meet:accessControl']['meet:meetingPassword']) {
					$MeetingData['meetingPassword'] = (string)  $newmetting['meet:accessControl']['meet:meetingPassword'];
				}
				if($newmetting['meet:metaData']['meet:confName']) {
					$MeetingData['confName'] = (string) $newmetting['meet:metaData']['meet:confName'];
				}
				if($newmetting['meet:participants']['meet:maxUserNumber']) {
					$MeetingData['maxUserNumber'] = (string) $newmetting['meet:participants']['meet:maxUserNumber'];
				}
				if($newmetting['meet:schedule']['meet:startDate']) {
					$MeetingData['startDate'] = (string) $newmetting['meet:schedule']['meet:startDate'];
				}
				if($newmetting['meet:schedule']['meet:timeZone']) {
					$MeetingData['timeZone'] = (string) $newmetting['meet:schedule']['meet:timeZone'];
				}
				if($newmetting['meet:schedule']['meet:duration']) {
					$MeetingData['duration'] = (string) $newmetting['meet:schedule']['meet:duration'];
				}
				if($newmetting['meet:schedule']['meet:hostWebExID']) {
					$MeetingData['hostWebExID'] = (string) $newmetting['meet:schedule']['meet:hostWebExID'];
				}
				if($newmetting['meet:telephony']['meet:telephonySupport']) {
					$MeetingData['telephonySupport'] = (string) $newmetting['meet:telephony']['meet:telephonySupport'];
				}
				if($newmetting['meet:schedule']['meet:timeZoneID']) {
					$MeetingData['timeZoneID'] = (string) $newmetting['meet:schedule']['meet:timeZoneID'];
				}
				if($newmetting['meet:metaData']['meet:meetingType']) {
					$MeetingData['meetingType'] = (string) $newmetting['meet:metaData']['meet:meetingType'];
				}
		//	}
			//print_r($MeetingData);
			return $MeetingData;
		}else{
			return false;
		} 
	/*	$xml 						= simplexml_load_string($response);
		$Data 						= new stdClass;
		$Data->header 				= new stdClass;
		$Data->header->response 	= new stdClass;
		$Data->bodyContent 			= array();

		$node 								= $xml->children(self::API_SCHEMA_SERVICE);
		$Data->header->response->result 	= (string) $node[0]->response->result;
		$Data->header->response->gsbStatus 	= (string) $node[0]->response->gsbStatus;

		$node_meeting = $node[1]->bodyContent; 
		
		if(is_array($node_meeting) && !empty($node_meeting)){
			foreach($node_meeting->children(self::API_SCHEMA_MEETING) as $meeting) {
				//echo "<pre>";print_r($meeting);echo "*--------------------------------*";
				if($meeting->meetingPassword) {
					$MeetingData['meetingPassword'] = (string) $meeting->meetingPassword;
				}
				if($meeting->confName) {
					$MeetingData['confName'] = (string) $meeting->confName;
				}
				if($meeting->maxUserNumber) {
					$MeetingData['maxUserNumber'] = (string) $meeting->maxUserNumber;
				}
				if($meeting->startDate) {
					$MeetingData['startDate'] = (string) $meeting->startDate;
				}
				if($meeting->timeZone) {
					$MeetingData['timeZone'] = (string) $meeting->timeZone;
				}
				if($meeting->duration) {
					$MeetingData['duration'] = (string) $meeting->duration;
				}
				if($meeting->hostWebExID) {
					$MeetingData['hostWebExID'] = (string) $meeting->hostWebExID;
				}
				if($meeting->telephonySupport) {
					$MeetingData['telephonySupport'] = (string) $meeting->telephonySupport;
				}
				if($meeting->timeZoneID) {
					$MeetingData['timeZoneID'] = (string) $meeting->timeZoneID;
				}
				if($meeting->meetingType) {
					$MeetingData['meetingType'] = (string) $meeting->meetingType;
				}
			}
			//print_r($response);
			return $MeetingData;
		}else{
			return false;
		}*/
	}
	
	function xmlToArray($xml, $options = array()) { 
        $defaults = array( 
          'namespaceRecursive' => false,  //setting to true will get xml doc namespaces recursively 
          'removeNamespace' => false,     //set to true if you want to remove the namespace from resulting keys (recommend setting namespaceSeparator = '' when this is set to true) 
          'namespaceSeparator' => ':',    //you may want this to be something other than a colon
          'attributePrefix' => '@',       //to distinguish between attributes and nodes with the same name
          'alwaysArray' => array(),       //array of xml tag names which should always become arrays
          'autoArray' => true,            //only create arrays for tags which appear more than once
          'textContent' => '$',           //key used for the text content of elements
          'autoText' => true,             //skip textContent key if node has no attributes or child nodes
          'keySearch' => false,           //optional search and replace on tag and attribute names
          'keyReplace' => false           //replace values for above search values (as passed to str_replace())
        );

        $options = array_merge($defaults, $options);
        $namespaces = $xml->getDocNamespaces($options['namespaceRecursive']);
        $namespaces[''] = null; //add base (empty) namespace
    
       //get attributes from all namespaces
       $attributesArray = array();
       foreach ($namespaces as $prefix => $namespace) {
            if ($options['removeNamespace']) $prefix = "";
            foreach ($xml->attributes($namespace) as $attributeName => $attribute) {
                //replace characters in attribute name
                if ($options['keySearch']) $attributeName =
                    str_replace($options['keySearch'], $options['keyReplace'], $attributeName);
                    $attributeKey = $options['attributePrefix']
                    . ($prefix ? $prefix . $options['namespaceSeparator'] : '')
                    . $attributeName;
                    $attributesArray[$attributeKey] = (string)$attribute;
            }
        }
 
        //get child nodes from all namespaces
        $tagsArray = array();
        foreach ($namespaces as $prefix => $namespace) { 
            if ($options['removeNamespace']) $prefix = ""; 
            foreach ($xml->children($namespace) as $childXml) { 
                //recurse into child nodes 
                $childArray = $this->xmlToArray($childXml, $options); 
                list($childTagName, $childProperties) = each($childArray);  
                //replace characters in tag name 
                if ($options['keySearch']) $childTagName = str_replace($options['keySearch'], $options['keyReplace'], $childTagName);
    
                //add namespace prefix, if any 
                if ($prefix) $childTagName = $prefix . $options['namespaceSeparator'] . $childTagName;
      
                if (!isset($tagsArray[$childTagName])) { 
                    //only entry with this key 
                    //test if tags of this type should always be arrays, no matter the element count 
                    $tagsArray[$childTagName] = 
                            in_array($childTagName, $options['alwaysArray']) || !$options['autoArray'] 
                            ? array($childProperties) : $childProperties;
                } elseif (
                   is_array($tagsArray[$childTagName]) && array_keys($tagsArray[$childTagName])
                    === range(0, count($tagsArray[$childTagName]) - 1)
                ) {
                 //key already exists and is integer indexed array
                 $tagsArray[$childTagName][] = $childProperties;
               } else {
                   //key exists so convert to integer indexed array with previous value in position 0
                     $tagsArray[$childTagName] = array($tagsArray[$childTagName], $childProperties);
               }
             }
        }
 
     //get text content of node 
        $textContentArray = array(); 
        $plainText = trim((string)$xml); 
        if ($plainText !== '') $textContentArray[$options['textContent']] = $plainText; 
        
        //stick it all together 
        $propertiesArray = !$options['autoText'] || $attributesArray || $tagsArray || ($plainText === '') 
                ? array_merge($attributesArray, $tagsArray, $textContentArray) : $plainText; 
        //return node as array 
        return array( 
            $xml->getName() => $propertiesArray 
        ); 
    }
	
}
