'use strict';

var
	_ = require('underscore'),
	$ = require('jquery'),
	ko = require('knockout'),
	
	TextUtils = require('%PathToCoreWebclientModule%/js/utils/Text.js'),
	UrlUtils = require('%PathToCoreWebclientModule%/js/utils/Url.js'),
	Utils = require('%PathToCoreWebclientModule%/js/utils/Common.js'),
	
	Ajax = require('%PathToCoreWebclientModule%/js/Ajax.js'),
	Api = require('%PathToCoreWebclientModule%/js/Api.js'),
	App = require('%PathToCoreWebclientModule%/js/App.js'),
	Browser = require('%PathToCoreWebclientModule%/js/Browser.js'),
	CAbstractScreenView = require('%PathToCoreWebclientModule%/js/views/CAbstractScreenView.js'),
	UserSettings = require('%PathToCoreWebclientModule%/js/Settings.js'),
	
	Settings = require('modules/%ModuleName%/js/Settings.js'),
	
	$html = $('html')
;

/**
 * @constructor
 */
function CLoginView()
{
	CAbstractScreenView.call(this, '%ModuleName%');

	this.sCustomLogoUrl = Settings.CustomLogoUrl;
	this.sInfoText = Settings.InfoText;
	this.sBottomInfoHtmlText = Settings.BottomInfoHtmlText;
	
	this.username = ko.observable('');
	this.password = ko.observable('');

	this.usernameDom = ko.observable(null);
	this.passwordDom = ko.observable(null);
	
	this.usernameFocus = ko.observable(false);
	this.passwordFocus = ko.observable(false);

	this.loading = ko.observable(false);

	this.bUseSignMe = (Settings.LoginSignMeType === Enums.LoginSignMeType.Unuse);
	this.signMe = ko.observable(Enums.LoginSignMeType.DefaultOn === Settings.LoginSignMeType);
	this.signMeFocused = ko.observable(false);

	this.canBeLogin = ko.computed(function () {
		return !this.loading();
	}, this);

	this.signInButtonText = ko.computed(function () {
		return this.loading() ? TextUtils.i18n('COREWEBCLIENT/ACTION_SIGN_IN_IN_PROGRESS') : TextUtils.i18n('COREWEBCLIENT/ACTION_SIGN_IN');
	}, this);

	this.loginCommand = Utils.createCommand(this, this.signIn, this.canBeLogin);

	this.username(Settings.DemoLogin || '');
	this.password(Settings.DemoPassword || '');
	
	this.shake = ko.observable(false).extend({'autoResetToFalse': 800});
	
	this.bRtl = UserSettings.IsRTL;
	this.aLanguages = UserSettings.LanguageList;
	this.currentLanguage = ko.observable(UserSettings.Language);
	this.bAllowChangeLanguage = Settings.AllowChangeLanguage && !App.isMobile();
	this.bUseDropdownLanguagesView = Settings.UseDropdownLanguagesView;
	this.headingSelectLanguage = ko.computed(function () {
		var sSiteName = UserSettings.SiteName;
		if (_.isEmpty(sSiteName))
		{
			sSiteName = TextUtils.i18n('%MODULENAME%/HEADING_DEFAULT_SITENAME');
		}
		return TextUtils.i18n('%MODULENAME%/HEADING_SELECT_LANGUAGE', {'SITENAME': sSiteName});
	}, this);
	
	this.domains = ko.observableArray([]);
	this.selectedDomain = ko.observable('');
	this.firstDomain = ko.computed(function () {
		return this.domains().length > 0 ? this.domains()[0] : '';
	}, this);
	this.selectedServer = ko.observable('');
	
	this.beforeButtonsControllers = ko.observableArray([]);
	App.broadcastEvent('AnonymousUserForm::PopulateBeforeButtonsControllers', { ModuleName: '%ModuleName%', RegisterBeforeButtonsController: this.registerBeforeButtonsController.bind(this) });
	
	App.broadcastEvent('%ModuleName%::ConstructView::after', {'Name': this.ViewConstructorName, 'View': this});
}

_.extendOwn(CLoginView.prototype, CAbstractScreenView.prototype);

CLoginView.prototype.ViewTemplate = '%ModuleName%_LoginView';
CLoginView.prototype.ViewConstructorName = 'CLoginView';

CLoginView.prototype.onBind = function ()
{
	$html.addClass('non-adjustable-valign');
};

/**
 * Focuses username input after view showing.
 */
CLoginView.prototype.onShow = function ()
{
	_.delay(_.bind(function(){
		if (this.username() === '')
		{
			this.usernameFocus(true);
		}
	},this), 1);
	Ajax.send('%ModuleName%', 'GetMailDomains', {}, function (oResponse, oRequest) {
		if (_.isArray(oResponse.Result))
		{
			this.domains(oResponse.Result);
		}
	}, this);
};

/**
 * Checks username input value and sends sign-in request to server.
 */
CLoginView.prototype.signIn = function ()
{
	// sometimes nockoutjs conflicts with saved passwords in FF
	this.username($(this.usernameDom()).val());
	this.password($(this.passwordDom()).val());
	
	if (!this.loading())
	{
		var
			sDomain = '',
			sLogin = '',
			sPassword = $.trim(this.password()),
			koForFocus = null
		;

		sLogin = $.trim(this.username());
		if (sLogin.length === 0)
		{
			koForFocus = this.usernameFocus;
		}
		else
		{
			sDomain = this.domains().length > 1 ? this.selectedDomain() :  this.firstDomain();
		}

		if (sLogin.length > 0 && sPassword.length > 0)
		{
			var oParameters = {
				'Domain': sDomain,
				'Login': sLogin,
				'Password': sPassword,
				'Language': $.cookie('aurora-selected-lang') || '',
				'SignMe': this.signMe()
			};

			App.broadcastEvent('AnonymousUserForm::PopulateFormSubmitParameters', { Module: '%ModuleName%', Parameters: oParameters });
			
			this.loading(true);

			Ajax.send('%ModuleName%', 'Login', oParameters, this.onSystemLoginResponse, this);
		}
		else
		{
			if (koForFocus)
			{
				koForFocus(true);
			}
			else if (sPassword.length === 0)
			{
				this.passwordFocus(true);
			}
			this.shake(true);
		}
	}
};

/**
 * Receives data from the server. Shows error and shakes form if server has returned false-result.
 * Otherwise clears search-string if it don't contain "reset-pass", "invite-auth" and "oauth" parameters and reloads page.
 * 
 * @param {Object} oResponse Data obtained from the server.
 * @param {Object} oRequest Data has been transferred to the server.
 */
CLoginView.prototype.onSystemLoginResponseBase = function (oResponse, oRequest)
{
	if (false === oResponse.Result)
	{
		this.loading(false);
		this.shake(true);
		
		Api.showErrorByCode(oResponse, TextUtils.i18n('COREWEBCLIENT/ERROR_PASS_INCORRECT'));
	}
	else
	{
		$.removeCookie('aurora-selected-lang');
		
		if (window.location.search !== '' &&
			UrlUtils.getRequestParam('reset-pass') === null &&
			UrlUtils.getRequestParam('invite-auth') === null &&
			UrlUtils.getRequestParam('oauth') === null)
		{
			UrlUtils.clearAndReloadLocation(Browser.ie8AndBelow, true);
		}
		else
		{
			UrlUtils.clearAndReloadLocation(Browser.ie8AndBelow, false);
		}
	}
};

/**
 * @param {string} sLanguage
 */
CLoginView.prototype.changeLanguage = function (sLanguage)
{
	if (sLanguage && this.bAllowChangeLanguage)
	{
		$.cookie('aurora-lang-on-login', sLanguage, { expires: 30 });
		$.cookie('aurora-selected-lang', sLanguage, { expires: 30 });
		window.location.reload();
	}
};

/**
 * @param {Object} oComponent
 */
CLoginView.prototype.registerBeforeButtonsController = function (oComponent)
{
	this.beforeButtonsControllers.push(oComponent);
};

/**
 * @param {Object} oResponse
 * @param {Object} oRequest
 */
CLoginView.prototype.onSystemLoginResponse = function (oResponse, oRequest)
{
	this.onSystemLoginResponseBase(oResponse, oRequest);
};

module.exports = new CLoginView();
