<?php
/**
 * eXtreme Message Board
 * XMB 1.9.12
 *
 * Developed And Maintained By The XMB Group
 * Copyright (c) 2001-2024, The XMB Group
 * https://www.xmbforum2.com/
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License
 * as published by the Free Software Foundation; either version 2
 * of the License, or (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program.  If not, see <https://www.gnu.org/licenses/>.
 *
 **/

if (!defined('IN_CODE')) {
    header('HTTP/1.0 403 Forbidden');
    exit("Not allowed to run this file directly.");
}

define('SQL_NUM', MYSQLI_NUM);
define('SQL_BOTH', MYSQLI_BOTH);
define('SQL_ASSOC', MYSQLI_ASSOC);

/**
 * Represents a single MySQL connection and provides abstracted query methods.
 */
class dbstuff {
    private $db         = '';   // string - Name of the database used by this connection.
    private $duration   = 0.0;  // float - Cumulative time used by synchronous query commands.
    private $last_id    = 0;    // int|string - The ID generated by INSERT or UPDATE commands, also known as LAST_INSERT_ID.  Stored for DEBUG mode only.
    private $last_rows  = 0;    // int|string - Number of rows affected by the last INSERT, UPDATE, REPLACE or DELETE query.  Stored for DEBUG mode only.
    private $link       = null; // mysqli - Connection object.
    private $querynum   = 0;    // int - Count of commands sent on this connection.
    private $querylist  = [];   // array - Log of all SQL commands sent.  Stored for DEBUG mode only.
    private $querytimes = [];   // array - Log of all SQL execution times.  Stored for DEBUG mode only.
    private $timer      = 0.0;  // float - Date/time the last query started.  Class scope not needed, just simplifies code.
    private $test_error = '';   // string - Any error message collected by test_connect().

    public function __construct() {
        // Force older versions of PHP to behave like PHP v8.1.  This assumes there are no incompatible mysqli scripts running.
        mysqli_report(MYSQLI_REPORT_ERROR | MYSQLI_REPORT_STRICT);
    }
    
    /**
     * Checks PHP dependencies
     *
     * @since 1.9.12.06
     * @return bool
     */
    public function installed(): bool {
        return extension_loaded('mysqli');
    }

    /**
     * Establishes a connection to the MySQL server.
     *
     * @param string $dbhost
     * @param string $dbuser
     * @param string $dbpw
     * @param string $dbname
     * @param bool   $pconnect Keep the connection open after the script ends.
     * @param bool   $force_db Generate a fatal error if the $dbname database doesn't exist on the server.
     * @return bool  Whether or not the database was found after connecting.
     */
    public function connect(string $dbhost, string $dbuser, string $dbpw, string $dbname, bool $pconnect = false, bool $force_db = false): bool {
        // Verify compatiblity.
        if (!$this->installed()) {
            header('HTTP/1.0 500 Internal Server Error');
            exit('Error: The PHP mysqli extension is missing.');
        }

        if ($pconnect) {
            $dbhost = "p:$dbhost";
        }

        if ($force_db) {
            $database = $dbname;
        } else {
            $database = '';
        }

        try {
            $this->link = new mysqli($dbhost, $dbuser, $dbpw, $database);
        } catch (mysqli_sql_exception $e) {
            $msg = "<h3>Database connection error!</h3>\n"
                 . "A connection to the Database could not be established.<br />\n"
                 . "Please check the MySQL username, password, database name and host.<br />\n"
                 . "Make sure <i>config.php</i> is correctly configured.<br />\n";
            $sql = '';
            $this->panic($e, $sql, $msg);
        }

        // Always force single byte mode so the PHP mysql client doesn't throw non-UTF input errors.
        try {
            $result = $this->link->set_charset('latin1');
        } catch (mysqli_sql_exception $e) {
            $msg = "<h3>Database connection error!</h3>\n"
                 . "The database connection could not be configured for XMB.<br />\n"
                 . "Please ensure the mysqli_set_charset function is working.<br />\n";
            $sql = '';
            $this->panic($e, $sql, $msg);
        }

        if ($force_db) {
            $this->db = $dbname;
            return true;
        } else {
            return $this->select_db($dbname, $force_db);
        }
    }

    /**
     * Attempts a connection and does not generate error messages.
     *
     * @since 1.9.12.06
     * @param string $dbhost
     * @param string $dbuser
     * @param string $dbpw
     * @param string $dbname
     * @return bool  Whether or not the connection was made and the database was found.
     */
    public function test_connect(string $dbhost, string $dbuser, string $dbpw, string $dbname): bool {
        if (!$this->installed()) return false;
        try {
            $this->link = new mysqli($dbhost, $dbuser, $dbpw, $dbname);
            $this->link->set_charset('latin1');
        } catch (mysqli_sql_exception $e) {
            $this->test_error = $e->getMessage();
            return false;
        }
        
        $this->db = $dbname;
        $this->test_error = '';
        return true;
    }
    
    /**
     * Gets any error message that was encountered during the last call to test_connect().
     *
     * Error messages are likely to contain sensitive file path info.
     * This method is intended for use by Super Administrators and install/upgrade scripts only.
     *
     * @since 1.9.12.06
     * @return string Error message or empty string.
     */
    public function get_test_error(): string {
        return $this->test_error;
    }

    /**
     * Closes a connection that is no longer needed.
     *
     * @since 1.9.12.06
     */
    public function close() {
        $this->link->close();
    }

    /**
     * Sets the name of the database to be used on this connection.
     *
     * @param string $database The full name of the MySQL database.
     * @param bool $force Optional. Specifies error mode. Dies if true.
     * @return bool TRUE on success, FALSE on failure with !$force.
     */
    public function select_db($database, $force = true) {
        try {
            $this->link->select_db($database);
            $this->db = $database;
            return true;
        } catch (mysqli_sql_exception $e) {
            if ($force) {
                $this->panic($e);
            } else {
                return false;
            }
        }
    }

    /**
     * Searches for an accessible database containing the XMB settings table.
     *
     * @param string $tablepre The settings table name prefix.
     * @return bool
     */
    public function find_database($tablepre) {
        $dbs = $this->query('SHOW DATABASES');
        while($db = $this->fetch_array($dbs)) {
            if ('information_schema' == $db['Database']) {
                continue;
            }
            $q = $this->query("SHOW TABLES FROM `{$db['Database']}`");

            while ($table = $this->fetch_array($q)) {
                if ($tablepre.'settings' == $table[0]) {
                    if ($this->select_db($db['Database'], false)) {
                        $this->free_result($dbs);
                        $this->free_result($q);
                        return true;
                    }
                }
            }
            $this->free_result($q);
        }
        $this->free_result($dbs);
        return false;
    }

    public function error() {
        return $this->link->error;
    }

    public function free_result($query) {
        try {
            $query->free();
        } catch (mysqli_sql_exception $e) {
            $this->panic($e);
        }
        return true;
    }
	
    /**
     * Fetch an array representing the next row of a result.
     *
     * The array type can be associative, numeric, or both.
     *
     * @param mysqli_result $query
     * @param int $type The type of indexing to add to the array: SQL_ASSOC, SQL_NUM, or SQL_BOTH
     * @return array|null Returns an array representing the fetched row, or null if there are no more rows.
     */
    public function fetch_array($query, $type=SQL_ASSOC) {
        try {
            return $query->fetch_array($type);
        } catch (mysqli_sql_exception $e) {
            $this->panic($e);
        }
    }

    public function field_name($query, $field) {
        try {
            return $query->fetch_field_direct($field)->name;
        } catch (mysqli_sql_exception $e) {
            $this->panic($e);
        }
    }

    /**
     * Returns the length of a field as specified in the database schema.
     *
     * @since 1.9.11.13
     * @param resource $query The result of a query.
     * @param int $field The field_offset starts at 0.
     * @return int
     */
    public function field_len($query, $field) {
        try {
            return $query->fetch_field_direct($field)->length;
        } catch (mysqli_sql_exception $e) {
            $this->panic($e);
        }
    }

    /**
     * Handle all MySQLi errors.
     *
     * @param Exception $e
     * @param string $sql Optional.  The full SQL command that caused the error, if any.
     * @param string $msg Optional.  HTML help message to display before the error.
     */
    private function panic(Exception $e, string $sql = '', string $msg = '') {
        if (!headers_sent()) {
            header('HTTP/1.0 500 Internal Server Error');
        }
        
        echo $msg;

        if (DEBUG || LOG_MYSQL_ERRORS) {
            $error = $e->getMessage();
            $errno = $e->getCode();
        }
        
        if (LOG_MYSQL_ERRORS) {
            $log_advice = "Please check the error log for details.<br />\n";
        } else {
            $log_advice = "Please set LOG_MYSQL_ERRORS to true in config.php.<br />\n";
        }

    	if (DEBUG && (!defined('X_SADMIN') || X_SADMIN)) {
            require_once(ROOT.'include/validate.inc.php');

            // MySQL error text may contain sensitive file path info.
            if (defined('X_SADMIN')) {
                echo 'MySQL encountered the following error: ' . cdataOut($error) . "<br />\n(errno = $errno)<br /><br />\n";
                if ($sql != '') {
                    echo "In the following query:<br />\n<pre>" . cdataOut($sql) . "</pre>\n";
                }
                echo "<strong>Stack trace:</strong>\n<pre>";
                debug_print_backtrace();
                echo '</pre>';
            } elseif ($sql != '') {
                echo "MySQL encountered an error in the following query:<br />\n<pre>" . cdataOut($sql) . "</pre>\n", $log_advice;
            } elseif ($msg != '') {
                echo $log_advice;
            } else {
                echo "MySQL encountered an error.<br />\n", $log_advice;
            }
        } else {
            echo "The system has failed to process your request.<br />\n", $log_advice;
            if (defined('X_SADMIN') && X_SADMIN && ! DEBUG) {
                echo "To display details, please set DEBUG to true in config.php.<br />\n";
            }
    	}
        if (LOG_MYSQL_ERRORS) {
            $log = "MySQL encountered the following error:\n$error\n(errno = $errno)\n";
            if ($sql != '') {
                if ((1153 == $errno || 2006 == $errno) && strlen($sql) > 16000) {
                    $log .= "In the following query (log truncated):\n" . substr($sql, 0, 16000) . "\n";
                } else {
                    $log .= "In the following query:\n$sql\n";
                }
            }

            $trace = $e->getTrace();
            $depth = 1; // Go back before dbstuff::panic() and see who called dbstuff::query().
            $filename = $trace[$depth]['file'];
            $linenum = $trace[$depth]['line'];
            $function = $trace[$depth]['function'];
            $log .= "\$db->{$function}() was called by {$filename} on line {$linenum}";

            if (!ini_get('log_errors')) {
                ini_set('log_errors', true);
                ini_set('error_log', 'error_log');
            }
            error_log($log);
        }
        exit;
    }

    /**
     * Can be used to make any expression query-safe, but see next function.
     *
     * Example:
     *  $sqlinput = $db->escape($rawinput);
     *  $db->query("UPDATE a SET b = 'Hello, my name is $sqlinput'");
     *
     * @param string $rawstring
     * @return string
     */
    public function escape(string $rawstring): string {
        try {
            return $this->link->real_escape_string($rawstring);
        } catch (mysqli_sql_exception $e) {
            $this->panic($e);
        }
    }

    /**
     * Preferred for performance when escaping any string variable.
     *
     * Note this only works when the raw value can be discarded.
     *
     * Example:
     *  $db->escape_fast($rawinput);
     *  $db->query("UPDATE a SET b = 'Hello, my name is $rawinput'");
     *
     * @since 1.9.11.12
     * @param string $sql Read/Write Variable
     */
    public function escape_fast(string &$sql) {
        try {
            $sql = $this->link->real_escape_string($sql);
        } catch (mysqli_sql_exception $e) {
            $this->panic($e);
        }
    }

    public function like_escape(string $rawstring): string {
        try {
            return $this->link->real_escape_string( str_replace(array('\\', '%', '_'), array('\\\\', '\\%', '\\_'), $rawstring) );
        } catch (mysqli_sql_exception $e) {
            $this->panic($e);
        }
    }

    public function regexp_escape(string $rawstring): string {
        try {
            return $this->link->real_escape_string(preg_quote($rawstring));
        } catch (mysqli_sql_exception $e) {
            $this->panic($e);
        }
    }

    /**
     * Executes a MySQL Query
     *
     * @param string $sql Unique MySQL query (multiple queries are not supported). The query string should not end with a semicolon.
     * @param bool $panic XMB will die and use dbstuff::panic() in case of any MySQL error unless this param is set to FALSE.
     * @return mixed Returns a MySQL resource or a bool, depending on the query type and error status.
     */
    public function query($sql, $panic = true) {
        $this->start_timer();
        try {
            $query = $this->link->query($sql);
        } catch (mysqli_sql_exception $e) {
            if ($panic) {
                $this->panic($e, $sql);
            } else {
                $query = false;
            }
        }
        $this->querytimes[] = $this->stop_timer();
        $this->querynum++;
    	if (DEBUG) {
            if (LOG_MYSQL_ERRORS) {
                $this->last_id = $this->link->insert_id;
                $this->last_rows = $this->link->affected_rows;
                $warnings = $this->link->warning_count;

                if ($warnings > 0) {
                    if (!ini_get('log_errors')) {
                        ini_set('log_errors', TRUE);
                        ini_set('error_log', 'error_log');
                    }
                    if (strlen($sql) > 16000) {
                        $output = "MySQL generated $warnings warnings in the following query (log truncated):\n" . substr($sql, 0, 16000) . "\n";
                    } else {
                        $output = "MySQL generated $warnings warnings in the following query:\n$sql\n";
                    }
                    $query3 = $this->link->query('SHOW WARNINGS');
                    while ($row = $this->fetch_array($query3)) {
                        $output .= var_export($row, TRUE)."\n";
                    }
                    error_log($output);
                    $this->free_result($query3);
                }
            }
            if (!defined('X_SADMIN') || X_SADMIN) {
                $this->querylist[] = $sql;
            }
        }
        return $query;
    }

    /**
     * Sends a MySQL query without fetching the result rows.
     *
     * You cannot use mysqli_num_rows() and mysqli_data_seek() on a result set
     * returned from mysqli_use_result(). You also have to call
     * mysqli_free_result() before you can send a new query to MySQL.
     *
     * @param string $sql Unique MySQL query (multiple queries are not supported). The query string should not end with a semicolon.
     * @param bool $panic XMB will die and use dbstuff::panic() in case of any MySQL error unless this param is set to FALSE.
     * @return mixed Returns a MySQL resource or a bool, depending on the query type and error status.
     */
    public function unbuffered_query($sql, $panic = true) {
        $this->start_timer();
        try {
            $query = $this->link->query($sql, MYSQLI_USE_RESULT);
        } catch (mysqli_sql_exception $e) {
            if ($panic) {
                $this->panic($e, $sql);
            } else {
                $query = false;
            }
        }
        $this->querynum++;
    	if (DEBUG && (!defined('X_SADMIN') || X_SADMIN)) {
            $this->querylist[] = $sql;
        }
        $this->querytimes[] = $this->stop_timer();
        return $query;
    }

    public function fetch_tables($dbname = null) {
        if ($dbname === null) {
            $dbname = $this->db;
        }
        $this->select_db($dbname);

        $array = array();
        $q = $this->query("SHOW TABLES");
        while($table = $this->fetch_row($q)) {
            $array[] = $table[0];
        }
        $this->free_result($q);
        return $array;
    }

    /**
     * Retrieves the contents of one cell from a MySQL result set.
     *
     * @param resource $query
     * @param int      $row   The row number from the result that's being retrieved.
     * @param mixed    $field The name or offset of the field being retrieved.
     * @return string
     */
    public function result($query, $row, $field = 0) {
        try {
            $query->data_seek($row);
            return $query->fetch_array()[$field];
        } catch (mysqli_sql_exception $e) {
            $this->panic($e);
        }
    }

    /**
     * Retrieves the row count from a query result.
     */
    public function num_rows(mysqli_result $query): int {
        $count = $query->num_rows;

        if (!is_int($count)) throw new UnexpectedValueException('Row count was not an int');

        return $count;
    }

    /**
     * Retrieves the column count from a query result.
     */
    public function num_fields(mysqli_result $query): int {
        return $query->field_count;
    }

    public function insert_id() {
        return $this->link->insert_id;
    }

    /**
     * Fetch an enumerated array representing the next row of a result.
     */
    public function fetch_row($query) {
        try {
            return $query->fetch_row();
        } catch (mysqli_sql_exception $e) {
            $this->panic($e);
        }
    }

    public function data_seek($query, $row) {
        try {
            return $query->data_seek($row);
        } catch (mysqli_sql_exception $e) {
            $this->panic($e);
        }
    }

    public function affected_rows(): int {
        $count = $this->link->affected_rows;

        if (!is_int($count)) throw new UnexpectedValueException('Row count was not an int');

        return $count;
    }

    /**
     * DEPRECATED by XMB 1.9.12
     *
     * dbstuff::time() was totally unrelated to the MySQL data types named TIME and TIMESTAMP.
     * Its purpose was ambiguous and usage seemed fully unnecessary.
     */
    function time($time=NULL) {
        trigger_error('dbstuff::time() is deprecated in this version of XMB', E_USER_DEPRECATED);
        if ($time === NULL) {
            $time = time();
        }
        return "LPAD('".$time."', '15', '0')";
    }

    private function start_timer() {
        $mtime = explode(" ", microtime());
        $this->timer = (float) $mtime[1] + (float) $mtime[0];
        return true;
    }

    /**
     * Calculate time since start_timer and add it to duration.
     * 
     * @return int Time since start_timer.
     */
    private function stop_timer() {
        $mtime = explode(" ", microtime());
        $endtime = (float) $mtime[1] + (float) $mtime[0];
        $taken = ($endtime - $this->timer);
        $this->duration += $taken;
        $this->timer = 0;
        return $taken;
    }

    /**
     * Retrieve the MySQL server version number.
     *
     * @return string
     */
    public function server_version(){
        return $this->link->server_info;
    }

    public function getDuration(): float {
        return $this->duration;
    }

    public function getQueryCount(): int {
        return $this->querynum;
    }

    public function getQueryList(): array {
        return $this->querylist;
    }

    public function getQueryTimes(): array {
        return $this->querytimes;
    }
}

/**
 * DEPRECATED by XMB 1.9.12.06
 *
 * This callback has been replaced by exception handlers for improved compatibility with PHP 8.1.
 *
 * @param int $errno
 * @param string $errstr
 */
function xmb_mysql_error($errno, $errstr) {
    trigger_error('xmb_mysql_error() is deprecated in this version of XMB', E_USER_DEPRECATED);
    $output = '';
    {
        $trace = debug_backtrace();
        if (isset($trace[2]['function'])) { // Catch MySQL error
            $depth = 2;
        } else { // Catch syntax error
            $depth = 1;
        }
        $functionname = $trace[$depth]['function'];
        $filename = $trace[$depth]['file'];
        $linenum = $trace[$depth]['line'];
        $output = "MySQLi encountered the following error: $errstr in \$db->{$functionname}() called by {$filename} on line {$linenum}";
        unset($trace, $functionname, $filename, $linenum);
    }

    if (!headers_sent()) {
        header('HTTP/1.0 500 Internal Server Error');
    }
	if (DEBUG && (!defined('X_SADMIN') || X_SADMIN)) {
        require_once(ROOT.'include/validate.inc.php');
		echo "<pre>".cdataOut($output)."</pre>";
    } else {
        echo "<pre>The system has failed to process your request. If you're an administrator, please set the DEBUG flag to true in config.php.</pre>";
	}
    if (LOG_MYSQL_ERRORS) {
        if (!ini_get('log_errors')) {
            ini_set('log_errors', TRUE);
            ini_set('error_log', 'error_log');
        }
        error_log($output);
    }
    exit;
}

return;
