<?php
/**
 * OAuth mail authorization api file.
 *
 * @package API
 *
 * @copyright YetiForce S.A.
 * @license YetiForce Public License 7.0 (licenses/LicenseEN.txt or yetiforce.com)
 * @author Radosław Skrzypczak <r.skrzypczak@yetiforce.com>
 */

namespace Api\OAuth\BaseAction;

use App\Mail\Account\Repository;

class MailAccount extends \Api\Core\BaseAction
{
	/** {@inheritdoc}  */
	public $allowedMethod = ['GET'];

	/** {@inheritdoc} */
	public function updateSession(array $data = []): void
	{
	}

	/**
	 * @OA\Get(
	 *		path="/webservice/OAuth/MailAccount",
	 *		summary="OAuth mail authorization api",
	 *		@OA\Response(response=200, description="Result"),
	 * ),
	 *
	 * @return void
	 */
	public function get()
	{
		$code = $this->controller->request->getRaw('code');
		$state = $this->controller->request->get('state');
		$hash = $state ? sha1($state) : '';
		$key = "OAuth.State.{$hash}";

		$data = \App\Session::get($key);
		\App\Session::delete($key);
		$recordId = $data['recordId'];
		$redirectUri = $data['redirectUri'];
		$class = $data['repositoryClass'];

		$recordId = $data['recordId'];
		$code = $this->controller->request->getRaw('code');
		try {
			$repository = (new Repository())->get($class);
			$accountEntity = $repository->findById($recordId);
			$mailAccount = new \App\Mail\Account($accountEntity, $repository);
			if ($this->controller->request->get('error') && !$code) {
				$message = $this->controller->request->get('error_description') ?: 'Authentication error';
				\App\Log::error($message);
				$accountEntity->setLogs($message);
			} else {
				$token = $mailAccount->getAccessToken(['code' => $code]);
				$provider = $mailAccount->getOAuthProvider();
				$resourceOwner = $provider->getResourceOwner();
				if (($aud = $resourceOwner->toArray()['aud'] ?? '') && $aud !== $mailAccount->getServer()->get('client_id')) {
					$message = 'Attempted to authenticate the wrong data aud: ' . $aud;
					$accountEntity->setLogs($message);
					\App\Log::error($message);
				} else {
					$accountEntity->setPassword($token);
					$reToken = $mailAccount->getRefreshToken();
					if ($reToken) {
						$accountEntity->setRefreshToken($reToken);
					}
					$accountEntity->setExpireTime($mailAccount->getExpireTime());
				}
			}

			$repository->update($accountEntity);
		} catch (\Throwable $th) {
			\App\Log::error($th->__toString());
			$message = $th->getMessage();
			if ($th instanceof \App\Exceptions\AppException) {
				$message = $th->getDisplayMessage();
			}
			$accountEntity->setLogs($message);
			$repository->update($accountEntity);
		}

		header('location: ' . $redirectUri);
		exit;
	}

	/** {@inheritdoc}  */
	protected function checkPermission(): void
	{
		\App\Session::init();
		$state = $this->controller->request->get('state');
		$hash = $state ? sha1($state) : '';
		if (!$state || !\App\Session::has("OAuth.State.{$hash}") || $state !== \App\Session::get("OAuth.State.{$hash}")['state']) {
			throw new \Api\Core\Exception('No permission or wrong data', 401);
		}
	}

	/** {@inheritdoc}  */
	protected function checkPermissionToModule(): void
	{
	}
}
