<?php
/**
 * Reference query inventory field file.
 *
 * @package UIType
 *
 * @copyright YetiForce S.A.
 * @license YetiForce Public License 7.0 (licenses/LicenseEN.txt or yetiforce.com)
 * @author Radosław Skrzypczak <r.skrzypczak@yetiforce.com>
 * @author    Mariusz Krzaczkowski <m.krzaczkowski@yetiforce.com>
 */

declare(strict_types=1);

namespace App\Conditions\QueryFields\Inventory;

/**
 * Reference query inventory field class.
 */
class ReferenceField extends BaseField
{
	/**
	 * Get related column name.
	 *
	 * @return string[]
	 */
	public function getRelatedTableName(): array
	{
		return $this->getRelatedTables($this->fieldModel->getReferenceModules(), $this->fieldModel->getName());
	}

	/**
	 * Get formatted column references from related records.
	 *
	 * @param array  $modules
	 * @param string $fieldName
	 *
	 * @return string[]
	 */
	public function getRelatedTables(array $modules, string $fieldName): array
	{
		$relatedTableName = [];
		if ($modules) {
			$moduleName = $this->queryGenerator->getModuleModel();
			$invTableName = $moduleName->getInventoryModel()->getDataTableName();
			$tableName = $moduleName->getBaseTableName();
			$tableIndex = $moduleName->getBaseTableIndex();
			$this->queryGenerator->addJoin(['LEFT JOIN', $invTableName, "{$tableName}.{$tableIndex} = {$invTableName}.crmid"]);
		}
		foreach ($modules as $moduleName) {
			$formattedTables = [];
			$entityFieldInfo = \App\Module::getEntityInfo($moduleName);
			$relatedModuleModel = \Vtiger_Module_Model::getInstance($moduleName);
			$relTableIndexes = $relatedModuleModel->getEntityInstance()->tab_name_index;
			foreach ($entityFieldInfo['fieldnameArr'] as $column) {
				if ($relField = $relatedModuleModel->getFieldByColumn($column)) {
					$referenceTable = $relField->getTableName() . $fieldName;
					$this->queryGenerator->addJoin(['LEFT JOIN',
						"{$relField->getTableName()} {$referenceTable}",
						"{$this->getColumnName()} = {$referenceTable}.{$relTableIndexes[$relField->getTableName()]}",
					]);
					$formattedTables[] = "{$referenceTable}.{$column}";
				}
			}
			$relatedTableName[$moduleName] = \count($formattedTables) > 1 ? new \yii\db\Expression('CONCAT(' . implode(",' ',", $formattedTables) . ')') : current($formattedTables);
		}
		return $relatedTableName;
	}

	/** {@inheritdoc} */
	public function operatorA(): array
	{
		if (\App\Config::performance('SEARCH_REFERENCE_BY_AJAX')) {
			if (false === strpos($this->value, '##')) {
				return [$this->getColumnName() => $this->value];
			}
			$condition = ['or'];
			foreach (explode('##', $this->value) as $value) {
				$condition[] = [$this->getColumnName() => $value];
			}
			return $condition;
		}
		return parent::operatorA();
	}

	/** {@inheritdoc} */
	public function operatorE(): array
	{
		$condition = ['or'];
		foreach ($this->getRelatedTableName() as $formattedName) {
			$condition[] = ['=', $formattedName, $this->getValue()];
		}
		return $condition;
	}

	/**
	 * Equals Id operator.
	 *
	 * @return array
	 */
	public function operatorEid()
	{
		return [$this->getColumnName() => $this->getValue()];
	}

	/** {@inheritdoc} */
	public function operatorN(): array
	{
		$condition = ['or'];
		foreach ($this->getRelatedTableName() as $formattedName) {
			$condition[] = ['<>', $formattedName, $this->getValue()];
		}
		return $condition;
	}

	/** {@inheritdoc} */
	public function operatorS()
	{
		$condition = ['or'];
		foreach ($this->getRelatedTableName() as $formattedName) {
			$condition[] = ['like', $formattedName, $this->getValue() . '%', false];
		}
		return $condition;
	}

	/** {@inheritdoc} */
	public function operatorEw()
	{
		$condition = ['or'];
		foreach ($this->getRelatedTableName() as $formattedName) {
			$condition[] = ['like', $formattedName, '%' . $this->getValue(), false];
		}
		return $condition;
	}

	/** {@inheritdoc} */
	public function operatorC(): array
	{
		$condition = ['or'];
		foreach ($this->getRelatedTableName() as $formattedName) {
			$condition[] = ['like', $formattedName, $this->getValue()];
		}
		return $condition;
	}

	/** {@inheritdoc} */
	public function operatorK(): array
	{
		$condition = ['or'];
		foreach ($this->getRelatedTableName() as $formattedName) {
			$condition[] = ['not like', $formattedName, $this->getValue()];
		}
		return $condition;
	}

	/** {@inheritdoc} */
	public function operatorY(): array
	{
		return ['or',
			[$this->getColumnName() => null],
			['=', $this->getColumnName(), ''],
			['=', $this->getColumnName(), 0],
		];
	}

	/** {@inheritdoc} */
	public function operatorNy(): array
	{
		return ['and',
			['not', [$this->getColumnName() => null]],
			['<>', $this->getColumnName(), ''],
			['<>', $this->getColumnName(), 0],
		];
	}

	/** {@inheritdoc} */
	public function getOrderBy($order = false): array
	{
		$condition = [];
		if ($order && 'DESC' === strtoupper($order)) {
			foreach ($this->getRelatedTableName() as $formattedName) {
				$condition[(string) $formattedName] = SORT_DESC;
			}
		} else {
			foreach ($this->getRelatedTableName() as $formattedName) {
				$condition[(string) $formattedName] = SORT_ASC;
			}
		}
		return $condition;
	}
}
