<?php
/**
 * System installer.
 *
 * @package App
 *
 * @copyright YetiForce S.A.
 * @license   YetiForce Public License 7.0 (licenses/LicenseEN.txt or yetiforce.com)
 * @author    Klaudia Łozowska <k.lozowska@yetiforce.com>
 */

namespace App\Installer;

use App\Cache;
use App\Config;
use App\ConfigFile;
use App\Db;
use App\Db\Migrations\Doctrine\Migrator;
use App\Db\Migrations\MigratorInterface;
use App\Db\Mongo;
use App\Db\Query;
use App\Encryption;
use App\Exceptions\AppException;
use App\Exceptions\IllegalValue;
use App\Exceptions\NoPermitted;
use App\Language;
use App\Log;
use App\Module;
use App\UserPrivilegesFile;
use App\Validator;
use App\Version;
use yii\db\Exception;

/**
 * Class Installer.
 */
class Installer
{
	/** @var string[] System requirements check list */
	public const SYS_REQS_CHECK = [
		'libraries' => 'Libraries',
		'security' => 'Recommended security settings',
		'stability' => 'Recommended PHP settings',
		'performance' => 'Performance verification',
		'publicDirectoryAccess' => 'Verification of permissions for directories',
		'environment' => 'Environment information',
		'writableFilesAndFolders' => 'Writable files and folders',
	];

	/** @var string[] Date formats. */
	public const DATE_FORMATS = [
		'yyyy-mm-dd',
		'dd-mm-yyyy',
		'mm-dd-yyyy',
		'yyyy.mm.dd',
		'dd.mm.yyyy',
		'mm.dd.yyyy',
		'yyyy/mm/dd',
		'dd/mm/yyyy',
		'mm/dd/yyyy',
	];

	/** @var string[] Config files to skip */
	private const CONFIG_FILES_TO_SKIP = ['performance', 'debug', 'security', 'module', 'component'];

	/** @var MigratorInterface|null DB migrator. */
	private ?MigratorInterface $migrator = null;

	/**
	 * Constructor.
	 */
	public function __construct()
	{
		$this->setMigrator();
	}

	/**
	 * Get migrator.
	 *
	 * @return ?MigratorInterface
	 */
	public function getMigrator(): ?MigratorInterface
	{
		$this->setMigrator();

		return $this->migrator;
	}

	/**
	 * Is CRM installed.
	 *
	 * @throws IllegalValue
	 */
	public function isInstalled(): bool
	{
		return $this->isDbInitialized() && $this->areConfigFilesCreated();
	}

	/**
	 * Is CRM database initialized.
	 *
	 * @return bool
	 */
	public function isDbInitialized(): bool
	{
		$this->setMigrator();

		return 0 === $this->migrator?->pendingMigrationsCount();
	}

	/**
	 * Are config files created.
	 *
	 * @throws IllegalValue
	 */
	public function areConfigFilesCreated(): bool
	{
		foreach (array_diff(ConfigFile::TYPES, self::CONFIG_FILES_TO_SKIP) as $type) {
			if (!(new ConfigFile($type))->exists()) {
				return false;
			}
		}

		$dataReader = (new Query())
			->select(['name'])
			->from('vtiger_tab')
			->createCommand()
			->query();

		while ($moduleName = $dataReader->readColumn(0)) {
			$filePath = 'modules' . \DIRECTORY_SEPARATOR . $moduleName . \DIRECTORY_SEPARATOR . 'ConfigTemplate.php';
			if (file_exists($filePath) && !(new ConfigFile('module', $moduleName))->exists()) {
				return false;
			}
		}

		$path = ROOT_DIRECTORY
			. \DIRECTORY_SEPARATOR
			. 'config'
			. \DIRECTORY_SEPARATOR
			. 'Components'
			. \DIRECTORY_SEPARATOR
			. 'ConfigTemplates.php';

		$componentsData = require "$path";
		foreach ($componentsData as $component => $data) {
			if (!(new ConfigFile('component', $component))->exists()) {
				return false;
			}
		}

		return true;
	}

	/**
	 * Gets languages for installation.
	 *
	 * @return array
	 */
	public static function getLanguages(): array
	{
		$languages = [];
		foreach ((new \DirectoryIterator('install/languages/')) as $item) {
			if (
				$item->isDir()
				&& !$item->isDot()
				&& file_exists($item->getPathname() . \DIRECTORY_SEPARATOR . 'Install.json')
			) {
				$languages[$item->getBasename()] = [
					'displayName' => Language::getDisplayName($item->getBasename()),
					'region' => strtolower(Language::getRegion($item->getBasename())),
				];
			}
		}

		return $languages;
	}

	/**
	 * Set default language.
	 *
	 * @param string $language
	 *
	 * @return void
	 */
	public function setLanguage(string $language): void
	{
		if (
			$language
			&& Language::DEFAULT_LANG !== $language
			&& ($languages = Languages::getAll())
			&& isset($languages[$language])
			&& (int) $languages[$language]['progress'] > 60
			&& Languages::download($language)
			&& Language::getLangInfo($language)
		) {
			\Settings_LangManagement_Module_Model::setAsDefault($language);
		}
	}

	/**
	 * Gets currencies.
	 *
	 * @return array
	 */
	public static function getCurrencies(): array
	{
		$currencies = require 'install/models/Currencies.php';

		return empty($currencies) ? [] : $currencies;
	}

	/**
	 * Set currency.
	 *
	 * @param string $currency
	 *
	 * @throws Exception
	 *
	 * @return void
	 */
	public function setCurrency(string $currency): void
	{
		$currencies = self::getCurrencies();
		if ($currencyData = ($currencies[$currency] ?? null)) {
			Db::getInstance()->createCommand()
				->update('vtiger_currency_info', [
					'currency_name' => $currency,
					'currency_code' => $currencyData[0],
					'currency_symbol' => $currencyData[1],
				])->execute();
		}
	}

	/**
	 * Function checks the database connection.
	 *
	 * @param array $config
	 *
	 * @return array
	 */
	public static function checkDbConnection(array $config): array
	{
		$dbType = $config['db_type'] ?? 'mysql';
		$dbServer = $config['db_server'] ?? null;
		$dbUsername = $config['db_username'] ?? null;
		$dbPassword = $config['db_password'] ?? null;
		$dbName = $config['db_name'] ?? null;
		$dbPort = $config['db_port'] ?? null;

		if (!$dbServer || !$dbUsername || !$dbPassword || !$dbName || !$dbPort) {
			return ['flag' => false];
		}

		$dbTypeStatus = false; // is there a db type?
		$dbServerStatus = false; // does the db server connection exist?
		$dbExistStatus = false; // does the database exist?
		$dbUtf8Support = false; // does the database support utf8?

		if ($dbType) {
			$conn = false;
			$pdoException = '';

			try {
				Db::setConfig([
					'dsn' => $dbType . ':host=' . $dbServer . ';charset=utf8;port=' . $dbPort . ';dbname=' . $dbName,
					'host' => $dbServer,
					'port' => $dbPort,
					'dbName' => $dbName,
					'tablePrefix' => 'yf_',
					'username' => $dbUsername,
					'password' => $dbPassword,
					'charset' => 'utf8',
					'attributes' => [
						\PDO::ATTR_ERRMODE => \PDO::ERRMODE_EXCEPTION,
					],
				]);
				$db = Db::getInstance();
				$conn = $db->getMasterPdo();
			} catch (\Throwable $e) {
				$pdoException = $e->getMessage();
				Log::error($e->__toString(), 'Install');
			}

			$dbTypeStatus = true;

			if ($conn) {
				$dbServerStatus = true;
				if (Validator::isMySQL($dbType)) {
					$stmt = $conn->query("SHOW VARIABLES LIKE 'version'");
					$res = $stmt->fetch(\PDO::FETCH_ASSOC);
					$mysqlServerVersion = $res['Value'];
				}
				try {
					$stmt = $conn->query("SELECT SCHEMA_NAME FROM INFORMATION_SCHEMA.SCHEMATA WHERE SCHEMA_NAME = '$dbName'");
				} catch (\Throwable $e) {
					$pdoException = $e->getMessage();
					Log::error($e->__toString(), 'Install');
					$dbServerStatus = false;
				}

				if (1 == $stmt->rowCount()) {
					$dbExistStatus = true;
				}
			}
		}

		$dbCheckResult = [];
		$dbCheckResult['db_utf8_support'] = $dbUtf8Support;

		$errorMsgInfo = '';

		if (!$dbTypeStatus || !$dbServerStatus) {
			$errorMsg = Language::translate('ERR_DATABASE_CONNECTION_FAILED', 'Install') . '. '
				. Language::translate('ERR_INVALID_MYSQL_PARAMETERS', 'Install');

			$errorMsgInfo = Language::translate('MSG_LIST_REASONS', 'Install') . ':<br />
					-  ' . Language::translate('MSG_DB_PARAMETERS_INVALID', 'Install') . '
					<br />-  ' . Language::translate('MSG_DB_USER_NOT_AUTHORIZED', 'Install');

			$errorMsgInfo .= "<br /><br />$pdoException";
		} elseif (Validator::isMySQL($dbType) && version_compare($mysqlServerVersion, '5.1', '<')) {
			$errorMsg = $mysqlServerVersion . ' -> ' . Language::translate('ERR_INVALID_MYSQL_VERSION', 'Install');
		} elseif (!$dbExistStatus) {
			$errorMsg = $dbName . ' -> ' . Language::translate('ERR_DB_NOT_FOUND', 'Install');
		} else {
			$dbCheckResult['flag'] = true;

			return $dbCheckResult;
		}
		$dbCheckResult['flag'] = false;
		$dbCheckResult['error_msg'] = $errorMsg;
		$dbCheckResult['error_msg_info'] = $errorMsgInfo;

		return $dbCheckResult;
	}

	/**
	 * Function checks the mongo database connection.
	 *
	 * @param array $config
	 *
	 * @return array
	 */
	public static function checkMongoConnection(array $config): array
	{
		$mongoType = $config['mongo_type'] ?? 'mongodb';
		$mongoServer = $config['mongo_server'] ?? '';
		$mongoUsername = $config['mongo_username'] ?? '';
		$mongoPassword = $config['mongo_password'] ?? '';
		$mongoName = $config['mongo_name'] ?? '';
		$mongoPort = $config['mongo_port'] ?? '';

		if (!$mongoServer || !$mongoUsername || !$mongoPassword || !$mongoName || !$mongoPort) {
			return ['flag' => false];
		}

		// Checking for database connection parameters
		$openException = '';
		try {
			$db = new Mongo([
				'dsn' => sprintf(
					'%s://%s:%s@%s:%d/%s',
					$mongoType,
					urlencode($mongoUsername),
					urlencode($mongoPassword),
					$mongoServer,
					$mongoPort,
					$mongoName
				),
				'tablePrefix' => 'yf_',
			]);
			$db->open();
		} catch (\Throwable $e) {
			$openException = $e->getMessage();
			Log::error($e->__toString(), 'Install');
		}

		$mongoCheckResult = [];
		if ($openException) {
			$errorMsg = Language::translate('ERR_DATABASE_CONNECTION_FAILED', 'Install') . '. '
				. Language::translate('ERR_INVALID_MONGO_PARAMETERS', 'Install');
			$errorMsgInfo = Language::translate('MSG_LIST_REASONS', 'Install') . ':<br />
					-  ' . Language::translate('MSG_DB_PARAMETERS_INVALID', 'Install') . '
					<br />-  ' . Language::translate('MSG_DB_USER_NOT_AUTHORIZED', 'Install');
			$errorMsgInfo .= "<br /><br />$openException";
		} else {
			$mongoCheckResult['flag'] = true;

			return $mongoCheckResult;
		}
		$mongoCheckResult['flag'] = false;
		$mongoCheckResult['error_msg'] = $errorMsg;
		$mongoCheckResult['error_msg_info'] = $errorMsgInfo;

		return $mongoCheckResult;
	}

	/**
	 * Create config files.
	 *
	 * @throws IllegalValue
	 * @throws AppException
	 *
	 * @return void
	 */
	public function createConfigFiles(): void
	{
		$skip = ['main', 'db', ...self::CONFIG_FILES_TO_SKIP];
		foreach (array_diff(ConfigFile::TYPES, $skip) as $type) {
			(new ConfigFile($type))->create();
		}

		$dirPath = ROOT_DIRECTORY . \DIRECTORY_SEPARATOR . 'config' . \DIRECTORY_SEPARATOR . 'Modules';
		if (!is_dir($dirPath)) {
			mkdir($dirPath);
		}

		$dataReader = (new Query())
			->select(['name'])
			->from('vtiger_tab')
			->createCommand()
			->query();

		while ($moduleName = $dataReader->readColumn(0)) {
			$filePath = 'modules' . \DIRECTORY_SEPARATOR . $moduleName . \DIRECTORY_SEPARATOR . 'ConfigTemplate.php';
			if (file_exists($filePath)) {
				(new ConfigFile('module', $moduleName))->create();
			}
		}

		$path = ROOT_DIRECTORY . \DIRECTORY_SEPARATOR . 'config' . \DIRECTORY_SEPARATOR . 'Components' . \DIRECTORY_SEPARATOR . 'ConfigTemplates.php';
		$componentsData = require "$path";

		foreach ($componentsData as $component => $data) {
			(new ConfigFile('component', $component))->create();
		}
	}

	/**
	 * Update CRM version.
	 *
	 * @throws Exception
	 */
	public function updateVersion(): void
	{
		Db::getInstance()->createCommand()
			->update('vtiger_version', [
				'current_version' => Version::get(),
				'old_version' => Version::get(),
			])->execute();
	}

	/**
	 * Update admin user.
	 *
	 * @param string $username
	 * @param string $password
	 * @param string $email
	 * @param string $firstName
	 * @param string $lastName
	 * @param string $dateFormat
	 *
	 * @throws Exception
	 * @throws \ReflectionException
	 *
	 * @return void
	 */
	public function prepareAdminUser(
		string $username,
		string $password,
		string $email,
		string $firstName,
		string $lastName,
		string $dateFormat,
	): void {
		Db::getInstance()->createCommand()
			->update('vtiger_users', [
				'user_name' => $username,
				'date_format' => $dateFormat,
				'time_zone' => Config::main('default_timezone'),
				'first_name' => $firstName,
				'last_name' => $lastName,
				'email1' => $email,
				'accesskey' => Encryption::generatePassword(20, 'lbn'),
				'language' => Config::main('default_language'),
				'force_password_change' => 0,
			])->execute();

		$userRecordModel = \Users_Record_Model::getInstanceById(1, 'Users');
		$userRecordModel->set('changeUserPassword', true);
		$userRecordModel->set('user_password', $password);
		$userRecordModel->save();

		require_once 'app/UserPrivilegesFile.php';
		UserPrivilegesFile::createUserPrivilegesfile(1);
	}

	/**
	 * Recalculate sharing rules for users.
	 *
	 * @throws NoPermitted
	 */
	public function recalculateSharingRules(): void
	{
		require_once 'include/utils/CommonUtils.php';
		require_once 'include/fields/DateTimeField.php';
		require_once 'include/fields/DateTimeRange.php';
		require_once 'include/fields/CurrencyField.php';
		require_once 'include/CRMEntity.php';
		require_once 'modules/Vtiger/CRMEntity.php';
		require_once 'include/runtime/Cache.php';
		require_once 'modules/Vtiger/helpers/Util.php';
		require_once 'modules/PickList/DependentPickListUtils.php';
		require_once 'modules/Users/Users.php';
		require_once 'include/Webservices/Utils.php';
		UserPrivilegesFile::recalculateAll();
		Cache::clear();
		Cache::clearOpcache();
		Module::createModuleMetaFile();
	}

	/**
	 * Adjust modules to business profile.
	 *
	 * @param int[] $modulePackages
	 *
	 * @return void
	 */
	public static function adjustModules(array $modulePackages): void
	{
		$moduleManagerModel = new \Settings_ModuleManager_Module_Model();

		$packages = Module::getPackages();
		foreach ($packages as $package) {
			$packageId = (int) $package['id'];

			foreach (Module::getByPackage($packageId) as $module) {
				$moduleName = $module['name'];
				if (\in_array($packageId, $modulePackages)) {
					if (!Module::isModuleActive($moduleName)) {
						$moduleManagerModel->enableModule($moduleName);
					}

					continue;
				}

				$moduleManagerModel->disableModule($moduleName);
			}
		}
	}

	/**
	 * Set migrator.
	 */
	private function setMigrator(): void
	{
		if (!$this->migrator && (new ConfigFile('db'))->exists()) {
			$this->migrator = new Migrator();
		}
	}
}
