<?php

/**
 * Mail servers module model class.
 *
 * @copyright YetiForce S.A.
 * @license   YetiForce Public License 7.0 (licenses/LicenseEN.txt or yetiforce.com)
 * @author    Radosław Skrzypczak <r.skrzypczak@yetiforce.com>
 */
class Settings_OSSMailScanner_Module_Model extends Settings_Vtiger_Module_Model
{
	/** {@inheritdoc} */
	public $name = 'OSSMailScanner';
	/** {@inheritdoc} */
	public $baseTable = 's_#__mail_account';
	/** {@inheritdoc} */
	public $baseIndex = 'id';

	/** {@inheritdoc} */
	public $listFields = ['login' => 'FL_MAILBOX_LOGIN', 'status' => 'FL_MAILBOX_STATUS',  'scanner_actions' => 'FL_SCANNER_ACTIONS', 'owner_id' => 'FL_OWNER', 'last_login' => 'FL_LAST_LOGIN', 'logs' => 'FL_LOGS'];

	/** @var string[] Fields name for edit view */
	public $editFields = [
		'name', 'mail_server_id', 'login', 'password', 'status', 'scanner_actions', 'scanner_fields', 'owner_id'
	];

	/**
	 * Function to get the url for Create view of the module.
	 *
	 * @return string - url
	 */
	public function getCreateRecordUrl()
	{
		return 'index.php?module=' . $this->getName() . '&parent=Settings&view=Edit';
	}

	/**
	 * Function to get the url for edit view of the module.
	 *
	 * @return string - url
	 */
	public function getEditViewUrl()
	{
		return 'index.php?module=' . $this->getName() . '&parent=Settings&view=Edit';
	}

	/**
	 * Function to get the url for default view of the module.
	 *
	 * @return string URL
	 */
	public function getDefaultUrl()
	{
		return 'index.php?module=' . $this->getName() . '&parent=Settings&view=List';
	}

	/**
	 * Function verifies if it is possible to sort by given field in list view.
	 *
	 * @param string $fieldName
	 *
	 * @return bool
	 */
	public function isSortByName($fieldName)
	{
		return \in_array($fieldName, ['name', 'status', 'login']);
	}

	/** {@inheritdoc} */
	public function getListFields(): array
	{
		if (!isset($this->listFieldModels)) {
			$fields = $this->listFields;
			$fieldObjects = [];
			foreach ($fields as $fieldName => $fieldLabel) {
				$fieldObject = $this->getFieldInstanceByName($fieldName);
				if (!$this->isSortByName($fieldName)) {
					$fieldObject->set('sort', true);
				}
				$fieldObjects[$fieldName] = $fieldObject;
			}
			$this->listFieldModels = $fieldObjects;
		}
		return $this->listFieldModels;
	}

	/**
	 * Editable fields.
	 *
	 * @return array
	 */
	public function getEditableFields(): array
	{
		return array_merge($this->editFields, ['folders']);
	}

	public function getFormFields()
	{
		return array_flip($this->editFields);
	}

	/**
	 * Get block icon.
	 *
	 * @param string $name
	 *
	 * @return string
	 */
	public function getBlockIcon($name): string
	{
		return '';
	}

	/**
	 * Get fields instance by name.
	 *
	 * @param string $name
	 *
	 * @return Vtiger_Field_Model
	 */
	public function getFieldInstanceByName($name)
	{
		$params = [];
		switch ($name) {
			case 'name':
				$params = [
					'name' => $name,
					'label' => 'FL_MAILBOX_SUBJECT',
					'uitype' => 1,
					'typeofdata' => 'V~M',
					'maximumlength' => '100',
					'purifyType' => \App\Purifier::TEXT,
					'blockLabel' => 'BL_BASE',
					'table' => $this->getBaseTable()
				];
				break;
			case 'login':
				$params = [
					'name' => $name,
					'label' => 'FL_MAILBOX_LOGIN',
					'uitype' => 1,
					'typeofdata' => 'V~M',
					'maximumlength' => '255',
					'purifyType' => \App\Purifier::ALNUM,
					'blockLabel' => 'BL_BASE',
					'table' => $this->getBaseTable()
				];
				break;
			case 'status':
				$params = [
					'name' => $name,
					'label' => 'FL_MAILBOX_STATUS',
					'uitype' => 16,
					'typeofdata' => 'C~O',
					'maximumlength' => '255',
					'purifyType' => \App\Purifier::BOOL,
					'blockLabel' => 'BL_BASE',
					'table' => $this->getBaseTable(),
					'defaultvalue' => 'PLL_ACTIVE',
					'picklistValues' => [
						'PLL_ACTIVE' => \App\Language::translate('PLL_ACTIVE', $this->getName(true)),
						'PLL_INACTIVE' => \App\Language::translate('PLL_INACTIVE', $this->getName(true)),
						'PLL_LOCKED' => \App\Language::translate('PLL_LOCKED', $this->getName(true))
					]
				];
				break;
			case 'password':
				$params = [
					'name' => $name,
					'label' => 'FL_MAILBOX_PASSWORD',
					'uitype' => 99,
					'typeofdata' => 'V~O',
					'maximumlength' => '255',
					'purifyType' => 'raw',
					'blockLabel' => 'BL_IMAP',
					'fromOutsideList' => true,
					'table' => $this->getBaseTable()
				];
				break;
			case 'last_login':
				$params = [
					'name' => $name,
					'label' => 'FL_LAST_LOGIN',
					'uitype' => 5,
					'typeofdata' => 'V~O',
					'maximumlength' => '255',
					'purifyType' => 'raw',
					'blockLabel' => 'BL_IMAP',
					'fromOutsideList' => true,
					'table' => $this->getBaseTable()
				];
				break;
			case 'mail_server_id':
				$params = [
					'name' => $name,
					'label' => 'FL_MAILBOX_SERVER',
					'uitype' => 16,
					'typeofdata' => 'I~M',
					'maximumlength' => '128',
					'purifyType' => \App\Purifier::INTEGER,
					'blockLabel' => 'BL_SMTP',
					'table' => $this->getBaseTable()
				];
				$params['picklistValues'] = array_map(fn ($server) => $server['name'], \App\Mail\Server::getAll(\App\Mail\Server::STATUS_ACTIVE));
				break;
			case 'owner_id':
				$params = [
					'name' => $name,
					'label' => 'FL_OWNER',
					'uitype' => 53,
					'typeofdata' => 'I~O',
					'maximumlength' => '0,65535',
					'purifyType' => \App\Purifier::INTEGER,
					'blockLabel' => 'BL_IMAP',
					'table' => $this->getBaseTable()
				];
				break;
			case 'scanner_actions':
				$params = [
					'name' => $name,
					'label' => 'FL_SCANNER_ACTIONS',
					'uitype' => 322,
					'typeofdata' => 'V~O',
					'maximumlength' => '255',
					'purifyType' => \App\Purifier::STANDARD,
					'blockLabel' => 'BL_IMAP',
					'tdClass' => 'text-wrap',
					'defaultvalue' => '',
					'table' => $this->getBaseTable()
				];
				break;
			case 'scanner_fields':
				$params = [
					'name' => $name,
					'label' => 'FL_SCANNER_FIELDS',
					'uitype' => 323,
					'typeofdata' => 'V~O',
					'maximumlength' => '65535',
					'purifyType' => \App\Purifier::TEXT,
					'blockLabel' => 'BL_MAIL_SCANNER',
					'defaultvalue' => '',
					'table' => $this->getBaseTable()
				];
				break;
			case 'folders':
				$params = [
					'name' => $name,
					'label' => 'FL_MAILBOX_FOLDERS',
					'uitype' => 89,
					'typeofdata' => 'V~O',
					'maximumlength' => '65535',
					'purifyType' => \App\Purifier::TEXT,
					'blockLabel' => 'BL_IMAP',
					'defaultvalue' => '',
					'table' => $this->getBaseTable()
				];
				break;
			case 'logs':
				$params = [
					'name' => $name,
					'label' => 'FL_LOGS',
					'uitype' => 21,
					'typeofdata' => 'V~O',
					'maximumlength' => '65535',
					'purifyType' => \App\Purifier::STANDARD,
					'blockLabel' => 'BL_SMTP',
					'defaultvalue' => '',
					'table' => $this->getBaseTable()
				];
				break;
			default:
				break;
		}

		return $params ? \Vtiger_Field_Model::init($this->getName(true), $params, $name) : null;
	}
}
