<?php
/**
 * Module builder file.
 *
 * @package Cli
 *
 * @copyright YetiForce S.A.
 * @license   YetiForce Public License 7.0 (licenses/LicenseEN.txt or yetiforce.com)
 * @author    Klaudia Łozowska <k.lozowska@yetiforce.com>
 */

namespace App\Cli;

use App\Db\Query;
use vtlib\Field;
use vtlib\Module;

/**
 * Module builder cli class.
 */
class ModuleBuilder extends Base
{
	/** {@inheritdoc} */
	public string $moduleName = 'System';

	/** {@inheritdoc} */
	protected array $methods = [
		'createFiles' => 'Create module files based on database entries',
	];

	/**
	 * Create module files based on database entries.
	 *
	 * @return void
	 */
	public function createFiles(): void
	{
		$dataReader = (new Query())
			->select(['vtiger_tab.*', 'vtiger_field.fieldid'])
			->from('vtiger_tab')
			->leftJoin('vtiger_field', 'vtiger_field.tabid = vtiger_tab.tabid AND vtiger_field.uitype = 2')
			->where('vtiger_tab.isentitytype = 1')
			->andWhere('vtiger_field.fieldid IS NOT NULL')
			->createCommand()
			->query();

		$modulesWithMissingFiles = 0;
		while ($moduleData = $dataReader->read()) {
			$moduleName = $moduleData['name'];
			if (!self::checkModuleFilesExist($moduleName)) {
				$this->climate->info("Generating files for module: $moduleName");
				++$modulesWithMissingFiles;

				$mainField = Field::getInstance($moduleData['fieldid']);
				try {
					Module::getInstance($moduleName)->createFiles($mainField);
				} catch (\Throwable $e) {
					$this->climate->bold($e->getMessage());
				}
			}
		}

		if (0 === $modulesWithMissingFiles) {
			$this->climate->bold('No modules to generate files for.');
		}

		$this->climate->lightYellow()->border('─', 200);

		if (!$this->climate->arguments->defined('action')) {
			$this->returnToActionList();
		}
	}

	/**
	 * Checks if module files exist.
	 *
	 * @param string $moduleName
	 *
	 * @return bool
	 */
	private static function checkModuleFilesExist(string $moduleName): bool
	{
		return file_exists("modules/$moduleName/$moduleName.php")
			|| file_exists("custom/modules/$moduleName/$moduleName.php");
	}
}
