<?php
/**
 * Headers controller file.
 *
 * @package Controller
 *
 * @copyright YetiForce S.A.
 * @license   YetiForce Public License 7.0 (licenses/LicenseEN.txt or yetiforce.com)
 * @author    Mariusz Krzaczkowski <m.krzaczkowski@yetiforce.com>
 * @author    Radosław Skrzypczak <r.skrzypczak@yetiforce.com>
 */

namespace App\Controller;

use App\Config;
use App\Encryption;
use App\RequestUtil;
use App\Session;
use Config\Main;
use Config\Security;

/**
 * Headers controller class.
 */
class Headers
{
	/**
	 * Default CSP header values.
	 *
	 * @var string[]
	 */
	public array $csp = [
		'default-src' => '\'self\' blob:',
		'img-src' => '\'self\' data:',
		'font-src' => '\'self\' data:',
		'script-src' => '\'self\' \'unsafe-inline\' blob:',
		'form-action' => '\'self\'',
		'frame-ancestors' => '\'self\'',
		'frame-src' => '\'self\' mailto: tel:',
		'style-src' => '\'self\' \'unsafe-inline\'',
		'connect-src' => '\'self\'',
	];

	/**
	 * Headers instance..
	 *
	 * @var self
	 */
	public static Headers $instance;

	/**
	 * Default header values.
	 *
	 * @var string[]
	 */
	protected array $headers = [
		'Access-Control-Allow-Methods' => 'GET, POST',
		'Access-Control-Allow-Origin' => '*',
		'Expires' => '-',
		'Last-Modified' => '-',
		'Pragma' => 'no-cache',
		'Cache-Control' => 'private, no-cache, no-store, must-revalidate, post-check=0, pre-check=0',
		'Content-Type' => 'text/html; charset=UTF-8',
		'Referrer-Policy' => 'no-referrer',
		'Expect-Ct' => 'enforce; max-age=3600',
		'X-Frame-Options' => 'sameorigin',
		'X-Content-Type-Options' => 'nosniff',
		'X-Robots-Tag' => 'none',
		'X-Permitted-Cross-Domain-Policies' => 'none',
	];

	/**
	 * Headers to delete.
	 *
	 * @var string[]
	 */
	protected array $headersToDelete = ['X-Powered-By', 'Server'];

	/**
	 * Construct, loads default headers depending on the browser and environment.
	 *
	 * @throws \ReflectionException
	 */
	public function __construct()
	{
		$browser = RequestUtil::getBrowserInfo();
		$this->headers['Expires'] = gmdate('D, d M Y H:i:s') . ' GMT';
		$this->headers['Last-Modified'] = gmdate('D, d M Y H:i:s') . ' GMT';
		if ($browser->ie) {
			$this->headers['X-Ua-Compatible'] = 'IE=11,edge';
			if ($browser->https) {
				$this->headers['Pragma'] = 'private';
				$this->headers['Cache-Control'] = 'private, must-revalidate';
			}
		}
		if ($browser->https) {
			$this->headers['Strict-Transport-Security'] = 'max-age=31536000; includeSubDomains; preload';
		}
		if (Config::security('cspHeaderActive')) {
			$this->loadCsp();
		}
		if ($keys = Config::security('hpkpKeysHeader')) {
			$this->headers['Public-Key-Pins'] = 'pin-sha256="' . implode('"; pin-sha256="', $keys) . '"; max-age=10000;';
		}
	}

	/**
	 * Get headers instance.
	 *
	 * @return self
	 */
	public static function getInstance(): self
	{
		if (isset(self::$instance)) {
			return self::$instance;
		}
		return self::$instance = new self();
	}

	/**
	 * Set header.
	 *
	 * @param string $key
	 * @param string $value
	 */
	public function setHeader(string $key, string $value): void
	{
		$this->headers[$key] = $value;
	}

	/**
	 * Send headers.
	 *
	 * @return void
	 * @throws \ReflectionException
	 */
	public function send(): void
	{
		if (headers_sent()) {
			return;
		}
		foreach ($this->getHeaders() as $value) {
			header($value);
		}
		foreach ($this->headersToDelete as $name) {
			header_remove($name);
		}
	}

	/**
	 * Get headers string.
	 *
	 * @return string[]
	 * @throws \ReflectionException
	 */
	public function getHeaders(): array
	{
		if (Config::security('cspHeaderActive')) {
			$this->headers['Content-Security-Policy'] = $this->getCspHeader();
		}
		$return = [];
		foreach ($this->headers as $name => $value) {
			$return[] = "$name: $value";
		}
		return $return;
	}

	/**
	 * Load CSP directive.
	 *
	 * @return void
	 */
	public function loadCsp(): void
	{
		if (Security::$generallyAllowedDomains) {
			$this->csp['default-src'] .= ' ' . implode(' ', Security::$generallyAllowedDomains);
		}
		if (class_exists('Config\Main') && Main::$regApiBaseUrl) {
			$this->csp['img-src'] .= ' ' . implode(' ', [Main::$regApiBaseUrl]);
		}
		if (Security::$allowedImageDomains) {
			$this->csp['img-src'] .= ' ' . implode(' ', Security::$allowedImageDomains);
		}
		if (Security::$allowedScriptDomains) {
			$this->csp['script-src'] .= ' ' . implode(' ', Security::$allowedScriptDomains);
		}
		if (Security::$allowedFormDomains) {
			$this->csp['form-action'] .= ' ' . implode(' ', Security::$allowedFormDomains);
		}
		if (Security::$allowedFrameDomains) {
			$this->csp['frame-ancestors'] .= ' ' . implode(' ', Security::$allowedFrameDomains);
		}
		if (Security::$allowedConnectDomains) {
			$this->csp['connect-src'] .= ' ' . implode(' ', Security::$allowedConnectDomains);
		}
		if (Security::$allowedStylesheetDomains) {
			$this->csp['style-src'] .= ' ' . implode(' ', Security::$allowedStylesheetDomains);
		}
		if (Security::$allowedFontDomains) {
			$this->csp['font-src'] .= ' ' . implode(' ', Security::$allowedFontDomains);
		}
		if (Security::$allowedDomainsLoadInFrame) {
			$this->csp['frame-src'] .= ' ' . implode(' ', Security::$allowedDomainsLoadInFrame);
		}
	}

	/**
	 * Get CSP headers string.
	 *
	 * @return string
	 */
	public function getCspHeader(): string
	{
		$scp = '';
		foreach ($this->csp as $key => $value) {
			$scp .= "$key $value; ";
		}
		return $scp;
	}

	/**
	 * Generate Content Security Policy token.
	 *
	 * @return void
	 */
	public static function generateCspToken(): void
	{
		Session::set('CSP_TOKEN', hash('sha256', Encryption::generatePassword()));
	}
}
