<?php
/**
 * Class for combine/extract field coordinate string
 *
 * @package App
 *
 * @copyright YetiForce S.A.
 * @license YetiForce Public License 7.0 (licenses/LicenseEN.txt or yetiforce.com)
 * @author Leszek Koziatek <l.koziatek@yetiforce.com>
 */
declare(strict_types=1);

namespace App\FieldCoordinatorTransformer;

use Vtiger_Field_Model;
use Vtiger_Module_Model;

/** QueryGeneratorFieldTransformer class */
final class QueryGeneratorFieldTransformer implements TransformerInterface
{
	public const INVENTORY_NAME = 'INVENTORY';

	/** Combine field coordinate from field relation */
	public static function combine(
		string $fieldName,
		?string $moduleName = null,
		?string $referenceField = null,
		bool $isInventory = false,
	): string {
		return implode(':', array_filter([
			$fieldName,
			$moduleName,
			$referenceField,
			$isInventory ? self::INVENTORY_NAME : null,
		], static fn ($element): bool => $element !== null));
	}

	/**
	 * Get field relation from field coordinate string
	 *
	 * @param string $fieldCoordinates - String in format %s:%s:%s:%s or shorter
	 * @return array{fieldName: string, moduleName: string, referenceField: ?string, advancedType: bool}
	 */
	public static function extract(string $fieldCoordinates): array
	{
		[$fieldName, $moduleName, $referenceField, $advancedType] = array_pad(explode(':', $fieldCoordinates), 4, null);

		return [
			'fieldName' => $fieldName,
			'moduleName' => $moduleName,
			'referenceField' => $referenceField,
			'advancedType' => $advancedType,
		];
	}

	/** Return if field is inventory field */
	public static function isInventory(string $fieldCoordinates): bool
	{
		$parts = self::extract($fieldCoordinates);

		return null === $parts['advancedType'] && self::INVENTORY_NAME === $parts['referenceField'];
	}
}
