<?php
/**
 * Mail account repository.
 *
 * @package \App\ModTracker
 *
 * @copyright YetiForce S.A.
 * @license   YetiForce Public License 7.0 (licenses/LicenseEN.txt or yetiforce.com)
 * @author    Radosław Skrzypczak <r.skrzypczak@yetiforce.com>
 */
declare(strict_types=1);

namespace App\Mail\Account\Repository;

use App\Db;
use App\Db\Query;
use App\Mail\Account\Entity\AccountPrivateEntity;
use App\Mail\Account\Entity\Enum\MailBoxType;
use App\Mail\Account\Entity\Enum\Status;
use App\User;

/**
 * Mail account repository class.
 */
class AccountPrivateRepository implements AccountRepositoryInterface
{
	public const TABLE_NAME = 's_#__mail_account';
	protected const AUTOLOGIN_TABLE_NAME = 'roundcube_users_autologin';
	protected const FOLDER_TABLE_NAME = 'u_#__mailscanner_folders';

	private readonly Db $db;

	public function __construct()
	{
		$this->db = Db::getInstance();
	}

	public function findByUserId(int $userId): ?AccountPrivateEntity
	{
		$data = (new Query())->from(self::TABLE_NAME)->where(['private' => 1, 'owner_id' => $userId])->one() ?: [];

		return $data ? (new AccountPrivateEntity())->setData(...$data) : null;
	}

	public function getByUserId(int $userId): AccountPrivateEntity
	{
		return $this->findByUserId($userId) ?? new AccountPrivateEntity();
	}

	public function findById(int $id): ?AccountPrivateEntity
	{
		$data = (new Query())->from(self::TABLE_NAME)->where(['id' => $id])->one() ?: [];

		return $data ? (new AccountPrivateEntity())->setData(...$data) : null;
	}

	public function getAll(): array
	{
		$response = [];
		$autologinRepository = new AutologinRepository();

		$dataReader = (new Query())->from(self::TABLE_NAME)
			->where(['private' => MailBoxType::PUBLIC->value])
			->createCommand()->query();
		while ($row = $dataReader->read()) {
			$users = $autologinRepository->findById($row['id']);
			$row['autologinUsers'] = $users;
			$response[$row['id']] = (new AccountPrivateEntity())->setData(...$row);
		}

		return $response;
	}

	public function getAllByStatus(Status $status, int $startFromId = 0): \Iterator
	{
		$query = (new Query())->from(self::TABLE_NAME)
			->where(['status' => $status->value])
			->orderBy(['id' => SORT_ASC]);
		if ($startFromId) {
			$query->andWhere(['>', 'id', $startFromId]);
		}

		$dataReader = $query->createCommand()->query();
		while ($row = $dataReader->read()) {
			$entity = (new AccountPrivateEntity())->setData(...$row);
			if (!$entity->isPrivate() || User::isExists($entity->getOwnerId())) {
				yield $entity;
			}
		}
		$dataReader->close();
	}

	public function update(AccountPrivateEntity $entry): string|int
	{
		$result = 0;
		if (!$data = $entry->getValuesForSave()) {
			return $result;
		}

		$transaction = $this->db->beginTransaction();
		try {
			$result = $this->db->createCommand()->update(self::TABLE_NAME, $data, ['id' => $entry->getId()])->execute();
			if (isset($data['folders'])) {
				$this->saveFolders($entry);
			}
			$transaction->commit();
		} catch (\Throwable $ex) {
			$transaction->rollBack();
			throw $ex;
		}

		return $result;
	}

	public function create(AccountPrivateEntity $entry): string|int
	{
		$result = 0;
		$data = $entry->getValuesForSave();
		$data['private'] = 1;

		$transaction = $this->db->beginTransaction();
		try {
			$result = $this->db->createCommand()->insert(self::TABLE_NAME, $data)->execute();
			$id = (int) $this->db->getLastInsertID(self::TABLE_NAME . '_id_seq');
			$entry->setId($id);
			if (isset($data['folders'])) {
				$this->saveFolders($entry);
			}
			$transaction->commit();
		} catch (\Throwable $ex) {
			$transaction->rollBack();
			throw $ex;
		}

		return $result;
	}

	public function save(AccountPrivateEntity $entry): string|int
	{
		if ($entry->getId()) {
			$result = $this->update($entry);
		} else {
			$result = $this->create($entry);
		}

		return $result;
	}

	/**
	 * Remove user mail account.
	 *
	 * @param int $id
	 * @param int $userId
	 *
	 * @return void
	 */
	public function dropPrivateMailboxByUserId(int $userId)
	{
		$query = (new Query())
			->select(['id'])
			->from(self::TABLE_NAME)
			->where(['owner_id' => $userId, 'private' => MailBoxType::PRIVATE->value]);

		$dataReader = $query->createCommand()->query();
		while ($id = $dataReader->readColumn(0)) {
			$this->db->createCommand()->delete(static::TABLE_NAME, ['id' => $id])->execute();
			$this->db->createCommand()->delete('roundcube_users', ['crm_ma_id' => $id])->execute();
		}
		$dataReader->close();
		\App\Cache::delete('Imap-getRoundCubeUsersEmails', 'All');
	}

	protected function saveFolders(AccountPrivateEntity $entry)
	{
		$id = $entry->getId();
		$current = $entry->getFolders();
		$previous = (new Query())->select(['name'])->from(static::FOLDER_TABLE_NAME)
			->where(['id' => $entry->getId()])->column() ?: [];

		if ($remove = array_diff($previous, $current)) {
			$this->db->createCommand()->delete(static::FOLDER_TABLE_NAME, ['id' => $id, 'name' => $remove])->execute();
		}
		if ($add = array_diff($current, $previous)) {
			$insert = [];
			foreach ($add as $folderName) {
				$insert[] = [$id, $folderName];
			}
			$this->db->createCommand()->batchInsert(static::FOLDER_TABLE_NAME, ['id', 'name'], $insert)->execute();
		}
	}
}
