<?php
/**
 * Google geocoding driver file to get coordinates.
 *
 * @package App
 *
 * @copyright YetiForce S.A.
 * @license   YetiForce Public License 7.0 (licenses/LicenseEN.txt or yetiforce.com)
 * @author    Mariusz Krzaczkowski <m.krzaczkowski@yetiforce.com>
 *
 * @see      https://developers.google.com/maps/documentation/geocoding/
 */

namespace App\Map\Coordinates;

/**
 * Google geocoding driver class to get coordinates.
 */
class GoogleGeocoding extends Base
{
	/**
	 * API URL.
	 */
	private const API_URL = 'https://maps.googleapis.com/maps/api/geocode/json';

	/**
	 * @inheritdoc
	 */
	protected string $label = 'LBL_COORDINATES_GOOGLE_GEOCODING';

	/**
	 * @inheritdoc
	 */
	protected string $docUrl = 'https://developers.google.com/maps/documentation/geocoding/';

	/**
	 * @inheritdoc
	 */
	protected array $formFields = [
		'api_key' => [
			'validator' => [['name' => 'AlphaNumeric']],
			'uitype' => 99,
			'label' => 'LBL_API_KEY',
			'purifyType' => \App\Purifier::ALNUM_EXTENDED,
			'maximumlength' => '200',
			'typeofdata' => 'V~M',
		],
	];

	/**
	 * @var string[] Address params to build address query.
	 */
	private const ADDRESS_PARAMS = ['street', 'city', 'county', 'state'];

	/** {@inheritdoc} */
	public function getCoordinates(array $addressInfo, bool $onlyOne = true): ?array
	{
		if (empty($addressInfo) || !\App\RequestUtil::isNetConnection() || empty($this->getConfig()['api_key'])) {
			return null;
		}
		$url = $this->getUrl($addressInfo);
		\App\Log::beginProfile("GET|GoogleGeocoding::getCoordinates|{$url}", __NAMESPACE__);
		$response = \App\RequestHttp::getClient()->get($url, [
			'http_errors' => false,
		]);
		\App\Log::endProfile("GET|GoogleGeocoding::getCoordinates|{$url}", __NAMESPACE__);

		$body = \App\Json::decode($response->getBody()->getContents());
		if (200 != $response->getStatusCode() || !empty($body['error_message'])) {
			\App\Log::warning(
				'Error: ' . $url . ' | ' . $response->getStatusCode() . ' ' . $response->getReasonPhrase(),
				__CLASS__
			);
			return [];
		}
		if (isset($body['status'], $body['results'][0])) {
			foreach ($body['results'] as $row) {
				$coordinate = [
					'lat' => $row['geometry']['location']['lat'],
					'lon' => $row['geometry']['location']['lng']
				];
				if ($onlyOne) {
					$coordinates = $coordinate;
					break;
				}
				$coordinates[] = $coordinate;
			}
		}
		return $coordinates ?? [];
	}

	/** {@inheritdoc} */
	public function getCoordinatesByValue(string $value): array
	{
		return $this->getCoordinates(['q' => $value]);
	}

	/**
	 * Get url to Geocoding API.
	 *
	 * @param array $addressInfo
	 *
	 * @return string
	 */
	private function getUrl(array $addressInfo): string
	{
		if (isset($addressInfo['q'])) {
			$address = $addressInfo['q'];
		} else {
			$address = '';
			foreach (self::ADDRESS_PARAMS as $value) {
				if ('' !== $addressInfo[$value]) {
					$address .= $addressInfo[$value] . ',';
				}
			}
		}
		$params = [
			'address' => rtrim($address, ','),
			'language' => \App\Language::getShortLanguageName(),
			'key' => $this->getConfig()['api_key'],
		];
		if (!empty($addressInfo['country'])) {
			$params['components'] = 'country:' . $addressInfo['country'];
		}
		return self::API_URL . '?' . http_build_query($params);
	}
}
