<?php
/**
 * Dependency Injection pattern main class
 *
 * @package App
 *
 * @copyright YetiForce S.A.
 * @license YetiForce Public License 7.0 (licenses/LicenseEN.txt or yetiforce.com)
 * @author Łukasz Krawczyk <l.krawczyk@yetiforce.com>
 */

declare(strict_types=1);

namespace App\Report\DependencyInjection;

use App\Report\Attribute\ServiceAggregator;
use App\Report\Resources\Config\Services;
use App\Report\Utils\ReflectionParameterUtil;
use ReflectionClass;
use ReflectionParameter;

/** Container class */
final class Container
{
	/** @var array<object> */
	private array $services = [];

	public function __construct()
	{
		foreach (Services::getDefinitions() as $definition) {
			$this->set($definition);
		}
	}

	/**
	 * @template T
	 * @var class-string<T> $class
	 * @return T|null
	 */
	public function get(string $class): ?object
	{
		return $this->services[$class] ?? $this->set($class);
	}

	/** Set class to DI pattern */
	private function set(string $class): object
	{
		$reflection = new ReflectionClass($alias = $class);

		if (true === $reflection->isInterface()) {
			$reflection = new ReflectionClass($class = Services::getDefinition($class));
		}

		$parameters = $reflection->getConstructor()?->getParameters() ?? [];

		$parameters = array_map(
			fn (ReflectionParameter $parameter): mixed => $this->resolveParameter($parameter),
			$parameters,
		);

		return $this->services[$alias] = new $class(...$parameters);
	}

	/** Resolve parameter */
	private function resolveParameter(ReflectionParameter $parameter): mixed
	{
		$type = $parameter->getType()->getName();

		$aggregator = ReflectionParameterUtil::getAttribute($parameter, ServiceAggregator::class);

		if ($aggregator instanceof ServiceAggregator) {
			return $this->resolveAggregator($aggregator);
		}

		return $this->get($type);
	}

	/** Resolve aggregator */
	private function resolveAggregator(ServiceAggregator $aggregator): iterable
	{
		$callback = static function (array $services, object $service) use ($aggregator): array {
			if ($service instanceof $aggregator->type) {
				$services[] = $service;
			}

			return $services;
		};

		return array_reduce($this->services, $callback, []);
	}
}
