<?php

/**
 * YetiForce shop file.
 * Modifying this file or functions that affect the footer appearance will violate the license terms!!!
 *
 * @package App
 *
 * @copyright YetiForce S.A.
 * @license   YetiForce Public License 7.0 (licenses/LicenseEN.txt or yetiforce.com)
 * @author    Mariusz Krzaczkowski <m.krzaczkowski@yetiforce.com>
 * @author    Radosław Skrzypczak <r.skrzypczak@yetiforce.com>
 * @author    Klaudia Łozowska <k.lozowska@yetiforce.com>
 */

namespace App\YetiForce;

use App\Cache;
use App\Json;
use App\Language;
use App\Log;
use App\Version;
use App\YetiForce\Shop\AbstractBaseProduct;
use Config\Main;

/**
 * YetiForce shop class.
 */
final class Shop extends AbstractBase
{
	/** @var string[] Premium icons. */
	public const PREMIUM_ICONS = [
		1 => 'yfi-premium color-red-600',
		2 => 'yfi-enterprise color-yellow-600',
		3 => 'yfi-partners color-grey-600',
	];

	/** @var array Product categories. */
	public const PRODUCT_CATEGORIES = [
		'All' => ['label' => 'LBL_CAT_ALL', 'icon' => 'yfi-all-shop'],
		'Extensions' => ['label' => 'LBL_CAT_ADDONS', 'icon' => 'yfi-adds-on-shop'],
		'Integrations' => ['label' => 'LBL_CAT_INTEGRATIONS', 'icon' => 'yfi-integration-shop'],
	];

	/**
	 * Product instance cache.
	 *
	 * @var array
	 */
	public static array $productCache = [];

	/**
	 * Get api URL.
	 *
	 * @return string
	 */
	public static function getApiUrl(): string
	{
		return Main::$regApiBaseUrl . '/store';
	}

	/**
	 * Get products.
	 *
	 * @return Shop\AbstractBaseProduct[]
	 */
	public function getProducts(): array
	{
		if (empty(self::$productCache)) {
			$this->load();
		}

		return self::$productCache;
	}

	/**
	 * Get variable payments.
	 *
	 * @return array
	 * @throws \ReflectionException
	 */
	public static function getVariablePayments(): array
	{
		return [
			'business' => 'paypal@yetiforce.com',
			'rm' => 2,
			'image_url' => 'https://public.yetiforce.com/shop/logo.png',
			'return' => Main::$site_URL . 'index.php?module=YetiForce&parent=Settings&view=Shop&status=success',
			'cancel_return' => Main::$site_URL . 'index.php?module=YetiForce&parent=Settings&view=Shop&status=fail',
			'custom' => Register::getInstanceKey() . '|' . Register::getCrmKey(),
		];
	}

	/**
	 * Verification of product.
	 *
	 * @param string $productName
	 *
	 * @return bool
	 */
	public static function check(string $productName): bool
	{
		$className = self::getProductClass($productName);
		$product = new $className($productName);
		return $product->getStatus();
	}

	/**
	 * Verification of product with a message.
	 *
	 * @param string $productName
	 *
	 * @return array
	 * @throws \Exception
	 */
	public static function checkWithMessage(string $productName): array
	{
		if (Cache::staticHas('Shop|checkWithMessage', $productName)) {
			return Cache::staticGet('Shop|checkWithMessage', $productName);
		}
		$status = $message = false;
		$productDetails = Register::getProduct($productName);
		if ($productDetails) {
			$interval = (new \DateTime('now', new \DateTimeZone('GMT')))->diff(new \DateTime($productDetails['expires_at'], new \DateTimeZone('GMT')));
			$status = $interval->invert && $interval->days > 0;
			if (!$status) {
				$message = 'LBL_SUBSCRIPTION_HAS_EXPIRED';
			}
		}

		Cache::staticSave('Shop|checkWithMessage', $productName, [$status, $message]);
		return [$status, $message];
	}

	/**
	 * Check alert to show for product.
	 *
	 * @param string $productName
	 *
	 * @return string
	 */
	public static function checkAlert(string $productName): string
	{
		$className = self::getProductClass($productName);
		$product = new $className($productName);
		return $product->getAlertMessage();
	}

	/**
	 * Get paypal URL.
	 * https://www.paypal.com/cgi-bin/webscr
	 * https://www.sandbox.paypal.com/cgi-bin/webscr.
	 *
	 * @return string
	 */
	public static function getPaypalUrl(): string
	{
		return 'https://www.paypal.com/cgi-bin/webscr';
	}

	/**
	 * Verify or show a message about invalid products.
	 *
	 * @param bool $getNames
	 *
	 * @return string
	 */
	public static function verify(bool $getNames = false): string
	{
		$names = [];
		foreach (Register::getProducts() as $row) {
			$productName = $row['product'];
			$className = self::getProductClass($productName);
			$product = new $className($productName);
			if ($product->isExpired()) {
				$names[$productName] = $productName;
				if (!$getNames) {
					$names[$productName] = Language::translate($productName, 'Settings::YetiForce');
				}
			}
		}

		return implode(', ', $names);
	}

	/**
	 * Get all available products.
	 *
	 * @return void
	 */
	public function load(): void
	{
		self::$productCache = [];
		$this->success = false;
		try {
			$client = new ApiClient();
			$client->send(self::getApiUrl() . '/' . Version::get() . '/products', 'GET');
			$this->error = $client->getError();
			if (!$this->error && 200 === $client->getStatusCode() && !Json::isEmpty($client->getResponseBody())) {
				$this->setProducts(Json::decode($client->getResponseBody()));
				$this->success = true;
			}
		} catch (\Throwable $e) {
			$this->success = false;
			$this->error = $e->getMessage();
			Log::error($e->getMessage(), __METHOD__);
		}
	}

	/**
	 * Get product by ID.
	 *
	 * @param string $productId
	 *
	 * @return void
	 */
	public function loadProduct(string $productId): void
	{
		$this->success = false;
		try {
			$client = new ApiClient();
			$client->send(self::getApiUrl() . '/' . Version::get() . "/products/$productId", 'GET');
			$this->error = $client->getError();
			if (!$this->error && 200 === $client->getStatusCode() && !Json::isEmpty($client->getResponseBody())) {
				$this->setProducts([Json::decode($client->getResponseBody())]);
				$this->success = true;
			}
		} catch (\Throwable $e) {
			$this->success = false;
			$this->error = $e->getMessage();
			Log::error($e->getMessage(), __METHOD__);
		}
	}

	/**
	 * Get products.
	 *
	 * @param string $name
	 * @param string $productId
	 *
	 * @return AbstractBaseProduct|null
	 */
	public static function getProduct(string $name, string $productId = ''): ?AbstractBaseProduct
	{
		if (empty(self::$productCache[$name])) {
			if ($productId) {
				(new self())->loadProduct($productId);
			} else {
				(new self())->load();
			}
		}

		return self::$productCache[$name] ?? null;
	}

	/**
	 * Get product class.
	 *
	 * @param string $name
	 *
	 * @return string
	 */
	private static function getProductClass(string $name): string
	{
		$className = '\\App\\YetiForce\\Shop\\Product\\' . $name;
		if (!class_exists($className)) {
			$className = '\\App\\YetiForce\\Shop\\Product\\YetiForceBase';
		}
		return $className;
	}

	/**
	 * Set Products to cache.
	 *
	 * @param array $products
	 *
	 * @return void
	 */
	private function setProducts(array $products): void
	{
		foreach ($products as $productData) {
			$name = $productData['name'] ?? '';
			$className = self::getProductClass($name);
			if (($product = $className::fromArray($productData)) && $product->isAvailable()) {
				self::$productCache[$product->getName()] = $product;
			}
		}
	}
}
