<?php

/**
 * ActivityStateModal view Class.
 *
 * @package   View
 *
 * @copyright YetiForce S.A.
 * @license   YetiForce Public License 7.0 (licenses/LicenseEN.txt or yetiforce.com)
 * @author    Radosław Skrzypczak <r.skrzypczak@yetiforce.com>
 * @author    Mariusz Krzaczkowski <m.krzaczkowski@yetiforce.com>
 * @author    Arkadiusz Adach <a.adach@yetiforce.com>
 */

use App\Exceptions\NoPermittedToRecord;

class Calendar_ActivityStateModal_View extends Vtiger_BasicModal_View
{
	/** @var Vtiger_Record_Model|null Record model.*/
	protected ?Vtiger_Record_Model $record;

	/**
	 * Get tpl path file.
	 *
	 * @return string
	 */
	protected function getTpl(): string
	{
		return 'ActivityStateModal.tpl';
	}

	/** {@inheritdoc} */
	public function checkPermission(App\Request $request): void
	{
		$this->record = $request->isEmpty('record', true)
			? null
			: Vtiger_Record_Model::getInstanceById($request->getInteger('record'), $request->getModule());

		if (!$this->record || !$this->record->isEditable()) {
			throw new NoPermittedToRecord('ERR_NO_PERMISSIONS_FOR_THE_RECORD', 406);
		}
	}

	/** {@inheritdoc} */
	public function process(App\Request $request): void
	{
		$moduleName = $request->getModule();

		$viewer = $this->getViewer($request);
		$viewer->assign('LINKS', $this->getLinks());
		$viewer->assign('RECORD', $this->record);
		$viewer->assign('TIME_POSTPONE', \App\Config::module('Calendar', 'timePostponeIntervals', []));
		$viewer->assign('SCRIPTS', $this->getScripts($request));
		$viewer->view($this->getTpl(), $moduleName);
	}

	/** {@inheritdoc} */
	public function getScripts(App\Request $request): array
	{
		return $this->checkAndConvertJsScripts([
			'modules.' . $request->getModule() . '.resources.ActivityStateModal'
		]);
	}

	/**
	 * Gets links.
	 *
	 * @return array
	 */
	public function getLinks(): array
	{
		$links = [];

		if ($this->record->isEditable() && \App\Mail::checkInternalMailClient()) {
			$linkDataUrl = "index.php?module="
				. $this->record->getModuleName()
				. "&view=SendInvitationModal&record="
				. $this->record->getId();

			$links[] = Vtiger_Link_Model::getInstanceFromValues([
				'linklabel' => 'LBL_SEND_CALENDAR',
				'linkdata' => ['url' => $linkDataUrl],
				'linkicon' => 'yfi-send-invitation',
				'linkclass' => 'btn-light btn-sm js-show-modal',
			]);
		}

		if ($this->record->isViewable()) {
			$links[] = Vtiger_Link_Model::getInstanceFromValues([
				'linklabel' => 'LBL_SHOW_COMPLETE_DETAILS',
				'linkurl' => $this->record->getDetailViewUrl(),
				'linkicon' => 'fas fa-th-list',
				'linkclass' => 'btn-sm btn-light',
			]);
		}

		if ($this->record->isEditable()) {
			$links[] = Vtiger_Link_Model::getInstanceFromValues([
				'linklabel' => 'LBL_EDIT',
				'linkurl' => $this->record->getEditViewUrl(),
				'linkicon' => 'yfi yfi-full-editing-view',
				'linkclass' => 'btn-sm btn-light',
			]);
		}

		return $links;
	}
}
