<?php
/**
 * Cron task to update coordinates in records.
 *
 * @package Cron
 *
 * @copyright YetiForce S.A.
 * @license YetiForce Public License 7.0 (licenses/LicenseEN.txt or yetiforce.com)
 * @author Tomasz Kur <t.kur@yetiforce.com>
 * @author Mariusz Krzaczkowski <m.krzaczkowski@yetiforce.com>
 */

use App\Config;
use App\Db;
use App\Db\Query;
use App\Json;
use App\Map\Coordinates;

/**
 * OpenStreetMap_UpdaterRecordsCoordinates_Cron class.
 */
class OpenStreetMap_UpdaterRecordsCoordinates_Cron extends \App\CronHandler
{
	/** {@inheritdoc} */
	public function process()
	{
		$coordinatesConnector = Coordinates::getActiveInstance();
		if (null === $coordinatesConnector) {
			$this->addTaskLog("Nothing to process: no active coordinates driver");
			return;
		}

		$limit = Config::module('OpenStreetMap', 'cronMaxUpdatedAddresses', 100);

		$dataReader = (new Query())->from(Coordinates::UPDATER_TABLE_NAME)
			->limit($limit)
			->createCommand()
			->query();

		while ($row = $dataReader->read()) {
			$typeAddress = $row['type'];
			$recordId = (int) $row['crmid'];

			$coordinate = $coordinatesConnector->getCoordinates(Json::decode($row['address']));

			if (empty($coordinate)) {
				$this->delete($recordId, $typeAddress);

				continue;
			}

			Coordinates::saveCoordinates($recordId, $typeAddress, $coordinate);

			$this->delete($recordId, $typeAddress);

			if ($this->checkTimeout()) {
				break;
			}
		}

		$dataReader->close();
	}

	/**
	 * Deletes data used in looking for coordinates.
	 *
	 * @param int    $recordId
	 * @param string $typeAddress
	 *
	 * @throws \yii\db\Exception
	 */
	private function delete(int $recordId, string $typeAddress): void
	{
		Db::getInstance()
			->createCommand()
			->delete(
				Coordinates::UPDATER_TABLE_NAME,
				[
					'crmid' => $recordId,
					'type' => $typeAddress
				]
			)->execute();
	}
}
