<?php
/**
 * Generate report xls file.
 *
 * @package   Actions
 *
 * @copyright YetiForce S.A.
 * @license   YetiForce Public License 7.0 (licenses/LicenseEN.txt or yetiforce.com)
 * @author    Lukasz Krawczyk <l.krawczyk@yetiforce.com>
 * @author    Adrian Kon <a.kon@yetiforce.com>
 */
declare(strict_types=1);

use App\DocumentBuilder\FileGenerator;
use App\DocumentBuilder\Generator\XlsxGenerator;
use App\DocumentBuilder\Integrations\DocumentModuleIntegration;
use App\Report\Provider\FieldsProvider;
use App\Report\Provider\ReportProvider;
use App\Report\Report;
use App\Report\Repository\TemplateRepository;
use App\Request;

/** Generate report class */
final class ReportTemplate_GenerateReport_Action extends \App\Controller\Action
{

	/** {@inheritdoc} */
	public function checkPermission(Request $request)
	{
		if (!$request->isEmpty('record', true) && !\App\Privilege::isPermitted($request->getModule(), 'DetailView', $request->getInteger('record'))) {
			throw new \App\Exceptions\NoPermittedToRecord('ERR_NO_PERMISSIONS_FOR_THE_RECORD', 406);
		}
	}

	/** {@inheritdoc} */
    public function process(Request $request): void
    {
        $response = new Vtiger_Response();

        try {
            $report = Report::get(TemplateRepository::class)->find($request->getInteger('record'));
            $fileGenerator = (new FileGenerator())->setFileGenerator(new XlsxGenerator());
            $fieldsProvider = Report::get(FieldsProvider::class);
            $dataProvider = Report::get(ReportProvider::class);

            $data = $dataProvider->provide($report->getId(), App\User::getCurrentUserId());
			$headers = $fieldsProvider->provide($report->getId(), count($data) > 0 ? $data[0] : []);

            $fileGenerator->addData($data)->generateFile(
                $report,
                $headers,
                $fieldsProvider->wrappedFields($report->getId()),
            );

            $filePath = $fileGenerator->getFilePath($fileGenerator->getReportFilename($report));

            $document = (new DocumentModuleIntegration())->createDocumentEntry($report, $filePath);

            $result = [
                'success' => true,
                'message' => sprintf('%s has been generated', $report->getName()),
                'file' => $document->getDownloadFileURL(),
            ];
        } catch (Exception $exception) {
            $result = [
                'success' => false,
                'message' => sprintf('%s could not be generated', $report->getName()),
            ];
        }

        $response->setResult($result);
        $response->emit();
    }
}
