<?php

/**
 * Settings map config action file.
 *
 * @package   Settings
 *
 * @copyright YetiForce S.A.
 * @license YetiForce Public License 7.0 (licenses/LicenseEN.txt or yetiforce.com)
 * @author Mariusz Krzaczkowski <m.krzaczkowski@yetiforce.com>
 */

use App\Exceptions\AppException;
use App\Exceptions\IllegalValue;
use App\Exceptions\NoPermitted;
use App\Map;
use App\Purifier;

/**
 * Settings map config action class.
 */
class Settings_Map_Config_Action extends Settings_Vtiger_Basic_Action
{
	/** {@inheritdoc} */
	public function __construct()
	{
		parent::__construct();
		$this->exposeMethod('setTileLayer');
		$this->exposeMethod('setCoordinate');
		$this->exposeMethod('setRouting');
		$this->exposeMethod('saveConfiguration');
	}

	/**
	 * Set tile layer provider url.
	 *
	 * @param App\Request $request
	 *
	 * @throws ReflectionException
	 * @throws Throwable
	 * @throws AppException
	 * @throws IllegalValue
	 * @throws NoPermitted
	 * @throws \yii\db\Exception
	 */
	public function setTileLayer(App\Request $request): void
	{
		$newProviderName = $request->getByType('value', \App\Purifier::STANDARD);
		$oldProviderDomains = \App\Map\Layer::getActiveInstance()?->getAssetsDomains();
		$newProviderDomains = \App\Map\Layer::getInstance($newProviderName)->getAssetsDomains();

		$securityValues = [
			'allowedImageDomains' => \App\Config::security('allowedImageDomains', []),
			'allowedScriptDomains' => \App\Config::security('allowedScriptDomains', []),
			'allowedConnectDomains' => \App\Config::security('allowedConnectDomains', []),
			'allowedStylesheetDomains' => \App\Config::security('allowedStylesheetDomains', []),
			'allowedFontDomains' => \App\Config::security('allowedFontDomains', []),
		];

		foreach ($securityValues as $key => $currentValues) {
			foreach ($oldProviderDomains[$key] ?? [] as $oldValue) {
				$subKey = array_search($oldValue, $currentValues);
				if (false !== $subKey) {
					unset($securityValues[$key][$subKey]);
				}
			}

			foreach ($newProviderDomains[$key] ?? [] as $newValue) {
				if (!\in_array($newValue, $securityValues[$key])) {
					$securityValues[$key][] = $newValue;
				}
			}
		}

		try {
			\App\Map\Layer::switchActive($newProviderName);

			$security = new \App\ConfigFile('security');
			foreach ($securityValues as $key => $values) {
				$security->set($key, array_values($values));
			}
			$security->create();
		} catch (\Throwable $th) {
			\App\Log::error('Error: ' . $th->getMessage(), __CLASS__);
			throw $th;
		}

		$result = ['success' => true, 'message' => \App\Language::translate('LBL_CHANGES_SAVED')];
		$response = new Vtiger_Response();
		$response->setResult($result);
		$response->emit();
	}

	/**
	 * Set coordinate provider.
	 *
	 * @param App\Request $request
	 *
	 * @throws Throwable
	 * @throws NoPermitted
	 * @throws \yii\db\Exception
	 */
	public function setCoordinate(App\Request $request): void
	{
		try {
			\App\Map\Coordinates::switchActive($request->getByType('value', \App\Purifier::STANDARD));
		} catch (\Throwable $th) {
			\App\Log::error('Error: ' . $th->getMessage(), __CLASS__);
			throw $th;
		}

		$result = ['success' => true, 'message' => \App\Language::translate('LBL_CHANGES_SAVED')];
		$response = new Vtiger_Response();
		$response->setResult($result);
		$response->emit();
	}

	/**
	 * Set routing provider.
	 *
	 * @param App\Request $request
	 *
	 * @throws Throwable
	 * @throws NoPermitted
	 * @throws \yii\db\Exception
	 */
	public function setRouting(App\Request $request): void
	{
		try {
			\App\Map\Routing::switchActive($request->getByType('value', \App\Purifier::STANDARD));
		} catch (\Throwable $th) {
			\App\Log::error('Error: ' . $th->getMessage(), __CLASS__);
			throw $th;
		}

		$result = ['success' => true, 'message' => \App\Language::translate('LBL_CHANGES_SAVED')];
		$response = new Vtiger_Response();
		$response->setResult($result);
		$response->emit();
	}

	/**
	 * Save configuration.
	 *
	 * @param App\Request $request
	 *
	 * @return void
	 * @throws IllegalValue
	 */
	public function saveConfiguration(App\Request $request): void
	{
		$connectorClass = match ($request->getByType('tab', \App\Purifier::STANDARD)) {
			'TileLayer' => Map\Layer::class,
			'Coordinates' => Map\Coordinates::class,
			'Routing' => Map\Routing::class,
			'default' => throw new IllegalValue('ERR_ILLEGAL_VALUE'),
		};

		$connector = new $connectorClass;
		$changeSet = [];
		$driverName = $request->getByType('driver', Purifier::STANDARD);
		$driver = $connector::getInstance($driverName);

		foreach ($driver->getFormFields() as $fieldName => $field) {
			if ($request->has($fieldName)) {
                $purifyType = $field->get('purifyType');
				$value = $request->getByType($fieldName, $purifyType);

				if ($fieldName === 'api_key') {
					$changeSet['api_key'] = \App\Encryption::getInstance()->encrypt($value);

					continue;
				}
				$changeSet['params'][$fieldName] = $value;
			}
		}

		if ($changeSet['params'] ?? null) {
			$changeSet['params'] = json_encode($changeSet['params']);
		}

		(new $connectorClass)->saveConfig($driverName, $changeSet);

		$result = ['success' => true, 'message' => \App\Language::translate('LBL_CHANGES_SAVED')];
		$response = new Vtiger_Response();
		$response->setResult($result);
		$response->emit();
	}
}
