<?php
/* +**********************************************************************************
 * The contents of this file are subject to the vtiger CRM Public License Version 1.1
 * ("License"); You may not use this file except in compliance with the License
 * The Original Code is:  vtiger CRM Open Source
 * The Initial Developer of the Original Code is vtiger.
 * Portions created by vtiger are Copyright (C) vtiger.
 * All Rights Reserved.
 * Contributor(s): YetiForce S.A.
 * ********************************************************************************** */

use App\Purifier;
use App\UserAuth;

class Users_Login_Action extends \App\Controller\Action
{
	use \App\Controller\ExposeMethod;

	/** {@inheritdoc} */
	public function __construct()
	{
		parent::__construct();
		$this->exposeMethod('mfa');
		$this->exposeMethod('auth');
		$this->exposeMethod('authorize');
		$this->exposeMethod('install');
		if ($nonce = \App\Session::get('CSP_TOKEN')) {
			$this->headers->csp['script-src'] .= " 'nonce-{$nonce}'";
		}
		$this->headers->csp['default-src'] = '\'self\'';
		$this->headers->csp['script-src'] = str_replace([
			' \'unsafe-inline\'', ' blob:',
		], '', $this->headers->csp['script-src']);
		$this->headers->csp['form-action'] = '\'self\'';
		$this->headers->csp['style-src'] = '\'self\'';
		$this->headers->csp['base-uri'] = '\'self\'';
		$this->headers->csp['object-src'] = '\'none\'';
	}

	/** {@inheritdoc} */
	public function loginRequired()
	{
		return false;
	}

	/** {@inheritdoc} */
	public function validateRequest(App\Request $request)
	{
		$request->validateWriteAccess('auth' === $request->getMode());
	}

	/** {@inheritdoc} */
	public function checkPermission(App\Request $request)
	{
		if (\App\Session::has('authenticated_user_id')) {
			throw new \App\Exceptions\NoPermitted('ERR_PERMISSION_DENIED', 403);
		}
	}

	/** {@inheritdoc} */
	public function process(App\Request $request)
	{
		$bfInstance = Settings_BruteForce_Module_Model::getCleanInstance();
		if ($bfInstance->isActive() && $bfInstance->isBlockedIp()) {
			$bfInstance->incAttempts();
			Users_Module_Model::getInstance('Users')->saveLoginHistory(strtolower($request->getByType('username', 'Text')), 'ERR_LOGIN_DESP_IP_BLOCK');
			header('location: index.php?module=Users&view=Login');
			return false;
		}

		if ($mode = $request->getMode()) {
			$this->invokeExposedMethod($mode, $request);
		} else {
			$this->login($request);
		}
	}

	/**
	 * Redirects the user for authorization.
	 *
	 * @param \App\Request $request
	 *
	 * @return void
	 */
	public function authorize(App\Request $request): void
	{
		try {
			$providerName = $request->getByType('provider', \App\Purifier::ALNUM);
			if (empty($providerName) || !($provider = UserAuth::getProviderByName($providerName)) || !$provider->isActive()
			|| !$provider instanceof \App\Authenticator\SSOProvider) {
				throw new \App\Exceptions\NoPermitted('ERR_PERMISSION_DENIED', 403);
			}
			$this->initDataFromRequest($request);
			$url = $provider->startAuthorization();
			header("Location: {$url}");
		} catch (\Throwable $th) {
			\App\Log::error($th->getMessage(), 'UserAuthentication');
			\App\Session::set('UserLoginMessage', \App\Language::translate('LBL_SSO_ERROR', 'Users'));
			\App\Session::set('UserLoginMessageType', 'error');
			$this->failedLogin($request, 'sso');
			header('Location: index.php');
		}
	}

	/**
	 * Login using SSO. This is called after the user is redirected back from the SSO provider.
	 *
	 * @param App\Request $request
	 *
	 * @return void
	 */
	public function auth(App\Request $request): void
	{
		try {
			$code = $request->getRaw('code');
			$state = $request->get('state');
			$this->initDataFromRequest($request);
			$providerName = $request->getByType('provider', \App\Purifier::STANDARD);
			if (empty($code) || empty($state) || empty($providerName) || !($provider = UserAuth::getProviderByName($providerName))
			|| !$provider->isActive() || !$provider instanceof \App\Authenticator\SSOProvider) {
				throw new \App\Exceptions\NoPermitted('ERR_PERMISSION_DENIED', 403);
			}

			if ($userId = $provider->finishAuthorization($code, $state)) {
				$provider->setUserModel(\App\User::getUserModel($userId));
				$this->afterLogin($request, $userId);
				$provider->logIn();
				$provider->postProcess();
				$this->redirectUser($userId);
			} else {
				throw new \App\Exceptions\Unauthorized("User not exists in: {$providerName}", 401);
			}
		} catch (\Throwable $th) {
			\App\Log::error($th->__toString(), 'UserAuthentication');
			\App\Session::set('UserLoginMessage', \App\Language::translate('LBL_SSO_ERROR', 'Users'));
			\App\Session::set('UserLoginMessageType', 'error');
			$this->failedLogin($request, 'sso');
			header('Location: index.php');
		}
	}

	/**
	 * MFA login.
	 *
	 * @param App\Request $request
	 *
	 * @return void
	 */
	public function mfa(App\Request $request): void
	{
		$userId = \App\Session::get('2faUserId');
		$userModel = \App\User::getUserModel($userId);
		$authenticator = $userModel->getAuthenticator();
		if (!$authenticator->isMFA()) {
			\App\Session::delete('2faUserId');
			\App\Session::delete('LoginAuthyMethod');
			$this->failedLogin($request, '2fa');
			header('location: index.php');
		} elseif ($authenticator->verifyMFA((string) $request->getByType('user_code', Purifier::DIGITS))) {
			$authenticator->logIn();
			$authenticator->postProcess();
			\App\Session::delete('2faUserId');
			\App\Session::delete('LoginAuthyMethod');
			$this->redirectUser();
		} else {
			\App\Session::set('UserLoginMessage', \App\Language::translate('LBL_2FA_WRONG_CODE', 'Users'));
			\App\Session::set('UserLoginMessageType', 'error');
			$this->failedLogin($request, '2fa');
		}
	}

	/**
	 * User login to the system.
	 *
	 * @param \App\Request $request
	 *
	 * @return void
	 */
	public function login(App\Request $request): void
	{
		$userName = $request->getByType('username', 'Text');
		$password = $request->getRaw('password');

		$this->initDataFromRequest($request);
		$userId = (int) \App\User::getUserIdByName($userName);
		$userModel = \App\User::getUserModel($userId);
		$authenticator = $userModel->getAuthenticator();
		if ($authenticator->verify($password, $userName)) {
			$authenticator->preProcess();
			$this->afterLogin($request, $userId);
			if ($authenticator->isMFA()) {
				\App\Session::set('LoginAuthyMethod', '2fa');
				\App\Session::set('2faUserId', $userModel->getId());
				if (\App\Session::has('UserLoginMessage')) {
					\App\Session::delete('UserLoginMessage');
				}
				header('location: index.php');
			} else {
				$authenticator->logIn();
				$authenticator->postProcess();
				$this->redirectUser($userId);
			}
		} else {
			\App\Session::set('UserLoginMessage', App\Language::translate('LBL_INVALID_USER_OR_PASSWORD', 'Users'));
			$this->failedLogin($request, 'login');
		}
	}

	/**
	 * Login after install.
	 *
	 * @param App\Request $request
	 *
	 * @return void
	 */
	public function install(App\Request $request)
	{
		$this->cleanInstallationFiles();
		$this->login($request);
	}

	/**
	 * After login function.
	 *
	 * @param \App\Request $request
	 * @param string       $userName
	 * @param int          $userId
	 */
	public function afterLogin(App\Request $request, int $userId): void
	{
		\App\Controller\Headers::generateCspToken();
		if (\Config\Security::$loginSessionRegenerate) {
			\App\Session::regenerateId(true); // to overcome session id reuse.
		}

		\App\Session::set('app_unique_key', App\Config::main('application_unique_key'));
		$userModel = \App\User::getUserModel($userId);
		\App\Session::set('user_name', $userModel->getDetail('user_name') ?: '');
		\App\Session::set('full_user_name', $userModel->getName() ?: '');

		$eventHandler = new \App\EventHandler();
		$eventHandler->setRecordModel(Vtiger_Record_Model::getInstanceById($userId, 'Users'));
		$eventHandler->setParams(['userModel' => $userModel, 'password' => $request->getRaw('password')]);
		$eventHandler->setModuleName('Users');
		$eventHandler->trigger('UsersAfterLogin');
	}

	/**
	 * Initialize data from request.
	 *
	 * @param App\Request $request
	 *
	 * @return void
	 */
	public function initDataFromRequest(App\Request $request)
	{
		if ($request->has('fingerprint') && !$request->isEmpty('fingerprint')) {
			\App\Session::set('fingerprint', $request->getByType('fingerprint', Purifier::ALNUM2));
		}
		if ($request->has('loginLanguage') && App\Config::main('langInLoginView')) {
			\App\Session::set('language', $request->getByType('loginLanguage'));
		}
		if ($request->has('layout')) {
			\App\Session::set('layout', $request->getByType('layout'));
		}
		\App\Session::set('user_agent', \App\Request::_getServer('HTTP_USER_AGENT', ''));
	}

	/**
	 * Clean installation files.
	 */
	public function cleanInstallationFiles(): void
	{
		\vtlib\Functions::recurseDelete('install');
		\vtlib\Functions::recurseDelete('public_html/install');
		\vtlib\Functions::recurseDelete('tests');
		\vtlib\Functions::recurseDelete('.github');
		\vtlib\Functions::recurseDelete('.gitattributes');
		\vtlib\Functions::recurseDelete('.gitignore');
		\vtlib\Functions::recurseDelete('.travis.yml');
		\vtlib\Functions::recurseDelete('codecov.yml');
		\vtlib\Functions::recurseDelete('.gitlab-ci.yml');
		\vtlib\Functions::recurseDelete('.php_cs.dist');
		\vtlib\Functions::recurseDelete('.scrutinizer.yml');
		\vtlib\Functions::recurseDelete('.sensiolabs.yml');
		\vtlib\Functions::recurseDelete('.prettierrc.js');
		\vtlib\Functions::recurseDelete('.editorconfig');
		\vtlib\Functions::recurseDelete('.whitesource');
		\vtlib\Functions::recurseDelete('whitesource.config.json');
		\vtlib\Functions::recurseDelete('jsconfig.json');
		\vtlib\Functions::recurseDelete('sonar-project.properties');
		\vtlib\Functions::recurseDelete('docker-compose.yml');
		\vtlib\Functions::recurseDelete('Dockerfile');
		\vtlib\Functions::recurseDelete('crowdin.yml');
	}

	/**
	 * Failed login function.
	 *
	 * @param \App\Request $request
	 * @param string       $type
	 */
	public function failedLogin(App\Request $request, string $type): void
	{
		$status = ['2fa' => 'ERR_WRONG_2FA_CODE', 'sso' => 'ERR_SSO_LOGIN', 'login' => 'Failed login'][$type] ?? 'Failed login';
		$bfInstance = Settings_BruteForce_Module_Model::getCleanInstance();
		if ($bfInstance->isActive()) {
			$bfInstance->updateBlockedIp();
			if ($bfInstance->isBlockedIp()) {
				$bfInstance->sendNotificationEmail();
				\App\Session::set('UserLoginMessage', App\Language::translate('LBL_TOO_MANY_FAILED_LOGIN_ATTEMPTS', 'Users'));
				\App\Session::set('UserLoginMessageType', 'error');
				$status = ['2fa' => 'ERR_2FA_IP_BLOCK', 'sso' => 'ERR_SSO_IP_BLOCK', 'login' => 'ERR_LOGIN_IP_BLOCK'][$type] ?? 'ERR_LOGIN_IP_BLOCK';
			}
		}
		$userName = $request->getRaw('username');
		if (!$userName) {
			$userName = \App\Session::get('user_name');
		}
		Users_Module_Model::getInstance('Users')->saveLoginHistory(Purifier::encodeHtml($userName ?? ''), $status);
		header('location: index.php?module=Users&view=Login');
	}

	/**
	 * Redirect the user to a different page.
	 *
	 * @param int $userId
	 */
	private function redirectUser(int $userId = 0): void
	{
		if ($param = ($_SESSION['return_params'] ?? false)) {
			unset($_SESSION['return_params']);
			header('location: index.php?' . $param);
		} elseif (App\Config::performance('SHOW_ADMIN_PANEL') && $userId && \App\User::getUserModel($userId)->isAdmin()) {
			header('location: index.php?module=Vtiger&parent=Settings&view=Index');
		} else {
			header('location: index.php');
		}
	}
}
