<?php

/**
 * User mailbox run test action file.
 *
 * @copyright YetiForce S.A.
 * @license   YetiForce Public License 7.0 (licenses/LicenseEN.txt or yetiforce.com)
 * @author    Radosław Skrzypczak <r.skrzypczak@yetiforce.com>
 */

use App\Request;
use Webklex\PHPIMAP\Exceptions\AuthFailedException;
use Webklex\PHPIMAP\Exceptions\ConnectionFailedException;
use Webklex\PHPIMAP\Exceptions\ImapBadRequestException;
use Webklex\PHPIMAP\Exceptions\ImapServerErrorException;
use Webklex\PHPIMAP\Exceptions\ResponseException;
use Webklex\PHPIMAP\Exceptions\RuntimeException;

/**
 * User mailbox run test action class.
 */
class Users_RunTest_Action extends Users_Mailbox_Action
{
	/** {@inheritdoc} */
	public function checkPermission(App\Request $request)
	{
		$recordId = $request->getInteger('record');
		$repository = new \App\Mail\Account\Repository\AccountPrivateRepository();
		$accountEntity = $repository->findByUserId($recordId);
		$userModel = \App\User::getCurrentUserModel();

		if ((int) $userModel->getId() !== $recordId || ($accountEntity && !$accountEntity->isPrivate())) {
			throw new \App\Exceptions\NoPermittedToRecord('ERR_NO_PERMISSIONS_FOR_THE_RECORD', 406);
		}
	}

	/** {@inheritDoc} */
	public function process(Request $request)
	{
		$moduleName = $request->getModule();
		$mailbox = $this->getModelFromRequest($request);
		$mailAccount = $mailbox->getAccount();

		try {
			if ($mailAccount->getServer()->isOAuth() && !$mailAccount->getRefreshToken()) {
				$message = \App\Language::translate('LBL_SAVE_BEFORE_VERIFYING_CONNECTION', $moduleName);
				$type = 'info';
				$result = false;
			} else {
				$imap = $mailAccount->openImap();
				$result = $imap->isConnected();
				$imap->disconnect();
				$message = \App\Language::translate('LBL_CONNECTION_VERIFIED_SUCCESSFULLY', $moduleName);
			}
		} catch (ConnectionFailedException|
			AuthFailedException|
			ImapBadRequestException|
			ImapServerErrorException|
			RuntimeException|
			ResponseException $th
		) {
			\App\Log::error($th->__toString());
			$message = $th->getMessage();
			$result = false;
		} catch (\Throwable $th) {
			\App\Log::error($th->__toString());
			$message = \App\Language::translate('LBL_ERROR', $moduleName);
			$result = false;
		}

		$response = new Vtiger_Response();
		$response->setResult(['result' => $result, 'message' => $message, 'type' => $type ?? null]);
		$response->emit();
	}
}
